;
; test_scired2.pro
;
; Last change: Thu Apr 28 15:42:56 2011
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

function compute_opd, lst, pos, baseline
  ;; lst = local sidereal time in seconds, double
  ;; pos = struct with RA,Dec of star 1,2, in hours,deg

  phi = -24.62794830d * !dpi/180.	;; latitude of paranal in rad

  lstrad = double(lst)/3600. * !dpi/12.
  HA = lstrad - pos.RA * !dpi/12.
  Dec= pos.Dec * !dpi/180.

  star_S = cos(Dec) * cos(HA) * sin(phi) - cos(phi) * sin(Dec)
  star_W = cos(Dec) * sin(HA)
  star_Z = cos(Dec) * cos(HA) * cos(phi) + sin(phi) * sin(Dec)

  ;;print, star_W, star_S, star_Z

  opd = baseline[0] * star_W $
      + baseline[1] * star_S $
      + baseline[2] * star_Z

  return, opd	;; in meters
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO test_scired2, basename, FILE=outfile

  if N_elements(basename) lt 1 then basename= 'PACMA.2008-04-01T04:'
  if not keyword_set(outfile) then outfile="scired2_testresults.tex"

  file = file_search(basename+'*LEVEL3.fits')
  if file[0] eq '' then begin
     print,"No file found matching "+basename+'*_LEVEL3.fits'

     infile= file_search(basename+"*_LEVEL2.fits")
     infile= infile[0]
     if infile eq '' then begin
        print,"No LEVEL2 file found - I give up!"
        return
     endif
  endif else begin
     print,"files: ",file
     print,"Reading ",file[0]
     lvl3 = fbt_open(file[0])
     infile = (lvl3.fileobj->prihead())->getpar('PRO REC1 CAL1 NAME')
     print,'infile ',infile
  endelse

  openw,outunit,outfile,/get_lun
  printf,outunit,"%% scired2 testresults"
  printf,outunit,""
  printf,outunit,'{\bf Input data}:'
  printf,outunit,'\begin{verbatim}'
  printf,outunit,'Size     Creation date        Filename'
  close,outunit & free_lun,outunit

  spawn,'ls -l '+basename+"*_LEVEL2.fits | sed -e 's/^.*users //'"
  spawn,'ls -l '+basename+"*_LEVEL2.fits | sed -e 's/^.*users //' >> "+outfile

  openw,outunit,outfile,/get_lun,/APPEND
  printf,outunit,'\end{verbatim}'
  printf,outunit,''

  sof = strmid(infile,0, strlen(infile)-strlen(".fits"))
  sof = sof[0]
  printf,outunit,'{\bf Set of Frames}:'
  printf,outunit,'\verbatiminput{../testdata/sci/'+sof+'.sof}'
  printf,outunit,''
  ;;printf,outunit,'The LEVEL2 files are created by the recipe pacma\_scired1,'
  ;;printf,outunit,'the other files were created by hand.'
  printf,outunit,'For this test, we re-use the data created for testing pacma\_sciave and pacma\_scired1.'
  printf,outunit,'The simulation consists of two stars at the positions listed in the table below.'
  printf,outunit,'The positions are known perfectly, there is no difference between the positions'
  printf,outunit,'listed in the FITS header and the positions used to simulate the OPD and DOPD.'
  printf,outunit,''

  printf,outunit,'{\bf Simulation Parameters relevant for this test}:'
  printf,outunit,'\begin{longtable}{lll}'
  printf,outunit,'Parameter & Value & Comment\\'
  printf,outunit,'\hline\endhead'
  printf,outunit,'\hline\endfoot'

  rawfile = strmid(infile,0, strlen(infile)-strlen("_LEVEL2.fits"))+'.fits'
  print,"raw file ",rawfile
  lvl0= fbt_open(rawfile)

  prihead= lvl0.fileobj->prihead()
  print_header_keywords, outunit, prihead, '^HIERARCH ESO OCS TARG[1-9] [A-Z0-9_ ]+[A-Z0-9_]', SKIP=13

  head= lvl0.metrology_data->head()
  print_header_keywords, outunit, head, '^HIERARCH ESO SIM [A-Z0-9_ ]+[A-Z0-9_]'

  printf,outunit,'\hline'
  printf,outunit,'\end{longtable}'
  printf,outunit,'\bigskip'
  fbt_close,lvl0

  print_cmd_stuff, outunit, 'scired2', '../sciave/'+sof+'.out'

  if keyword_set(lvl3) then begin
     list_fbt_extensions,outunit,lvl3
     printf,outunit,'\smallskip'
     printf,outunit,''
     printf,outunit,'\clearpage'
     printf,outunit,'{\bf Output Data Contents}:\\'
     printf,outunit,'%'
     printf,outunit,'The important results are in the table with the misleading name \verb|SCIENCE_RESULT|, which has this structure:'
     printf,outunit,''
     printf,outunit,'\begin{center}'

     l3l3 = lvl3.science_result->readrows()
     dump_table_structure, outunit, l3l3
     printf,outunit,'\end{center}'
     printf,outunit,'%'
     printf,outunit,'Primary Header: The following table contains the QC-keywords in the primary header.'
                                ;printf,outunit,'For the keywords containing effective wavenumbers, the expected value and the difference'
                                ;printf,outunit,'are also computed.  Differences within acceptable limits are printed in {\color{green}green}.'
     printf,outunit,'\begin{longtable}{lccc}'
     printf,outunit,'Keyword & Value & Description\\'
     printf,outunit,'\hline\endhead'
     printf,outunit,'\hline\endfoot'

     prihead= lvl3.fileobj->prihead()
     print_header_keywords, outunit, prihead, '^HIERARCH ESO QC [A-Z0-9_ ]+[A-Z0-9_]', SKIP=13

     printf,outunit,'\hline'
     printf,outunit,'\end{longtable}'
     printf,outunit,'\bigskip'

  ;;;;;

     l3l3= l3l3[indgen(N_elements(l3l3)/50-1)*50]	;; every 50th point

     MJD0= prihead->getpar('MJD-OBS')
     lst = lst_from_date(MJD0 + l3l3.time_aVe[0]/86400d) * 3600d	;; seconds
     pos_A= { RA:  prihead->getpar('HIERARCH ESO OCS TARG1 RA'),$
              Dec: prihead->getpar('HIERARCH ESO OCS TARG1 DEC') }	;; hr, deg
     opd_A= compute_opd( lst, pos_A, [100.,0.,0.])

     pos_B= { RA:  prihead->getpar('HIERARCH ESO OCS TARG2 RA'),$
              Dec: prihead->getpar('HIERARCH ESO OCS TARG2 DEC') }	;; hr, deg
     opd_B= compute_opd( lst, pos_B, [100.,0.,0.])

     dopd = opd_B - opd_A

     ymax= max([ l3l3.total_lvl2, l3l3.dopd_model, l3l3.no_instr_calc, l3l3.with_instr_calc, l3l3.diff_delay],$
               min= ymin)
     xmax= max( l3sr.time_ave, min= xmin)
     xr= (xmax-xmin)/10	 &  xmin -= xr  & xmax += xr

     if !D.name eq 'PS' then device,/color,landscape=1,file='Level3_Level3.ps'

     tek_color
     plot, l3l3.time_ave[0], l3l3.total_lvl2[0], xtitle='TIME_AVE [sec]', xstyle=2,$
           yrange=[ymin,ymax],xrange=[xmin,xmax],psym=4,color=0
     oplot,l3l3.time_ave, l3l3.dopd_Model,	psym=1,	color=2
     oplot,l3l3.time_ave, l3l3.no_instr_calc,	psym=5, color=3
     oplot,l3l3.time_ave, l3l3.with_instr_calc, psym=6, color=4
     oplot,l3l3.time_ave, l3l3.diff_delay,	psym=7, color=5

     xx = xmin+xr/2
     ystep = (ymax-ymin)/20.
     plots, xx,ymax-ystep  , psym=4		& XYouts, xx+10,ymax-ystep,  "TOTAL_LVL2"
     plots, xx,ymax-ystep*2, psym=1,color=2	& XYouts, xx+10,ymax-ystep*2,"DOPD_MODEL",color=2
     plots, xx,ymax-ystep*3, psym=5,color=3	& XYouts, xx+10,ymax-ystep*3,"NO_INSTR_CALC",color=3
     plots, xx,ymax-ystep*4, psym=6,color=4	& XYouts, xx+10,ymax-ystep*4,"WITH_INSTR_CALC",color=4
     plots, xx,ymax-ystep*5, psym=7,color=5	& XYouts, xx+10,ymax-ystep*5,"DIFF_DELAY",color=5

     printf,outunit,'\clearpage'
     printf,outunit,''
     printf,outunit,"The figure shows the contents of table ``LEVEL3''."
     printf,outunit,'The column TOTAL\_LVL2 is plotted with black diamonds,'
     printf,outunit,'the column DOPD\_MODEL with red plus signs,'
     printf,outunit,'NO\_INSTR\_CALC with green triangles,'
     printf,outunit,'WITH\_INSTR\_CALC with blue squares,'
     printf,outunit,'and DIFF\_DELAY with cyan Xs.'

     printf,outunit,'\bigskip'
     printf,outunit,'\begin{center}'
     printf,outunit,'\includegraphics[height=0.5\textheight,angle=180]{../sciave/Level3_Level3.pdf}'
     printf,outunit,'\end{center}'

     if !d.name eq "PS" then begin
        device,/close
        print,'ps2pdf Level3_Level3.ps'
        spawn,'ps2pdf Level3_Level3.ps'
        device,/colo,/land,file='Baseline_angle.ps'
     endif

     ;;;;;

     RA = (pos_A.RA + pos_B.RA )/2.
     Dec= (pos_A.Dec+ pos_B.Dec)/2.
     ha = lst/3600.d - RA
     baseline = [0.,0.,0.]
     baseline[0]= prihead->getpar('HIERARCH ESO ISS CONF T2X')-prihead->getpar('HIERARCH ESO ISS CONF T1X')
     baseline[1]= prihead->getpar('HIERARCH ESO ISS CONF T2Y')-prihead->getpar('HIERARCH ESO ISS CONF T1Y')
     baseline[2]= prihead->getpar('HIERARCH ESO ISS CONF T2Z')-prihead->getpar('HIERARCH ESO ISS CONF T1Z')

     project_baseline, ha, Dec, baseline, base_len, base_pa, Debug=0
     idx= where(base_pa lt   0.)  &  if idx[0] ge 0 then base_pa[idx] += 360.
     idx= where(base_pa ge 180.)  &  if idx[0] ge 0 then base_pa[idx] -= 180.

     recipe_pa= l3l3.base_angle[0]
     idx= where(recipe_pa lt   0.)  &  if idx[0] ge 0 then recipe_pa[idx] += 360.
     idx= where(recipe_pa ge 180.)  &  if idx[0] ge 0 then recipe_pa[idx] -= 180.

     ymin= min( [recipe_pa,base_pa], max=ymax)
     print,"angle:",ymin,"...",ymax
     plot, l3l3.time_ave[0], recipe_pa, ysty=16, yrange= [ymin,ymax],$
           xtitle='TIME_AVE', ytitle='projected baseline angle [degrees]', psym=1
     oplot,l3l3.time_ave[0], base_pa, color=3

     ;;print, l3l3[0].base_angle[0], l3l3[N_elements(l3l3)-1].base_angle[0]
     ;;print, base_pa[0], base_pa[N_elements(base_pa)-1]

     printf,outunit,'\clearpage'
     printf,outunit,''
     printf,outunit,"The next two figures shows the position angle and length of the baseline projected onto the sky."
     printf,outunit,"The values found in the recipe result are plotted in white,"
     printf,outunit,"while the expected values are green."
     printf,outunit,"The expected values were computed by the simple algorithm used in the MIDI data reduction software."

     printf,outunit,'\bigskip'
     printf,outunit,'\begin{center}'
     printf,outunit,'\includegraphics[height=0.46\textheight,angle=180]{../sciave/Baseline_angle.pdf}'
     printf,outunit,'\end{center}'

     if !d.name eq "PS" then begin
        device,/close
        print,'ps2pdf Baseline_angle.ps'
        spawn,'ps2pdf Baseline_angle.ps'
        device,/colo,/land,file='Baseline_length.ps'
     endif

     ymin= min( [l3l3.base_length[0],base_len], max=ymax)
     print,"length:",ymin,"...",ymax
     plot, l3l3.time_ave[0], l3l3.base_length[0], ysty=16, yrange= [floor(ymin),ceil(ymax)],$
           xtitle='TIME_AVE', ytitle='projected baseline length [m]', psym=1
     oplot,l3l3.time_ave[0], base_len, color=3

     printf,outunit,'\bigskip'
     printf,outunit,'\begin{center}'
     printf,outunit,'\includegraphics[height=0.46\textheight,angle=180]{../sciave/Baseline_length.pdf}'
     printf,outunit,'\end{center}'

     if !d.name eq "PS" then begin
        device,/close
        print,'ps2pdf Baseline_length.ps'
        spawn,'ps2pdf Baseline_length.ps'
     endif

     fbt_close,lvl3

  endif else begin
     ;; no LEVEL3-file, find out what the recipe said
     printf,outunit,'{\bf Results}:\\'
     printf,outunit,'The recipe produced the following output:'
     printf,outunit,'\begin{verbatim}'
     close,outunit & free_lun,outunit

     spawn,'esorex --suppress-prefix pacma_scired2 '+sof[0]+'.sof >> '+outfile+' 2>&1',$
           result,errRes
     ;;help,result,errRes
     print,"Result: ",result,"//"
     print,"ErrResult: ",errres,"//"

     openw,outunit,outfile,/get_lun,/APPEND
     printf,outunit,Result,errRes
     printf,outunit,'\end{verbatim}'
  endelse

  close,outunit & free_lun,outunit
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

function vec2RADec, vec

  RA = atan(vec[1],vec[0]) * 12.d/!dpi	;; hours
  Dec= asin(vec[2]) * 180.d/!dpi	;; deg
  return, { RA:RA, Dec:Dec }
end


pro test_astrometry, basepatt, FILE=outfile

  if N_elements(basepatt) lt 1 then basepatt= 'PACMA.20*-*-01T*_LEVEL3.fits'
  if not keyword_set(outfile) then outfile="astrometry_testresults.tex"

  file = file_search(basepatt)
  if file[0] eq '' then begin
     print,"No file found matching "+basepatt
     return
  endif

  print,"files: ",file

  openw,outunit,outfile,/get_lun
  printf,outunit,"%% astrometric testresults"
  printf,outunit,""
  printf,outunit,'The table shows the expected changes (E) and computed corrections (C) to the DOPD because of the astrometric effects.'
  printf,outunit,'\bigskip'
  printf,outunit,""

     printf,outunit,'\begin{longtable}{rlccccc}'
     printf,outunit,'MJD-OBS & & AB\_DD & LTD\_DD & LD\_DD & PAR\_DD & PM\_DD \\'
     printf,outunit,'\hline\endhead'
     printf,outunit,'\hline\endfoot'

  for iFile=0,N_elements(file)-1 do begin
     print,"Reading ",file[iFile]
     ;;printf,outunit,'File \verb|',file[iFile],'|\span\span\span\span\\'

     lvl3 = fbt_open(file[iFile])
     ;infile = (lvl3.fileobj->prihead())->getpar('PRO REC1 CAL1 NAME')
     ;print,'infile ',infile

     prihead= lvl3.astrom_corr->prihead()
     MJD0   = prihead->getpar('MJD-OBS')
     t1name = strtrim(prihead->getpar('HIERARCH ESO OCS TARG1 NAME'),2)
     t2name = strtrim(prihead->getpar('HIERARCH ESO OCS TARG2 NAME'),2)

     t1fits= fbt_open(t1name+'_positions.fits')
     t1dada= t1fits.star_positions_Corrected->readrows()
     t1ph  = t1fits.fileobj->prihead()
     t1pos0= { RA:  t1ph->getpar('HIERARCH ESO OCS TARG1 RA')/15.,$
               Dec: t1ph->getpar('HIERARCH ESO OCS TARG1 DEC') }
     fbt_close,t1fits

     t2fits= fbt_open(t2name+'_positions.fits')
     t2dada= t2fits.star_positions_Corrected->readrows()
     t2ph  = t2fits.fileobj->prihead()
     t2pos0= { RA:  t2ph->getpar('HIERARCH ESO OCS TARG1 RA')/15.,$
               Dec: t2ph->getpar('HIERARCH ESO OCS TARG1 DEC') }
     fbt_close,t2fits

     dada = lvl3.astrom_corr->readrows(1)

     ;print, format='(F15.6)',MJD_oops
     ;print, format='(F15.6)',t1dada[0:5].mjd[0]
     ;print, format='(F15.6)',t1dada[0:5].mjd[0]-MJD_oops
     idx1= where( abs(t1dada.mjd[0]-MJD0) lt 1./1440.)
     idx2= where( abs(t2dada.mjd[0]-MJD0) lt 1./1440.)

     if idx1[0] lt 0 or idx2[0] lt 0 then begin
        printf,outunit,MJD0,' & No matching simulation found\span\span\span\span\\'
     endif else begin
        t1dada = t1dada[idx1[0]]
        t2dada = t2dada[idx2[0]]

        lst = lst_from_date(MJD0 + dada.time_ave[0]/86400d) * 3600d	;; seconds
        opd_z1= compute_opd( lst, t1pos0, [100.,0.,0.])
        opd_z2= compute_opd( lst, t2pos0, [100.,0.,0.])
        dopd_z= opd_z1 - opd_z2

        opd_b1= compute_opd( lst, vec2RADec(t1dada.barycentric), [100.,0.,0.])
        opd_b2= compute_opd( lst, vec2RADec(t2dada.barycentric), [100.,0.,0.])
        dopd_b= opd_b1 - opd_b2

        opd_g1= compute_opd( lst, vec2RADec(t1dada.geocentric), [100.,0.,0.])
        opd_g2= compute_opd( lst, vec2RADec(t2dada.geocentric), [100.,0.,0.])
        dopd_g= opd_g1 - opd_g2

        opd_d1= compute_opd( lst, vec2RADec(t1dada.deflected), [100.,0.,0.])
        opd_d2= compute_opd( lst, vec2RADec(t2dada.deflected), [100.,0.,0.])
        dopd_d= opd_d1 - opd_d2

        opd_a1= compute_opd( lst, vec2RADec(t1dada.aberrated), [100.,0.,0.])
        opd_a2= compute_opd( lst, vec2RADec(t2dada.aberrated), [100.,0.,0.])
        dopd_a= opd_a1 - opd_a2

        printf,outunit,MJD0,' & E & ',dopd_a-dopd_d,' &  & ',dopd_d-dopd_g,' & ',dopd_g-dopd_b,' & ',dopd_b-dopd_z,'\\'

     endelse
     printf,outunit,' & C & ',dada.ab_dd[0],' & ',dada.ltd_dd[0],' & ',$
            dada.ld_dd[0],' & ',dada.par_dd[0],' & ',dada.pm_dd[0],'\\'
     printf,outunit,'\hline'

     fbt_close,lvl3
  endfor

     printf,outunit,'\end{longtable}'
     printf,outunit,'\bigskip'

  close,outunit & free_lun,outunit
end

pro calc_sepvec

  t1fits= fbt_open('Prinda_positions.fits')
  t1dada= t1fits.star_positions_Corrected->readrows(1)
  fbt_close,t1fits

  t2fits= fbt_open('Secuma_positions.fits')
  t2dada= t2fits.star_positions_Corrected->readrows(1)
  fbt_close,t2fits

  ;help,t1dada
  ;help,t1dada,/str
  print,"Prinda:"
  print,t1dada
  print,"Secuma:"
  print,t2dada

  print,"separation vector:"
  print,"Barycentric:", t2dada.barycentric-t1dada.barycentric
  print,"Geocentric: ", t2dada.geocentric -t1dada.geocentric
  print,"Deflected:  ", t2dada.deflected  -t1dada.deflected
  print,"Aberrated:  ", t2dada.aberrated  -t1dada.aberrated
end
