;
; utilities.pro
; Created:     Fri Apr 27 15:11:13 2007 by Rkoehler@lx40
; Last change: Wed Jun 15 16:14:57 2011
;
; Copyright 2008 Rainer Koehler
;
; This file is part of Pacmart.
;
; Pacmart is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; Pacmart is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with Pacmart; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; the most important function in this world of changing names:
;;
Function tag_exists, struct, tag

  idx = where( strupcase(tag) eq Tag_names(struct))
  return,(idx[0] gt -1)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO print_cmd_stuff, outunit, recipe, outname, CMDSEARCH=cmdsearch

  if not keyword_set(cmdsearch) then cmdsearch=recipe

  printf,outunit,''
  printf,outunit,'{\bf Recipe:} pacma\_'+recipe
  printf,outunit,''

  openr,cmdunit,"esorex.sh",/get_lun,ERROR=err
  if err eq 0 then begin
     printf,outunit,'{\bf Command line(s):}'
     printf,outunit,'\begin{verbatim}'
     line= ''
     while not eof(cmdunit) do begin
        readf,cmdunit,line
        if strpos(line,cmdsearch) ge 0 then printf,outunit,line
     endwhile
     close,cmdunit & free_lun,cmdunit
     printf,outunit,'\end{verbatim}'
     printf,outunit,''
  endif
  printf,outunit,'{\bf Test conducted on:} '+systime(0,(file_info(outname)).mtime)
  printf,outunit,''

  if recipe ne 'baseline' then begin
     printf,outunit,'{\bf Output printed on the screen:}'
     printf,outunit,'\verbatiminput{'+outname+'}'
  endif else $
     printf,outunit,'{\bf Output printed on the screen omitted}'

  printf,outunit,''
end

PRO enum_fbt_extensions, outunit, fbt

  printf,outunit,'\begin{enumerate}'
  tnames= tag_names(fbt)
  ;; index 0 and 1 are filename and fileobj
  for i=2,N_Tags(fbt)-1 do begin
     printf,outunit,'\item \verb|'+tnames[i]+'|'
  endfor
  printf,outunit,'\end{enumerate}'
end

PRO list_fbt_extensions, outunit, fbt

  if N_elements(fbt) le 0 then begin
     printf,outunit,"The recipe did not produce an output file"
     return
  endif
  printf,outunit,'{\bf Test conducted on:} '+systime(0,(file_info(fbt.filename)).mtime)
  ;;printf,outunit,'\bigskip'
  printf,outunit,''
  printf,outunit,'{\bf Result:} The recipe produced the file \verb|'+fbt.filename+'|,'
  printf,outunit,'which contains the following tables:'
  printf,outunit,'\begin{verbatim}'
  tnames= tag_names(fbt)
  ;; index 0 and 1 are filename and fileobj
  for i=2,N_Tags(fbt)-1 do begin
     printf,outunit,'        '+tnames[i]
  endfor
  printf,outunit,'\end{verbatim}'
end

PRO dump_table_structure, outunit, tab

  if size(tab,/TName) eq 'STRUCT' then begin

     printf,outunit,'\begin{tabular}{ll}'
     printf,outunit,'\hline\noalign{\vspace{2pt}}'
     printf,outunit,'Column Name & Data Type\\'
     printf,outunit,'\hline\noalign{\vspace{2pt}}'
     tnames= tag_names(tab)
     for i=0,N_Tags(tab)-1 do begin
        N = N_elements(tab[0].(i))
        Nstr= N gt 1 ? strtrim(N,2) : ''
        tname= strjoin( strsplit(tnames[i],'_',/Extract),'\_')
        printf,outunit,'        '+tname+' & '+Nstr+' '+strlowcase(size(tab[0].(i),/tname))+'\\'
     endfor
     printf,outunit,'\hline'
     printf,outunit,'\end{tabular}'
     printf,outunit,''
  endif else $
     printf,outunit,"table has no columns - maybe it's empty?"
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO print_header_keywords, outunit, head, pattern, SKIP=skip

  if not keyword_set(skip) then skip=17	 ;; removes HIERARSCH ESO SIM
  keywords= head->sthead()
  for k=0,N_elements(keywords)-1 do begin
     ;;print,k,keywords[k]
     key = stregex(keywords[k],pattern,/Extract)
     if key ne '' then begin
        val = head->getpar(key,COMM=comment)
        if N_elements(comment) le 0 then comment = '(no comment)'
        key = strmid(key,skip)	;; get rid of HIERARSCH ESO ...
        key = strjoin( strsplit(key,'_',/Extract),'\_')
        key = strjoin( strsplit(key,' ',/Extract),'.')
        printf,outunit, key,'	& ',val,'	& ',comment,'	\\'
     endif
  endfor
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO pwd
  cd,current=c
  print,c
end

PRO dir, arg
  if N_params() gt 0 then  spawn,"ls -l -- "+arg  else  spawn,"ls -l"
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Modified Julian Date
;;
;; Note the order of arguments: largest unit first!
;;

Function MJulDay, Year, Month, Day, Hour, Minute, Second

  return, JulDay(Month, Day, Year, Hour, Minute, Second) - 2400000.5d
end

Function MJulDate, MJD, yr,mon,day,hr,min,sec
  caldat, MJD+2400000.5d, mon, day, yr, hr, min, sec
  return, string(Format="(C(CDI,'-',CMoA,'-',CYI,', ',CHI02,':',CMI02,':',CSF06.3))", MJD+2400000.5d)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; convert ESO-timestring to pieces
;; input format:  yyyy-mm-ddThh:mm:ss.sss
;; It is strongly recommend to use single quotes, since double quotes
;; start octal numbers in IDL!
;;
Function eso_date, time_string

  if n_elements(time_string) eq 0 then $
    return, bin_date()	;; let bin_date produce the current date and time

  return, strsplit(time_string, '-T:', /extract)
end
;;
;; convert ESO-timestring to MJD
;;
Function eso_MJD, time_string

  st= eso_date(time_string)	;; year, month, day, hour, min, sec
  ;;print,"Date:",time_string,st

  MJD0  = MJulday(st[0],st[1],st[2], 0,0,0)	;; year, mon, day, hms
  MJD0 += (st[3] + (st[4] + st[5]/60.d) / 60.d) / 24.d	;; add later to preserve precision
  return, MJD0
end
;;
;; Create an ESO-string representing a given MJD
;;
Function MJD_to_ESO_string, MJD

  JD   = MJD+2400000.5d
  Dfrac= MJD+0.5	;; fraction of day, ignore integer part

  ;; calendar format codes - in the true spirit of "code"
  ;; Note that we use different variables for the day and the time
  ;; this should leave enough digits in the time for millisec-precision
  ;; I've had enough of IDL-format codes, there won't be any milliseconds!

  return, string(Format=$
                 '(C(CYI4.4,"-",CMoI2.2,"-",CDI2.2),"T",C(CHI2.2,":",CMI2.2,":",CSI2.2),".000")',$
                   JD, Dfrac )
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; hours_to_hms
;;
function hours_to_hms, hours

  sign= (hours ge 0.) ? '':'-'
  hours= abs(hours)

  return, string(Format='(A,I02,":",I02,":",F06.3)',$
                 sign, fix(hours), fix(hours*60.) mod 60, (hours * 3600.) mod 60.)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Compute LST from date and time
;;
;; MJD0 = Modified Julian Date at 0h UT
;; UT = Universal Time
;;
;; see R. Green: Spherical Astronomy, p.242 and p.31

function lst_from_date, MJD, geolong

  if size(geolong,/tname) eq 'UNDEFINED' then geolong= -70.40479659d

  ;; time in julian centuries since 1.1.2000, 12h UT
  Tu = (double(floor(MJD)) - MJulday(2000,1,1, 12,0,0)) / 36525.d
  UT = (MJD-floor(MJD))*24.d

  GMST= 6. + 41./60. + 50.54841/3600. $
	+ 8640184.812866d/3600. * Tu $
	+ 0.093104d/3600.  * Tu * Tu $
	- 6.2d-6/3600. * Tu * Tu * Tu $
	+ 1.002737909350795d * UT

  LST = GMST + geolong/15.	;; longitude of paranal

  return, LST mod 24.
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro project_baseline, t_in, de_in, baseline, base_len, base_pa, DEBUG=debug
  ; t = hourangle in hours, de = Declination in degrees
  ; baseline = [W,S,Up]
  ; base_len is given in meters
  ; base_pa in degrees

  if keyword_set(debug) then print,"projBaseline: HA =",t_in,", Dec =",de_in

  t  = t_in * !dPI/12.
  de = de_in* !dPI/180.
  sinH = sin(t)
  cosH = cos(t)

  ; from ESO exposure time calculator:
  ;longitude=  70.40479659;
  latitude = -24.62794830

  sin_la = sin(latitude * !dPI/180.)
  cos_la = cos(latitude * !dPI/180.)

  ; coordinates of baseline in horizontal system
  BaseN = double(-baseline[1])
  BaseE = double(-baseline[0])

  if keyword_set(debug) then print,"projBaseline: BaseN,E =",BaseN,BaseE

  ; rotate around y-axis into equatorial system
  r1 = BaseN * (-sin_la)          ; + c * cos_la
  r2 = BaseE
  r3 = BaseN * (-cos_la)

  if keyword_set(debug) then print,"projBaseline: r=",r1,r2,r3

  ; rotate into star-centered system
  a_E =   r1 * sinH + r2 * cosH
  b_N = (-r1 * cosH + r2 * sinH) * sin(de) - r3 * cos(de)
  ; length along line of sight doesn't matter

  if keyword_set(debug) then print,"projBaseline: E =",a_E,", N =",b_N

  ;;print,"Baseline:",baseline
  ;;N= N_elements(a_E)
  ;;mit= min(abs(t),tidx)
  ;;print,a_E[tidx],b_N[tidx]
  ;;print,a_E[0],b_N[0]
  ;;print,a_E[N/2],b_N[N/2]
  ;;print,a_E[N-1],b_N[N-1]

  base_len = sqrt( a_E * a_E + b_N * b_N );
  base_pa  = atan(a_E, b_N) * 180./!PI;
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; struct with all the tables in a PRIMA/Pacma(n)/GM6k-file

Function Pacman_binfile, MJD0

  file= { Pacman_file, $
          arrdesc: fitstable_create('ARRAY_DESCRIPTION'),$
          arrgeom: fitstable_create('ARRAY_GEOMETRY'),$
          optrain: fitstable_create('OPTICAL_TRAIN'),$
          MDL:     fitstable_create('MAIN_DELAY'),$
          det_A:   fitstable_create('IMAGING_DETECTOR_FSUA'),$
          dataA:   fitstable_create('IMAGING_DATA_FSUA'),$
          det_B:   fitstable_create('IMAGING_DETECTOR_FSUB'),$
          dataB:   fitstable_create('IMAGING_DATA_FSUB'),$
          metro:   fitstable_create('METROLOGY_DATA'),$
          metroB:  fitstable_create('METROLOGY_DATA_FSUB'),$
          DDL:     fitstable_create('DDL'), $
          OPDC:    fitstable_create('OPDC'),$
          DOPDC:   fitstable_create('DOPDC') $
        }

  prihead= file.arrdesc->prihead()
  prihead->addPar,'DATE-OBS', MJD_to_ESO_string(MJD0), COMMENT=" Simulating date"
  prihead->addPar,'MJD-OBS', MJD0, FORMAT="F16.10", Comment=" MJD of Oops"
  prihead->addPar,'UTC', (MJD0 mod 1.)*86400., $
    Comment= " " + hours_to_hms((MJD0 mod 1.)*24.) + " UTC at start (sec)"

  return, file
end

PRO Pacman_binfile_cleanup, file

  obj_destroy, file.arrdesc
  obj_destroy, file.arrgeom
  obj_destroy, file.optrain
  obj_destroy, file.MDL
  obj_destroy, file.det_A
  obj_destroy, file.dataA
  obj_destroy, file.det_B
  obj_destroy, file.dataB
  obj_destroy, file.metro
  obj_destroy, file.metroB
  obj_destroy, file.DDL
  obj_destroy, file.OPDC
  obj_destroy, file.DOPDC
end


PRO Pacman_binfile_create_arrdesc, pfile, filename
  pfile.arrdesc->newFile, filename
end

PRO Pacman_binfile_create_arrgeom, pfile
  pfile.arrgeom->appendtofile, pfile.arrdesc
end

PRO Pacman_binfile_create_optrain, pfile
  pfile.optrain->appendtofile, pfile.arrgeom
end

PRO Pacman_binfile_create_MDL, pfile
  ;;pfile.MDL->appendtofile, pfile.optrain
  ;;print,"Refusing to create Main Delay table"
end

PRO Pacman_binfile_create_DetA, pfile
  pfile.det_A->appendtofile, pfile.optrain
end

PRO Pacman_binfile_create_DataA, pfile
  pfile.dataA->appendtofile, pfile.det_A
end

PRO Pacman_binfile_create_DetB, pfile
  pfile.det_B->appendtofile, pfile.dataA
end

PRO Pacman_binfile_create_DataB, pfile
  pfile.dataB->appendtofile, pfile.det_B
end

PRO Pacman_binfile_create_metro, pfile
  pfile.metro->appendtofile, pfile.dataB
end

PRO Pacman_binfile_create_metroB, pfile
  pfile.metroB->appendtofile,pfile.metro
end

PRO Pacman_binfile_create_DDL, pfile
  ;;pfile.DDL->appendtofile, pfile.metroB
  ;;print,"Refusing to create DehDehEll table"
end

PRO Pacman_binfile_create_OPDC, pfile
  pfile.OPDC->appendtofile,pfile.metroB
end

PRO Pacman_binfile_create_DOPDC, pfile
  pfile.DOPDC->appendtofile,pfile.OPDC
end

PRO Pacman_binfile_close, pfile
  pfile.DOPDC->close
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; standard stuff in primary header
;;
PRO standard_primary_header, prihead, dita, ditb, OBS_Start=obs_start, SIM_CODE=sim_code,$
                             CATG=dpr_catg, TECH=dpr_tech, TYPE=dpr_type

  if not keyword_set(obs_start) then obs_start= 'now'
  tpl_name= 'PACMA do something'

  if keyword_set(sim_code) then begin
      ;; according to gronk:
     ;;keys= $
     ;;   [ { Code: 'LD', TPL: 'cal_lab_dark', CATG: 'CALIB', TYPE: 'DARK',      TECH: 'IMAGE' },$
     ;;     { Code: 'LF', TPL: 'cal_lab_flat', CATG: 'CALIB', TYPE: 'FLAT,LAMP', TECH: 'IMAGE' },$
     ;;     { Code: 'FR', TPL: 'cal_lab_FSUresponse', CATG: 'CALIB', TYPE: 'FLAT,LAMP', TECH: 'INTERFEROMETRY,SPECTRUM' },$
     ;;     { Code: 'SD', TPL: 'cal_sky_dark', CATG: 'CALIB', TYPE: 'SKY',              TECH: 'IMAGE' },$
     ;;     { Code: 'SF', TPL: 'cal_sky_flat', CATG: 'CALIB', TYPE: 'OBJECT,PHOTOMETRY',TECH: 'IMAGE' },$
     ;;     { Code: 'VR', TPL: 'cal_sky_VLTIresponse', CATG: 'CALIB', TYPE: 'OBJECT', TECH: 'INTERFEROMETRY,SPECTRUM' },$
     ;;     { Code: 'SR', TPL: 'obs_spectrum',       CATG: 'SCIENCE', TYPE: 'OBJECT', TECH: 'INTERFEROMETRY,SPECTRUM' },$
     ;;     { Code: 'AM', TPL: 'obs_Astrometry',        CATG: 'SCIENCE', TYPE: 'OBJECT,ASTROMETRY', TECH: 'INTERFEROMETRY' },$
     ;;     { Code: 'AS', TPL: 'obs_Astrometry_wSteps', CATG: 'SCIENCE', TYPE: 'OBJECT,ASTROMETRY', TECH: 'INTERFEROMETRY,JITTER'},$
     ;;     { Code: 'EN', TPL: 'rec_Environment',	      CATG: 'CALIB', TYPE: 'SENSORS', TECH: 'OTHER' } $
     ;;   ]
      ;; in COMM#5 and #6:
     keys= $
        [ { Code: 'LD', TPL: 'cal_lab_dark',	    CATG: 'CALIB', TYPE: 'DARK',      TECH: 'IMAGE' },$
          { Code: 'LF', TPL: 'cal_lab_flat',	    CATG: 'CALIB', TYPE: 'FLAT,LAMP', TECH: 'IMAGE' },$
          { Code: 'FR', TPL: 'cal_lab_FSUresponse', CATG: 'CALIB', TYPE: 'FLAT,LAMP', TECH: 'INTERFEROMETRY,SPECTRUM' },$
          { Code: 'SD', TPL: 'cal_sky_Background',  CATG: 'CALIB', TYPE: 'SKY',	      TECH: 'IMAGE' },$
          { Code: 'SF', TPL: 'cal_sky_flat', 	    CATG: 'CALIB', TYPE: 'FLAT,SKY',  TECH: 'IMAGE' },$
          { Code: 'VR', TPL: 'cal_sky_VLTIresponse',CATG: 'CALIB', TYPE: 'OBJECT',    TECH: 'INTERFEROMETRY,SPECTRUM' },$
          { Code: 'SR', TPL: 'obs_spectrum',         CATG: 'SCIENCE', TYPE: 'OBJECT', TECH: 'INTERFEROMETRY,SPECTRUM' },$
          { Code: 'AM', TPL: 'obs_Astrometry',       CATG: 'SCIENCE', TYPE: 'OBJECT', TECH: 'INTERFEROMETRY' },$
          { Code: 'AS', TPL: 'obs_Astrometry_wSteps',CATG: 'SCIENCE', TYPE: 'OBJECT,ASTROMETRY', TECH: 'INTERFEROMETRY,JITTER'},$
          { Code: 'EN', TPL: 'rec_Environment',	     CATG: 'CALIB', TYPE: 'SENSORS', TECH: 'OTHER' } $
        ]

     idx= where(keys.Code eq strupcase(sim_code))
     if idx[0] gt -1 then begin
        tpl_name= 'PACSIM_'+keys[idx[0]].TPL
        dpr_catg= keys[idx[0]].CATG
        dpr_tech= keys[idx[0]].TECH
        dpr_type= keys[idx[0]].TYPE
        print,"stdprimhdr: ",sim_code,"/",tpl_name,"/",dpr_catg
     endif
  endif

  prihead->addESOpar,'DEL FT SENSOR','FSUB',Comment=' Fringe Tracker Sensor Name'
  prihead->addESOpar,'DEL FT STATUS','OFF', Comment=' Fringe Tracker Status'

  if keyword_set(dpr_catg) then prihead->addESOpar,'DPR CATG',dpr_catg,Comment=' Oopservation category'
  if keyword_set(dpr_tech) then prihead->addESOpar,'DPR TECH',dpr_tech,Comment=' observation technique'
  if keyword_set(dpr_type) then prihead->addESOpar,'DPR TYPE',dpr_type,Comment=' observation type'

  insmode = 'CONFUSED'
  if dita ne 0 then begin
     insmode = (ditb ne 0) ? 'NORMAL' : 'SINGLE_A'
  end else if ditb ne 0 then insmode= 'SINGLE_B'

  prihead->addESOPar,'INS MODE', insmode, COMMENT=" instrument mode (normal or swapped)"
  prihead->addESOPar,'INS FSU1 STATE', 'NORMAL', Comment=' Current state of FSU1'
  prihead->addESOPar,'INS FSU2 STATE', 'NORMAL', Comment=' Current state of FSU2'

  prihead->addESOPar,'ISS ALT', 0., Comment=' Altitude angle at start'
  prihead->addESOPar,'ISS AZ',  0., Comment=' Azimuth angle at start'
  prihead->addESOPar,'ISS RA',  0., Comment=' Refernece RA as given by the user'

  prihead->addESOPar,'ISS ALT',	0d, Comment=' deg Altitude angles at start'
  prihead->addESOPar,'ISS AZ',	0d, Comment=' deg Azimuth angles at start'
  prihead->addESOPar,'ISS REF RA', 0d, Comment=' Reference Alpha as given by user'
  prihead->addESOPar,'ISS REF DEC',0d, Comment=' Reference Delta as given by user'
  prihead->addESOPar,'ISS GEOELEV',2681.0000, Comment=' VLTI site elevation [m]'
  prihead->addESOPar,'ISS GEOLAT',  -24.6279, Comment=' VLTI site latitude (UV zero) [deg]'
  prihead->addESOPar,'ISS GEOLON',  -70.4048, Comment=' VLTI site longitude (UV zero) [deg]'
  prihead->addESOPar,'ISS OPER', 'JdJ', Comment=' Name of VLTI operator'
  prihead->addESOPar,'ISS PARANG END',	0d, Comment=' deg Parallactic angle at end'
  prihead->addESOPar,'ISS PARANG START',0d, Comment=' deg Parallactic angle at start'
  prihead->addESOPar,'ISS AIRM END',	0d, Comment=' Airmass at end'
  prihead->addESOPar,'ISS AIRM START',	0d, Comment=' Airmass at start'
  prihead->addESOPar,'ISS AMBI FWHM START',0d, Comment=' arcsec Observatory seeing'
  prihead->addESOPar,'ISS AMBI FWHM END',  0d, Comment=' arcsec Observatory seeing'
  prihead->addESOPar,'ISS AMBI TAU0 START',0d, Comment=' seconds Coherence time'
  prihead->addESOPar,'ISS AMBI TAU0 END',  0d, Comment=' seconds Coherence time'
  prihead->addESOPar,'ISS AMBI TEMP',	0d, Comment=' C Observatory ambient temperature '
  prihead->addESOPar,'ISS AMBI RHUM',	0d, Comment=' percentage Relative humidity'
  prihead->addESOPar,'ISS AMBI WINDDIR',0d, Comment=' deg Wind direction'
  prihead->addESOPar,'ISS AMBI WINDSP',	0d, Comment=' m/s Observatory wind speed'
  prihead->addESOPar,'ISS AMBI PRES',	0d, Comment=' hPa Observatory ambient air pressure'
  prihead->addESOPar,'ISS AMBI TEMPDEW',0d, Comment=' C Observatory ambient dew temperature'
  prihead->addESOPar,'ISS AMBI LRATE',	0d, Comment=' K/m Observatory ambient lapse rate'
  prihead->addESOPar,'ISS TRAK STATUS','trecking', Comment=' Tracking status'
  prihead->addESOPar,'ISS TRAK RATEA',	0d, Comment=' arcsec/sec Tracking rate in RA'
  prihead->addESOPar,'ISS TRAK RATED',	0d, Comment=' arcsec/sec Tracking rate in DEC'
  prihead->addESOPar,'ISS CHOP ST',   'F*', Comment=' Chopping status'
  prihead->addESOPar,'ISS CHOP FREQ',	0d, Comment=' Hz Frequency of cycle'
  prihead->addESOPar,'ISS CHOP THROW',	0d, Comment=' arcsec Amplitude of chopping'
  prihead->addESOPar,'ISS CHOP PVRATIO',0d, Comment=' ratio Peak-to-valley ratio'
  prihead->addESOPar,'ISS CHOP POSANG',	0d, Comment=' deg Orientation on sky'
  prihead->addESOPar,'ISS CHOP OFFSCHOP',0d, Comment=' arcsec Offset'
  prihead->addESOPar,'ISS CHOP GUIDE',  'n/a',Comment=' Guiding beam for chopping'
  prihead->addESOPar,'ISS CHOP NULLING','n/a',Comment=' Out of phase chopping'

  prihead->addESOPar,'ISS PUPIL 1U',	0d, Comment=' m U location output pupil beam 1'
  prihead->addESOPar,'ISS PUPIL 1V',	0d, Comment=' m V location output pupil beam 1'
  prihead->addESOPar,'ISS PUPIL 1W',	0d, Comment=' m W location output pupil beam 1'
  prihead->addESOPar,'ISS PUPIL 2U',	0d, Comment=' m U location output pupil beam 2'
  prihead->addESOPar,'ISS PUPIL 2V',	0d, Comment=' m V location output pupil beam 2'
  prihead->addESOPar,'ISS PUPIL 2W',	0d, Comment=' m W location output pupil beam 2'
  prihead->addESOPar,'ISS ID',		'', Comment=' ISS version string'
  prihead->addESOPar,'ISS DID', 	'', Comment=' Data dictionary name'
  prihead->addESOPar,'ISS DATE',	'', Comment=' VLTI release version and date'
  prihead->addESOPar,'ISS CONF NTEL',	0 , Comment=' Number of telescopes used'
  prihead->addESOPar,'ISS CONF T1X',	0d, Comment=' m Telescope X coordinate - OPD model'
  prihead->addESOPar,'ISS CONF T1Y',	0d, Comment=' m Telescope Y coordinate - OPD model'
  prihead->addESOPar,'ISS CONF T1Z',	0d, Comment=' m Telescope Z coordinate - OPD model'
  prihead->addESOPar,'ISS CONF A1L',	0d, Comment=' m Static optical path length - OPD model'
  prihead->addESOPar,'ISS CONF T1NAME',  '', Comment=' Telescope name'
  prihead->addESOPar,'ISS CONF STATION1','', Comment=' Station of telescope 1'
  prihead->addESOPar,'ISS CONF DL1',	 '', Comment=' Delay line used'
  prihead->addESOPar,'ISS CONF INPUT1',	  0, Comment=' Input channel in lab'
  prihead->addESOPar,'ISS CONF SWYARD1', '', Comment=' Switchyard configuration'
  prihead->addESOPar,'ISS CONF INPUT12',  0, Comment=' Input channel in lab of secondary beam'
  prihead->addESOPar,'ISS CONF SWYARD12','', Comment='Switchyard config for channel 1 secondary beam'
  prihead->addESOPar,'ISS CONF T2X',	 0d, Comment=' m Telescope X coordinate - OPD model'
  prihead->addESOPar,'ISS CONF T2Y',	 0d, Comment=' m Telescope Y coordinate - OPD model'
  prihead->addESOPar,'ISS CONF T2Z',	 0d, Comment=' m Telescope Z coordinate - OPD model'
  prihead->addESOPar,'ISS CONF A2L',	 0d, Comment=' m Static optical path length - OPD model'
  prihead->addESOPar,'ISS CONF T2NAME',  '', Comment=' Telescope name'
  prihead->addESOPar,'ISS CONF STATION2','', Comment=' Station of telescope 2'
  prihead->addESOPar,'ISS CONF DL2',	 '', Comment=' Delay line used'
  prihead->addESOPar,'ISS CONF INPUT2',	  0, Comment=' Input channel in lab'
  prihead->addESOPar,'ISS CONF SWYARD2', '', Comment='Switchyard configuration'
  prihead->addESOPar,'ISS CONF INPUT22',  0, Comment=' Input channel in lab of secondary beam'
  prihead->addESOPar,'ISS CONF SWYARD22','', Comment='Switchyard config for secondary beam'
  prihead->addESOPar,'ISS TEMP LAB1',	0d, Comment=' C Temp in lab'
  prihead->addESOPar,'ISS TEMP LAB2',	0d, Comment=' C Temp in lab'
  prihead->addESOPar,'ISS TEMP LAB3',	0d, Comment=' C Temp in lab eastern wall'
  prihead->addESOPar,'ISS TEMP LAB4',	0d, Comment=' C Temp in lab'
  prihead->addESOPar,'ISS TEMP TUN1',	0d, Comment=' C Temp in tunnel A west'
  prihead->addESOPar,'ISS TEMP TUN2',	0d, Comment=' C Temp in tunnel centre'
  prihead->addESOPar,'ISS TEMP TUN3',	0d, Comment=' C Temp in tunnel centre'
  prihead->addESOPar,'ISS TEMP TUN4',	0d, Comment=' C Temp in tunnel L east'
  prihead->addESOPar,'ISS PBL12 END',	0d, Comment=' m Projected baseline T1 T2 at end'
  prihead->addESOPar,'ISS PBL12 START',	0d, Comment=' m Projected baseline T1 T2 at start'
  prihead->addESOPar,'ISS PBLA12 END',	0d, Comment=' deg Projected baseline orientation angle T1 T2 at end'
  prihead->addESOPar,'ISS PBLA12 START',0d, Comment=' deg Projected baseline orientation angle T1 T2 at start'
  prihead->addESOPar,'ISS LEO LAMP',       '', Comment=' ARAL LEO calibration lamp (NONE or name)'
  prihead->addESOPar,'ISS LEO OPTI1 NAME', '', Comment=' ARAL LEO beam splitter A (OUT, AUTOTEST, AUTOCOLL)'
  prihead->addESOPar,'ISS LEO OPTI2 NAME', '', Comment=' ARAL LEO beam splitter B (OUT, AUTOTEST, AUTOCOLL)'
  prihead->addESOPar,'ISS IAS OPTI3 NAME', '', Comment=' ARAL IAS ALI1 slide position (OUT, BEAM B TCCD, SHUTTER)'
  prihead->addESOPar,'ISS IAS OPTI4 NAME', '', Comment=' ARAL IAS ALI5 slide position (OUT, BEAM A TCCD, SHUTTER)'
  prihead->addESOPar,'ISS IAS OPTI6 NAME', '', Comment=' ARAL IAS TCCD lens and focus setting'
  prihead->addESOPar,'ISS IAS OPTI7 NAME', '', Comment=' ARAL Iris notch filter setting (IN,OUT)'
  prihead->addESOPar,'ISS IAS OPTI8 NAME', '', Comment=' ARAL Iris Fly Eye and Telephoto (PSF,TTS)'
  prihead->addESOPar,'ISS ARAL MIRR1 NAME','', Comment=' ARAL Switchyard Unit 1 (DIRECT,BC,DDL,BC DDL)'
  prihead->addESOPar,'ISS ARAL MIRR3 NAME','', Comment=' ARAL Switchyard Unit 3 (DIRECT,BC,DDL,BC DDL)'
  prihead->addESOPar,'ISS ARAL MIRR5 NAME','', Comment=' ARAL Switchyard Unit 5 (DIRECT,BC,DDL,BC DDL)'
  prihead->addESOPar,'ISS IAS MIRR9 NAME', '', Comment=' ARAL IAS ALI slide position (OUT,BEAM A, BEAM B)'
  prihead->addESOPar,'ISS ARAL MIRR10 NAME', '', Comment='ARAL VINCI feeding optics beam A (FREE,MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR11 NAME', '', Comment='ARAL VINCI feeding optics beam B (FREE,MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR12 NAME', '', Comment='ARAL MIDI feeding optics rot beam A (FREE,MIRR,DICHR1,DICHR2)'
  prihead->addESOPar,'ISS ARAL MIRR13 NAME', '', Comment='ARAL MIDI feeding optics rot beam B (FREE,MIRR,DICHR1,DICHR2)'
  prihead->addESOPar,'ISS ARAL MIRR14 NAME', '', Comment='ARAL MIDI feeding optics trans beam A (IN,OUT)'
  prihead->addESOPar,'ISS ARAL MIRR15 NAME', '', Comment='ARAL MIDI feeding optics trans beam B (IN,OUT)'
  prihead->addESOPar,'ISS ARAL MIRR16 NAME', '', Comment='ARAL Finito feeding optics A (FREE,MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR17 NAME', '', Comment='ARAL Finito feeding optics B (FREE,MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR18 NAME', '', Comment='ARAL Finito feeding optics C (FREE,MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR19 NAME', '', Comment='ARAL Amber feeding optics A (FREE,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR20 NAME', '', Comment='ARAL Amber feeding optics B (FREE,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR21 NAME', '', Comment='ARAL Amber feeding optics C (FREE,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR22 NAME', '', Comment='ARAL Iris feeding optics 1 (MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR23 NAME', '', Comment='ARAL Iris feeding optics 2 (MIRR,DICHR)'
  prihead->addESOPar,'ISS ARAL MIRR24 NAME', '', Comment='ARAL Iris feeding optics 5 (IN,OUT)'
  prihead->addESOPar,'ISS ARAL MIRR25 NAME', '', Comment='ARAL Iris feeding optics A1 (T,R)'
  prihead->addESOPar,'ISS ARAL MIRR26 NAME', '', Comment='ARAL Iris feeding optics A2 (T,R)'
  prihead->addESOPar,'ISS ARAL IRISF NAME', '', Comment='ARAL Iris filter (J,H,K)'
  prihead->addESOPar,'ISS ARAL MODE',   '', Comment='Instrument mode used'
  prihead->addESOPar,'ISS ARAL CONFIG', '', Comment='Light path configuration'
  prihead->addESOPar,'ISS IAS SELECTED','', Comment='Selected image alignment sensor (TCCD,IRIS)'
  prihead->addESOPar,'ISS IAS IRIS ERRSTDXi',	0d, Comment=' Standard deviation of the IRIS x positions'
  prihead->addESOPar,'ISS IAS IRIS ERRSTDYi',	0d, Comment=' Standard deviation of the IRIS y positions'
  prihead->addESOPar,'ISS IAS IRIS ERRSTDFLUXi',0d, Comment=' Standard deviation of the IRIS flux'
  prihead->addESOPar,'ISS IAS IRIS GUID',	'', Comment=' IRIS guiding mode'
  prihead->addESOPar,'ISS MARCEL TEMP',	      700., Comment=' Temparature of MARCEL calibration lamp'
  ;;
  ;; FSUA
  ;;
  prihead->addESOPar,'ISS PRI FSU1 NAME', dita ne 0? 'FSUA':'none', Comment=' After discussion'
  prihead->addESOPar,'ISS PRI FSU1 ID',   '936756821t4682957378', Comment=' Serial Killer Number'
  prihead->addESOPar,'ISS PRI FSU1 NDIT', 1, Comment=' number of ints before out'
  prihead->addESOPar,'ISS PRI FSU1 DIT',dita, Comment=' time of sub-int (sec)'
  prihead->addESOPar,'ISS PRI FSU1 GDFAC', 0, Comment=' there is no GD measurement'
  prihead->addESOPar,'ISS PRI FSU1 EXPNO', 6472, Comment=' Unique exposure ID number'
  prihead->addESOPar,'ISS PRI FSU1 STATE', 'ONLINE', Comment=' Current FSU state'
  prihead->addESOPar,'ISS PRI FSU1 ROLE', 'UNKNOWN', Comment=' I dunno'
  prihead->addESOPar,'ISS PRI FSU1 FREQ', (dita ne 0.)? double(1./(dita+50.d-6)) : 0.d, Comment=' FSU1 freq'
  ;; spectral_resp.pro subtracts 50.d-6 when computing dit from rate
  ;;
  ;; FSUB
  ;;
  prihead->addESOPar,'ISS PRI FSU2 NAME', ditb ne 0? 'FSUB':'none', Comment=' No discussion'
  prihead->addESOPar,'ISS PRI FSU2 ID',   '936756827t4682957378', Comment=' Serial Killer Number'
  prihead->addESOPar,'ISS PRI FSU2 NDIT', 1, Comment=' number of ints before out'
  prihead->addESOPar,'ISS PRI FSU2 DIT',ditb, Comment=' time of sub-int (sec)'
  prihead->addESOPar,'ISS PRI FSU2 GDFAC', 0, Comment=' there is no GD measurement'
  prihead->addESOPar,'ISS PRI FSU2 EXPNO', 6472, Comment=' Unique exposure ID number'
  prihead->addESOPar,'ISS PRI FSU2 STATE', 'ONLINE', Comment=' Current FSU state'
  prihead->addESOPar,'ISS PRI FSU2 ROLE', 'UNKNOWN', Comment=' I dunno'
  prihead->addESOPar,'ISS PRI FSU2 FREQ', (ditb ne 0.)? double(1./(ditb+50.d-6)) : 0.d, Comment=' FSU2 freq'
  ;;
  ;; PRIMET
  ;;
  prihead->addESOPar,'ISS PRI MET WLEN', 1.3d-6, Comment=' PRIMETROLOGY LASER WLEN [m]'
  prihead->addESOPar,'ISS MDL WLEN',  0.6328d-6, Comment=' Wavelength of DL metro [m]'
  ;;
  ;; Information from OB
  ;;
  prihead->addESOPar,'OBS DID', 'ESO-VLT-DIC.OOPS',	Comment=' OBS Dic'
  prihead->addESOPar,'OBS EXECTIME',	0,		Comment=' Expected execcution time [sec]'
  prihead->addESOPar,'OBS ID',long(systime(/seconds)-1.2e9), Comment=' OB ID'
  prihead->addESOPar,'OBS NAME',	'gronk-1',	Comment=' OB Name'
  prihead->addESOPar,'OBS GRP', 	'O',		Comment=' Linked blocks'
  prihead->addESOPar,'OBS OBSERVER',	'anonymous',	Comment=' Ob-server name'
  prihead->addESOPar,'OBS PI-COI ID', 	0,	 	Comment=' ESO INT PI COI ID'
  prihead->addESOPar,'OBS PI-COI NAME', 'Ptolemaeus',	Comment=' PI COI NAME'
  prihead->addESOPar,'OBS PROG ID', 	'099.A-0001', 	Comment=' ESO Prog ID'
  prihead->addESOPar,'OBS TPLNO',	0,		Comment=' Template No. within OB'
  prihead->addESOPar,'OBS TARG NAME',	'Gronk',	Comment=' OB Target Name'
  prihead->addESOPar,'OBS START',	obs_start,	Comment=' OB Start time'

  prihead->addESOPar,'OBS USER STATE',	'confused',	Comment=' State of user'
  prihead->addESOPar,'OBS USER LAST WORD', 'gronk',	Comment=' Last word user uttered'
  prihead->addESOPar,'OBS SENSE', 	'optional',	Comment=' Failure is not an option'

  prihead->addESOPar,'OCS TARG EPOCH',	2000.0d,	Comment=' Epoch'
  prihead->addESOPar,'OCS TARG EPOCHSYSTEM', 'J',	Comment=' Epochsystem (Julian)'
  prihead->addESOPar,'OCS TARG EQUINOX', 2000.0d,	Comment=' Equinox'

  prihead->addESOPar,'OCS TARG1 TYPE', 'unknown', Comment='Type of Target'
  prihead->addESOPar,'OCS TARG1 ID',   'unknown', Comment=" ID of target"
  prihead->addESOPar,'OCS TARG1 BCAL', 'no',      Comment=" BCal of target"
  prihead->addESOPar,'OCS TARG2 TYPE', 'unknown', Comment='Type of Target'
  prihead->addESOPar,'OCS TARG2 ID',   'unknown', Comment=" ID of target"
  prihead->addESOPar,'OCS TARG2 BCAL', 'no',      Comment=" BCal of target"

  prihead->addESOPar,'TPL DID',	'ESO-VLT-DIC.temp',	Comment=' TPL Dic'
  prihead->addESOPar,'TPL ID',	tpl_name, Comment=' TPL sicknature'
  prihead->addESOPar,'TPL NAME',tpl_name, Comment=' TPL name'
  prihead->addESOPar,'TPL NEXP', 1,	  Comment=' No. of expos within TPL'
  prihead->addESOPar,'TPL EXPNO',1,	  Comment=' Expo number within TPL'
  prihead->addESOPar,'TPL VERSION','Draft',Comment=' Version of the TPL (must be DRAFT)'
  prihead->addESOPar,'TPL PRESEQ', 'PACMA_do_something.seq',Comment=' Sequencer script'
  prihead->addESOPar,'TPL START',  'now', Comment=' TPL start time'

  if keyword_set(sim_code) then prihead->addESOpar,'SIM CODE',sim_code,Comment=' Simulation Code'
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function standard_array_description
  return, { array_description_row,$
            bintable:    string(format='(A-32)',"OPTICAL_TRAIN"),$
            opti_name:   string(format='(A-32)',"TELESCOPE"),$
            value:	   42,$
            description: string(format='(A-128)',"TELESCOPE_NAME_INGO"),$
            reference:   string(format='(A-128)',"VLT-ICD-ESO-12345-6789, p.42, links unten") }
end

Function standard_array_geometry
  return, { array_geometry_row,$
            Tel_Name: string(format='(A-8)','Ingo'),$
            Sta_Name: string(format='(A-8)','draussen'),$
            Sta_Index: 42,$
            Diameter:  1.7364,$
            StaXYZ: [ -1D99, 2D99, 36.D6 ],$
            MNTSTA: 0 }
end

Function standard_optical_train
  return, { OPTICAL_TRAIN_row,$
            Index: 1,$
            Tel_Name: 'INGO',$
            OPTI_NAME1: string(format='(A-32)','REAR APERTURE'),$
            Value1: 666 }
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; standard stuff in IMAGING_DETECTOR_FSU table
;;
PRO standard_img_detector_header, head, role, gain
  head->addPar, 'ROLE', role,	 Comment=' FSU role'
  head->addPar, 'NDETECT', 1,	 Comment=' No. of detectors used in the instrument (mostly 1)'
  head->addPar, 'NREGION', 4,	 Comment=' Number of regions used (always 4)'
  head->addPar, 'NUM_DIM', 'all',Comment=' Number of dimensions in the detector array'
  head->addPar, 'MAXTEL', 0,	 Comment=' Maximum number of contributing telescopes'

  head->addESOPar, 'ISS DID', 'JICD-1.0', Comment=' Data Dictionary name'
  head->addESOPar, 'ISS ID' , '1.0',	  Comment=' Data Dictionary version number'
  head->addESOPar, 'ISS DIP', '089-3200-6467', Comment=' Data Dictionary phone number'

  head->addESOPar, 'DET BITS', 16,	    Comment=' bit per pixel'
  head->addESOPar, 'DET CHIP NAME', 'Chio', Comment=' detector chip name'
  head->addESOPar, 'DET CHIP FLAVOR','X-Cut',Comment=' flavor of chip'
  head->addESOPar, 'DET CHIP GAIN', double(gain), Comment=' Detector gain (e^-1/ADU)'
  head->addESOPar, 'DET CHIP NX', 47,	    Comment=' No pix along X'
  head->addESOPar, 'DET CHIP PXSPACE', 1e3, Comment=' Pixel-Pixel spacing (m)'
  head->addESOPar, 'DET DID', 'Chio',	    Comment=' Data Dictionary Name'
  head->addESOPar, 'DET ID', 'Windows Vista', Comment=' Detector SW idea'
end

PRO standard_img_detector_table, detbl
  imaging_detector_FSU_row__define, tabdat
  tabdat.region= 1 & tabdat.regname='A' & tabdat.corner= [ 1, 1] & detbl->writeRows, tabdat
  tabdat.region= 2 & tabdat.regname='B' & tabdat.corner= [42, 1] & detbl->writeRows, tabdat
  tabdat.region= 3 & tabdat.regname='C' & tabdat.corner= [ 1,42] & detbl->writeRows, tabdat
  tabdat.region= 4 & tabdat.regname='D' & tabdat.corner= [42,42] & detbl->writeRows, tabdat
  ;;tabdat.region= 5 & tabdat.regname='Q' & tabdat.corner= [-1,-1] & detbl->writeRows, tabdat
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro standard_main_delay_header, head, sim_rate

  head->addESOPar, 'ISS DID', 'JICD-1.0', Comment=' Data Dictionary name'
  head->addESOPar, 'ISS ID' , '1.0',	  Comment=' Data Dictionary version number'
  head->addESOPar, 'ISS DIP', '089-3200-6467', Comment=' Data Dictionary phone number'
  head->addPar, 'MAXTEL', 0, Comment=' Number of virtual delay lines'
  head->addPar, 'DLNAME1', 'Nick', Comment=' Virtual name of delay line 1'
  head->addPar, 'DLNAME2', 'Nack', Comment=' Virtual name of delay line 2'
  head->addPar, 'DLBDAY1', '1962-05-14', Comment=' Virtual Birthday of delay line 1'
  head->addPar, 'DLBDAY2', 'unknown',    Comment=' Virtual Birthday of delay line 2'
  head->addPar, 'WLEN', 0.6328, Comment=' Wavelength of DL metro (micrometer)'

  head->addESOpar, 'SIM RATE', sim_rate, Comment=' data rate'
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro standard_metrology_header, head, sim_rate, sim_zero, sim_noise, FSUB=fsub

  c     = 299792458.	;; speed of light in m/s
  lambda= 1.3d-6

  lambda= 0.5d-6	;; CHANGED TO FULFILL NICK'S EXPECTATIONS.
			;; PROBABLY WRONG!  FIX ME, FIX ME, FIX ME PLEEEEAAAAASE!

  freq  = c/lambda

  head->addESOPar, 'ISS DID', 'JICD-1.0', Comment=' Data Dictionary name'
  head->addESOPar, 'ISS ID' , '1.0',	  Comment=' Data Dictionary version number'
  head->addESOPar, 'ISS DIP', '089-3200-6467', Comment=' Data Dictionary phone number'
  head->addPar, 'ZERO_VAL', 42.d, Comment=" Last zero'ed measured position"
  head->addPar, 'INDREF', 0., Comment=' Index of refraction'
  head->addPar, 'LASER_F', freq, Comment=' Celibate Laser wavelength (Hz)'
  head->addPar, 'F_SHIFT', 0.,   Comment=' Frequency shift between the metro lines'
  head->addPar, 'C',	c,	 Comment=' Speed of light (not of thought)'
  head->addPar, 'WLEN', lambda*1d6, Comment=' Celibate Laser wavelength (mircometer)'

  head->addPar, 'IP3865', 'IP3',  Comment=' VLTI input channel connected to +38.65Mhz'
  head->addPar, 'IP3800', 'IP1',  Comment=' VLTI input channel connected to 38Mhz'
  head->addPar, 'IP3955', 'IP4',  Comment=' VLTI input channel connected to -39.55 Mhz'
  head->addPar, 'IP4000', 'IP2',  Comment=' VLTI input channel connected to -40 Mhz'
  head->addPar, 'DEV650', 'FSUA', Comment=' Device to which Channel A of PRIMET is connected'
  head->addPar, 'DEV450', 'FSUB', Comment=' Device to which Channel B of PRIMET is connected'
  head->addPar, 'DEFOPDA', 'IP3-IP1',  Comment=' PRIMET definition of positive OPD for channel A'
  head->addPar, 'DEFOPDB', 'IP4-IP2',  Comment=' PRIMET definition of positive OPD for channel B'
  head->addPar, 'DELT_DEF', keyword_set(fsub) ? '-FSUB':'CHA-FSUB',$
                Comment=' Definition of positive DOPD'

  head->addESOPar,'SIM RATE', sim_rate, COMMENT=" data rate (Hz)"
  head->addESOPar,'SIM PRIMET0', sim_zero[0],$
				 Comment=" PRIMET-zeropoint constant term (m)"
  head->addESOPar,'SIM PRIMET1', sim_zero[1],$
				 Comment=" PRIMET-zeropoint linear term (m/s)"
  head->addESOPar,'SIM PRIMET NOISE',sim_noise,$
				 Comment=" PRIMET Gaussian noise (stddev in m)"
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO standard_targ_header, prihead, targid, type, targpos, FSU_cfg

  pos= { starpos }
  struct_assign, targpos, pos

  prihead->addESOPar,'ISS PRI FSU'+targid+' ROLE', type, Comment='Type of Target'

  ;; there's only one epoch in the header, so the last star wins :-)
  prihead->addESOPar,'OCS TARG EPOCH', pos.epoch, Comment=' Epoch'

  otkey= 'OCS TARG' + targid

  prihead->addESOPar,otkey+' TYPE', type, Comment='Type of Target'
  prihead->addESOPar,otkey+' NAME', pos.Name, Comment=" Name of target"
  prihead->addESOPar,otkey+' ID',   pos.Name, Comment=" ID of target"

  RAstr = " " + hours_to_hms(pos.RA) + " RA "+type+" (deg)"
  prihead->addESOPar,otkey+' ALPHA', pos.RA*15., Comment= RAstr
  prihead->addESOPar,otkey+' RA',    pos.RA*15., Comment= RAstr

  Decstr= " " + hours_to_hms(pos.Dec)+ " DEC "+type+" (deg)"
  prihead->addESOPar,otkey+' DEC',   pos.Dec, Comment= Decstr
  prihead->addESOPar,otkey+' DELTA', pos.Dec, Comment= Decstr

  prihead->addESOPar,otkey+' PMA',	pos.pmRA,  Comment= " proper motion in RA"
  prihead->addESOPar,otkey+' PMD',	pos.pmDec, Comment= " proper motion in Dec"
  prihead->addESOPar,otkey+' PARALLAX', pos.Parallax, Comment= " pralallax"

  prihead->addESOPar,otkey+' VMAG',  99.d,	 Comment= " Magnitude in V band"
  prihead->addESOPar,otkey+' KMAG',  99.d,	 Comment= " Magnitude in V band"
  prihead->addESOPar,otkey+' BCAL',  "yes",	 Comment= " Is this a baseline calibrator"
  prihead->addESOPar,otkey+' SPT',   'A2',	 Comment= " Spectral type"
  prihead->addESOPar,otkey+' TMP', FSU_cfg.star_temp, Comment= " Ineffective temperature"
  prihead->addESOPar,otkey+' RADVEL', pos.v_rad, Comment= " Radial Velocity"
  prihead->addESOPar,otkey+' WLENGTH',0.d,	 Comment= " Effective Wavelength"
  prihead->addESOPar,otkey+' VSINI', 9.d9,	 Comment= " Projected rotational velocity"
  prihead->addESOPar,otkey+' FEH',   9.d9,	 Comment= " Metallicity"
  prihead->addESOPar,otkey+' VIS', FSU_cfg.visi, Comment= " Visibillicity"
  prihead->addESOPar,otkey+' DMAS',FSU_cfg.star_angdiam, Comment= " Angular diameter [mas]"
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Mathar, R.J. 2007, Journal of Optics A, Pure and Applied Optics, Volume 9,
;;	Issue 5, 470-476.
;;
Function phaseRefIndexAirIR, temper, airprs, humidy, kappa

  Tref = 17.5d		 ;; Reference temperature (C)
  Pref = 75000.0d	 ;; Reference pressure (Pa)
  Href = 10.0d		 ;; Reference relative humidity (%)
  Kapparef= 1.0d6 / 2.25 ;; Reference wavenumber (m^{-1})

  ;; Check the inputs, insult the user if they're not ok

  if ( kappa lt 1.0d+06/2.5  or  Kappa gt 1.0d+06/1.3 ) then begin
     print, "The wavenumber ",kappa," has not the honor to be in the K-band"
     print, "You are a piece of baktag!  I will not work for you!!"
     return, -1.
  endif

  ;; Initialize the Mathar coefficients

  cref= [+0.200192D-03, +0.113474D-09, -0.424595D-14, +0.100957D-16, -0.293315D-20, +0.307228D-24 ]
  cT  = [+0.588625D-01, -0.385766D-07, +0.888019D-10, -0.567650D-13, +0.166615D-16, -0.174845D-20 ]
  cTT = [-3.015130D+00, +0.406167D-03, -0.514544D-06, +0.343161D-09, -0.101189D-12, +0.106749D-16 ]
  cP  = [+0.267085D-08, +0.135941D-14, +0.135295D-18, +0.818218D-23, -0.222957D-26, +0.249964D-30 ]
  cPP = [+0.609186D-17, +0.519024D-23, -0.419477D-27, +0.434120D-30, -0.122445D-33, +0.134816D-37 ]
  cH  = [-0.103945D-07, +0.136858D-11, -0.171039D-14, +0.112908D-17, -0.329925D-21, +0.344747D-25 ]
  cHH = [+0.573256D-12, +0.186367D-16, -0.228150D-19, +0.150947D-22, -0.441214D-26, +0.461209D-30 ]
  cTP = [+0.779176D-06, +0.396499D-12, +0.395114D-16, +0.233587D-20, -0.636441D-24, +0.716868D-28 ]
  cTH = [+0.497859D-04, -0.661752D-08, +0.832034D-11, -0.551793D-14, +0.161899D-17, -0.169901D-21 ]
  cPH = [-0.206567D-15, +0.106141D-20, -0.149982D-23, +0.984046D-27, -0.288266D-30, +0.299105D-34 ]

  ;; Calculate the phase refractive index of air

  Toff = (1.d/(temper+273.15) - 1.d/(Tref+273.15))	;; convert to Kelvin
  Poff = (airprs - Pref)
  Hoff = (humidy - Href)
  Kapoff= (Kappa-Kapparef) * 0.01	;; convert to cm^{-1}

  n = 1.0d	;; Initialize to the vacuum value

  for i= 0,5 do begin	;; Nick-debugged
     c = cref[i] + $
         cT[i]  * Toff   + $
         cTT[i] * Toff^2 + $
         cP[i]  * Poff   + $
         cPP[i] * Poff^2 + $
         cH[i]  * Hoff   + $
         cHH[i] * Hoff^2 + $
         cTP[i] * Toff * Poff + $
         cTH[i] * Toff * Hoff + $
         cPH[i] * Poff * Hoff
     n += c * Kapoff^i
  endfor

  return, n
end
;;
;; Mathar, R.J. 2007, Journal of Optics A, Pure and Applied Optics, Volume 9,
;;	Issue 5, 470-476.
;;
Function groupRefIndexAirIR, temper, airprs, humidy, kappa

  Tref = 17.5d		 ;; Reference temperature (C)
  Pref = 75000.0d	 ;; Reference pressure (Pa)
  Href = 10.0d		 ;; Reference relative humidity (%)
  Kapparef= 1.0d6 / 2.25 ;; Reference wavenumber (m^{-1})

  ;; Check the inputs, insult the user if they're not ok

  if ( kappa lt 1.0d+06/2.5  or  Kappa gt 1.0d+06/1.3 ) then begin
     print, "The wavenumber ",kappa," has not the honor to be in the K-band"
     print, "You are a piece of baktag!  I will not work for you!!"
     return, -1.
  endif

  ;; Initialize the Mathar coefficients

  cref= [+0.200192D-03, +0.113474D-09, -0.424595D-14, +0.100957D-16, -0.293315D-20, +0.307228D-24 ]
  cT  = [+0.588625D-01, -0.385766D-07, +0.888019D-10, -0.567650D-13, +0.166615D-16, -0.174845D-20 ]
  cTT = [-3.015130D+00, +0.406167D-03, -0.514544D-06, +0.343161D-09, -0.101189D-12, +0.106749D-16 ]
  cP  = [+0.267085D-08, +0.135941D-14, +0.135295D-18, +0.818218D-23, -0.222957D-26, +0.249964D-30 ]
  cPP = [+0.609186D-17, +0.519024D-23, -0.419477D-27, +0.434120D-30, -0.122445D-33, +0.134816D-37 ]
  cH  = [-0.103945D-07, +0.136858D-11, -0.171039D-14, +0.112908D-17, -0.329925D-21, +0.344747D-25 ]
  cHH = [+0.573256D-12, +0.186367D-16, -0.228150D-19, +0.150947D-22, -0.441214D-26, +0.461209D-30 ]
  cTP = [+0.779176D-06, +0.396499D-12, +0.395114D-16, +0.233587D-20, -0.636441D-24, +0.716868D-28 ]
  cTH = [+0.497859D-04, -0.661752D-08, +0.832034D-11, -0.551793D-14, +0.161899D-17, -0.169901D-21 ]
  cPH = [-0.206567D-15, +0.106141D-20, -0.149982D-23, +0.984046D-27, -0.288266D-30, +0.299105D-34 ]

  ;; Calculate the phase refractive index and its derivative of air

  Toff = (1.d/(temper+273.15) - 1.d/(Tref+273.15))	;; convert to Kelvin
  Poff = (airprs - Pref)
  Hoff = (humidy - Href)
  Kapoff= (Kappa-Kapparef) * 0.01	;; convert to cm^{-1}

  nPhase = 1.0d		;; Initialize to the vacuum value
  nPrime = 0.0d

  for i= 0,5 do begin
     c = cref[i] + $
         cT[i]  * Toff   + $
         cTT[i] * Toff^2 + $
         cP[i]  * Poff   + $
         cPP[i] * Poff^2 + $
         cH[i]  * Hoff   + $
         cHH[i] * Hoff^2 + $
         cTP[i] * Toff * Poff + $
         cTH[i] * Toff * Hoff + $
         cPH[i] * Poff * Hoff
     nPhase += c * Kapoff^i
     nPrime += c * double(i) * Kapoff^(i-1)
  endfor

  return, nPhase + Kappa*0.01 * nPrime
end
;;
;; Result: dispersion factor, where d_measured = dispf * d_real
;;
Function disperse_metroB, MJD0, time, NICK=nick
  ;; time in seconds

  hour = (MJD0-floor(MJD0))*24.d + 12.d	;; hours since last UT-noon
  etime= hour * 3600.d + time		;; time used by environment

  temper= double(environment_value( 1,etime))		;; Celsius!
  airprs= double(environment_value(31,etime)*100.)	;; hPa -> Pa
  humidy= double(environment_value(35,etime))

  kappa_FSU= 4.364493d+05	;; 1/m
  kapprimet= 7.692307d+05

  dispf= dblarr(N_elements(time))

  ;;print,"first timestamp :",time[0],etime[0],temper[0], airprs[0], humidy[0]

  if keyword_set(nick) then begin
     for i=0L,N_elements(time)-1 do begin
        ;;print,i,temper[i], airprs[i], humidy[i], kappa_FSU, kapprimet
        nFSU= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','groupRefIndexAirIR',$
                            temper[i], airprs[i], humidy[i], kappa_FSU,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
        nPMT= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','phaseRefIndexAirIR',$
                            temper[i], airprs[i], humidy[i], kapprimet,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
        dispf[i]= nPMT/nFSU
        if i eq 0 then print,"Nick's first dispersion:", nFSU,nPMT,dispf[0]
     endfor
     print," last dispersion:",nFSU,nPMT,dispf[i-1]
  endif else begin
     ;;print,format='("My first dispersion:",$)'
     nFSU= groupRefIndexAirIR( temper, airprs, humidy, kappa_FSU)
     nPMT= phaseRefIndexAirIR( temper, airprs, humidy, kapprimet)
     dispf = nPMT/nFSU
     ;;print, nFSU[0],nPMT[0],dispf[0]
  endelse
  return,dispf
end


pro test_disperse

  temp0= 17.5d
  pres0= 75000.d
  humi0= 10.d

  kappa= 4.364493d+05	;; FSU, 1/m
  kappa= 7.692307d+05	;; PRImet

  for temp=0.d, 30.d, 3. do begin
     nick= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','phaseRefIndexAirIR',$
                         temp, pres0, humi0, kappa,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
     mainz= phaseRefIndexAirIR(temp, pres0, humi0, kappa)
     print,temp,nick,mainz,nick-mainz
  endfor
  for pres= 70000.d,80000.d,1000.d do begin
     nick= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','phaseRefIndexAirIR',$
                         temp0, pres, humi0, kappa,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
     mainz= phaseRefIndexAirIR(temp0, pres, humi0, kappa)
     print,pres,nick,mainz,nick-mainz
  endfor
  for humi=0.d,100.,10. do begin
     nick= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','phaseRefIndexAirIR',$
                         temp0, pres0, humi, kappa,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
     mainz= phaseRefIndexAirIR(temp0, pres0, humi, kappa)
     print,humi,nick,mainz,nick-mainz
  endfor
  print,"GROUP"
  for temp=0.d, 30.d, 3. do begin
     nick= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','groupRefIndexAirIR',$
                         temp, pres0, humi0, kappa,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
     mainz= groupRefIndexAirIR(temp, pres0, humi0, kappa)
     print,temp,nick,mainz,nick-mainz
  endfor
  for pres= 70000.d,80000.d,1000.d do begin
     nick= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','groupRefIndexAirIR',$
                         temp0, pres, humi0, kappa,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
     mainz= groupRefIndexAirIR(temp0, pres, humi0, kappa)
     print,pres,nick,mainz,nick-mainz
  endfor
  for humi=0.d,100.,10. do begin
     nick= call_external('/home/rkoehler/PRIMA/simulator/refIndexAir.so','groupRefIndexAirIR',$
                         temp0, pres0, humi, kappa,/AUTO_GLUE,/CDECL,/ALL_VALUE,/D_VALUE)
     mainz= groupRefIndexAirIR(temp0, pres0, humi, kappa)
     print,humi,nick,mainz,nick-mainz
  endfor
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

pro baseline, filename, TIMES=times, LENGTH=base_len, ANGLE=base_pa, QUIET=quiet

  im = Obj_new('fitsfile',filename[0],'READ')
  he = im->prihead()
  objectname= STRCOMPRESS(he->getpar('OBS TARG NAME'),/remove_all)
  esolen = StrCompress(he->getpar('HIERARCH ESO ISS PBL12 START'),/remove_all)
  esoPA  = StrCompress(he->getpar('HIERARCH ESO ISS PBLA12 START'),/remove_all)

  stat1  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION1'),/remove_all)
  stat2  = StrCompress(he->getpar('HIERARCH ESO ISS CONF STATION2'),/remove_all)

  tel1x  = double(he->getpar('HIERARCH ESO ISS CONF T1X'))
  tel1y  = double(he->getpar('HIERARCH ESO ISS CONF T1Y'))
  tel1z  = double(he->getpar('HIERARCH ESO ISS CONF T1Z'))
  tel2x  = double(he->getpar('HIERARCH ESO ISS CONF T2X'))
  tel2y  = double(he->getpar('HIERARCH ESO ISS CONF T2Y'))
  tel2z  = double(he->getpar('HIERARCH ESO ISS CONF T2Z'))

  date   = StrCompress(he->getpar('DATE-OBS'),/remove_all)
  exptime= fix(he->getpar('EXPTIME')) > 1
  lst    = double(he->getpar('LST'))
  RA     = double(he->getpar('RA') )
  Dec    = double(he->getpar('DEC'))
  obj_destroy,im

  if (stat1 gt stat2) then begin
      tmp = stat1 & stat1 = stat2 & stat2 = tmp
  endif

  if not keyword_set(quiet) then $
     print, "Baseline:            ", stat1, " - ", stat2

  stations = stat1+"-"+stat2

  BaselineLen= -1.

  case stations of
  "U1-U2": begin & BaselineLen=  56.569 & BaselinePA=  26.016 & end
  "U1-U3": begin & BaselineLen= 102.434 & BaselinePA=  32.369 & end
  "U1-U4": begin & BaselineLen= 130.231 & BaselinePA=  60.396 & end
  "U2-U3": begin & BaselineLen=  46.635 & BaselinePA=  40.082 & end
  "U2-U4": begin & BaselineLen=  89.443 & BaselinePA=  81.321 & end
  "U3-U4": begin & BaselineLen=  62.463 & BaselinePA= 110.803 & end
  "A0-B0": begin & BaselineLen=   8.010 & BaselinePA=   71.059 & end
  "A0-B1": begin & BaselineLen=  17.891 & BaselinePA=  134.419 & end
  "A0-B2": begin & BaselineLen=  25.299 & BaselinePA=  142.558 & end
  "A0-B3": begin & BaselineLen=  33.003 & BaselinePA=  146.970 & end
  "A0-B4": begin & BaselineLen=  40.794 & BaselinePA=  149.692 & end
  "A0-B5": begin & BaselineLen=  48.663 & BaselinePA=  151.542 & end
  "A0-C0": begin & BaselineLen=  15.999 & BaselinePA=   71.016 & end
  "A0-C1": begin & BaselineLen=  22.625 & BaselinePA=  116.014 & end
  "A0-C2": begin & BaselineLen=  28.849 & BaselinePA=  127.334 & end
  "A0-C3": begin & BaselineLen=  35.774 & BaselinePA=  134.450 & end
  "A0-D0": begin & BaselineLen=  32.011 & BaselinePA=   71.014 & end
  "A0-D1": begin & BaselineLen=  45.264 & BaselinePA=  116.008 & end
  "A0-D2": begin & BaselineLen=  57.694 & BaselinePA=  127.316 & end
  "A0-E0": begin & BaselineLen=  48.012 & BaselinePA=   71.020 & end
  "A0-G0": begin & BaselineLen=  64.018 & BaselinePA=   71.020 & end
  "A0-G1": begin & BaselineLen=  90.522 & BaselinePA=  116.005 & end
  "A0-G2": begin & BaselineLen=  68.352 & BaselinePA=   50.451 & end
  "A0-H0": begin & BaselineLen=  96.016 & BaselinePA=   71.012 & end
  "A0-I1": begin & BaselineLen= 111.423 & BaselinePA=   92.045 & end
  "A0-J1": begin & BaselineLen= 122.389 & BaselinePA=   82.315 & end
  "A0-J2": begin & BaselineLen= 129.257 & BaselinePA=   92.811 & end
  "A0-J3": begin & BaselineLen= 132.443 & BaselinePA=   45.999 & end
  "A0-J4": begin & BaselineLen= 139.961 & BaselinePA=   40.054 & end
  "A0-J5": begin & BaselineLen= 153.688 & BaselinePA=   32.360 & end
  "A0-J6": begin & BaselineLen= 169.720 & BaselinePA=   26.019 & end
  "A0-K0": begin & BaselineLen= 128.003 & BaselinePA=   71.013 & end
  "A0-L0": begin & BaselineLen= 136.022 & BaselinePA=   71.010 & end
  "A0-M0": begin & BaselineLen= 144.014 & BaselinePA=   71.011 & end
  "A0-U1": begin & BaselineLen=  35.789 & BaselinePA=    7.573 & end
  "A0-U2": begin & BaselineLen=  91.225 & BaselinePA=   18.886 & end
  "A0-U3": begin & BaselineLen= 135.756 & BaselinePA=   26.021 & end
  "A0-U4": begin & BaselineLen= 154.511 & BaselinePA=   49.761 & end
  "A1-B0": begin & BaselineLen=  17.895 & BaselinePA=    7.607 & end
  "A1-B1": begin & BaselineLen=   8.010 & BaselinePA=   70.944 & end
  "A1-B2": begin & BaselineLen=  11.314 & BaselinePA=  115.944 & end
  "A1-B3": begin & BaselineLen=  17.900 & BaselinePA=  134.434 & end
  "A1-B4": begin & BaselineLen=  25.294 & BaselinePA=  142.554 & end
  "A1-B5": begin & BaselineLen=  32.979 & BaselinePA=  146.959 & end
  "A1-C0": begin & BaselineLen=  22.632 & BaselinePA=   26.000 & end
  "A1-C1": begin & BaselineLen=  15.999 & BaselinePA=   70.980 & end
  "A1-C2": begin & BaselineLen=  17.887 & BaselinePA=   97.577 & end
  "A1-C3": begin & BaselineLen=  22.619 & BaselinePA=  115.998 & end
  "A1-D0": begin & BaselineLen=  35.791 & BaselinePA=   44.446 & end
  "A1-D1": begin & BaselineLen=  35.784 & BaselinePA=   97.565 & end
  "A1-D2": begin & BaselineLen=  45.256 & BaselinePA=  115.998 & end
  "A1-E0": begin & BaselineLen=  50.609 & BaselinePA=   52.580 & end
  "A1-G0": begin & BaselineLen=  65.988 & BaselinePA=   56.980 & end
  "A1-G1": begin & BaselineLen=  80.010 & BaselinePA=  107.871 & end
  "A1-G2": begin & BaselineLen=  75.478 & BaselinePA=   38.997 & end
  "A1-H0": begin & BaselineLen=  97.342 & BaselinePA=   61.547 & end
  "A1-I1": begin & BaselineLen= 106.730 & BaselinePA=   83.998 & end
  "A1-J1": begin & BaselineLen= 120.281 & BaselinePA=   74.816 & end
  "A1-J2": begin & BaselineLen= 124.206 & BaselinePA=   85.938 & end
  "A1-J3": begin & BaselineLen= 139.966 & BaselinePA=   40.050 & end
  "A1-J4": begin & BaselineLen= 148.831 & BaselinePA=   34.762 & end
  "A1-J5": begin & BaselineLen= 164.164 & BaselinePA=   27.993 & end
  "A1-J6": begin & BaselineLen= 181.392 & BaselinePA=   22.441 & end
  "A1-K0": begin & BaselineLen= 129.001 & BaselinePA=   63.885 & end
  "A1-L0": begin & BaselineLen= 136.962 & BaselinePA=   64.298 & end
  "A1-M0": begin & BaselineLen= 144.902 & BaselinePA=   64.668 & end
  "A1-U1": begin & BaselineLen=  50.617 & BaselinePA=   -0.555 & end
  "A1-U2": begin & BaselineLen= 104.325 & BaselinePA=   13.481 & end
  "A1-U3": begin & BaselineLen= 147.509 & BaselinePA=   21.620 & end
  "A1-U4": begin & BaselineLen= 161.007 & BaselinePA=   44.444 & end
  "B0-C0": begin & BaselineLen=   7.989 & BaselinePA=   70.973 & end
  "B0-C1": begin & BaselineLen=  17.876 & BaselinePA=  134.471 & end
  "B0-C2": begin & BaselineLen=  25.295 & BaselinePA=  142.605 & end
  "B0-C3": begin & BaselineLen=  32.973 & BaselinePA=  146.995 & end
  "B0-D0": begin & BaselineLen=  24.001 & BaselinePA=   70.999 & end
  "B0-D1": begin & BaselineLen=  39.997 & BaselinePA=  124.142 & end
  "B0-D2": begin & BaselineLen=  53.660 & BaselinePA=  134.447 & end
  "B0-E0": begin & BaselineLen=  40.002 & BaselinePA=   71.012 & end
  "B0-G0": begin & BaselineLen=  56.008 & BaselinePA=   71.014 & end
  "B0-G1": begin & BaselineLen=  85.042 & BaselinePA=  119.821 & end
  "B0-G2": begin & BaselineLen=  60.920 & BaselinePA=   47.798 & end
  "B0-H0": begin & BaselineLen=  88.006 & BaselinePA=   71.008 & end
  "B0-I1": begin & BaselineLen= 103.984 & BaselinePA=   93.626 & end
  "B0-J1": begin & BaselineLen= 114.544 & BaselinePA=   83.097 & end
  "B0-J2": begin & BaselineLen= 121.853 & BaselinePA=   94.207 & end
  "B0-J3": begin & BaselineLen= 125.233 & BaselinePA=   44.446 & end
  "B0-J4": begin & BaselineLen= 133.159 & BaselinePA=   38.278 & end
  "B0-J5": begin & BaselineLen= 147.522 & BaselinePA=   30.415 & end
  "B0-J6": begin & BaselineLen= 164.158 & BaselinePA=   24.041 & end
  "B0-K0": begin & BaselineLen= 119.993 & BaselinePA=   71.010 & end
  "B0-L0": begin & BaselineLen= 128.012 & BaselinePA=   71.007 & end
  "B0-M0": begin & BaselineLen= 136.004 & BaselinePA=   71.008 & end
  "B0-U1": begin & BaselineLen=  33.001 & BaselinePA=   -4.971 & end
  "B0-U2": begin & BaselineLen=  86.544 & BaselinePA=   14.694 & end
  "B0-U3": begin & BaselineLen= 130.219 & BaselinePA=   23.526 & end
  "B0-U4": begin & BaselineLen= 147.077 & BaselinePA=   48.628 & end
  "B1-C0": begin & BaselineLen=  17.882 & BaselinePA=    7.552 & end
  "B1-C1": begin & BaselineLen=   7.989 & BaselinePA=   71.016 & end
  "B1-C2": begin & BaselineLen=  11.312 & BaselinePA=  116.084 & end
  "B1-C3": begin & BaselineLen=  17.883 & BaselinePA=  134.481 & end
  "B1-D0": begin & BaselineLen=  28.845 & BaselinePA=   37.329 & end
  "B1-D1": begin & BaselineLen=  28.847 & BaselinePA=  104.712 & end
  "B1-D2": begin & BaselineLen=  40.001 & BaselinePA=  124.146 & end
  "B1-E0": begin & BaselineLen=  43.081 & BaselinePA=   49.222 & end
  "B1-G0": begin & BaselineLen=  58.247 & BaselinePA=   55.079 & end
  "B1-G1": begin & BaselineLen=  73.764 & BaselinePA=  111.611 & end
  "B1-G2": begin & BaselineLen=  68.812 & BaselinePA=   35.466 & end
  "B1-H0": begin & BaselineLen=  89.449 & BaselinePA=   60.709 & end
  "B1-I1": begin & BaselineLen=  98.943 & BaselinePA=   85.046 & end
  "B1-J1": begin & BaselineLen= 112.291 & BaselinePA=   75.092 & end
  "B1-J2": begin & BaselineLen= 116.487 & BaselinePA=   86.958 & end
  "B1-J3": begin & BaselineLen= 133.156 & BaselinePA=   38.280 & end
  "B1-J4": begin & BaselineLen= 142.444 & BaselinePA=   32.859 & end
  "B1-J5": begin & BaselineLen= 158.395 & BaselinePA=   26.018 & end
  "B1-J6": begin & BaselineLen= 176.187 & BaselinePA=   20.490 & end
  "B1-K0": begin & BaselineLen= 121.056 & BaselinePA=   63.419 & end
  "B1-L0": begin & BaselineLen= 129.010 & BaselinePA=   63.886 & end
  "B1-M0": begin & BaselineLen= 136.943 & BaselinePA=   64.302 & end
  "B1-U1": begin & BaselineLen=  48.671 & BaselinePA=   -9.534 & end
  "B1-U2": begin & BaselineLen= 100.245 & BaselinePA=    9.619 & end
  "B1-U3": begin & BaselineLen= 142.418 & BaselinePA=   19.175 & end
  "B1-U4": begin & BaselineLen= 153.880 & BaselinePA=   43.114 & end
  "B2-C0": begin & BaselineLen=  25.293 & BaselinePA=   -0.571 & end
  "B2-C1": begin & BaselineLen=  11.306 & BaselinePA=   25.977 & end
  "B2-C2": begin & BaselineLen=   7.989 & BaselinePA=   71.073 & end
  "B2-C3": begin & BaselineLen=  11.305 & BaselinePA=  116.052 & end
  "B2-D0": begin & BaselineLen=  33.941 & BaselinePA=   26.018 & end
  "B2-D1": begin & BaselineLen=  25.300 & BaselinePA=   89.459 & end
  "B2-D2": begin & BaselineLen=  33.943 & BaselinePA=  116.016 & end
  "B2-E0": begin & BaselineLen=  46.647 & BaselinePA=   40.059 & end
  "B2-G0": begin & BaselineLen=  60.931 & BaselinePA=   47.826 & end
  "B2-G1": begin & BaselineLen=  68.827 & BaselinePA=  106.548 & end
  "B2-G2": begin & BaselineLen=  73.751 & BaselinePA=   30.403 & end
  "B2-H0": begin & BaselineLen=  91.221 & BaselinePA=   55.759 & end
  "B2-I1": begin & BaselineLen=  97.314 & BaselinePA=   80.471 & end
  "B2-J1": begin & BaselineLen= 112.007 & BaselinePA=   71.006 & end
  "B2-J2": begin & BaselineLen= 114.548 & BaselinePA=   83.107 & end
  "B2-J3": begin & BaselineLen= 137.647 & BaselinePA=   35.478 & end
  "B2-J4": begin & BaselineLen= 147.521 & BaselinePA=   30.415 & end
  "B2-J5": begin & BaselineLen= 164.150 & BaselinePA=   24.043 & end
  "B2-J6": begin & BaselineLen= 182.433 & BaselinePA=   18.893 & end
  "B2-K0": begin & BaselineLen= 122.371 & BaselinePA=   59.703 & end
  "B2-L0": begin & BaselineLen= 130.245 & BaselinePA=   60.392 & end
  "B2-M0": begin & BaselineLen= 138.107 & BaselinePA=   61.004 & end
  "B2-U1": begin & BaselineLen=  56.578 & BaselinePA=  -10.864 & end
  "B2-U2": begin & BaselineLen= 107.337 & BaselinePA=    7.574 & end
  "B2-U3": begin & BaselineLen= 148.791 & BaselinePA=   17.271 & end
  "B2-U4": begin & BaselineLen= 157.782 & BaselinePA=   40.545 & end
  "B3-C0": begin & BaselineLen=  32.998 & BaselinePA=   -4.973 & end
  "B3-C1": begin & BaselineLen=  17.900 & BaselinePA=    7.524 & end
  "B3-C2": begin & BaselineLen=  11.313 & BaselinePA=   25.941 & end
  "B3-C3": begin & BaselineLen=   7.989 & BaselinePA=   70.880 & end
  "B3-D0": begin & BaselineLen=  40.014 & BaselinePA=   17.872 & end
  "B3-D1": begin & BaselineLen=  24.001 & BaselinePA=   70.983 & end
  "B3-D2": begin & BaselineLen=  28.836 & BaselinePA=  104.677 & end
  "B3-E0": begin & BaselineLen=  51.235 & BaselinePA=   32.346 & end
  "B3-G0": begin & BaselineLen=  64.511 & BaselinePA=   41.266 & end
  "B3-G1": begin & BaselineLen=  64.498 & BaselinePA=  100.741 & end
  "B3-G2": begin & BaselineLen=  79.204 & BaselinePA=   25.996 & end
  "B3-H0": begin & BaselineLen=  93.651 & BaselinePA=   51.021 & end
  "B3-I1": begin & BaselineLen=  96.322 & BaselinePA=   75.761 & end
  "B3-J1": begin & BaselineLen= 112.295 & BaselinePA=   66.912 & end
  "B3-J2": begin & BaselineLen= 113.141 & BaselinePA=   79.134 & end
  "B3-J3": begin & BaselineLen= 142.457 & BaselinePA=   32.853 & end
  "B3-J4": begin & BaselineLen= 152.860 & BaselinePA=   28.133 & end
  "B3-J5": begin & BaselineLen= 170.099 & BaselinePA=   22.200 & end
  "B3-J6": begin & BaselineLen= 188.826 & BaselinePA=   17.399 & end
  "B3-K0": begin & BaselineLen= 124.193 & BaselinePA=   56.073 & end
  "B3-L0": begin & BaselineLen= 131.959 & BaselinePA=   56.968 & end
  "B3-M0": begin & BaselineLen= 139.725 & BaselinePA=   57.764 & end
  "B3-U1": begin & BaselineLen=  64.526 & BaselinePA=  -11.870 & end
  "B3-U2": begin & BaselineLen= 114.565 & BaselinePA=    5.781 & end
  "B3-U3": begin & BaselineLen= 155.329 & BaselinePA=   15.522 & end
  "B3-U4": begin & BaselineLen= 161.996 & BaselinePA=   38.100 & end
  "B4-C0": begin & BaselineLen=  40.790 & BaselinePA=   -7.689 & end
  "B4-C1": begin & BaselineLen=  25.297 & BaselinePA=   -0.574 & end
  "B4-C2": begin & BaselineLen=  17.878 & BaselinePA=    7.558 & end
  "B4-C3": begin & BaselineLen=  11.308 & BaselinePA=   25.966 & end
  "B4-D0": begin & BaselineLen=  46.649 & BaselinePA=   11.980 & end
  "B4-D1": begin & BaselineLen=  25.299 & BaselinePA=   52.586 & end
  "B4-D2": begin & BaselineLen=  25.299 & BaselinePA=   89.448 & end
  "B4-E0": begin & BaselineLen=  56.568 & BaselinePA=   26.020 & end
  "B4-G0": begin & BaselineLen=  68.823 & BaselinePA=   35.485 & end
  "B4-G1": begin & BaselineLen=  60.935 & BaselinePA=   94.208 & end
  "B4-G2": begin & BaselineLen=  85.039 & BaselinePA=   22.190 & end
  "B4-H0": begin & BaselineLen=  96.672 & BaselinePA=   46.570 & end
  "B4-I1": begin & BaselineLen=  95.992 & BaselinePA=   71.006 & end
  "B4-J1": begin & BaselineLen= 113.147 & BaselinePA=   62.876 & end
  "B4-J2": begin & BaselineLen= 112.292 & BaselinePA=   75.097 & end
  "B4-J3": begin & BaselineLen= 147.524 & BaselinePA=   30.414 & end
  "B4-J4": begin & BaselineLen= 158.401 & BaselinePA=   26.016 & end
  "B4-J5": begin & BaselineLen= 176.186 & BaselinePA=   20.490 & end
  "B4-J6": begin & BaselineLen= 195.311 & BaselinePA=   16.009 & end
  "B4-K0": begin & BaselineLen= 126.487 & BaselinePA=   52.577 & end
  "B4-L0": begin & BaselineLen= 134.120 & BaselinePA=   53.657 & end
  "B4-M0": begin & BaselineLen= 141.768 & BaselinePA=   54.622 & end
  "B4-U1": begin & BaselineLen=  72.455 & BaselinePA=  -12.652 & end
  "B4-U2": begin & BaselineLen= 121.861 & BaselinePA=    4.208 & end
  "B4-U3": begin & BaselineLen= 161.971 & BaselinePA=   13.922 & end
  "B4-U4": begin & BaselineLen= 166.469 & BaselinePA=   35.793 & end
  "B5-C0": begin & BaselineLen=  48.659 & BaselinePA=   -9.534 & end
  "B5-C1": begin & BaselineLen=  32.983 & BaselinePA=   -4.967 & end
  "B5-C2": begin & BaselineLen=  25.288 & BaselinePA=   -0.568 & end
  "B5-C3": begin & BaselineLen=  17.885 & BaselinePA=    7.547 & end
  "B5-D0": begin & BaselineLen=  53.666 & BaselinePA=    7.582 & end
  "B5-D1": begin & BaselineLen=  28.844 & BaselinePA=   37.332 & end
  "B5-D2": begin & BaselineLen=  24.001 & BaselinePA=   71.016 & end
  "B5-E0": begin & BaselineLen=  62.480 & BaselinePA=   20.825 & end
  "B5-G0": begin & BaselineLen=  73.759 & BaselinePA=   30.422 & end
  "B5-G1": begin & BaselineLen=  58.251 & BaselinePA=   86.957 & end
  "B5-G2": begin & BaselineLen=  91.213 & BaselinePA=   18.881 & end
  "B5-H0": begin & BaselineLen= 100.247 & BaselinePA=   42.405 & end
  "B5-I1": begin & BaselineLen=  96.326 & BaselinePA=   66.243 & end
  "B5-J1": begin & BaselineLen= 114.554 & BaselinePA=   58.912 & end
  "B5-J2": begin & BaselineLen= 112.007 & BaselinePA=   71.012 & end
  "B5-J3": begin & BaselineLen= 152.850 & BaselinePA=   28.137 & end
  "B5-J4": begin & BaselineLen= 164.155 & BaselinePA=   24.042 & end
  "B5-J5": begin & BaselineLen= 182.432 & BaselinePA=   18.893 & end
  "B5-J6": begin & BaselineLen= 201.916 & BaselinePA=   14.707 & end
  "B5-K0": begin & BaselineLen= 129.240 & BaselinePA=   49.211 & end
  "B5-L0": begin & BaselineLen= 136.720 & BaselinePA=   50.456 & end
  "B5-M0": begin & BaselineLen= 144.230 & BaselinePA=   51.572 & end
  "B5-U1": begin & BaselineLen=  80.410 & BaselinePA=  -13.281 & end
  "B5-U2": begin & BaselineLen= 129.252 & BaselinePA=    2.812 & end
  "B5-U3": begin & BaselineLen= 168.743 & BaselinePA=   12.446 & end
  "B5-U4": begin & BaselineLen= 171.208 & BaselinePA=   33.606 & end
  "C0-D0": begin & BaselineLen=  16.012 & BaselinePA=   71.012 & end
  "C0-D1": begin & BaselineLen=  35.784 & BaselinePA=  134.435 & end
  "C0-D2": begin & BaselineLen=  50.599 & BaselinePA=  142.568 & end
  "C0-E0": begin & BaselineLen=  32.013 & BaselinePA=   71.021 & end
  "C0-G0": begin & BaselineLen=  48.019 & BaselinePA=   71.021 & end
  "C0-G1": begin & BaselineLen=  80.011 & BaselinePA=  124.132 & end
  "C0-G2": begin & BaselineLen=  53.667 & BaselinePA=   44.440 & end
  "C0-H0": begin & BaselineLen=  80.017 & BaselinePA=   71.012 & end
  "C0-I1": begin & BaselineLen=  96.660 & BaselinePA=   95.451 & end
  "C0-J1": begin & BaselineLen= 106.746 & BaselinePA=   83.997 & end
  "C0-J2": begin & BaselineLen= 114.556 & BaselinePA=   95.784 & end
  "C0-J3": begin & BaselineLen= 118.139 & BaselinePA=   42.716 & end
  "C0-J4": begin & BaselineLen= 126.509 & BaselinePA=   36.323 & end
  "C0-J5": begin & BaselineLen= 141.548 & BaselinePA=   28.312 & end
  "C0-J6": begin & BaselineLen= 158.810 & BaselinePA=   21.935 & end
  "C0-K0": begin & BaselineLen= 112.004 & BaselinePA=   71.012 & end
  "C0-L0": begin & BaselineLen= 120.023 & BaselinePA=   71.009 & end
  "C0-M0": begin & BaselineLen= 128.015 & BaselinePA=   71.010 & end
  "C0-U1": begin & BaselineLen=  32.013 & BaselinePA=  -18.980 & end
  "C0-U2": begin & BaselineLen=  82.377 & BaselinePA=   10.067 & end
  "C0-U3": begin & BaselineLen= 124.955 & BaselinePA=   20.827 & end
  "C0-U4": begin & BaselineLen= 139.721 & BaselinePA=   47.382 & end
  "C1-D0": begin & BaselineLen=  22.635 & BaselinePA=   26.039 & end
  "C1-D1": begin & BaselineLen=  22.639 & BaselinePA=  116.002 & end
  "C1-D2": begin & BaselineLen=  35.783 & BaselinePA=  134.434 & end
  "C1-E0": begin & BaselineLen=  35.786 & BaselinePA=   44.467 & end
  "C1-G0": begin & BaselineLen=  50.613 & BaselinePA=   52.595 & end
  "C1-G1": begin & BaselineLen=  67.897 & BaselinePA=  116.002 & end
  "C1-G2": begin & BaselineLen=  62.485 & BaselinePA=   31.203 & end
  "C1-H0": begin & BaselineLen=  81.602 & BaselinePA=   59.706 & end
  "C1-I1": begin & BaselineLen=  91.213 & BaselinePA=   86.262 & end
  "C1-J1": begin & BaselineLen= 104.324 & BaselinePA=   75.404 & end
  "C1-J2": begin & BaselineLen= 108.827 & BaselinePA=   88.113 & end
  "C1-J3": begin & BaselineLen= 126.510 & BaselinePA=   36.323 & end
  "C1-J4": begin & BaselineLen= 136.251 & BaselinePA=   30.783 & end
  "C1-J5": begin & BaselineLen= 152.851 & BaselinePA=   23.900 & end
  "C1-J6": begin & BaselineLen= 171.219 & BaselinePA=   18.426 & end
  "C1-K0": begin & BaselineLen= 113.142 & BaselinePA=   62.884 & end
  "C1-L0": begin & BaselineLen= 121.086 & BaselinePA=   63.417 & end
  "C1-M0": begin & BaselineLen= 129.012 & BaselinePA=   63.887 & end
  "C1-U1": begin & BaselineLen=  48.011 & BaselinePA=  -18.982 & end
  "C1-U2": begin & BaselineLen=  96.675 & BaselinePA=    5.458 & end
  "C1-U3": begin & BaselineLen= 137.626 & BaselinePA=   16.559 & end
  "C1-U4": begin & BaselineLen= 146.868 & BaselinePA=   41.655 & end
  "C2-D0": begin & BaselineLen=  28.857 & BaselinePA=   14.718 & end
  "C2-D1": begin & BaselineLen=  17.897 & BaselinePA=   97.552 & end
  "C2-D2": begin & BaselineLen=  28.845 & BaselinePA=  127.298 & end
  "C2-E0": begin & BaselineLen=  40.012 & BaselinePA=   34.154 & end
  "C2-G0": begin & BaselineLen=  53.683 & BaselinePA=   44.458 & end
  "C2-G1": begin & BaselineLen=  62.493 & BaselinePA=  110.802 & end
  "C2-G2": begin & BaselineLen=  67.891 & BaselinePA=   26.005 & end
  "C2-H0": begin & BaselineLen=  83.542 & BaselinePA=   54.312 & end
  "C2-I1": begin & BaselineLen=  89.442 & BaselinePA=   81.307 & end
  "C2-J1": begin & BaselineLen= 104.018 & BaselinePA=   71.001 & end
  "C2-J2": begin & BaselineLen= 106.748 & BaselinePA=   84.001 & end
  "C2-J3": begin & BaselineLen= 131.233 & BaselinePA=   33.447 & end
  "C2-J4": begin & BaselineLen= 141.556 & BaselinePA=   28.308 & end
  "C2-J5": begin & BaselineLen= 158.812 & BaselinePA=   21.934 & end
  "C2-J6": begin & BaselineLen= 177.647 & BaselinePA=   16.857 & end
  "C2-K0": begin & BaselineLen= 114.549 & BaselinePA=   58.915 & end
  "C2-L0": begin & BaselineLen= 122.403 & BaselinePA=   59.699 & end
  "C2-M0": begin & BaselineLen= 130.249 & BaselinePA=   60.389 & end
  "C2-U1": begin & BaselineLen=  56.019 & BaselinePA=  -18.982 & end
  "C2-U2": begin & BaselineLen= 104.018 & BaselinePA=    3.633 & end
  "C2-U3": begin & BaselineLen= 144.217 & BaselinePA=   14.709 & end
  "C2-U4": begin & BaselineLen= 150.955 & BaselinePA=   39.005 & end
  "C3-D0": begin & BaselineLen=  35.781 & BaselinePA=    7.600 & end
  "C3-D1": begin & BaselineLen=  16.012 & BaselinePA=   71.034 & end
  "C3-D2": begin & BaselineLen=  22.637 & BaselinePA=  115.998 & end
  "C3-E0": begin & BaselineLen=  45.260 & BaselinePA=   26.033 & end
  "C3-G0": begin & BaselineLen=  57.701 & BaselinePA=   37.342 & end
  "C3-G1": begin & BaselineLen=  57.707 & BaselinePA=  104.694 & end
  "C3-G2": begin & BaselineLen=  73.760 & BaselinePA=   21.612 & end
  "C3-H0": begin & BaselineLen=  86.180 & BaselinePA=   49.217 & end
  "C3-I1": begin & BaselineLen=  88.365 & BaselinePA=   76.202 & end
  "C3-J1": begin & BaselineLen= 104.327 & BaselinePA=   66.608 & end
  "C3-J2": begin & BaselineLen= 105.241 & BaselinePA=   79.758 & end
  "C3-J3": begin & BaselineLen= 136.253 & BaselinePA=   30.783 & end
  "C3-J4": begin & BaselineLen= 147.093 & BaselinePA=   26.020 & end
  "C3-J5": begin & BaselineLen= 164.933 & BaselinePA=   20.115 & end
  "C3-J6": begin & BaselineLen= 184.184 & BaselinePA=   15.401 & end
  "C3-K0": begin & BaselineLen= 116.487 & BaselinePA=   55.069 & end
  "C3-L0": begin & BaselineLen= 124.219 & BaselinePA=   56.082 & end
  "C3-M0": begin & BaselineLen= 131.956 & BaselinePA=   56.977 & end
  "C3-U1": begin & BaselineLen=  64.010 & BaselinePA=  -18.982 & end
  "C3-U2": begin & BaselineLen= 111.437 & BaselinePA=    2.053 & end
  "C3-U3": begin & BaselineLen= 150.931 & BaselinePA=   13.026 & end
  "C3-U4": begin & BaselineLen= 155.339 & BaselinePA=   36.505 & end
  "D0-E0": begin & BaselineLen=  16.001 & BaselinePA=   71.030 & end
  "D0-G0": begin & BaselineLen=  32.007 & BaselinePA=   71.025 & end
  "D0-G1": begin & BaselineLen=  71.557 & BaselinePA=  134.443 & end
  "D0-G2": begin & BaselineLen=  39.993 & BaselinePA=   34.123 & end
  "D0-H0": begin & BaselineLen=  64.005 & BaselinePA=   71.012 & end
  "D0-I1": begin & BaselineLen=  82.350 & BaselinePA=  100.065 & end
  "D0-J1": begin & BaselineLen=  91.215 & BaselinePA=   86.258 & end
  "D0-J2": begin & BaselineLen= 100.242 & BaselinePA=   99.621 & end
  "D0-J3": begin & BaselineLen= 104.317 & BaselinePA=   38.543 & end
  "D0-J4": begin & BaselineLen= 113.709 & BaselinePA=   31.727 & end
  "D0-J5": begin & BaselineLen= 130.234 & BaselinePA=   23.529 & end
  "D0-J6": begin & BaselineLen= 148.814 & BaselinePA=   17.271 & end
  "D0-K0": begin & BaselineLen=  95.992 & BaselinePA=   71.012 & end
  "D0-L0": begin & BaselineLen= 104.011 & BaselinePA=   71.008 & end
  "D0-M0": begin & BaselineLen= 112.003 & BaselinePA=   71.010 & end
  "D0-U1": begin & BaselineLen=  35.792 & BaselinePA=  -45.555 & end
  "D0-U2": begin & BaselineLen=  75.903 & BaselinePA=   -0.559 & end
  "D0-U3": begin & BaselineLen= 115.360 & BaselinePA=   14.707 & end
  "D0-U4": begin & BaselineLen= 125.216 & BaselinePA=   44.444 & end
  "D1-E0": begin & BaselineLen=  35.777 & BaselinePA=    7.583 & end
  "D1-G0": begin & BaselineLen=  45.258 & BaselinePA=   26.024 & end
  "D1-G1": begin & BaselineLen=  45.258 & BaselinePA=  116.003 & end
  "D1-G2": begin & BaselineLen=  64.501 & BaselinePA=   10.744 & end
  "D1-H0": begin & BaselineLen=  71.562 & BaselinePA=   44.447 & end
  "D1-I1": begin & BaselineLen=  72.432 & BaselinePA=   77.343 & end
  "D1-J1": begin & BaselineLen=  88.371 & BaselinePA=   65.807 & end
  "D1-J2": begin & BaselineLen=  89.447 & BaselinePA=   81.314 & end
  "D1-J3": begin & BaselineLen= 124.463 & BaselinePA=   26.014 & end
  "D1-J4": begin & BaselineLen= 136.245 & BaselinePA=   21.252 & end
  "D1-J5": begin & BaselineLen= 155.337 & BaselinePA=   15.526 & end
  "D1-J6": begin & BaselineLen= 175.643 & BaselinePA=   11.085 & end
  "D1-K0": begin & BaselineLen= 101.188 & BaselinePA=   52.575 & end
  "D1-L0": begin & BaselineLen= 108.827 & BaselinePA=   53.906 & end
  "D1-M0": begin & BaselineLen= 116.489 & BaselinePA=   55.064 & end
  "D1-U1": begin & BaselineLen=  65.987 & BaselinePA=  -33.026 & end
  "D1-U2": begin & BaselineLen= 106.746 & BaselinePA=   -5.996 & end
  "D1-U3": begin & BaselineLen= 143.093 & BaselinePA=    7.580 & end
  "D1-U4": begin & BaselineLen= 142.437 & BaselinePA=   32.852 & end
  "D2-E0": begin & BaselineLen=  50.593 & BaselinePA=   -0.547 & end
  "D2-G0": begin & BaselineLen=  57.688 & BaselinePA=   14.715 & end
  "D2-G1": begin & BaselineLen=  35.785 & BaselinePA=   97.571 & end
  "D2-G2": begin & BaselineLen=  78.793 & BaselinePA=    4.966 & end
  "D2-H0": begin & BaselineLen=  80.007 & BaselinePA=   34.145 & end
  "D2-I1": begin & BaselineLen=  72.436 & BaselinePA=   64.663 & end
  "D2-J1": begin & BaselineLen=  91.225 & BaselinePA=   55.750 & end
  "D2-J2": begin & BaselineLen=  88.006 & BaselinePA=   71.011 & end
  "D2-J3": begin & BaselineLen= 136.245 & BaselinePA=   21.252 & end
  "D2-J4": begin & BaselineLen= 148.816 & BaselinePA=   17.271 & end
  "D2-J5": begin & BaselineLen= 168.762 & BaselinePA=   12.447 & end
  "D2-J6": begin & BaselineLen= 189.657 & BaselinePA=    8.663 & end
  "D2-K0": begin & BaselineLen= 107.327 & BaselinePA=   44.446 & end
  "D2-L0": begin & BaselineLen= 114.558 & BaselinePA=   46.237 & end
  "D2-M0": begin & BaselineLen= 121.860 & BaselinePA=   47.813 & end
  "D2-U1": begin & BaselineLen=  81.598 & BaselinePA=  -30.299 & end
  "D2-U2": begin & BaselineLen= 122.386 & BaselinePA=   -7.680 & end
  "D2-U3": begin & BaselineLen= 157.564 & BaselinePA=    4.978 & end
  "D2-U4": begin & BaselineLen= 152.841 & BaselinePA=   28.131 & end
  "E0-G0": begin & BaselineLen=  16.006 & BaselinePA=   71.020 & end
  "E0-G1": begin & BaselineLen=  65.966 & BaselinePA=  146.971 & end
  "E0-G2": begin & BaselineLen=  28.846 & BaselinePA=   14.665 & end
  "E0-H0": begin & BaselineLen=  48.004 & BaselinePA=   71.005 & end
  "E0-I1": begin & BaselineLen=  68.799 & BaselinePA=  106.546 & end
  "E0-J1": begin & BaselineLen=  75.892 & BaselinePA=   89.433 & end
  "E0-J2": begin & BaselineLen=  86.531 & BaselinePA=  104.698 & end
  "E0-J3": begin & BaselineLen=  91.225 & BaselinePA=   33.137 & end
  "E0-J4": begin & BaselineLen= 101.833 & BaselinePA=   26.014 & end
  "E0-J5": begin & BaselineLen= 120.005 & BaselinePA=   17.887 & end
  "E0-J6": begin & BaselineLen= 139.951 & BaselinePA=   11.980 & end
  "E0-K0": begin & BaselineLen=  79.991 & BaselinePA=   71.009 & end
  "E0-L0": begin & BaselineLen=  88.010 & BaselinePA=   71.004 & end
  "E0-M0": begin & BaselineLen=  96.002 & BaselinePA=   71.006 & end
  "E0-U1": begin & BaselineLen=  45.274 & BaselinePA=  -63.980 & end
  "E0-U2": begin & BaselineLen=  72.458 & BaselinePA=  -12.654 & end
  "E0-U3": begin & BaselineLen= 107.317 & BaselinePA=    7.579 & end
  "E0-U4": begin & BaselineLen= 111.138 & BaselinePA=   40.750 & end
  "G0-H0": begin & BaselineLen=  31.998 & BaselinePA=   70.998 & end
  "G0-I1": begin & BaselineLen=  56.543 & BaselinePA=  116.013 & end
  "G0-J1": begin & BaselineLen=  60.915 & BaselinePA=   94.193 & end
  "G0-J2": begin & BaselineLen=  73.748 & BaselinePA=  111.611 & end
  "G0-J3": begin & BaselineLen=  79.205 & BaselinePA=   26.009 & end
  "G0-J4": begin & BaselineLen=  91.221 & BaselinePA=   18.887 & end
  "G0-J5": begin & BaselineLen= 111.143 & BaselinePA=   11.271 & end
  "G0-J6": begin & BaselineLen= 132.430 & BaselinePA=    6.031 & end
  "G0-K0": begin & BaselineLen=  63.985 & BaselinePA=   71.006 & end
  "G0-L0": begin & BaselineLen=  72.004 & BaselinePA=   71.001 & end
  "G0-M0": begin & BaselineLen=  79.996 & BaselinePA=   71.004 & end
  "G0-U1": begin & BaselineLen=  57.713 & BaselinePA=  -75.289 & end
  "G0-U2": begin & BaselineLen=  72.462 & BaselinePA=  -25.336 & end
  "G0-U3": begin & BaselineLen= 101.178 & BaselinePA=   -0.556 & end
  "G0-U4": begin & BaselineLen=  97.648 & BaselinePA=   36.010 & end
  "G1-H0": begin & BaselineLen=  71.555 & BaselinePA=    7.576 & end
  "G1-I1": begin & BaselineLen=  46.638 & BaselinePA=   40.027 & end
  "G1-J1": begin & BaselineLen=  68.826 & BaselinePA=   35.464 & end
  "G1-J2": begin & BaselineLen=  58.238 & BaselinePA=   55.065 & end
  "G1-J3": begin & BaselineLen= 132.427 & BaselinePA=    6.030 & end
  "G1-J4": begin & BaselineLen= 147.079 & BaselinePA=    3.394 & end
  "G1-J5": begin & BaselineLen= 169.513 & BaselinePA=    0.305 & end
  "G1-J6": begin & BaselineLen= 192.332 & BaselinePA=   -2.058 & end
  "G1-K0": begin & BaselineLen=  90.500 & BaselinePA=   26.006 & end
  "G1-L0": begin & BaselineLen=  96.341 & BaselinePA=   29.378 & end
  "G1-M0": begin & BaselineLen= 102.451 & BaselinePA=   32.349 & end
  "G1-U1": begin & BaselineLen= 107.349 & BaselinePA=  -45.556 & end
  "G1-U2": begin & BaselineLen= 136.246 & BaselinePA=  -22.359 & end
  "G1-U3": begin & BaselineLen= 163.148 & BaselinePA=   -7.679 & end
  "G1-U4": begin & BaselineLen= 144.219 & BaselinePA=   14.697 & end
  "G2-H0": begin & BaselineLen=  40.018 & BaselinePA=  107.873 & end
  "G2-I1": begin & BaselineLen=  75.470 & BaselinePA=  129.004 & end
  "G2-J1": begin & BaselineLen=  73.765 & BaselinePA=  111.600 & end
  "G2-J2": begin & BaselineLen=  91.229 & BaselinePA=  123.131 & end
  "G2-J3": begin & BaselineLen=  64.516 & BaselinePA=   41.281 & end
  "G2-J4": begin & BaselineLen=  73.770 & BaselinePA=   30.428 & end
  "G2-J5": begin & BaselineLen=  91.219 & BaselinePA=   18.905 & end
  "G2-J6": begin & BaselineLen= 111.144 & BaselinePA=   11.284 & end
  "G2-K0": begin & BaselineLen=  68.360 & BaselinePA=   91.572 & end
  "G2-L0": begin & BaselineLen=  75.918 & BaselinePA=   89.441 & end
  "G2-M0": begin & BaselineLen=  83.539 & BaselinePA=   87.710 & end
  "G2-U1": begin & BaselineLen=  48.658 & BaselinePA=  -99.517 & end
  "G2-U2": begin & BaselineLen=  48.664 & BaselinePA=  -28.440 & end
  "G2-U3": begin & BaselineLen=  78.771 & BaselinePA=    4.990 & end
  "G2-U4": begin & BaselineLen=  86.168 & BaselinePA=   49.214 & end
  "H0-I1": begin & BaselineLen=  40.780 & BaselinePA=  149.723 & end
  "H0-J1": begin & BaselineLen=  33.931 & BaselinePA=  115.997 & end
  "H0-J2": begin & BaselineLen=  53.664 & BaselinePA=  134.449 & end
  "H0-J3": begin & BaselineLen=  60.929 & BaselinePA=    4.214 & end
  "H0-J4": begin & BaselineLen=  75.895 & BaselinePA=   -0.548 & end
  "H0-J5": begin & BaselineLen=  98.949 & BaselinePA=   -4.946 & end
  "H0-J6": begin & BaselineLen= 122.374 & BaselinePA=   -7.673 & end
  "H0-K0": begin & BaselineLen=  31.987 & BaselinePA=   71.014 & end
  "H0-L0": begin & BaselineLen=  40.006 & BaselinePA=   71.003 & end
  "H0-M0": begin & BaselineLen=  47.998 & BaselinePA=   71.008 & end
  "H0-U1": begin & BaselineLen=  86.179 & BaselinePA=  -87.182 & end
  "H0-U2": begin & BaselineLen=  82.378 & BaselinePA=  -48.045 & end
  "H0-U3": begin & BaselineLen=  95.980 & BaselinePA=  -18.992 & end
  "H0-U4": begin & BaselineLen=  73.752 & BaselinePA=   21.605 & end
  "I1-J1": begin & BaselineLen=  22.642 & BaselinePA=   26.034 & end
  "I1-J2": begin & BaselineLen=  17.906 & BaselinePA=   97.582 & end
  "I1-J3": begin & BaselineLen=  97.320 & BaselinePA=   -9.512 & end
  "I1-J4": begin & BaselineLen= 113.129 & BaselinePA=  -10.846 & end
  "I1-J5": begin & BaselineLen= 136.924 & BaselinePA=  -12.267 & end
  "I1-J6": begin & BaselineLen= 160.787 & BaselinePA=  -13.268 & end
  "I1-K0": begin & BaselineLen=  46.640 & BaselinePA=   11.986 & end
  "I1-L0": begin & BaselineLen=  51.237 & BaselinePA=   19.694 & end
  "I1-M0": begin & BaselineLen=  56.575 & BaselinePA=   26.027 & end
  "I1-U1": begin & BaselineLen= 113.700 & BaselinePA=  -69.696 & end
  "I1-U2": begin & BaselineLen= 121.850 & BaselinePA=  -42.184 & end
  "I1-U3": begin & BaselineLen= 136.205 & BaselinePA=  -22.351 & end
  "I1-U4": begin & BaselineLen= 103.997 & BaselinePA=    3.636 & end
  "J1-K0": begin & BaselineLen=  25.281 & BaselinePA=   -0.569 & end
  "J1-L0": begin & BaselineLen=  28.842 & BaselinePA=   14.721 & end
  "J1-M0": begin & BaselineLen=  33.933 & BaselinePA=   26.022 & end
  "J1-U1": begin & BaselineLen= 118.129 & BaselinePA=  -80.690 & end
  "J1-U2": begin & BaselineLen= 115.380 & BaselinePA=  -52.683 & end
  "J1-U3": begin & BaselineLen= 122.345 & BaselinePA=  -30.304 & end
  "J1-U4": begin & BaselineLen=  83.510 & BaselinePA=   -2.294 & end
  "J2-K0": begin & BaselineLen=  48.659 & BaselinePA=   -9.538 & end
  "J2-L0": begin & BaselineLen=  50.605 & BaselinePA=   -0.546 & end
  "J2-M0": begin & BaselineLen=  53.669 & BaselinePA=    7.576 & end
  "J2-U1": begin & BaselineLen= 131.225 & BaselinePA=  -71.418 & end
  "J2-U2": begin & BaselineLen= 136.012 & BaselinePA=  -47.061 & end
  "J2-U3": begin & BaselineLen= 145.967 & BaselinePA=  -28.453 & end
  "J2-U4": begin & BaselineLen= 106.735 & BaselinePA=   -5.998 & end
  "J3-K0": begin & BaselineLen=  56.569 & BaselinePA=  152.900 & end
  "J3-L0": begin & BaselineLen=  58.236 & BaselinePA=  145.064 & end
  "J3-M0": begin & BaselineLen=  60.921 & BaselinePA=  137.819 & end
  "J3-U1": begin & BaselineLen= 106.748 & BaselinePA= -121.975 & end
  "J3-U2": begin & BaselineLen=  65.986 & BaselinePA=  -94.948 & end
  "J3-U3": begin & BaselineLen=  46.635 & BaselinePA=  -49.978 & end
  "J3-U4": begin & BaselineLen=  23.984 & BaselinePA=   71.006 & end
  "J4-K0": begin & BaselineLen=  72.441 & BaselinePA=  154.687 & end
  "J4-L0": begin & BaselineLen=  73.749 & BaselinePA=  148.482 & end
  "J4-M0": begin & BaselineLen=  75.887 & BaselinePA=  142.582 & end
  "J4-U1": begin & BaselineLen= 111.440 & BaselinePA= -130.015 & end
  "J4-U2": begin & BaselineLen=  64.016 & BaselinePA= -108.978 & end
  "J4-U3": begin & BaselineLen=  33.937 & BaselinePA=  -64.026 & end
  "J4-U4": begin & BaselineLen=  28.827 & BaselinePA=  104.712 & end
  "J5-K0": begin & BaselineLen=  96.325 & BaselinePA=  156.260 & end
  "J5-L0": begin & BaselineLen=  97.310 & BaselinePA=  151.549 & end
  "J5-M0": begin & BaselineLen=  98.941 & BaselinePA=  146.980 & end
  "J5-U1": begin & BaselineLen= 122.122 & BaselinePA= -140.582 & end
  "J5-U2": begin & BaselineLen=  68.362 & BaselinePA= -129.525 & end
  "J5-U3": begin & BaselineLen=  24.015 & BaselinePA= -109.019 & end
  "J5-U4": begin & BaselineLen=  46.628 & BaselinePA=  130.061 & end
  "J6-K0": begin & BaselineLen= 120.261 & BaselinePA=  157.208 & end
  "J6-L0": begin & BaselineLen= 121.051 & BaselinePA=  153.418 & end
  "J6-M0": begin & BaselineLen= 122.366 & BaselinePA=  149.707 & end
  "J6-U1": begin & BaselineLen= 136.241 & BaselinePA= -149.213 & end
  "J6-U2": begin & BaselineLen=  80.007 & BaselinePA= -145.841 & end
  "J6-U3": begin & BaselineLen=  33.964 & BaselinePA= -153.987 & end
  "J6-U4": begin & BaselineLen=  68.337 & BaselinePA=  140.470 & end
  "K0-L0": begin & BaselineLen=   8.019 & BaselinePA=   70.959 & end
  "K0-M0": begin & BaselineLen=  16.011 & BaselinePA=   70.995 & end
  "K0-U1": begin & BaselineLen= 116.485 & BaselinePA=  -93.036 & end
  "K0-U2": begin & BaselineLen= 101.829 & BaselinePA=  -63.982 & end
  "K0-U3": begin & BaselineLen= 101.173 & BaselinePA=  -37.423 & end
  "K0-U4": begin & BaselineLen=  58.246 & BaselinePA=   -3.042 & end
  "L0-M0": begin & BaselineLen=   7.992 & BaselinePA=   71.030 & end
  "L0-U1": begin & BaselineLen= 124.213 & BaselinePA=  -94.056 & end
  "L0-U2": begin & BaselineLen= 107.643 & BaselinePA=  -67.005 & end
  "L0-U3": begin & BaselineLen= 103.980 & BaselinePA=  -41.620 & end
  "L0-U4": begin & BaselineLen=  56.564 & BaselinePA=  -10.875 & end
  "M0-U1": begin & BaselineLen= 131.952 & BaselinePA=  -94.949 & end
  "M0-U2": begin & BaselineLen= 113.711 & BaselinePA=  -69.699 & end
  "M0-U3": begin & BaselineLen= 107.312 & BaselinePA=  -45.561 & end
  "M0-U4": begin & BaselineLen=  56.000 & BaselinePA=  -18.997 & end
      else: print,"Unknown baseline "+stations
  endcase

  if BaselineLen gt 0. then begin
     ;; we have a valid baseline
     ;; coordinates of baseline in horizontal system
     BPAr  = BaselinePA * !PI/180.
     BaseN = BaselineLen * cos(BPAr)
     BaseE = BaselineLen * sin(BPAr)
  endif else begin
     ;; unknown baseline, use tel positions in header
     BaseN = Tel1Y-Tel2Y
     BaseE = Tel1X-Tel2X
  endelse

  if not keyword_set(quiet) then begin
     print, "Projected Baseline"
     print, "Time Length P.A."
  endif
  if not keyword_set(times) then times= dindgen(exptime)	;; one timestamp per sec

  nTimes = N_elements(times)
  base_len = dblarr(nTimes)
  base_pa  = dblarr(nTimes)

  for t=0,nTimes-1 do begin
     project_baseline, ( (LST+times[t])/3600. - RA/15.), Dec, -[BaseE,BaseN,0.], b_len, b_pa
     if b_pa lt 0. then b_pa += 360.
     base_len[t]= b_len
     base_pa[t] = b_pa

     if not keyword_set(quiet)  and  ((t mod 10) eq 0  or  (t eq nTimes-1)) then $
        print, Format='(F7.2,F7.2, I4,F7.2,F7.2)', (LST+times[t])/3600., RA/15., t, b_len, b_pa
  endfor
  if not keyword_set(quiet) then begin
     print,''
     print,"object:              ", objectname
     print,"Baseline:            ", stat1, " - ", stat2
     print,"                     E = ",strtrim(BaseE,2),", N = ",strtrim(BaseN,2)
     print,"ESO's 3D baseline:   ", Tel2X-Tel1X, Tel2Y-Tel1Y, Tel2Z-Tel1z
     print,"ESO's proj. bl len.: ", esolen
     print,"ESO's proj. bl PA:   ", esoPA
     print,"Date of observation: ", date
     print,"Local Sid Time:      ", LST, " sec =",LST/3600., " hours"
     print,"Right Ascension:     ", RA
     print,"Declination:         ", Dec
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
