@~/idl/read_parameters
@~/idl/rgb

function make_uv,image,pars
  npix=pars.npix
  npix_ft=pars.npix_ft
  if (npix gt npix_ft) then begin
    npix_ft=npix
    pars.npix_ft=npix_ft
  endif

  big_image=fltarr(npix_ft,npix_ft)
  big_image[0:n_elements(image[*,0])-1,0:n_elements(image[0,*])-1]=image

  ft_image=fft(big_image,-1)
  ft_image=shift((abs(ft_image)/abs(ft_image[0,0])),$
                      npix_ft/2,npix_ft/2)

  low=floor(npix_ft/2)-floor(npix/2)
  high=floor(npix_ft/2)+floor(npix/2)-1
  uv=ft_image[low:high,low:high]
  return,uv
end

function make_grid,npix
  im=fltarr(npix,npix)
  center=floor(npix/2)

  dummy=dindgen(npix)
  im1=im
  im2=im
  for i=0,npix-1 do begin
    im1[*,i]=dummy
    im2[i,*]=dummy
  endfor

  im1=im1-center
  im2=im2-center
  im=sqrt(im1^2.+im2^2.) ;; contains distance to center in pixels
  return,im
end

function make_2Dgrid,pars
  npix=pars.npix
  im=fltarr(npix,npix,2)
  center=floor(npix/2)

  dummy=dindgen(npix)
  im1=fltarr(npix,npix)
  im2=im1
  for i=0,npix-1 do begin
    im1[*,i]=dummy
    im2[i,*]=dummy
  endfor

  im1=im1-center
  im2=im2-center

  im[*,*,0]=im1*pars.pixsize ;; distances to x=0 (mas)
  im[*,*,1]=im2*pars.pixsize ;; distances to y=0 (mas)

  return,im
end

pro plot_trace,uv,pars,ps=ps
  n=n_elements(uv[0,*])
  cen=n/2

  k=pars.Baseline/(1d-6*pars.lambda) ;; (cycles/radian)
  k=k/206264.8062 ;; (cycles/arcsec)
  pixelsize_arcsec=1d-3*pars.pixsize
  k_pix=k*(pixelsize_arcsec*pars.npix_ft)

  ntrace=256
  trace=fltarr(2,ntrace)
  trace[0,*]=cen+k_pix*cos(dindgen(ntrace)*2.*!dpi/(ntrace-1.))
  trace[1,*]=cen+k_pix*sin(dindgen(ntrace)*2.*!dpi/(ntrace-1.))

  v2=reform(bilinear(uv,trace[0,*],trace[1,*]))^2
  sigma_fourier_space=sqrt(-k^2/(2.*alog(sqrt(v2))))
  sigma_image_space=1./(2.*!dpi*sigma_fourier_space) ;; in arcsec
  fwhm=1d3*2.355*sigma_image_space ;; fwhm in mas

  grid=360.*dindgen(ntrace)/(ntrace-1.)

  x=dindgen(ntrace)*4.*!dpi/(ntrace-1.)
  mean=(pars.a+(pars.a/pars.a_over_b))/2.
  ampl=pars.a-mean

  simple_model=mean+ampl*cos(x)

  if keyword_set(ps) then begin
    postscriptname='cosine_curve_vs_2D_gauss.ps'
    xsize=20.
    set_plot,'ps',/interpolate
    x_over_y=1.25 ;; ratio x_size/y_size
    device,file=postscriptname,/portrait,/color,bits_per_pixel=24,$
      font_size=16.,xsize=xsize,ysize=(xsize/x_over_y),yoffset=1
    charsize=1.4
  endif

  plot,grid,fwhm,xrange=[45,225],charsize=0.9,$
    xtitle='position angle (degrees)',$
    ytitle='fwhm gauss',$
    title='_____ real 2D gauss, _ _ _ cosine approximation, a/b=1.3'

  oplot,grid,simple_model,line=2

  if keyword_set(ps) then begin
    device,/close
    set_plot,'x'
    print,'wrote ',postscriptname
  endif


end


pro test_2D_gauss,ps=ps
  pars=read_parameters('parameters_test_2D_gauss.txt')
  grid=make_2Dgrid(pars)

  a=pars.a/2.355
  b=a/pars.a_over_b
  im=exp(-0.5*(grid[*,*,0]^2/a^2+grid[*,*,1]^2/b^2))
  uv=make_uv(im,pars)
  tvscl,uv
  plot_trace,uv,pars,ps=ps
end  

pro test_1D_gauss
  x=dindgen(4096)
  fwhm=1024.
  sig=fwhm/2.355

  y=exp(-x^2/(2.*sig^2))
  y=y/total(y)

  indx=where(x le fwhm/2.)
  print,total(y[indx])



end

