/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: issshadowMap.c,v 1.8 2001/09/13 09:12:41 vltsccm Exp $"
*
* who       when        what
* --------  ----------  ----------------------------------------------
* nhousen   2001-09-05  reduced number of included header files
* nhousen   2001-07-13  changed orientation of output maps
*                       (now: left-right = alt; rows = az)
* nhousen   2001-07-11  adapted error handling to VLT standards
* rwilhelm  2001-07-10  created (as isshadowTest.c)
*/

/************************************************************************
*   NAME
*   issshadowMap.c
*
*   SYNOPSIS
*
*
*   DESCRIPTION
*   Creates a file containing the coordinates of the horizon line for a 
*   specific telescope and position.
*
*   PARENT CLASS
*
*
*   DESCRIPTION
*   
*
*   PUBLIC METHODS
*
*
*   PUBLIC DATA MEMBERS
*
*
*   PROTECTED METHODS
*
*
*   PROTECTED DATA MEMBERS
*
*
*   PRIVATE METHODS
*
*
*   PRIVATE DATA MEMBERS
*
*
*   FILES
*
*   ENVIRONMENT
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS
*
*------------------------------------------------------------------------
*/

/* #define _POSIX_SOURCE 1 */
/* #include "vltPort.h" */

static char *rcsId="@(#) $Id: issshadowMap.c,v 1.8 2001/09/13 09:12:41 vltsccm Exp $";
static void *use_rcsId = ((void)&use_rcsId,(void *) &rcsId);

/* Standard ANSI C Includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "issshadowDefines.h"
#include "issshadowErrors.h"
/*
#include "err.h"
*/
/*#include "log.h"*/

int main(int argc, char* argv[])
{   
    /* LOCAL VARIABLES */
    /* --------------- */
    long i_alt, i_az; /* loop counters for altitude / azimuth loop */
    ccsERROR error;

    /* Command line input arguments */
    int teltype;
    int location;
    int noofRayRadial;
    int noofRayAngular;    
    PointingStruct *Pointing;
    char OutputFilename[MAXIMUM_STRING_LENGTH];
    char OutputFileFormat[MAXIMUM_STRING_LENGTH];

    float **HorizonMap;
    double shadow;

    /*ccsInit(argv[0],0,NULL,NULL,error);*/  
    errResetStack(&error);

    /* Allocate memory for the pointing pointer */
    Pointing = AllocateMemoryForPointingPointer();

    /* Parse command line arguments and create altitude and azimuth vectors */
    if ( ParseCLIArguments(argc, argv,
                      &teltype,
                      &location,
                      &noofRayRadial,
                      &noofRayAngular,
                      Pointing, 
                      OutputFilename,
                      OutputFileFormat,
		      &error) == FAILURE) 
    {

      errCloseStack(&error);
      return 1 ;
    }

    /* Allocate memory for horizon map */
    HorizonMap = AllocMemForFloatMatrix(Pointing->Dimension_Altitude,
                                        Pointing->Dimension_Azimuth);

    /* ===================================== */
    /* LOOP; ALL ALTITUDE AND AZIMUTH ANGLES */
    /* ===================================== */
    for(i_alt=0; i_alt<Pointing->Dimension_Altitude; i_alt++)
      {
          for(i_az=0; i_az<Pointing->Dimension_Azimuth; i_az++)
          {
              if ( issshadowAltAz(Pointing->AltitudeVector[i_alt], Pointing->AzimuthVector[i_az],
		  teltype, location, noofRayRadial, noofRayAngular, &shadow, &error) == FAILURE )
		{
		  errCloseStack(&error);
		  return 1;
		}
              HorizonMap[i_alt][i_az] = (float)shadow;
          }
      }

    /* ========================================= */
    /* END LOOP; ALL ALTITUDE AND AZIMUTH ANGLES */
    /* ========================================= */
 
    /* Write result file to disk */
    if ( WriteOutputFile(OutputFilename, OutputFileFormat,
			 HorizonMap, Pointing, &error) == FAILURE )
      {
	errCloseStack(&error);
	return 1;
      }

    /*fprintf(stderr,"Resulting horizon map of dimension %d x %d was written\n",
            Pointing->Dimension_Altitude, 
            Pointing->Dimension_Azimuth);
	    fprintf(stderr,"to the %s file %s.\n", OutputFileFormat, OutputFilename);*/
    /*logData(issshadowMOD, 0, "Horizon map was successfully created!");*/

    /* Free memory for some local variables */
    FreeMemForFloatMatrix(HorizonMap, Pointing->Dimension_Altitude);
    FreeMemForFloatVector(Pointing->AltitudeVector);
    FreeMemForFloatVector(Pointing->AzimuthVector);
    FreeMemoryForPointingPointer(Pointing);

    /*ccsExit(error);*/
    errResetStack(&error);
    return 0;
    
}


ccsCOMPL_STAT ParseCLIArguments(int argc, char* argv[],
                       int *teltype,
                       int *location,
                       int *noofRayRadial,
                       int *noofRayAngular,
                       PointingStruct *Pointing,
                       char *OutputFilename,
                       char *OutputFileFormat,
		       ccsERROR *error)
/*----------------------------------------------------------------------- */
/*  Parse control info from command line                                  */
/*   Input:                                                               */
/*      argc   Number of arguments from command line                      */
/*      argv   Array of strings from command line                         */
/*----------------------------------------------------------------------- */
{
    long i;
    char Argument[MAXIMUM_STRING_LENGTH];

    /* Parse command line */
    /* ------------------ */

    /* Set defaults                       */
    /* :::::::::::::::::::::::::::::::::: */
    strcpy(OutputFilename, "DefaultRun");
    strcpy(OutputFileFormat, "ASCII");
    *teltype = 1; /* telescope type UT */
    *location = 31; /* station U1 (--> UT1) */ 
    *noofRayRadial = 3;
    *noofRayAngular = 8;
    Pointing->Altitude_Min = 20.;
    Pointing->Altitude_Max = 80.;
    Pointing->Altitude_Increment = 1.;
    Pointing->Azimuth_Min  = 0.;
    Pointing->Azimuth_Max  = 360.;
    Pointing->Azimuth_Increment = 1.;
    

    for (i=1; i<argc; i++)
    {
        /* Get argument (starting with index 1 since index 0 argument is
           the app name itself */
        strcpy(Argument, argv[i]);
    
        if(strcmp(Argument, "-out")==0)
        {
            /* Output filename (used for horizon map result file)
               (WITHOUT (!) extension) */
            strcpy(OutputFilename, argv[++i]);
        }
        else if(strcmp(Argument, "-format")==0)
        {
            /* Format for the output file (horizon map) (ASCII or BINARY) */
            strcpy(OutputFileFormat, argv[++i]);
        }
        else if(strcmp(Argument, "-t")==0)
        {
            /* Telescope type ('UT', 'AT' or 'SID') */
            sscanf(argv[++i], "%d", teltype);      
        }
        else if(strcmp(Argument, "-s")==0)
        {
            /* Station (e.g. E0) */
            sscanf(argv[++i], "%d", location);
        }
        else if(strcmp(Argument, "-rays")==0)
        {
            /* Radial incrementation of ray grid */
            sscanf(argv[++i], "%d", noofRayRadial);
            /* Angular incrementation of ray grid */
            sscanf(argv[++i], "%d", noofRayAngular);
        }
        else if(strcmp(Argument, "-alt")==0)
        {
            /* Lower border for altitude range */
            sscanf(argv[++i], "%g", &Pointing->Altitude_Min);  /* IMPORTANT!!! sscanf returns a float (32 bit)!! */
            /* Upper border for altitude range */
            sscanf(argv[++i], "%g", &Pointing->Altitude_Max);
            /* Increment for altitude */
            sscanf(argv[++i], "%g", &Pointing->Altitude_Increment);
        }
        else if(strcmp(Argument, "-az")==0)
        {
            /* Lower border for azimuth range */
            sscanf(argv[++i], "%g", &Pointing->Azimuth_Min);
            /* Upper border for azimuth range */
            sscanf(argv[++i], "%g", &Pointing->Azimuth_Max);
            /* Increment for azimuth */
            sscanf(argv[++i], "%g", &Pointing->Azimuth_Increment);
        }
        else
        {
            /* Unknown argument */
	  /*issshadowTestUsage();*/
	  errAdd(error, issshadowMOD, issshadowERR_UNKNOWN_ARGUMENT,issshadowMOD,"", "");
	  return(FAILURE);
        }
    }
    
    /* Check pointing range and fill altitude and azimuth vectors */
    if ( CheckPointingRange(Pointing, error) == FAILURE )
    {
	return(FAILURE);
    }

    /* Check output file format option */    
    if ( CheckOutputFileFormat(OutputFileFormat, error) == FAILURE)
    {
        return(FAILURE);
    }

    return(SUCCESS);
}

void issshadowTestUsage(void)
{
    fprintf(stderr, "------------------------------------------------------------------\n");
    fprintf(stderr, "| issshadowMap                                                   |\n");
    fprintf(stderr, "|                                                                |\n");
    fprintf(stderr, "| This executable program has been created to create a           |\n");
    fprintf(stderr, "| horizon map for a given telescope and position.                |\n"); 
    fprintf(stderr, "------------------------------------------------------------------\n");
    fprintf(stderr, "Please use the following input options for running\n");
    fprintf(stderr, "the issshadowMap program:\n\n");
    fprintf(stderr, " -out <Name> ................ where <Name> is a string that is used for\n");
    fprintf(stderr, "                              the result file (horizon map).\n");
    fprintf(stderr, "                              { DEFAULT : <Name> = 'DefaultRun' }\n\n");
    fprintf(stderr, " -format <Format> ........... where <Format> is a string that specifies\n");
    fprintf(stderr, "                              the format of the output file. The two\n");
    fprintf(stderr, "                              possible values are:\n");
    fprintf(stderr, "                              <Format> = 'ASCII'  ==> for an ASCII output file,\n");
    fprintf(stderr, "                              <Format> = 'BINARY' ==> for a BINARY output file.\n");
    fprintf(stderr, "                              { DEFAULT : <Format> = 'ASCII'\n\n"); 
    fprintf(stderr, " -t <teltype> ............... where <teltype> defines the type of\n");
    fprintf(stderr, "                              telescope (possible values:\n");
    fprintf(stderr, "                              <teltype> = 1 <==> UT,\n");
    fprintf(stderr, "                              <teltype> = 4 <==> AT,\n");
    fprintf(stderr, "                              <teltype> = 5 <==> SID.\n"); 
    fprintf(stderr, "                              { DEFAULT : <teltype> = 1 (UT) }\n\n");
    fprintf(stderr, " -s <location> .............. where <location> defines the location\n");
    fprintf(stderr, "                              of the selected telescope in the VLTI\n");
    fprintf(stderr, "                              {u, v}-plane, in accordance with:\n");
    fprintf(stderr, "                              <location> = 1 <==> A0\n");
    fprintf(stderr, "                              <location> = 2 <==> A1\n");
    fprintf(stderr, "                                        ...\n");                             
    fprintf(stderr, "                              <location> = 30 <==> M0\n");
    fprintf(stderr, "                              <location> = 31 <==> U1\n");
    fprintf(stderr, "                                        ...\n"); 
    fprintf(stderr, "                              <location> = 34 <==> U4\n");
    fprintf(stderr, "                              { DEFAULT : <location> = 31 (U1) }\n\n");
    fprintf(stderr, " -rays <i_r> <i_a> .......... where <i_r> and <i_a> are the RADIAL and\n");
    fprintf(stderr, "                              ANGULAR incrementation of the ray grid used\n");
    fprintf(stderr, "                              for the ray tracing procedure.\n");
    fprintf(stderr, "                              Minimum allowed values are:\n");
    fprintf(stderr, "                              --> for <i_r> = 0 (i.e. one single ray)\n");
    fprintf(stderr, "                              --> for <i_a> = 1\n");
    fprintf(stderr, "                              { DEFAULTS : <i_r> = 3, <i_a> =8 }\n\n");
    fprintf(stderr, " -alt <min> <max> <delta> ... where <min> and <max> are the minimum and\n");
    fprintf(stderr, "                              maximum values for the altitude angle in\n");
    fprintf(stderr, "                              units of degrees. <delta> is the increment\n");
    fprintf(stderr, "                              of the altitude angle used for the horizon\n");
    fprintf(stderr, "                              map.\n");
    fprintf(stderr, "                              { DEFAULTS : <min> = 20, <max> = 80, <delta> = 1 }\n\n");
    fprintf(stderr, " -az <min> <max> <delta> .... where <min> and <max> are the minimum and\n");
    fprintf(stderr, "                              maximum values for the azimuth angle in\n");
    fprintf(stderr, "                              units of degrees. <delta> is the increment\n");
    fprintf(stderr, "                              of the azimuth angle used for the horizon\n");
    fprintf(stderr, "                              map.\n");
    fprintf(stderr, "                              { DEFAULTS : <min> = 0, <max> = 360, <delta> = 1 }\n\n");

    exit(0);
}

ccsCOMPL_STAT CheckPointingRange(PointingStruct *Pointing, ccsERROR *error)
{
    long i;

    /* Minimum altitude must be smaller than or equal maximum altitude */
    if(Pointing->Altitude_Min>Pointing->Altitude_Max)
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "Your altitude range vector is invalid. The minimum altitude\n");
        fprintf(stderr, "must be SMALLER OR EQUAL than the maximum altitude.\n");
        fprintf(stderr, "You supplied the following values:\n");
        fprintf(stderr, " > Minimum altitude = %6.2f deg\n", Pointing->Altitude_Min);
        fprintf(stderr, " > Maximum altitude = %6.2f deg\n", Pointing->Altitude_Max);*/
        errAdd(error,issshadowMOD,issshadowERR_INVALID_ALTITUDE_RANGE_VECTOR,__FILE__,"","");
	return(FAILURE);
    }

    /* Minimum azimuth must be smaller than or equal maximum azimuth */
    if(Pointing->Azimuth_Min>Pointing->Azimuth_Max)
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "Your azimuth range vector is invalid. The minimum azimuth\n");
        fprintf(stderr, "must be SMALLER OR EQUAL than the maximum azimuth.\n");
        fprintf(stderr, "You supplied the following values:\n");
        fprintf(stderr, " > Minimum azimuth = %6.2f deg\n", Pointing->Azimuth_Min);
        fprintf(stderr, " > Maximum azimuth = %6.2f deg\n", Pointing->Azimuth_Max);*/
        errAdd(error,issshadowMOD,issshadowERR_INVALID_AZIMUTH_RANGE_VECTOR,__FILE__,"","");
	return(FAILURE);
    }

    /* Check if altitude increment is zero */
    if(TestEquality(Pointing->Altitude_Increment, 0, 1.e-300)==0)
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The altitude increment cannot be (so close to) zero!\n");
        fprintf(stderr, "You supplied an altitude increment of %22.14e\n",
	Pointing->Altitude_Increment);*/
        errAdd(error,issshadowMOD,issshadowERR_ALTITUDE_INCREMENT_TOO_SMALL,__FILE__,"","");
	return(FAILURE);
    }

    /* Check if azimuth increment is zero */
    if(TestEquality(Pointing->Azimuth_Increment, 0, 1.e-300)==0)
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The azimuth increment cannot be (so close to) zero!\n");
        fprintf(stderr, "You supplied an azimuth increment of %22.14e\n",
	Pointing->Azimuth_Increment); */
        errAdd(error,issshadowMOD,issshadowERR_AZIMUTH_INCREMENT_TOO_SMALL,__FILE__,"","");
	return(FAILURE);
    }
    
    /* Compute the size of the altitude and azimuth vectors */
    Pointing->Dimension_Altitude = (long)
                                   ((Pointing->Altitude_Max - Pointing->Altitude_Min)/
                                    Pointing->Altitude_Increment) + 1;
    if(Pointing->Dimension_Altitude<1)
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The altitude increment you supplied (%22.14e) is too small for\n",
                Pointing->Altitude_Increment);
        fprintf(stderr, "the specified range [%6.2f, %6.2f] of altitude angles.\n",
	Pointing->Altitude_Max, Pointing->Altitude_Min);*/
        errAdd(error,issshadowMOD,issshadowERR_ALT_INCR_TOO_SMALL_FOR_RANGE,__FILE__,"","");
	return(FAILURE);
    }
    
    Pointing->Dimension_Azimuth  = (long)
                                   ((Pointing->Azimuth_Max - Pointing->Azimuth_Min)/
                                    Pointing->Azimuth_Increment) + 1;
    if(Pointing->Dimension_Azimuth<1)
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The azimuth increment you supplied (%22.14e) is too small for\n",
                Pointing->Azimuth_Increment);
        fprintf(stderr, "the specified range [%6.2f, %6.2f] of azimuth angles.\n",
	Pointing->Azimuth_Max, Pointing->Azimuth_Min);*/
        errAdd(error,issshadowMOD,issshadowERR_AZ_INCR_TOO_SMALL_FOR_RANGE,__FILE__,"","");
	return(FAILURE);
    }

    /* Create the altitude and azimuth vectors */
    /* --------------------------------------- */

    /* Allocate memory for the 2 vectors */
    Pointing->AltitudeVector = AllocMemForFloatVector(Pointing->Dimension_Altitude);
    Pointing->AzimuthVector  = AllocMemForFloatVector(Pointing->Dimension_Azimuth);

    /* Fill the vectors */
    for(i=0; i<Pointing->Dimension_Altitude; i++)
    {
        Pointing->AltitudeVector[i] = Pointing->Altitude_Min + i* Pointing->Altitude_Increment;
    }
    for(i=0; i<Pointing->Dimension_Azimuth; i++)
    {
        Pointing->AzimuthVector[i] = Pointing->Azimuth_Min + i* Pointing->Azimuth_Increment;
    }
    
    /* Check if minimum and maximum altitude angle are ok */
    if((Pointing->AltitudeVector[Pointing->Dimension_Altitude-1]>90)||
       (Pointing->AltitudeVector[0]<0))
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The altitude angle (in degrees) must be in the range between 0 and 90 degrees.\n");
        fprintf(stderr, "Your altitude vector ranges from %6.2f deg to %6.2f deg.\n",
	Pointing->AltitudeVector[0], Pointing->AltitudeVector[Pointing->Dimension_Altitude-1]);*/
        errAdd(error,issshadowMOD,issshadowERR_ALTITUDE_ANGLE_OUT_OF_RANGE,__FILE__,"","");
	return(FAILURE);
    }   
    
    return(SUCCESS);
}

ccsCOMPL_STAT CheckOutputFileFormat(char *OutputFileFormat, ccsERROR *error)
{
    if((strcmp(OutputFileFormat, "ASCII")!=0)&&(strcmp(OutputFileFormat, "BINARY")!=0))
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The two possible formats for the output file are ASCII and BINARY.\n");
        fprintf(stderr, "Specify the requested format by using the option '-format' followed\n");
        fprintf(stderr, "by on of the strings 'ASCII' or 'BINARY'.\n");
        fprintf(stderr, "\nYou supplied the string %s.\n", OutputFileFormat);*/
        errAdd(error,issshadowMOD,issshadowERR_INVALID_OUTPUT_FILE_FORMAT,__FILE__,"","");
	return(FAILURE);
    }
    return(SUCCESS);
}


ccsCOMPL_STAT WriteOutputFile(char* OutputFilename, char *OutputFileFormat,
                     float **HorizonMap, PointingStruct *Pointing, ccsERROR *error)
{
    FILE *FilePointer;
    long i_alt, i_az;


    if(strcmp(OutputFileFormat, "BINARY")==0)
    {
        FilePointer = fopen(OutputFilename, "wb");
        for(i_az=0; i_az<Pointing->Dimension_Azimuth; i_az++)
        {
            for(i_alt=0; i_alt<Pointing->Dimension_Altitude; i_alt++)
            {
                fwrite(&HorizonMap[i_alt][i_az], sizeof(float), 1, FilePointer);        
            }
        }
        fclose(FilePointer);
        return(SUCCESS);
    }
    if(strcmp(OutputFileFormat, "ASCII")==0)
    {
        FilePointer = fopen(OutputFilename, "w");
        for(i_az=0; i_az<Pointing->Dimension_Azimuth; i_az++)
        {
            for(i_alt=0; i_alt<Pointing->Dimension_Altitude; i_alt++)
            {
                fprintf(FilePointer, "%14.8e ", HorizonMap[i_alt][i_az]);        
            }
            fprintf(FilePointer, "\n");
        }
        fclose(FilePointer);
        return(SUCCESS);
    }
    
    /*fprintf(stderr, "*** ERROR Message ***\n");
    fprintf(stderr, "The two possible formats for the output file are ASCII and BINARY.\n");
    fprintf(stderr, "Specify the requested format by using the option '-format' followed\n");
    fprintf(stderr, "by on of the strings 'ASCII' or 'BINARY'.\n");
    fprintf(stderr, "\nYou supplied the string %s.\n", OutputFileFormat);
    fprintf(stderr, "\n NO OUTPUT FILE WRITTEN.\n"); */
    errAdd(error,issshadowMOD,issshadowERR_INVALID_OUTPUT_FILE_FORMAT,__FILE__,"","");
    return(FAILURE);
}
