/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: issshadowMathematics.c,v 1.8 2001/09/13 09:12:39 vltsccm Exp $"
*
* who       when        what
* --------  ----------  ----------------------------------------------
* nhousen   2001-09-05  reduced number of included header files
* rwilhelm  2001-07-10  created 
*/

/************************************************************************
*   NAME
*   issshadowMathematics.c
*
*   SYNOPSIS
*
*
*   DESCRIPTION
*   Contains mathematical routines that are used by the 'issshadowAltAz' routine.   
*
*   PARENT CLASS
*
*
*   DESCRIPTION
*   
*
*   PUBLIC METHODS
*
*   PUBLIC DATA MEMBERS
*
*
*   PROTECTED METHODS
*
*
*   PROTECTED DATA MEMBERS
*
*
*   PRIVATE METHODS
*
*
*   PRIVATE DATA MEMBERS
*
*
*   FILES
*
*   ENVIRONMENT
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS
*
*------------------------------------------------------------------------
*/

/* #define _POSIX_SOURCE 1 */
/* #include "vltPort.h" */
/*
static char *rcsId="@(#) $Id: issshadowMathematics.c,v 1.8 2001/09/13 09:12:39 vltsccm Exp $";
static void *use_rcsId = ((void)&use_rcsId,(void *) &rcsId);
*/

/* Standard ANSI C Includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <float.h>

#include "issshadowDefines.h"

long TestEquality(float Number1, float Number2, float Epsilon)
/* returns 0 if Number1 is equal to Number2 (within epsilon);
   else 1 */
{
    long TestResult;
    
    if((fabs(Number1-Number2))<=Epsilon)
    {
        TestResult=0;
    }
    else
    {
        TestResult=1;
    }
    return TestResult;
}

long TestPerpendicular(float *Vector1, float *Vector2, long Dimension, float Epsilon)
/* returns 0 if the 2 vectors of dimension "Dimension" X 1 are perpendicular;
   if they're not perpendicular the function returns 1;
   (The scalar product of the 2 vectors is calculated and it is tested if this
    scalar product is equal to zero within an "epsilon range" of size "Epsilon") */
{
    /* local variables */
    float Help;
    
    long TestResult; /* can take values 0 or 1; "answer variable" of this function */
    
    /* calculate scalar product of the 2 vectors */
    ScalarProduct(Vector1, Vector2, Dimension, &Help);
    
    /* test if scalar product is zero or not */
    if(TestEquality(Help, 0., Epsilon)==0)
    {
        /* the scalar product result is equal to zero */
        TestResult=0;
    }
    else
    {
        TestResult=1;
    }
    
    /* return TestResult */
    return TestResult;
}


void SetVector_3D(float *Vector, float Component_1, float Component_2, float Component_3)
{
    Vector[0] = Component_1;
    Vector[1] = Component_2;
    Vector[2] = Component_3;
    
    return;
}

void EquateVectors(float *Vector, long Dimension, float *OtherVector)
/* OtherVector is set to be equal to Vector; Dimensions: "Dimension" x 1 */
{
    long i;
    for(i=0; i<Dimension; i++)
    {
        OtherVector[i]=Vector[i];
    }
    return;
}

void AbsValueOfVector(float *Vector, long Dimension, float *AbsValue)
/* calculates the absolute value of a vector of size "Dimension x 1" */
{
    long i;
    float HelpVariable;
    
    *AbsValue=0;
    HelpVariable=0;
    
    for (i=0; i<Dimension; i++)
    {
        HelpVariable += Vector[i]*Vector[i];
    }
    *AbsValue=sqrt(HelpVariable);
    return;
}

void InvertVector(float *Vector, long Dimension)
/* a vector is multiplied by the factor "-1", i.e. the vector is inverted. */
{
    long i;
    
    for(i=0; i<Dimension; i++)
    {
        Vector[i] = -Vector[i];
    }
    return;
}

void NormalizeVector(float *Vector, long Dimension)
/* normalizes the vector "Vector" of dimension "Dimension X 1" */
{
    long i;
    float Help;
    
    /* Help gets the absolute value of Vector assigned */
    AbsValueOfVector(Vector, Dimension, &Help);
    
    if(TestEquality(Help, 0., 1.e-300)!=0)
    /* test if the absolute value of the vector is not zero (within 1.E-300) */
    {
        for (i=0; i<Dimension; i++)
        {
            Vector[i] /= Help;
        }
    }
    else
    {
        for (i=0; i<Dimension; i++)
        {
            Vector[i]=0.;
        }
    }
}

void VectorAddition(float *Vector1, float *Vector2, long Dimension, float *VectorResult)
/* calculates the sum of 2 vectors which both have the size "Dimension x 1"; the result
   is also a vector of size "Dimension x 1" (VectorResult) */
{
    long i;
    
    for (i=0; i<Dimension; i++)
    {
        VectorResult[i]=Vector1[i]+Vector2[i];
    }
}

void VectorSubtraction(float *Vector1, float *Vector2, long Dimension, float *VectorResult)
/* calculates the difference of 2 vectors "Vector1-Vector2" which both have the size 
   "Dimension x 1"; the result is also a vector of size "Dimension x 1" (VectorResult) */
{
    long i;
    
    for (i=0; i<Dimension; i++)
    {
        VectorResult[i]=Vector1[i]-Vector2[i];
    }
    return;
}

void ScalarProduct(float *Vector1, float *Vector2, long Dimension, float *ScalarResult)
/* calculates the scalar product of two vectors which have both the
   size (Dimension x 1) */
{ 
    long i;
    
    *ScalarResult=0.;
    for (i=0; i<Dimension; i++)
    {
        *ScalarResult += Vector1[i]*Vector2[i];
    }
    return;
}

void VectorProduct(float *Vector1, float *Vector2, float *VectorResult)
/* calculates the vector product of 2 vectors which both have the size "3 x 1"; 
   the result is also a vector of size "3 x 1" (VectorResult) */
{
    VectorResult[0] = Vector1[1]*Vector2[2]-Vector1[2]*Vector2[1];
    VectorResult[1] = Vector1[2]*Vector2[0]-Vector1[0]*Vector2[2];
    VectorResult[2] = Vector1[0]*Vector2[1]-Vector1[1]*Vector2[0];
    return;
}


void MultiplyVectorWithScalar (float *Vector, float Scalar, long Dimension, float *VectorResult)
/* multiplies a vector "Vector" of size "Dimension x 1" with the scalar "Scalar"; the result
   is also a vector of the same size ("VectorResult") */
{
    long i;
    
    for (i=0; i<Dimension; i++)
    {
        VectorResult[i]=Scalar*(Vector[i]);
    }
    return;
}


void MatrixVectorProduct(float **Matrix, float *Vector, long Dimension1, long Dimension2, float *VectorResult)
/* calculates the product of a matrix (size "Dimension1 x Dimension2")
   and a vector of size "Dimension2 x 1".
   The result is a vector of size "Dimension1 x 1" */
{
    long i,j;
    
    for (i=0; i<Dimension1; i++)
    /* loop over the lines (from i=0...Dimension1-1); that makes "Dimension1" lines totally */
    {
        VectorResult[i]=0.;
        for (j=0; j<Dimension2; j++)
        /* loop over the columns of the matrix (or the lines of the vector) */
        {
            VectorResult[i] += (Matrix[i][j])*Vector[j];
        }
    }
    return;
}

void SetUpTemporaryCoordinateSystem(float *Axis_3, float *Axis_1, float *Axis_2)
/* for a given axis "Axis_3" (unit lenghth assumed!!), this function finds two other coordinate
   axes "Axis_1" and "Axis_2", both perpendicular to the first axis. The three axis form
   a orthogonal triad;
   Make sure that memory for the 2 vectors "Axis_1" and "Axis_2" (both of size 3 x 1)
   has been allocated before calling this routine */
{
    float Help;
    
    /* It's sure that AT LEAST ONE of the components of "Axis_3" is not zero. */
    
    /* //////////////////////////////////////////////////////// */
    /* (1) COMPUTE FIRST AXIS                                   */
    /* //////////////////////////////////////////////////////// */
    
    /* +++++++++++++++++++++++++++++++++++++++++++++++++++ */
    /* case 1                                              */
    /* +++++++++++++++++++++++++++++++++++++++++++++++++++ */
    if((fabs(Axis_3[0])>=fabs(Axis_3[1]))&&
       (fabs(Axis_3[0])>=fabs(Axis_3[2])))
    /* first component of the third axis is dominant */
    {
        Help=1./(sqrt(1.+(Axis_3[1]/Axis_3[0])*
                         (Axis_3[1]/Axis_3[0])));
        
        Axis_1[0]=Help*(-Axis_3[1]/Axis_3[0]);
        Axis_1[1]=Help;
        Axis_1[2]=0.;
    }
    
    /* case 2 */
    if((fabs(Axis_3[1])>=fabs(Axis_3[0]))&&
       (fabs(Axis_3[1])>=fabs(Axis_3[2])))
    /* second component of the third axis is dominant */
    {
        Help=1./(sqrt(1.+(Axis_3[0]/Axis_3[1])*
                         (Axis_3[0]/Axis_3[1])));
        
        Axis_1[0]=Help;
        Axis_1[1]=Help*(-Axis_3[0]/Axis_3[1]);;
        Axis_1[2]=0.;
    }
    
    /* case 3 */
    if((fabs(Axis_3[2])>=fabs(Axis_3[0]))&&
       (fabs(Axis_3[2])>=fabs(Axis_3[1])))
    /* third component of the third axis is dominant */
    {
        Help=1./(sqrt(1.+(Axis_3[0]/Axis_3[2])*
                         (Axis_3[0]/Axis_3[2])));
        
        Axis_1[0]=Help;
        Axis_1[1]=0.;
        Axis_1[2]=Help*(-Axis_3[0]/Axis_3[2]);
    }
    
    /* //////////////////////////////////////////////////////// */
    /* (1) COMPUTE SECOND AXIS                                  */
    /* //////////////////////////////////////////////////////// */
    
    VectorProduct(Axis_3,Axis_1,Axis_2);
    
    return;     
}


void TransformVector_Local_a1a2a3_To_Global_xyz(float *LocalAxis1, float *LocalAxis2, float *LocalAxis3,
                                                float *Vector_Local, float *Vector_Global)
/* for a given vector in the local (surface-related) coordinate system {a1,a2,a3}, this routine computes
   the vector components in the global {x,y,z}-coordinate system.
   
   The transformation can be expressed as:
   
   x_glob = M * x_loc
   
   with
   
     x_glob = (x_x, x_y, x_z) = vector in global coordinates
                                
     x_loc  = (x_a1, x_a2, x_a3) = vector in local coordinates
     
     M      = 3 x 3 transformation matrix, given by:
     
                         ( a1_x   a2_x   a3_x )
                   M =   ( a1_y   a2_y   a3_y )
                         ( a1_z   a2_z   a3_z )
                         
*/     
     
   
{
    /* LOCAL VARIABLES */
    /* --------------- */
    float **TransformationMatrix;
    
    
    /* Allocate memory for the transfomation matrix {a1,a2,a3} --> {x,y,z} */
    TransformationMatrix = AllocMemForFloatMatrix(3, 3);
    
    /* Compute the transformation matrix */
    GetTransformationMatrix_Local3_To_Global3(LocalAxis1, LocalAxis2, LocalAxis3, TransformationMatrix);
                                              
    /* Compute the transformed point */
    MatrixVectorProduct(TransformationMatrix, Vector_Local, 3, 3, Vector_Global);
    
    /* Free memory for the transformation matrix */
    FreeMemForFloatMatrix(TransformationMatrix, 3);
                                               
    return;
}


void TransformPoint_Local_a1a2a3_To_Global_xyz(float *LocalAxis1, float *LocalAxis2, float *LocalAxis3,
                                               float *LocalOrigin, float *Point_Local, float *Point_Global)
/* for a given point in the local (surface-related) coordinate system {a1,a2,a3}, this routine computes
   the position in the global {x,y,z}-coordinate system.
   
   The transformation can be expressed as:
   
   p_glob = v + M * p_loc
   
   with
   
     p_glob = (p_x, p_y, p_z) = point in global coordinates
     
     v      = (v_x, v_y, v_z) = surface vertex in global coordinates
                                (= origin of local {a1,a2,a3}-system)
                                
     p_loc  = (p_a1, p_a2, p_a3) = point in local coordinates (with respect to the vertex)
     
     M      = 3 x 3 transformation matrix, given by:
     
                         ( a1_x   a2_x   a3_x )
                   M =   ( a1_y   a2_y   a3_y )
                         ( a1_z   a2_z   a3_z )
                         
*/     
     
   
{
    /* LOCAL VARIABLES */
    /* --------------- */
    float **TransformationMatrix;
    float HelpVector[3];
    
    
    /* Allocate memory for the transfomation matrix {a1,a2,a3} --> {x,y,z} */
    TransformationMatrix = AllocMemForFloatMatrix(3, 3);
    
    /* Compute the transformation matrix */
    GetTransformationMatrix_Local3_To_Global3(LocalAxis1, LocalAxis2, LocalAxis3, TransformationMatrix);
                                              
    /* Compute the transformed point */
    MatrixVectorProduct(TransformationMatrix, Point_Local, 3, 3, HelpVector);
    VectorAddition(HelpVector, LocalOrigin, 3, Point_Global);
    
    /* Free memory for the transformation matrix */
    FreeMemForFloatMatrix(TransformationMatrix, 3);
                                               
    return;
}

void GetTransformationMatrix_Local3_To_Global3(float *LocalAxis1, float *LocalAxis2, float *LocalAxis3,
                                               float **TransformMatrix)
/* computes the 3 x 3 transformation matrix for coordinate transformations from the local
   {Axis1,Axis2,Axis3}-coordinate system into the global coordinate system;
   Make sure that memory for the 3 x 3 matrix 'TransformMatrix' has been allocated before calling
   this function !! */
{
    /* The matrix is given by:
    
         a1_x   a2_x   a3_x
         a1_y   a2_y   a3_y
         a1_z   a2_z   a3_z
    */
    
    /* line 1 */     
    TransformMatrix[0][0] = LocalAxis1[0];
    TransformMatrix[0][1] = LocalAxis2[0];
    TransformMatrix[0][2] = LocalAxis3[0];
    
    /* line 2 */
    TransformMatrix[1][0] = LocalAxis1[1];
    TransformMatrix[1][1] = LocalAxis2[1];
    TransformMatrix[1][2] = LocalAxis3[1];
    
    /* line 3 */
    TransformMatrix[2][0] = LocalAxis1[2];
    TransformMatrix[2][1] = LocalAxis2[2];
    TransformMatrix[2][2] = LocalAxis3[2];
    
    return;
}

void GetTransformationMatrix_Global3_To_Local2(float *LocalAxis1, float *LocalAxis2,
                                               float **TransformMatrix)
/* computes the 2 x 3 transformation matrix for coordinate transformations from the global {x,y,z} system to
   the local {Axis1,Axis2}-coordinate system;
   Make sure that memory for the 2 x 3 matrix 'TransformMatrix' has been allocated before calling
   this function !! */
/* The matrix is given by:
    
         a1_x   a1_y   a1_z
         a2_x   a2_y   a2_z
*/
{
    TransformMatrix[0][0] = LocalAxis1[0];
    TransformMatrix[0][1] = LocalAxis1[1];
    TransformMatrix[0][2] = LocalAxis1[2];
    
    TransformMatrix[1][0] = LocalAxis2[0];
    TransformMatrix[1][1] = LocalAxis2[1];
    TransformMatrix[1][2] = LocalAxis2[2];
    
    return;
}


void TransformPoint_Global_xyz_To_Local_a1a2(float *LocalAxis1, float *LocalAxis2, float *LocalOrigin,
                                             float *Point_Global, float *Point_Local)
/* for a given point in the global {x,y,z}-coordinate system, this routine computes
   the position in the 2D - local (surface-related) coordinate system {a1,a2} whoose origin
   is the vertex of the surface.
   
   The transformation can be expressed as:
   
   p_loc = M * ( p_glob - v )
   
   with
   
     p_glob = (p_x, p_y, p_z) = point in global coordinates
     
     v      = (v_x, v_y, v_z) = origin of local coordinate system
                                = surface vertex in global coordinates
                                (= origin of local {a1,a2,a3}-system)
                                
     p_loc  = (p_a1, p_a2) = point in local coordinates (with respect to the vertex)
     
     M      = 2 x 3 transformation matrix, given by:
     
                         ( a1_x   a1_y   a1_z )
                   M =   ( a2_x   a2_y   a2_z )
                         
                         
*/     
     
   
{
    /* LOCAL VARIABLES */
    /* --------------- */
    float **TransformationMatrix; /* size 2 x 3 */
    float HelpVector[3];
    
    
    /* Allocate memory for the transfomation matrix {x,y,z} --> {a1,a2} */
    TransformationMatrix = AllocMemForFloatMatrix(2, 3);
    
    /* Compute the transformation matrix */
    GetTransformationMatrix_Global3_To_Local2(LocalAxis1, LocalAxis2, TransformationMatrix);
                                              
    /* Compute the transformed point */
    VectorSubtraction(Point_Global, LocalOrigin, 3, HelpVector);
    MatrixVectorProduct(TransformationMatrix, HelpVector, 2, 3, Point_Local);
    
    
    /* Free memory for the transformation matrix */
    FreeMemForFloatMatrix(TransformationMatrix, 2);
                                               
    return;
}


void RotateVector(float *Vector, float *RotationAxis, float Angle)
/* -------------------------------------------------------------------------------------- */
/* The 3 x 1 vector "Vector" is rotated around the axis "RotationAxis" by the angle       */
/* "Angle" (positive rotation sense defined by "right-hand-rule")                         */
/* -------------------------------------------------------------------------------------- */
{
    /* The following formula is used:
    
       ----------------------------------------------------
       |                                                  |
       |  x' = x_par + cos(a) x_per + sin(a) (i X x_per)  |  ("X" = vector product)
       |                                                  |
       ----------------------------------------------------
       
       where
       
       x     = vector to rotate = x_par + x_per
       x_par = part of x parallel to rotation axis i,
               -------------------------------------------
               |                                         |
               |  x_par = (x*i)i ("*" = scalar product)  |
               |                                         |
               -------------------------------------------
       x_per = part of x perpendicular to rotation axis i,
               x_per = x - x_par
       i     = rotation axis
       a     = rotation angle
       
       All vectors of size 3 x 1 !!
       
    */   
    
    /* LOCAL VARIABLES */
    /* --------------- */
    float Vector_PerpendicularToRotationAxis[3];
    float Vector_ParallelToRotationAxis[3];
    
    float Help;
    float HelpVector1[3],
           HelpVector2[3],
           HelpVector3[3];
    
    /* Compute the part of the vector parallel to the rotation axis */
    /* //////////////////////////////////////////////////////////// */
    ScalarProduct(Vector, RotationAxis, 3, &Help);
    MultiplyVectorWithScalar (RotationAxis, Help, 3, Vector_ParallelToRotationAxis);
    
    /* Compute the part of the vector perpendicular to the rotation axis */
    /* ///////////////////////////////////////////////////////////////// */
    VectorSubtraction(Vector, Vector_ParallelToRotationAxis, 3, Vector_PerpendicularToRotationAxis);
    
    /* The part parallel to the rotation axis stays un-modied.       */
    /* Only the part perpendicular is affected by the rotation.      */
    
    /* Rotate the part perpendicular to the axis of rotation */
    /* ///////////////////////////////////////////////////// */
    MultiplyVectorWithScalar (Vector_PerpendicularToRotationAxis, cos(Angle), 3, HelpVector1);
    VectorProduct(RotationAxis, Vector_PerpendicularToRotationAxis, HelpVector2);
    MultiplyVectorWithScalar (HelpVector2, sin(Angle), 3, HelpVector3);
    VectorAddition(HelpVector1, HelpVector3, 3, Vector_PerpendicularToRotationAxis);
    /* Now "Vector_PerpendicularToRotationAxis" holds the perp. part AFTER rotation! */
    
    /* Build the final vector (AFTER rotation) (--> stored in "Vector") */
    /* //////////////////////////////////////////////////////////////// */
    VectorAddition(Vector_ParallelToRotationAxis, Vector_PerpendicularToRotationAxis, 3, Vector);
    
    return;
}

void RotatePoint(float *Point, float *RotationAxis, float *RotationCenter, float Angle)
/* ----------------------------------------------------------------------------------------- */
/* A point defined by its location vector "Point" (connecting the origin and the point) is   */
/* rotated around a given center of rotation ("RotationCenter") by an angle "Angle".         */
/* The result is stored in "Point" --> the content of "Point" is overwritten!                */
/* ----------------------------------------------------------------------------------------- */
{
    float RotatingVector[3];
    /* the vector connecting the center of rotation and the point; it's given by
       "Point" - "RotationCenter"; This vector is passed to the routine "RotateVector". */
    
       
    /* Compute the vector interconnecting the center of rotation and the point to rotate */
    /* ///////////////////////////////////////////////////////////////////////////////// */   
    VectorSubtraction(Point, RotationCenter, 3, RotatingVector);
    
    /* Rotate the vector */
    /* ///////////////// */
    RotateVector(RotatingVector, RotationAxis, Angle);
    
    /* Now the vector "RotatingVector" points from the center of rotation to the rotated point */
    
    /* Construct the new location vector (for the rotated point) */
    /* ///////////////////////////////////////////////////////// */
    VectorAddition(RotationCenter, RotatingVector, 3, Point);
      
    return;
}

void QuadraticEquationSolutionSelect(float a, float b, float c, long *NumberOfSolutions, float *PathLength)
{
    /* some local variables */
    /* -------------------- */
    
    float q;
    /* help parameter used for quadratic equation solving (see Numerical Recipes, p. 184) */
    
    float discriminant;
    /* used for quadratic equation solving; discriminant = b*b-4*a*c */
    
    float Solution1, Solution2;
    /* the 2 solutions of the qaudratic equation */
    
    
    /* calculation of the discriminant */
    /* ------------------------------- */
    discriminant=b*b-4.*a*c;
        
    if(discriminant<0)
    {
        /* No (real) Solution */
        *NumberOfSolutions = 0;
        return;
    }
    
    /* If the routine reaches this point it is assured that the discriminant is non-negative ! */
        
    /* Calculation of parameter 'q'                                        */
    /*   see: 'NUMERICAL RECIPES IN C', Second Edition, 1992, page 183 ff. */
        
    if(b>=0)
    {
        q = -0.5*(b+sqrt(discriminant));
    }
    else
    {
        q = -0.5*(b-sqrt(discriminant));
    }
        
    /* the solutions are
                        Solution1 = q/a;
                        Solution2 = c/q;  */
        
    if((TestEquality(a,0.,DBL_MIN)!=0)&&(TestEquality(q,0.,DBL_MIN)!=0))
    /* both, a and q are not zero (within DBL_MIN) */
    {
        /* both solutions are allowed */
        Solution1 = q/a;
        Solution2 = c/q;
            
        if((Solution1>=0)&&(Solution2>=0))
        {
            /* both are positive */
            *NumberOfSolutions=2;
            PathLength[0]=Solution1; PathLength[1]=Solution2;
            return;
        }
        if((Solution1>=0)&&(Solution2<0))
        {
            /* only one is positive (solution 1) */
            *NumberOfSolutions=1;
            PathLength[0]=Solution1;
            return;
        }
        if((Solution2>=0)&&(Solution1<0))
        {
            /* only one is positive (solution 2) */
            *NumberOfSolutions=1;
            PathLength[0]=Solution2;
            return;
        }
        if((Solution1<0)&&(Solution2<0))
        {
            /* both are negative */
            *NumberOfSolutions=0;
            return;
        }
    }
    else
    {
        /* one or two solution(s) are not allowed (a and/or q are zero (within DBL_MIN) */
            
        if(TestEquality(a,0.,DBL_MIN)!=0)
        {
            /* solution1 is allowed */
                
            Solution1 = q/a;
            
            if(Solution1>=0)
            {
                /* solution 1 is positive */
                *NumberOfSolutions=1;
                PathLength[0]=Solution1;
                return;
            }
            else
            {
                /* solution 1 is negative */
                *NumberOfSolutions=0;
                return;
            }
        }
        if(TestEquality(q,0.,DBL_MIN)!=0)
        {
            /* solution2 is allowed */
                
            Solution2 = c/q;
                
            if(Solution2>=0)
            {
                /* solution 2 is positive */
                *NumberOfSolutions=1;
                PathLength[0]=Solution2;
                return;
            }
            else
            {
                /* solution 2 is negative */
                *NumberOfSolutions=0;
                return;
            }
        }
            
        /* both are not allowed */
        *NumberOfSolutions=0;
        return;
    }
    return; 
}
