/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: issshadowProgramFlow.c,v 1.8 2001/09/13 09:12:39 vltsccm Exp $"
*
* who       when       what
* --------  ---------- ----------------------------------------------
* nhousen   2001-09-06 small correction to function DeactivateUTCylinder
*                      to eliminate compilation warnings 
* nhousen   2001-07-12 reduced number of included header files
* nhousen   2001-07-12 adapted error handling to VLT standards
* rwilhelm  2001-07-10 created 
*/

/************************************************************************
*   NAME
*   issshadowProgramFlow.c
*
*   SYNOPSIS
*
*
*   DESCRIPTION
*   Contains general support routines which are used by the 'issshadowAltAz' routine.
* 
*   PARENT CLASS
*
*
*   DESCRIPTION
*   
*
*   PUBLIC METHODS
*
*   PUBLIC DATA MEMBERS
*
*
*   PROTECTED METHODS
*
*
*   PROTECTED DATA MEMBERS
*
*
*   PRIVATE METHODS
*
*
*   PRIVATE DATA MEMBERS
*
*
*   FILES
*
*   ENVIRONMENT
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS
*
*------------------------------------------------------------------------
*/

/* #define _POSIX_SOURCE 1 */
/* #include "vltPort.h" */
/*
static char *rcsId="@(#) $Id: issshadowProgramFlow.c,v 1.8 2001/09/13 09:12:39 vltsccm Exp $";
static void *use_rcsId = ((void)&use_rcsId,(void *) &rcsId);
*/

/* Standard ANSI C Includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "issshadowDefines.h"
#include "issshadowErrors.h"


void DefineVLTISiteCoordinateSystem(float *u_Axis, float *v_Axis, float *w_Axis)
{
    /* Define the VLTI site coordinate system {u, v, w} */
    u_Axis[0] = 1; u_Axis[1] = 0; u_Axis[2] = 0;
    v_Axis[0] = 0; v_Axis[1] = 1; v_Axis[2] = 0;
    w_Axis[0] = 0; w_Axis[1] = 0; w_Axis[2] = 1;
    return;
}




ccsCOMPL_STAT ParseInputArguments(double alt,
                         double az,
                         int teltype,
                         int location,
                         int noofRayRadial,
                         int noofRayAngular,
                         TelescopeStruct *Telescope,
                         BeamStruct *Beam,
			 ccsERROR *error)
{
    /* This function 'fills' the structures 'Telescope' and 'Beam' using
       the input information passed to the routine */
        
    /* ======================================================= */
    /* Telescope structure                                     */
    /* ======================================================= */

    /* Telescope type (UT, AT or SID) */
    if(teltype==1)
    {
        strcpy(Telescope->Type, "UT");
    }
    else if(teltype==4)
    {
        strcpy(Telescope->Type, "AT");
    }
    else if(teltype==5)
    {
        strcpy(Telescope->Type, "SID");
    }
    else
    {
        /* Illegal telescope type! --> Error */
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "Illegal value for the input parameter 'teltype': %d\n", teltype);
        fprintf(stderr, "Possible values for the input parameter\n");
        fprintf(stderr, "'teltype' are: 1 <==> UT\n");
        fprintf(stderr, "               4 <==> AT\n");
        fprintf(stderr, "               5 <==> SID\n");
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_TEL_TYPE, __FILE__, "", "");
        return(FAILURE);
    }

    /* Station number, station name and station position */
    if ( SetStationCharacteristics(location, Telescope, error) == FAILURE )
      {
	return(FAILURE);
      }

    /* ======================================================= */
    /* Beam structure                                          */
    /* ======================================================= */
    /* Radial incrementation */
    if(noofRayRadial>=0)
    {
        Beam->RadialIncrementation = noofRayRadial;
    }
    else
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "Invalid radial incrementation!\n");
        fprintf(stderr, "You supplied the value %d\n", noofRayRadial);
        fprintf(stderr, "The minimum allowed value is 0 (--> one single ray).\n");
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_RADIAL_INCR, __FILE__, "", "");
        return(FAILURE);
    }
    
    /* Angular incrementation */
    if(noofRayAngular>=1)
    {
        Beam->AngularIncrementation = noofRayAngular; 
    }
    else
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "Invalid angular incrementation!\n");
        fprintf(stderr, "You supplied the value %d\n", noofRayAngular);
        fprintf(stderr, "The minimum allowed value is 1 (--> no angular segmentation).\n");
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_ANGULAR_INCR, __FILE__, "", "");
        return(FAILURE);
    }

    /* ======================================================= */
    /* Pointing angles                                         */
    /* ======================================================= */
    if((alt>90)||(alt<0))
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The altitude angle (in degrees) must be in the range between 0 and 90 degrees.\n");
        fprintf(stderr, "The value passed to the 'issshadowAltAz' routine is 'alt' = %6.2f\n", alt);
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_ALT_ANGLE, __FILE__, "", "");
        return(FAILURE);
    }   
    /* VLT-ICD-ESO-150000-1918, Issue 1.0:
       It is possible to observe with ATs and UTs any object with a zenith distance comprised
       between 0.5 deg and 70 deg. However, all VLTI performance requirements and all system
       analyses are valid in the 0.5 deg -- 60 deg range. This means that observation between
       60 deg and 70 deg zenith distances are a priori possible, but performance may be degraded.*/
    if(((alt>89.5)||(alt<20))&&
       (strcmp(Telescope->Type, "SID")!=0))
    {
        /* just print a warning message ... */
        
      /*fprintf(stderr, "------------------------------------------------------\n"); 
        fprintf(stderr, "|                W A R N I N G  !!!                  |\n"); 
        fprintf(stderr, "|                                                    |\n"); 
        fprintf(stderr, "| According to VLT-ICD-ESO-150000-1918, Issue 1.0    |\n"); 
        fprintf(stderr, "| the range of possible altitude (=elevation) angles |\n"); 
        fprintf(stderr, "| for UTs and ATs is between 20 deg and 89.5 deg.    |\n"); 
        fprintf(stderr, "| (zenith distance range = 0.5 deg ... 70 deg)       |\n"); 
        fprintf(stderr, "|                                                    |\n"); 
        fprintf(stderr, "| You specified an altitude value of %6.2f          |\n", alt); 
        fprintf(stderr, "------------------------------------------------------\n");*/ 
      /*errAdd(error, issshadowMOD, issshadowERR_ALTITUDE_RANGE, __FILE__, "", "");*/
    }
    
    /* Azimuth angle */
    if((az>360)||(az<0))
    {
      /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "The azimuth angle (in degrees) must be in the range between 0 and 360 degrees.\n");
        fprintf(stderr, "The value passed to the 'issshadowAltAz' routine is 'az' = %6.2f\n", az);
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_AZIMUTH_RANGE, __FILE__, "", "");
        return(FAILURE);
    }

    return(SUCCESS);
}


ccsCOMPL_STAT DefineVignettingSurfaces(SurfaceStruct *Surface, long *NumberOfSurfaces,
                              float *u_Axis, float *v_Axis, float *w_Axis, ccsERROR *error)
{
    
    /* ************************************************************** */
    /* * SET NUMBER OF SURFACES (ADJUST THIS VALUE IF YOU ADD       * */
    /* * SURFACES TO THE ABOVE LIST!                                * */
    /* ************************************************************** */
    *NumberOfSurfaces = 73;  
    
    /* Check if number of surfaces it not too big */
    if(*NumberOfSurfaces>MAXIMUM_NUMBER_OF_SURFACES)
    {
      /*fprintf(stderr,"*** ERROR MESSAGE ***\n");
        fprintf(stderr,"The number of surfaces defined in the issshadow\n");
        fprintf(stderr,"routine 'DefineVignettingSurfaces' is too big:\n");
        fprintf(stderr,"The maximum number is defined in 'issshadowConstants.h'\n");
        fprintf(stderr,"(constant MAXIMUM_NUMBER_OF_SURFACES = %d).\n",
                MAXIMUM_NUMBER_OF_SURFACES);
        fprintf(stderr,"You have specified a number of %d.\n", *NumberOfSurfaces);
        exit(0); */
        errAdd(error, issshadowMOD, issshadowERR_TOO_MANY_SURFACES, __FILE__, "", "");
        return(FAILURE);
    }


    /* ================================================================================== */
    /* THE 4 UTs                                                                          */
    /* ================================================================================== */
    strcpy(Surface[0].Type, "Cylinder"); strcpy(Surface[0].Name, "UT1");
    Surface[0].CylinderCenter_uv[0] = -16.; Surface[0].CylinderCenter_uv[1] = -16.;
    Surface[0].Width = 29.; Surface[0].Height = 25.;
    Surface[0].Defined = 1;

    strcpy(Surface[1].Type, "Cylinder"); strcpy(Surface[1].Name, "UT2");
    Surface[1].CylinderCenter_uv[0] =  24.; Surface[1].CylinderCenter_uv[1] =  24.;
    Surface[1].Width = 29.; Surface[1].Height = 25.;
    Surface[1].Defined = 1;

    strcpy(Surface[2].Type, "Cylinder"); strcpy(Surface[2].Name, "UT3");
    Surface[2].CylinderCenter_uv[0] =  64.; Surface[2].CylinderCenter_uv[1] =  48.;
    Surface[2].Width = 29.; Surface[2].Height = 25.;
    Surface[2].Defined = 1;

    strcpy(Surface[3].Type, "Cylinder"); strcpy(Surface[3].Name, "UT4");
    Surface[3].CylinderCenter_uv[0] = 112.; Surface[3].CylinderCenter_uv[1] =   8.;
    Surface[3].Width = 29.; Surface[3].Height = 25.;
    Surface[3].Defined = 1;

    /* ================================================================================== */
    /* ATs ON THIRTY STATIONS                                                             */
    /* ================================================================================== */
    strcpy(Surface[4].Type, "Cylinder"); strcpy(Surface[4].Name, "AT_A0");
    Surface[4].CylinderCenter_uv[0] = -32.; Surface[4].CylinderCenter_uv[1] = -48.;
    Surface[4].Width = 5.25; Surface[4].Height = 7.;
    Surface[4].Defined = 0;

    strcpy(Surface[5].Type, "Cylinder"); strcpy(Surface[5].Name, "AT_A1");
    Surface[5].CylinderCenter_uv[0] = -32.; Surface[5].CylinderCenter_uv[1] = -64.;
    Surface[5].Width = 5.25; Surface[5].Height = 7.;
    Surface[5].Defined = 0;
    
    strcpy(Surface[6].Type, "Cylinder"); strcpy(Surface[6].Name, "AT_B0");
    Surface[6].CylinderCenter_uv[0] = -24.; Surface[6].CylinderCenter_uv[1] = -48.;
    Surface[6].Width = 5.25; Surface[6].Height = 7.;
    Surface[6].Defined = 0;

    strcpy(Surface[7].Type, "Cylinder"); strcpy(Surface[7].Name, "AT_B1");
    Surface[7].CylinderCenter_uv[0] = -24.; Surface[7].CylinderCenter_uv[1] = -64.;
    Surface[7].Width = 5.25; Surface[7].Height = 7.;
    Surface[7].Defined = 0;

    strcpy(Surface[8].Type, "Cylinder"); strcpy(Surface[8].Name, "AT_B2");
    Surface[8].CylinderCenter_uv[0] = -24.; Surface[8].CylinderCenter_uv[1] = -72.;
    Surface[8].Width = 5.25; Surface[8].Height = 7.;
    Surface[8].Defined = 0;

    strcpy(Surface[9].Type, "Cylinder"); strcpy(Surface[9].Name, "AT_B3");
    Surface[9].CylinderCenter_uv[0] = -24.; Surface[9].CylinderCenter_uv[1] = -80.;
    Surface[9].Width = 5.25; Surface[9].Height = 7.;
    Surface[9].Defined = 0;

    strcpy(Surface[10].Type, "Cylinder"); strcpy(Surface[10].Name, "AT_B4");
    Surface[10].CylinderCenter_uv[0] = -24.; Surface[10].CylinderCenter_uv[1] = -88.;
    Surface[10].Width = 5.25; Surface[10].Height = 7.;
    Surface[10].Defined = 0;

    strcpy(Surface[11].Type, "Cylinder"); strcpy(Surface[11].Name, "AT_B5");
    Surface[11].CylinderCenter_uv[0] = -24.; Surface[11].CylinderCenter_uv[1] = -96.;
    Surface[11].Width = 5.25; Surface[11].Height = 7.;
    Surface[11].Defined = 0;

    strcpy(Surface[12].Type, "Cylinder"); strcpy(Surface[12].Name, "AT_C0");
    Surface[12].CylinderCenter_uv[0] = -16.; Surface[12].CylinderCenter_uv[1] = -48.;
    Surface[12].Width = 5.25; Surface[12].Height = 7.;
    Surface[12].Defined = 0;

    strcpy(Surface[13].Type, "Cylinder"); strcpy(Surface[13].Name, "AT_C1");
    Surface[13].CylinderCenter_uv[0] = -16.; Surface[13].CylinderCenter_uv[1] = -64.;
    Surface[13].Width = 5.25; Surface[13].Height = 7.;
    Surface[13].Defined = 0;

    strcpy(Surface[14].Type, "Cylinder"); strcpy(Surface[14].Name, "AT_C2");
    Surface[14].CylinderCenter_uv[0] = -16.; Surface[14].CylinderCenter_uv[1] = -72.;
    Surface[14].Width = 5.25; Surface[14].Height = 7.;
    Surface[14].Defined = 0;

    strcpy(Surface[15].Type, "Cylinder"); strcpy(Surface[15].Name, "AT_C3");
    Surface[15].CylinderCenter_uv[0] = -16.; Surface[15].CylinderCenter_uv[1] = -80.;
    Surface[15].Width = 5.25; Surface[15].Height = 7.;
    Surface[15].Defined = 0;

    strcpy(Surface[16].Type, "Cylinder"); strcpy(Surface[16].Name, "AT_D0");
    Surface[16].CylinderCenter_uv[0] =   0.; Surface[16].CylinderCenter_uv[1] = -48.;
    Surface[16].Width = 5.25; Surface[16].Height = 7.;
    Surface[16].Defined = 0;

    strcpy(Surface[17].Type, "Cylinder"); strcpy(Surface[17].Name, "AT_D1");
    Surface[17].CylinderCenter_uv[0] =   0.; Surface[17].CylinderCenter_uv[1] = -80.;
    Surface[17].Width = 5.25; Surface[17].Height = 7.;
    Surface[17].Defined = 0;
    
    strcpy(Surface[18].Type, "Cylinder"); strcpy(Surface[18].Name, "AT_D2");
    Surface[18].CylinderCenter_uv[0] =   0.; Surface[18].CylinderCenter_uv[1] = -96.;
    Surface[18].Width = 5.25; Surface[18].Height = 7.;
    Surface[18].Defined = 0;

    strcpy(Surface[19].Type, "Cylinder"); strcpy(Surface[19].Name, "AT_E0");
    Surface[19].CylinderCenter_uv[0] =  16.; Surface[19].CylinderCenter_uv[1] = -48.;
    Surface[19].Width = 5.25; Surface[19].Height = 7.;
    Surface[19].Defined = 0;

    strcpy(Surface[20].Type, "Cylinder"); strcpy(Surface[20].Name, "AT_G0");
    Surface[20].CylinderCenter_uv[0] =  32.; Surface[20].CylinderCenter_uv[1] = -48.;
    Surface[20].Width = 5.25; Surface[20].Height = 7.;
    Surface[20].Defined = 0;

    strcpy(Surface[21].Type, "Cylinder"); strcpy(Surface[21].Name, "AT_G1");
    Surface[21].CylinderCenter_uv[0] =  32.; Surface[21].CylinderCenter_uv[1] = -112.;
    Surface[21].Width = 5.25; Surface[21].Height = 7.;
    Surface[21].Defined = 0;

    strcpy(Surface[22].Type, "Cylinder"); strcpy(Surface[22].Name, "AT_G2");
    Surface[22].CylinderCenter_uv[0] =  32.; Surface[22].CylinderCenter_uv[1] = -24.;
    Surface[22].Width = 5.25; Surface[22].Height = 7.;
    Surface[22].Defined = 0;

    strcpy(Surface[23].Type, "Cylinder"); strcpy(Surface[23].Name, "AT_H0");
    Surface[23].CylinderCenter_uv[0] =  64.; Surface[23].CylinderCenter_uv[1] = -48.;
    Surface[23].Width = 5.25; Surface[23].Height = 7.;
    Surface[23].Defined = 0;

    strcpy(Surface[24].Type, "Cylinder"); strcpy(Surface[24].Name, "AT_I1");
    Surface[24].CylinderCenter_uv[0] =  72.; Surface[24].CylinderCenter_uv[1] = -88.;
    Surface[24].Width = 5.25; Surface[24].Height = 7.;
    Surface[24].Defined = 0;

    strcpy(Surface[25].Type, "Cylinder"); strcpy(Surface[25].Name, "AT_J1");
    Surface[25].CylinderCenter_uv[0] =  88.; Surface[25].CylinderCenter_uv[1] = -72.;
    Surface[25].Width = 5.25; Surface[25].Height = 7.;
    Surface[25].Defined = 0;

    strcpy(Surface[26].Type, "Cylinder"); strcpy(Surface[26].Name, "AT_J2");
    Surface[26].CylinderCenter_uv[0] =  88.; Surface[26].CylinderCenter_uv[1] = -96.;
    Surface[26].Width = 5.25; Surface[26].Height = 7.;
    Surface[26].Defined = 0;

    strcpy(Surface[27].Type, "Cylinder"); strcpy(Surface[27].Name, "AT_J3");
    Surface[27].CylinderCenter_uv[0] =  88.; Surface[27].CylinderCenter_uv[1] =  8.;
    Surface[27].Width = 5.25; Surface[27].Height = 7.;
    Surface[27].Defined = 0;

    strcpy(Surface[28].Type, "Cylinder"); strcpy(Surface[28].Name, "AT_J4");
    Surface[28].CylinderCenter_uv[0] =  88.; Surface[28].CylinderCenter_uv[1] =  24.;
    Surface[28].Width = 5.25; Surface[28].Height = 7.;
    Surface[28].Defined = 0;

    strcpy(Surface[29].Type, "Cylinder"); strcpy(Surface[29].Name, "AT_J5");
    Surface[29].CylinderCenter_uv[0] =  88.; Surface[29].CylinderCenter_uv[1] =  48.;
    Surface[29].Width = 5.25; Surface[29].Height = 7.;
    Surface[29].Defined = 0;

    strcpy(Surface[30].Type, "Cylinder"); strcpy(Surface[30].Name, "AT_J6");
    Surface[30].CylinderCenter_uv[0] =  88.; Surface[30].CylinderCenter_uv[1] =  72.;
    Surface[30].Width = 5.25; Surface[30].Height = 7.;
    Surface[30].Defined = 0;

    strcpy(Surface[31].Type, "Cylinder"); strcpy(Surface[31].Name, "AT_K0");
    Surface[31].CylinderCenter_uv[0] =  96.; Surface[31].CylinderCenter_uv[1] = -48.;
    Surface[31].Width = 5.25; Surface[31].Height = 7.;
    Surface[31].Defined = 0;

    strcpy(Surface[32].Type, "Cylinder"); strcpy(Surface[32].Name, "AT_L0");
    Surface[32].CylinderCenter_uv[0] = 104.; Surface[32].CylinderCenter_uv[1] = -48.;
    Surface[32].Width = 5.25; Surface[32].Height = 7.;
    Surface[32].Defined = 0;

    strcpy(Surface[33].Type, "Cylinder"); strcpy(Surface[33].Name, "AT_M0");
    Surface[33].CylinderCenter_uv[0] = 112.; Surface[33].CylinderCenter_uv[1] = -48.;
    Surface[33].Width = 5.25; Surface[33].Height = 7.;
    Surface[33].Defined = 0;


    /* ================================================================================== */
    /* SIDs ON THIRTY STATIONS                                                            */
    /* ================================================================================== */
    strcpy(Surface[34].Type, "Cylinder"); strcpy(Surface[34].Name, "SID_A0");
    Surface[34].CylinderCenter_uv[0] = -32.; Surface[34].CylinderCenter_uv[1] = -48.4;
    Surface[34].Width = 0.62; Surface[34].Height = 1.3;
    Surface[34].Defined = 0;

    strcpy(Surface[35].Type, "Cylinder"); strcpy(Surface[35].Name, "SID_A1");
    Surface[35].CylinderCenter_uv[0] = -32.; Surface[35].CylinderCenter_uv[1] = -64.4;
    Surface[35].Width = 0.62; Surface[35].Height = 1.3;
    Surface[35].Defined = 0;
    
    strcpy(Surface[36].Type, "Cylinder"); strcpy(Surface[36].Name, "SID_B0");
    Surface[36].CylinderCenter_uv[0] = -24.; Surface[36].CylinderCenter_uv[1] = -48.4;
    Surface[36].Width = 0.362; Surface[36].Height = 1.3;
    Surface[36].Defined = 0;

    strcpy(Surface[37].Type, "Cylinder"); strcpy(Surface[37].Name, "SID_B1");
    Surface[37].CylinderCenter_uv[0] = -24.; Surface[37].CylinderCenter_uv[1] = -64.4;
    Surface[37].Width = 0.62; Surface[37].Height = 1.3;
    Surface[37].Defined = 0;

    strcpy(Surface[38].Type, "Cylinder"); strcpy(Surface[38].Name, "SID_B2");
    Surface[38].CylinderCenter_uv[0] = -24.; Surface[38].CylinderCenter_uv[1] = -72.4;
    Surface[38].Width = 0.62; Surface[38].Height = 1.3;
    Surface[38].Defined = 0;

    strcpy(Surface[39].Type, "Cylinder"); strcpy(Surface[39].Name, "SID_B3");
    Surface[39].CylinderCenter_uv[0] = -24.; Surface[39].CylinderCenter_uv[1] = -80.4;
    Surface[39].Width = 0.62; Surface[39].Height = 1.3;
    Surface[39].Defined = 0;

    strcpy(Surface[40].Type, "Cylinder"); strcpy(Surface[40].Name, "SID_B4");
    Surface[40].CylinderCenter_uv[0] = -24.; Surface[40].CylinderCenter_uv[1] = -88.4;
    Surface[40].Width = 0.62; Surface[40].Height = 1.3;
    Surface[40].Defined = 0;

    strcpy(Surface[41].Type, "Cylinder"); strcpy(Surface[41].Name, "SID_B5");
    Surface[41].CylinderCenter_uv[0] = -24.; Surface[41].CylinderCenter_uv[1] = -96.4;
    Surface[41].Width = 0.62; Surface[41].Height = 1.3;
    Surface[41].Defined = 0;

    strcpy(Surface[42].Type, "Cylinder"); strcpy(Surface[42].Name, "SID_C0");
    Surface[42].CylinderCenter_uv[0] = -16.; Surface[42].CylinderCenter_uv[1] = -48.4;
    Surface[42].Width = 0.62; Surface[42].Height = 1.3;
    Surface[42].Defined = 0;

    strcpy(Surface[43].Type, "Cylinder"); strcpy(Surface[43].Name, "SID_C1");
    Surface[43].CylinderCenter_uv[0] = -16.; Surface[43].CylinderCenter_uv[1] = -64.4;
    Surface[43].Width = 0.62; Surface[43].Height = 1.3;
    Surface[43].Defined = 0;

    strcpy(Surface[44].Type, "Cylinder"); strcpy(Surface[44].Name, "SID_C2");
    Surface[44].CylinderCenter_uv[0] = -16.; Surface[44].CylinderCenter_uv[1] = -72.4;
    Surface[44].Width = 0.62; Surface[44].Height = 1.3;
    Surface[44].Defined = 0;

    strcpy(Surface[45].Type, "Cylinder"); strcpy(Surface[45].Name, "SID_C3");
    Surface[45].CylinderCenter_uv[0] = -16.; Surface[45].CylinderCenter_uv[1] = -80.4;
    Surface[45].Width = 0.62; Surface[45].Height = 1.3;
    Surface[45].Defined = 0;

    strcpy(Surface[46].Type, "Cylinder"); strcpy(Surface[46].Name, "SID_D0");
    Surface[46].CylinderCenter_uv[0] =   0.; Surface[46].CylinderCenter_uv[1] = -48.4;
    Surface[46].Width = 0.62; Surface[46].Height = 1.3;
    Surface[46].Defined = 0;

    strcpy(Surface[47].Type, "Cylinder"); strcpy(Surface[47].Name, "SID_D1");
    Surface[47].CylinderCenter_uv[0] =   0.; Surface[47].CylinderCenter_uv[1] = -80.4;
    Surface[47].Width = 0.62; Surface[47].Height = 1.3;
    Surface[47].Defined = 0;
    
    strcpy(Surface[48].Type, "Cylinder"); strcpy(Surface[48].Name, "SID_D2");
    Surface[48].CylinderCenter_uv[0] =   0.; Surface[48].CylinderCenter_uv[1] = -96.4;
    Surface[48].Width = 0.62; Surface[48].Height = 1.3;
    Surface[48].Defined = 0;

    strcpy(Surface[49].Type, "Cylinder"); strcpy(Surface[49].Name, "SID_E0");
    Surface[49].CylinderCenter_uv[0] =  16.; Surface[49].CylinderCenter_uv[1] = -48.4;
    Surface[49].Width = 0.62; Surface[49].Height = 1.3;
    Surface[49].Defined = 0;

    strcpy(Surface[50].Type, "Cylinder"); strcpy(Surface[50].Name, "SID_G0");
    Surface[50].CylinderCenter_uv[0] =  32.; Surface[50].CylinderCenter_uv[1] = -48.4;
    Surface[50].Width = 0.62; Surface[50].Height = 1.3;
    Surface[50].Defined = 0;

    strcpy(Surface[51].Type, "Cylinder"); strcpy(Surface[51].Name, "SID_G1");
    Surface[51].CylinderCenter_uv[0] =  32.; Surface[51].CylinderCenter_uv[1] = -112.4;
    Surface[51].Width = 0.62; Surface[51].Height = 1.3;
    Surface[51].Defined = 0;

    strcpy(Surface[52].Type, "Cylinder"); strcpy(Surface[52].Name, "SID_G2");
    Surface[52].CylinderCenter_uv[0] =  32.; Surface[52].CylinderCenter_uv[1] = -24.4;
    Surface[52].Width = 0.62; Surface[52].Height = 1.3;
    Surface[52].Defined = 0;

    strcpy(Surface[53].Type, "Cylinder"); strcpy(Surface[53].Name, "SID_H0");
    Surface[53].CylinderCenter_uv[0] =  64.; Surface[53].CylinderCenter_uv[1] = -48.4;
    Surface[53].Width = 0.62; Surface[53].Height = 1.3;
    Surface[53].Defined = 0;

    strcpy(Surface[54].Type, "Cylinder"); strcpy(Surface[54].Name, "SID_I1");
    Surface[54].CylinderCenter_uv[0] =  72.; Surface[54].CylinderCenter_uv[1] = -88.4;
    Surface[54].Width = 0.62; Surface[54].Height = 1.3;
    Surface[54].Defined = 0;

    strcpy(Surface[55].Type, "Cylinder"); strcpy(Surface[55].Name, "SID_J1");
    Surface[55].CylinderCenter_uv[0] =  88.; Surface[55].CylinderCenter_uv[1] = -72.4;
    Surface[55].Width = 0.62; Surface[55].Height = 1.3;
    Surface[55].Defined = 0;

    strcpy(Surface[56].Type, "Cylinder"); strcpy(Surface[56].Name, "SID_J2");
    Surface[56].CylinderCenter_uv[0] =  88.; Surface[56].CylinderCenter_uv[1] = -96.4;
    Surface[56].Width = 0.62; Surface[56].Height = 1.3;
    Surface[56].Defined = 0;

    strcpy(Surface[57].Type, "Cylinder"); strcpy(Surface[57].Name, "SID_J3");
    Surface[57].CylinderCenter_uv[0] =  88.; Surface[57].CylinderCenter_uv[1] =  7.6;
    Surface[57].Width = 0.62; Surface[57].Height = 1.3;
    Surface[57].Defined = 0;

    strcpy(Surface[58].Type, "Cylinder"); strcpy(Surface[58].Name, "SID_J4");
    Surface[58].CylinderCenter_uv[0] =  88.; Surface[58].CylinderCenter_uv[1] =  23.6;
    Surface[58].Width = 0.62; Surface[58].Height = 1.3;
    Surface[58].Defined = 0;

    strcpy(Surface[59].Type, "Cylinder"); strcpy(Surface[59].Name, "SID_J5");
    Surface[59].CylinderCenter_uv[0] =  88.; Surface[59].CylinderCenter_uv[1] =  47.6;
    Surface[59].Width = 0.62; Surface[59].Height = 1.3;
    Surface[59].Defined = 0;

    strcpy(Surface[60].Type, "Cylinder"); strcpy(Surface[60].Name, "SID_J6");
    Surface[60].CylinderCenter_uv[0] =  88.; Surface[60].CylinderCenter_uv[1] =  71.6;
    Surface[60].Width = 0.62; Surface[60].Height = 1.3;
    Surface[60].Defined = 0;

    strcpy(Surface[61].Type, "Cylinder"); strcpy(Surface[61].Name, "SID_K0");
    Surface[61].CylinderCenter_uv[0] =  96.; Surface[61].CylinderCenter_uv[1] = -48.4;
    Surface[61].Width = 0.62; Surface[61].Height = 1.3;
    Surface[61].Defined = 0;

    strcpy(Surface[62].Type, "Cylinder"); strcpy(Surface[62].Name, "SID_L0");
    Surface[62].CylinderCenter_uv[0] = 104.; Surface[62].CylinderCenter_uv[1] = -48.4;
    Surface[62].Width = 0.62; Surface[62].Height = 1.3;
    Surface[62].Defined = 0;

    strcpy(Surface[63].Type, "Cylinder"); strcpy(Surface[63].Name, "SID_M0");
    Surface[63].CylinderCenter_uv[0] = 112.; Surface[63].CylinderCenter_uv[1] = -48.4;
    Surface[63].Width = 0.62; Surface[63].Height = 1.3;
    Surface[63].Defined = 0;


    /* ================================================================================== */
    /* INTERFEROMETRIC LABORATORY WALLS                                                   */
    /* ================================================================================== */
    strcpy(Surface[64].Type, "Wall"); strcpy(Surface[64].Name, "LabWall_SA");
    SetVector_3D(Surface[64].VertexPoint, 34.7, -36.3, 0.);
    SetVector_3D(Surface[64].Axis3,  0., -1.,  0.);
    SetVector_3D(Surface[64].Axis1,  1.,  0.,  0.);
    Surface[64].Width = 28.5; Surface[64].Height = 3.7;
    Surface[64].Defined = 1;

    strcpy(Surface[65].Type, "Wall"); strcpy(Surface[65].Name, "LabWall_EA");
    SetVector_3D(Surface[65].VertexPoint, 63.2, -36.3, 0.);
    SetVector_3D(Surface[65].Axis3,  1.,  0.,  0.);
    SetVector_3D(Surface[65].Axis1,  0.,  1.,  0.);
    Surface[65].Width = 7.9; Surface[65].Height = 3.7;
    Surface[65].Defined = 1;

    strcpy(Surface[66].Type, "Wall"); strcpy(Surface[66].Name, "LabWall_EB");
    SetVector_3D(Surface[66].VertexPoint, 47.3, -28.4, 0.);
    SetVector_3D(Surface[66].Axis3,  1.,  0.,  0.);
    SetVector_3D(Surface[66].Axis1,  0.,  1.,  0.);
    Surface[66].Width = 14.5; Surface[66].Height = 3.7;
    Surface[66].Defined = 1;

    strcpy(Surface[67].Type, "Wall"); strcpy(Surface[67].Name, "LabWall_EC");
    SetVector_3D(Surface[67].VertexPoint, 59.0, -28.4, 0.);
    SetVector_3D(Surface[67].Axis3,  1.,  0.,  0.);
    SetVector_3D(Surface[67].Axis1,  0.,  1.,  0.);
    Surface[67].Width = 21.0; Surface[67].Height = 2.31;
    Surface[67].Defined = 1;

    strcpy(Surface[68].Type, "Wall"); strcpy(Surface[68].Name, "LabWall_NA");
    SetVector_3D(Surface[68].VertexPoint, 59.0, -7.4, 0.);
    SetVector_3D(Surface[68].Axis3,  0.,  1.,  0.);
    SetVector_3D(Surface[68].Axis1, -1.,  0.,  0.);
    Surface[68].Width = 11.8; Surface[68].Height = 2.31;
    Surface[68].Defined = 1;

    strcpy(Surface[69].Type, "Wall"); strcpy(Surface[69].Name, "LabWall_NB");
    SetVector_3D(Surface[69].VertexPoint, 47.3, -13.8, 0.);
    SetVector_3D(Surface[69].Axis3,  0.,  1.,  0.);
    SetVector_3D(Surface[69].Axis1, -1.,  0.,  0.);
    Surface[69].Width = 12.5; Surface[69].Height = 3.7;
    Surface[69].Defined = 1;

    strcpy(Surface[70].Type, "Wall"); strcpy(Surface[70].Name, "LabWall_NC");
    SetVector_3D(Surface[70].VertexPoint, 63.2, -28.8, 0.);
    SetVector_3D(Surface[70].Axis3,  0.,  1.,  0.);
    SetVector_3D(Surface[70].Axis1, -1.,  0.,  0.);
    Surface[70].Width = 15.9; Surface[70].Height = 3.7;
    Surface[70].Defined = 1;

    strcpy(Surface[71].Type, "Wall"); strcpy(Surface[71].Name, "LabWall_WA");
    SetVector_3D(Surface[71].VertexPoint, 47.0, -7.4, 0.);
    SetVector_3D(Surface[71].Axis3, -1.,  0.,  0.);
    SetVector_3D(Surface[71].Axis1,  0., -1.,  0.);
    Surface[71].Width = 6.4; Surface[71].Height = 2.31;
    Surface[71].Defined = 1;
 
    strcpy(Surface[72].Type, "Wall"); strcpy(Surface[72].Name, "LabWall_WB");
    SetVector_3D(Surface[72].VertexPoint, 34.7, -13.8, 0.);
    SetVector_3D(Surface[72].Axis3, -1.,  0.,  0.);
    SetVector_3D(Surface[72].Axis1,  0., -1.,  0.);
    Surface[72].Width = 22.5; Surface[72].Height = 3.7;
    Surface[72].Defined = 1;
    
    return(SUCCESS);
}

ccsCOMPL_STAT CompleteSurfaceData(SurfaceStruct *Surface, long NumberOfSurfaces,
                         float *u_Axis, float *v_Axis, float *w_Axis, ccsERROR *error)
{
    long SurfaceIndex;
    float HelpVector1[3],
          HelpVector2[3];
    
    for(SurfaceIndex=0; SurfaceIndex<NumberOfSurfaces; SurfaceIndex++)
    {
        if(Surface[SurfaceIndex].Defined==1)
        {
            if(strcmp(Surface[SurfaceIndex].Type, "Cylinder")==0)
            {
                /* Set 3 local axes */
                EquateVectors(w_Axis, 3, Surface[SurfaceIndex].Axis1);
                EquateVectors(v_Axis, 3, Surface[SurfaceIndex].Axis2);
                EquateVectors(u_Axis, 3, Surface[SurfaceIndex].Axis3);
                InvertVector(Surface[SurfaceIndex].Axis3, 3);

                /* Set vertex */
                MultiplyVectorWithScalar (u_Axis,
                                          Surface[SurfaceIndex].Width/2.,
                                          3, HelpVector1);
                HelpVector2[0] = Surface[SurfaceIndex].CylinderCenter_uv[0];
                HelpVector2[1] = Surface[SurfaceIndex].CylinderCenter_uv[1];
                HelpVector2[2] = 0.; /* arbitrary ... */
                VectorAddition(HelpVector1, HelpVector2, 3, Surface[SurfaceIndex].VertexPoint);
            }

            if(strcmp(Surface[SurfaceIndex].Type, "Wall")==0)
            {
                /* For a 'Wall' the Axis2 must be computed from Axis1 and Axis3 */
                
                /* Normalize Axis 1 (w. r. t. local coord. system) */
                NormalizeVector(Surface[SurfaceIndex].Axis1, 3);
    
                /* Normalize Axis 3 (principal axis) (w. r. t. local coord. system) */
                NormalizeVector(Surface[SurfaceIndex].Axis3, 3);
                
                /* Check if Axis 1 is perpendicular to Axis3  */
                if(TestPerpendicular(Surface[SurfaceIndex].Axis1, 
                                     Surface[SurfaceIndex].Axis3, 3, 1.0e-10)!=0)
                /* Axis1 is NOT perpendicular to Axis3 */
                {
		  /*fprintf(stderr, "*** ERROR Message ***\n");
                    fprintf(stderr, "Axis1 of the 'Wall'-type surface %d (Name %s)",
                            SurfaceIndex,
                            Surface[SurfaceIndex].Name);
                    fprintf(stderr, "is NOT perpendicular to the surface normal (Axis3) of this surface.\n");
                    fprintf(stderr, "Axis1          = %e, %e, %e\n", Surface[SurfaceIndex].Axis1[0],
                                                                     Surface[SurfaceIndex].Axis1[1],
                                                                     Surface[SurfaceIndex].Axis1[2]);
                    fprintf(stderr, "Principal Axis = %e, %e, %e\n", Surface[SurfaceIndex].Axis3[0],
                                                                     Surface[SurfaceIndex].Axis3[1],
                                                                     Surface[SurfaceIndex].Axis3[2]);
                    fprintf(stderr, "Please modify your surface file.\n");
                    exit(0);*/   
                    errAdd(error, issshadowMOD, issshadowERR_SURFACE_FILE, __FILE__, "", "");
                    return(FAILURE);
                }
            
                /* calculate Axis 2:
                   --> Axis2 = Axis3 X Axis1 (w. r. t. local coord. system) */
                VectorProduct(Surface[SurfaceIndex].Axis3,
                              Surface[SurfaceIndex].Axis1, 
                              Surface[SurfaceIndex].Axis2);
            }
        }
    }
    return(SUCCESS);
}

void ActivateSIDPeriscopeCylinder(TelescopeStruct *Telescope, SurfaceStruct *Surface,
                                  long NumberOfSurfaces)
/* Goes through the list of surfaces, finds the surface that represents the Siderostat
   selected (and stored in 'Telescope' struct) and sets the obstructing surface to 'Defined'. */
{
    char HelpString[MAXIMUM_STRING_LENGTH];
    long SurfaceIndex;
  
    if(strcmp(Telescope->Type, "SID")==0)
    {
        sprintf(HelpString, Telescope->Type);
        strcat(HelpString, "_");
        strcat(HelpString, Telescope->StationName);

        for(SurfaceIndex=0; SurfaceIndex<NumberOfSurfaces; SurfaceIndex++)
        {
            if(strcmp(Surface[SurfaceIndex].Name, HelpString)==0)
            {
                Surface[SurfaceIndex].Defined=1;
            }
        }     
    }

    return;
}

void DeactivateUTCylinder(TelescopeStruct *Telescope, SurfaceStruct *Surface,
                                  long NumberOfSurfaces)
{
    char HelpString_1[MAXIMUM_STRING_LENGTH],
         HelpString_2[MAXIMUM_STRING_LENGTH];
    long SurfaceIndex;
  
    if(strcmp(Telescope->Type, "UT")==0)
    {
        sprintf(HelpString_1, Telescope->Type);
        sprintf(HelpString_2, "%1d", (int) Telescope->StationNumber);        
        strcat(HelpString_1, HelpString_2);

        for(SurfaceIndex=0; SurfaceIndex<NumberOfSurfaces; SurfaceIndex++)
        {
            if(strcmp(Surface[SurfaceIndex].Name, HelpString_1)==0)
            {
                Surface[SurfaceIndex].Defined=0;
            }
        }        
    }

    return;
}

ccsCOMPL_STAT SetStationCharacteristics(int location, TelescopeStruct *Telescope, ccsERROR *error)
{
    if(strcmp(Telescope->Type, "UT")==0)
    {
        /* ===================================================================================== */
        /* UNIT TELESCOPES (UT1 ... UT4)                                                         */
        /* ===================================================================================== */
        if(location==31) { strcpy(Telescope->StationName, "U1");
                           Telescope->StationNumber = 1;
                           Telescope->StationPosition_uv[0] = -16.;
                           Telescope->StationPosition_uv[1] = -16.;
                           return(SUCCESS); }
        if(location==32) { strcpy(Telescope->StationName, "U2");
                           Telescope->StationNumber = 2;
                           Telescope->StationPosition_uv[0] = 24.;
                           Telescope->StationPosition_uv[1] = 24.;
                           return(SUCCESS); }
        if(location==33) { strcpy(Telescope->StationName, "U3");
                           Telescope->StationNumber = 3;
                           Telescope->StationPosition_uv[0] = 64.;
                           Telescope->StationPosition_uv[1] = 48.;
                           return(SUCCESS); }
        if(location==34) { strcpy(Telescope->StationName, "U4");
                           Telescope->StationNumber = 4;
                           Telescope->StationPosition_uv[0] = 112.;
                           Telescope->StationPosition_uv[1] =   8.;
                           return(SUCCESS); }


        /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "For a UT ('teltype'=1), the input parameter 'location'\n");
        fprintf(stderr, "must be an integer value between 31 ('U1') and 34 ('U4').\n");
        fprintf(stderr, "You supplied the following 'location' value: %d\n", location);
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_UT_LOCATION, __FILE__, "", "");
        return(FAILURE);
   }

    if((strcmp(Telescope->Type, "AT")==0)||
       (strcmp(Telescope->Type, "SID")==0))
    {
        /* ===================================================================================== */
        /* AUXILIARY TELESCOPES (AT A0 ... AT M0) / SIDEROSTATS                                  */
        /* ===================================================================================== */

        /* A Series (@ u = -32 m) .............................................................. */
        if(location==1) {strcpy(Telescope->StationName, "A0");
                         Telescope->StationNumber = 5;
                         Telescope->StationPosition_uv[0] = -32.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }
        if(location==2) {strcpy(Telescope->StationName, "A1"); 
                         Telescope->StationNumber = 6;
                         Telescope->StationPosition_uv[0] = -32.;
                         Telescope->StationPosition_uv[1] = -64.;
                         return(SUCCESS); }

        /* B Series (@ u = -24 m) .............................................................. */
        if(location==3) {strcpy(Telescope->StationName, "B0"); 
                         Telescope->StationNumber = 7;
                         Telescope->StationPosition_uv[0] = -24.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }
        if(location==4) {strcpy(Telescope->StationName, "B1"); 
                         Telescope->StationNumber = 8;
                         Telescope->StationPosition_uv[0] = -24.;
                         Telescope->StationPosition_uv[1] = -64.;
                         return(SUCCESS); }
        if(location==5) {strcpy(Telescope->StationName, "B2"); 
                         Telescope->StationNumber = 9;
                         Telescope->StationPosition_uv[0] = -24.;
                         Telescope->StationPosition_uv[1] = -72.;
                         return(SUCCESS); }
        if(location==6) {strcpy(Telescope->StationName, "B3"); 
                         Telescope->StationNumber = 10;
                         Telescope->StationPosition_uv[0] = -24.;
                         Telescope->StationPosition_uv[1] = -80.;
                         return(SUCCESS); }
        if(location==7) {strcpy(Telescope->StationName, "B4"); 
                         Telescope->StationNumber = 11;
                         Telescope->StationPosition_uv[0] = -24.;
                         Telescope->StationPosition_uv[1] = -88.;
                         return(SUCCESS); }
        if(location==8) {strcpy(Telescope->StationName, "B5"); 
                         Telescope->StationNumber = 12;
                         Telescope->StationPosition_uv[0] = -24.;
                         Telescope->StationPosition_uv[1] = -96.;
                         return(SUCCESS); }
        

        /* C Series (@ u = -16 m) .............................................................. */
        if(location==9) {strcpy(Telescope->StationName, "C0"); 
                         Telescope->StationNumber = 13;
                         Telescope->StationPosition_uv[0] = -16.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }      
        if(location==10){strcpy(Telescope->StationName, "C1"); 
                         Telescope->StationNumber = 14;
                         Telescope->StationPosition_uv[0] = -16.;
                         Telescope->StationPosition_uv[1] = -64.;
                         return(SUCCESS); }  
        if(location==11){strcpy(Telescope->StationName, "C2"); 
                         Telescope->StationNumber = 15;
                         Telescope->StationPosition_uv[0] = -16.;
                         Telescope->StationPosition_uv[1] = -72.;
                         return(SUCCESS); } 
        if(location==12){strcpy(Telescope->StationName, "C3"); 
                         Telescope->StationNumber = 16;
                         Telescope->StationPosition_uv[0] = -16.;
                         Telescope->StationPosition_uv[1] = -80.;
                         return(SUCCESS); } 

        /* D Series (@ u = 0 m) ................................................................ */
        if(location==13){strcpy(Telescope->StationName, "D0"); 
                         Telescope->StationNumber = 17;
                         Telescope->StationPosition_uv[0] =   0.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }
        if(location==14){strcpy(Telescope->StationName, "D1"); 
                         Telescope->StationNumber = 18;
                         Telescope->StationPosition_uv[0] =   0.;
                         Telescope->StationPosition_uv[1] = -80.;
                         return(SUCCESS); }
        if(location==15){strcpy(Telescope->StationName, "D2"); 
                         Telescope->StationNumber = 19;
                         Telescope->StationPosition_uv[0] =   0.;
                         Telescope->StationPosition_uv[1] = -96.;
                         return(SUCCESS); }


        /* E Series (@ u = 16 m) ............................................................... */
        if(location==16){strcpy(Telescope->StationName, "E0"); 
                         Telescope->StationNumber = 20;
                         Telescope->StationPosition_uv[0] =  16.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }
        

        /* G Series (@ u = 32 m) ............................................................... */
        if(location==17){strcpy(Telescope->StationName, "G0"); 
                         Telescope->StationNumber = 21;
                         Telescope->StationPosition_uv[0] =  32.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }
        if(location==18){strcpy(Telescope->StationName, "G1"); 
                         Telescope->StationNumber = 22;
                         Telescope->StationPosition_uv[0] =  32.;
                         Telescope->StationPosition_uv[1] =-112.;
                         return(SUCCESS); }
        if(location==19){strcpy(Telescope->StationName, "G2"); 
                         Telescope->StationNumber = 23;
                         Telescope->StationPosition_uv[0] =  32.;
                         Telescope->StationPosition_uv[1] = -24.;
                         return(SUCCESS); }


        /* H Series (@ u = 64 m) ............................................................... */
        if(location==20){strcpy(Telescope->StationName, "H0"); 
                         Telescope->StationNumber = 24;
                         Telescope->StationPosition_uv[0] =  64.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }
        

        /* I Series (@ u = 72 m) ............................................................... */
        if(location==21){strcpy(Telescope->StationName, "I1"); 
                         Telescope->StationNumber = 25;
                         Telescope->StationPosition_uv[0] =  72.;
                         Telescope->StationPosition_uv[1] = -88.;
                         return(SUCCESS); }


        /* J Series (@ u = 88 m) ............................................................... */
        if(location==22){strcpy(Telescope->StationName, "J1"); 
                         Telescope->StationNumber = 26;
                         Telescope->StationPosition_uv[0] =  88.;
                         Telescope->StationPosition_uv[1] = -72.;
                         return(SUCCESS); }
        if(location==23){strcpy(Telescope->StationName, "J2"); 
                         Telescope->StationNumber = 27;
                         Telescope->StationPosition_uv[0] =  88.;
                         Telescope->StationPosition_uv[1] = -96.;
                         return(SUCCESS); }
        if(location==24){strcpy(Telescope->StationName, "J3"); 
                         Telescope->StationNumber = 28;
                         Telescope->StationPosition_uv[0] =  88.;
                         Telescope->StationPosition_uv[1] =   8.;
                         return(SUCCESS); }
        if(location==25){strcpy(Telescope->StationName, "J4"); 
                         Telescope->StationNumber = 29;
                         Telescope->StationPosition_uv[0] =  88.;
                         Telescope->StationPosition_uv[1] =  24.;
                         return(SUCCESS); }
        if(location==26){strcpy(Telescope->StationName, "J5"); 
                         Telescope->StationNumber = 30;
                         Telescope->StationPosition_uv[0] =  88.;
                         Telescope->StationPosition_uv[1] =  48.;
                         return(SUCCESS); }
        if(location==27){strcpy(Telescope->StationName, "J6"); 
                         Telescope->StationNumber = 31;
                         Telescope->StationPosition_uv[0] =  88.;
                         Telescope->StationPosition_uv[1] =  72.;
                         return(SUCCESS); }


        /* K Series (@ u = 96 m) ............................................................... */
        if(location==28){strcpy(Telescope->StationName, "K0"); 
                         Telescope->StationNumber = 32;
                         Telescope->StationPosition_uv[0] =  96.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }


        /* L Series (@ u = 104 m) ............................................................. */
        if(location==29){strcpy(Telescope->StationName, "L0"); 
                         Telescope->StationNumber = 33;
                         Telescope->StationPosition_uv[0] = 104.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }


        /* M Series (@ u = 112 m) ............................................................. */
        if(location==30){strcpy(Telescope->StationName, "M0"); 
                         Telescope->StationNumber = 34;
                         Telescope->StationPosition_uv[0] = 112.;
                         Telescope->StationPosition_uv[1] = -48.;
                         return(SUCCESS); }


        /*fprintf(stderr, "*** ERROR Message ***\n");
        fprintf(stderr, "Invalid value for input parameter 'location' for %s-type telescope.\n",
                Telescope->Type);
        fprintf(stderr, "You supplied the following value for 'location': %d\n", location);
        exit(0);*/
        errAdd(error, issshadowMOD, issshadowERR_ILLEGAL_LOCATION, __FILE__, "", "");
        return(FAILURE);
    }
    return(SUCCESS);
}
