/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: issshadowRayTrace.c,v 1.8 2001/09/13 09:12:40 vltsccm Exp $"
*
* who       when        what
* --------  ----------  ----------------------------------------------
* nhousen   2001-09-05  reduced number of included header files
* nhousen   2001-07-12  adapted error handling to VLT standards
* rwilhelm  2001-07-10  created 
*/

/************************************************************************
*   NAME
*   issshadowRayTrace.c
*
*   SYNOPSIS
*
*
*   DESCRIPTION
*   Contains ray tracing routines used within the 'issshadowAltAz' routine.
*
*   PARENT CLASS
*
*
*   DESCRIPTION
*   
*
*   PUBLIC METHODS
*
*   PUBLIC DATA MEMBERS
*
*
*   PROTECTED METHODS
*
*
*   PROTECTED DATA MEMBERS
*
*
*   PRIVATE METHODS
*
*
*   PRIVATE DATA MEMBERS
*
*
*   FILES
*
*   ENVIRONMENT
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS
*
*------------------------------------------------------------------------
*/

/* #define _POSIX_SOURCE 1 */
/* #include "vltPort.h" */
/*
static char *rcsId="@(#) $Id: issshadowRayTrace.c,v 1.8 2001/09/13 09:12:40 vltsccm Exp $";
static void *use_rcsId = ((void)&use_rcsId,(void *) &rcsId);
*/
/* Standard ANSI C Includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "issshadowDefines.h"
#include "issshadowErrors.h"

void InitializeRayGrid(TelescopeStruct *Telescope, BeamStruct *Beam)
{
    /* Compute number of rays */
    Beam->NumberOfRays = (Beam->RadialIncrementation)*(Beam->AngularIncrementation) + 1;

    /* Allocate memory for rays */
    Beam->Ray = AllocateMemoryForRays(Beam->NumberOfRays);

    /* Compute radial and angular spacing */
    if(strcmp(Telescope->Type, "UT")==0)
    {
        Beam->Delta_R = RADIUS_M1_UT/((float)Beam->RadialIncrementation);
    }
    if(strcmp(Telescope->Type, "AT")==0)
    {
        Beam->Delta_R = RADIUS_M1_AT/((float)Beam->RadialIncrementation);
    }
    if(strcmp(Telescope->Type, "SID")==0)
    {
        Beam->Delta_R = RADIUS_M1_SID/((float)Beam->RadialIncrementation);
    }
    Beam->Delta_Phi = TWO_PI/((float)Beam->AngularIncrementation);

    return;
}



void GenerateRayGrid(float *u_Axis, float *v_Axis, float *w_Axis,
                     TelescopeStruct *Telescope, BeamStruct *Beam,
                     float *East_Axis, float *North_Axis, float *Zenith_Axis,
                     float Altitude_Deg, float Azimuth_Deg)
{
    float Direction[3];
    float Axis1[3],
           Axis2[3];
    long RayIndex;
    float PrimaryVertex[3];
    float PositionLocal[3];
    float r, phi;
    long i_radial, i_angular; /* loop counters */
    
    /* Get direction of rays */
    GetRayDirection(East_Axis, North_Axis, Zenith_Axis,
                    Altitude_Deg, Azimuth_Deg, Direction);

    /* Setup ray grid Cartesian coordinate system (perpendicular to direction) */
    SetUpTemporaryCoordinateSystem(Direction, Axis1, Axis2);

    /* Compute the vertex of the primary (=origin of local coordinate system) */
    ComputePrimaryVertex(u_Axis, v_Axis, w_Axis, Telescope, Altitude_Deg, Azimuth_Deg,
                         PrimaryVertex);

    /* Generate the rays */
    /* ----------------- */
    RayIndex=0;
    /* Define the central ray */
    EquateVectors(PrimaryVertex, 3, Beam->Ray[RayIndex].Position);
    EquateVectors(Direction, 3, Beam->Ray[RayIndex].Direction);
    Beam->Ray[RayIndex].Active = 1;

    /* Define the rays located around the central ray */ 
    for(i_radial=1; i_radial<=Beam->RadialIncrementation; i_radial++)
    {
        for(i_angular=0; i_angular<Beam->AngularIncrementation; i_angular++)
        {
            /* Radius */
            r   = (float)i_radial * Beam->Delta_R;
            phi = (float)i_angular * Beam->Delta_Phi;
            PositionLocal[0] = r * cos(phi);
            PositionLocal[1] = r * sin(phi);
            PositionLocal[2] = 0; 
            
            RayIndex++;
            TransformPoint_Local_a1a2a3_To_Global_xyz(Axis1, Axis2, Direction,
                                                      PrimaryVertex, PositionLocal,
                                                      Beam->Ray[RayIndex].Position);

            /* All rays get the same direction ... */
            EquateVectors(Direction, 3, Beam->Ray[RayIndex].Direction);

            /* All rays get status flag set to '1' ==> 'Active' */
            Beam->Ray[RayIndex].Active = 1;
        }
    }    
    return;
}

void ComputePrimaryVertex(float *u_Axis, float *v_Axis, float *w_Axis,
                          TelescopeStruct *Telescope, float Altitude_Deg, float Azimuth_Deg,
                          float *Vertex)
{
    float RotationCenter[3];
    float Beta;
    
    /* 3 CASES ARE DISTINGUISHED: UT, AT and SID */
    /* ----------------------------------------- */

    /* ::::::::::::::::::::::::::::::::::::::::: */
    /* CASE 1: UT                                */
    /* ::::::::::::::::::::::::::::::::::::::::: */
    if(strcmp(Telescope->Type, "UT")==0)
    {
        /* Set the primary vertex for altitude 90 deg (Zenithwards) */
        Vertex[0] = Telescope->StationPosition_uv[0];
        Vertex[1] = Telescope->StationPosition_uv[1];
        Vertex[2] = 1.054400000000000e+01; /* M1 vertex w-coordinate for altitude = 90 deg */

        /* Set rotation center (=M3 vertex) */
        RotationCenter[0] = Telescope->StationPosition_uv[0];
        RotationCenter[1] = Telescope->StationPosition_uv[1];
        RotationCenter[2] = 1.304400000000000e+01;

        /* ALTITUDE ROTATION */
        
        /* To achieve a pointing to the given altitude angle,
           the mirror M1 has to be rotated around the POSITIVE
           u-axis by an angle of BETA = 90 deg - AltitudeAngle
           The rotation center is the vertex of M3. */
        Beta = (90. - Altitude_Deg) * DEG_2_RAD; /* in RADIANS !! */
        RotatePoint(Vertex, u_Axis, RotationCenter, Beta);

        /* AZIMUTH ROTATION */

        /* To achieve a pointing to the specified azimuth angle,
           the mirror M1 has to be rotated around the positive w-axis.
           The rotation center is given by the station center (u,v). (The third (w-)coordinate
           can be chosen freely since the rotation axis is the w-axis.
           The rotation angle is given by "Beta = VLTI_Configuration.Azimuth_Degree - 19.55 deg". */
        
        Beta = (Azimuth_Deg - SITE_ANGLE) * DEG_2_RAD; /* in RADIANS !! */
        RotatePoint(Vertex, w_Axis, RotationCenter, Beta);
        
        return; 
    }

    /* ::::::::::::::::::::::::::::::::::::::::: */
    /* CASE 2: AT                                */
    /* ::::::::::::::::::::::::::::::::::::::::: */
    if(strcmp(Telescope->Type, "AT")==0)
    {
        /* Set the primary vertex for altitude 90 deg (Zenithwards) */
        Vertex[0] = Telescope->StationPosition_uv[0];
        Vertex[1] = Telescope->StationPosition_uv[1];
        Vertex[2] = 3.839899775000001e+00; /* M1 vertex w-coordinate for altitude = 90 deg */

        /* Set rotation center (=M3 vertex) */
        RotationCenter[0] = Telescope->StationPosition_uv[0];
        RotationCenter[1] = Telescope->StationPosition_uv[1];
        RotationCenter[2] = 4.539899775000000e+00;

        /* ALTITUDE ROTATION */
        
        /* To achieve a pointing to the given altitude angle,
           the mirror M1 has to be rotated around the POSITIVE
           u-axis by an angle of BETA = 90 deg - AltitudeAngle
           The rotation center is the vertex of M3. */
        Beta = (90. - Altitude_Deg) * DEG_2_RAD; /* in RADIANS !! */
        RotatePoint(Vertex, u_Axis, RotationCenter, Beta);

        /* AZIMUTH ROTATION */

        /* To achieve a pointing to the specified azimuth angle,
           the mirror M1 has to be rotated around the positive w-axis.
           The rotation center is given by the station center (u,v). (The third (w-)coordinate
           can be chosen freely since the rotation axis is the w-axis.
           The rotation angle is given by "Beta = VLTI_Configuration.Azimuth_Degree - 19.55 deg". */
        
        Beta = (Azimuth_Deg - SITE_ANGLE) * DEG_2_RAD; /* in RADIANS !! */
        RotatePoint(Vertex, w_Axis, RotationCenter, Beta);
        
        return; 
    }

    /* ::::::::::::::::::::::::::::::::::::::::: */
    /* CASE 3: SID                               */
    /* ::::::::::::::::::::::::::::::::::::::::: */
    if(strcmp(Telescope->Type, "SID")==0)
    {
        Vertex[0] = Telescope->StationPosition_uv[0] + 0.7*sin(SITE_ANGLE*DEG_2_RAD);
        Vertex[1] = Telescope->StationPosition_uv[1] + 0.7*cos(SITE_ANGLE*DEG_2_RAD) - 0.4;
        Vertex[2] = 8.2500000000000000e-01;

        return;
    }
    return;
}

void GetRayDirection(float *East_Axis, float *North_Axis, float *Zenith_Axis,
                     float Altitude_Deg, float Azimuth_Deg, float *Direction)
{
    float Altitude_Rad;
    float Azimuth_Rad;
    float Cos_Altitude;
    float Direction_ENZ[3];
    
    /* this trig. value is needed twice, therefore it is computed before ... */

    /* Compute altitude and azimuth angles in units of RADIAN */
    Altitude_Rad = DEG_2_RAD * Altitude_Deg;
    Azimuth_Rad  = DEG_2_RAD * Azimuth_Deg;

    /* Step 1: Compute direction vector (3 x 1) in the {EAST, NORTH, ZENITH}-System */
    Cos_Altitude = cos(Altitude_Rad);

    Direction_ENZ[0] =  sin(Azimuth_Rad) * Cos_Altitude;
    Direction_ENZ[1] = -cos(Azimuth_Rad) * Cos_Altitude;
    Direction_ENZ[2] =  sin(Altitude_Rad);

    /* Step 2: transform direction into the VLTI site coordinate system */
    TransformVector_Local_a1a2a3_To_Global_xyz(East_Axis, North_Axis, Zenith_Axis,
                                               Direction_ENZ, Direction);

    return;
}


void GetEastNorthZenithAxes(float *East_Axis, float *North_Axis, float *Zenith_Axis)
{
    float Angle;
    /* the 19.55 deg angle that defines the orientation of the site coordinate system {u, v, w}
       w. r. t. the {EAST, NORTH, ZENITH}-system. The VLTI site coordinate system {u, y, w}
       results from the {EAST, NORTH, ZENITH}-system by rotation around ZENITH-axis (=w-axis)
       by an angle of +19.55 deg. */
    /* IMPORTANT:
       According to BKO's internal memo (VIF-01/050) from April 6, 2001, the angle is not
       19.55 deg but 18.984 deg +- 0.003 deg) */

    Angle = SITE_ANGLE * DEG_2_RAD;
    
    East_Axis[0]   = cos(Angle); East_Axis[1]   = -sin(Angle); East_Axis[2]   = 0.;
    North_Axis[0]  = sin(Angle); North_Axis[1]  = cos(Angle);  North_Axis[2]  = 0.;
    Zenith_Axis[0] = 0.;         Zenith_Axis[1] = 0.;          Zenith_Axis[2] = 1.;
    
    return;
}


ccsCOMPL_STAT PerformRayTracing(BeamStruct *Beam, SurfaceStruct *Surface,
                       long NumberOfSurfaces, ccsERROR *error)
/* This function computes the number of active (i.e. NOT obstructed) rays
   after ray tracing over all (defined) surfaces */ 
{
    /* LOCAL VARIABLES */
    /* --------------- */
    long RayIndex;
    /* Loop index for the ray tracing loop */
    long SurfaceIndex;
    /* Loop index for the surface loop */
    float PathLength[2];
    /* vector of optical path length solution(s); maximum size = 2 x 1 ==> 2 solutions */
    long NumberOfSolutions;
    /* number of valid solutions for the optical path length when computing the intersection
       of a ray with an obscuring surface (with infinite-size aperture) */


    /* Initialize number of active rays */
    Beam->NumberOfActiveRays = Beam->NumberOfRays;
            
    /* LOOP; ALL RAYS */
    /* -------------- */
    for(RayIndex=0; RayIndex<Beam->NumberOfRays; RayIndex++)
    {
        SurfaceIndex = 0;
        /* LOOP; ALL DEFINED SURFACES */
        /* :::::::::::::::::::::::::: */
        while((SurfaceIndex<NumberOfSurfaces)&&(Beam->Ray[RayIndex].Active==1))
        {
            if(Surface[SurfaceIndex].Defined==1)
            {
                if ( TraceRay(Beam, RayIndex, Surface, SurfaceIndex,
			      PathLength, &NumberOfSolutions, error) == FAILURE )
		  {
		    return(FAILURE);
		  }
                        
                /* Check the intersections (only if there are some ...) */
                if(NumberOfSolutions>0)
                {
                    if ( CheckIfRayObscured(Beam, RayIndex,
                                Surface, SurfaceIndex,
				PathLength, NumberOfSolutions, error) == FAILURE )
		      {
			return(FAILURE);
		      }
                }
            }
            if(Beam->Ray[RayIndex].Active==1)
            {
                /* ray has not hit surface: --> check next surface */
                SurfaceIndex++;
            }
            else
            {
                /* ray has been obscured */
                Beam->NumberOfActiveRays--;
            }
        }
        /* END LOOP; ALL DEFINED SURFACES */
        /* :::::::::::::::::::::::::::::: */
    }
    /* END LOOP; ALL RAYS */
    /* ------------------ */
    return(SUCCESS);
}



ccsCOMPL_STAT TraceRay(BeamStruct *Beam, long RayIndex, SurfaceStruct *Surface, long SurfaceIndex,
              float *PathLength, long *NumberOfSolutions, ccsERROR *error)
{
    /* CASE 1: SURFACE TYPE = PLANE */
    /* ---------------------------- */
    if(strcmp(Surface[SurfaceIndex].Type, "Wall")==0)
    {
        FindPathLength_FlatSurface(Beam, RayIndex, Surface, SurfaceIndex,
                                   NumberOfSolutions, PathLength);
        return(SUCCESS);
    }

    /* CASE 2: SURFACE TYPE = CYLINDER */
    /* ------------------------------- */
    if(strcmp(Surface[SurfaceIndex].Type, "Cylinder")==0)
    {
        FindPathLength_CylindricalSurface(Beam, RayIndex, Surface, SurfaceIndex,
                                          NumberOfSolutions, PathLength);
        return(SUCCESS);
    }

    /* The program should NEVER reach this point */
    /*fprintf(stderr, "*** ERROR Message ***\n");
    fprintf(stderr, "Bad internal error in issshadow routine 'TraceRay'.\n");
    fprintf(stderr, "Invalid surface type = %s\n", Surface[SurfaceIndex].Type);
    exit(0);*/
    errAdd(error, issshadowMOD, issshadowERR_TRACERAY, __FILE__, "", "");
    return(FAILURE);
}


void FindPathLength_FlatSurface(BeamStruct *Beam, long RayIndex, SurfaceStruct *Surface, long SurfaceIndex,
                                long *NumberOfSolutions, float *PathLength)
/* calculates the solution for the ray path length for the case of a flat surface; The local variable
   'RayPosition_wrt_Vertex' defined below is the ray position with respect to the surface vertex;
   The function returns the vector PathLength which holds the solution(s) and the integer variable
   'NumberOfSolutions' which gives the size of this vector (-> the number of solution(s)).
   In the case of a flat surface, the maximum value for 'NumberOfSolutions' is 1. */
{
    float Enumerator, Denominator;
    /* Help variables; needed to calculate the path length for the flat surface case */
    float RayPosition_wrt_Vertex[3];

    /* Compute ray position w.r.t. surface vertex */
    VectorSubtraction(Beam->Ray[RayIndex].Position, Surface[SurfaceIndex].VertexPoint, 3, RayPosition_wrt_Vertex);

    /* Compute enumerator and denominator of the quotient used to compute the path length */
    ScalarProduct(Surface[SurfaceIndex].Axis3, RayPosition_wrt_Vertex, 3, &Enumerator);
    ScalarProduct(Surface[SurfaceIndex].Axis3, Beam->Ray[RayIndex].Direction, 3, &Denominator);
    
    if(TestEquality(Denominator, 0., 1.e-10)==1)
    {
        /* Denominator is not (very close to) zero */
        PathLength[0]     = -Enumerator/Denominator;
        if(PathLength[0]>=0.)
        {
            *NumberOfSolutions = 1;
            return;
        }
        else
        {
            *NumberOfSolutions = 0;
            return;
        }
    }
    else
    {
        /* Denominator is equal to zero (within 1.e-10) --> Ray Direction is parallel to surface */
        *NumberOfSolutions = 0;
        return;
    }
    return;
}

void FindPathLength_CylindricalSurface(BeamStruct *Beam, long RayIndex, SurfaceStruct *Surface, long SurfaceIndex,
                                       long *NumberOfSolutions, float *PathLength)
{
    float a, b, c;
    float RayPosition_wrt_Vertex[3];

    /* Compute ray position w.r.t. vertex of cylindrical surface */
    VectorSubtraction(Beam->Ray[RayIndex].Position, Surface[SurfaceIndex].VertexPoint, 3,
                      RayPosition_wrt_Vertex);

    /* Compute quadratic equation coefficients a, b and c */
    Calculate_abc_CylindricalSurface(Surface, SurfaceIndex, Beam->Ray[RayIndex].Direction, RayPosition_wrt_Vertex,
                                     &a, &b, &c);

    /* now the solution(s) for the PathLength are known! (quadratic equation formula) */
    /* 'Classical Solution Formula' (WILL LEAD TO PROBLEMS) : (1/(2a)) * (-b +- Sqrt[ b^2 -4ac]) */
    QuadraticEquationSolutionSelect(a, b, c, NumberOfSolutions, PathLength);

    return;
}



void Calculate_abc_CylindricalSurface(SurfaceStruct *Surface, long SurfaceIndex, float *RayDirection,
                                      float *RayPosition_wrt_Vertex, float *a, float *b, float *c)
/* HelpRayDirection is the direction of the incident ray; HelpRayPosition is the
   start position of the incident ray in global coords BUT WITH RESPECT TO THE 
   SURFACE VERTEX !! */
{
    /* some help variables */
    float AlphaX, BetaX, AlphaY, BetaY;
    float Etha;
    float f;
    
    /* e = 0;  eccentricity zero (spherical cylinder) */
    f = Surface[SurfaceIndex].Width/2.; /* geometrical focal distance = radius of spherical cylinder */
    
    /* Axis3 is the surface normal at the vertex; Axis2 is an axis perp. to Axis3 and Axis1 while
       Axis1 is the cylinder axis */

    /* ORIGINAL CODE: OK BUT SLOWER */
    /*
    ScalarProduct(G->Element[ElementIndex].Axis2[SurfaceIndex], HelpRayPosition,  3, &AlphaX);
    ScalarProduct(G->Element[ElementIndex].Axis2[SurfaceIndex], HelpRayDirection, 3, &BetaX);
    
    ScalarProduct(G->Element[ElementIndex].Axis3[SurfaceIndex], HelpRayPosition,  3, &AlphaY);
    ScalarProduct(G->Element[ElementIndex].Axis3[SurfaceIndex], HelpRayDirection, 3, &BetaY);
    */

    /* NEW VERSION: Makes use of the fact that Axis2 = v-axis = (0,1,0) and Axis3 = -u-axis = (-1,0,0) */
    AlphaX = RayPosition_wrt_Vertex[1];
    BetaX  = RayDirection[1];

    AlphaY = -RayPosition_wrt_Vertex[0];
    BetaY  = -RayDirection[0];

    
    /* Pre-compute some quantities to speed up code */
    /* -------------------------------------------- */
  
    /* Chi  = (1.-e*e) = 1 for a spherical cylinder; */
    Etha = Surface[SurfaceIndex].Width; /* Etha = 2*f*(1+e) = 2f = r for a spherical cylinder */
    
    /* "a" */
    *a = BetaY*BetaY + BetaX*BetaX;
    
    /* "b" */
    *b = 2.*AlphaY*BetaY + 2.*AlphaX*BetaX -Etha*BetaY;
    
    /* "c" */
    *c = AlphaY*AlphaY + AlphaX*AlphaX - Etha*AlphaY;
    
    return;
}

ccsCOMPL_STAT CheckIfRayObscured(BeamStruct *Beam, long RayIndex,
                        SurfaceStruct *Surface, long SurfaceIndex,
                        float *PathLength, long NumberOfSolutions,
			ccsERROR *error)
{
    /* 'NumberOfSolutions' is either 1 or 2 */

    long SolutionIndex;
    float RayPosition[3];
    float HelpVector[3];
    long HitFlag;
    
    SolutionIndex = 0;
    
    while((SolutionIndex<NumberOfSolutions)&&(Beam->Ray[RayIndex].Active==1))
    {
        /* Compute the new ray position on the (mathematical) surface */
        MultiplyVectorWithScalar (Beam->Ray[RayIndex].Direction, PathLength[SolutionIndex], 3, HelpVector);
        VectorAddition(Beam->Ray[RayIndex].Position, HelpVector, 3, RayPosition);

        /* Check if ray hits the surface */
        if ( CheckSurfaceHit(RayPosition, Surface, SurfaceIndex, &HitFlag, error) == FAILURE)
	  {
	    return(FAILURE);
	  }

        if(HitFlag==1)
        {
            /* The ray is obscured */
            Beam->Ray[RayIndex].Active = 0;
        }
        else
        {
            /* Go to the next solution */
            SolutionIndex++;
        }
    }    
    return(SUCCESS);
}

ccsCOMPL_STAT CheckSurfaceHit(float *RayPosition, SurfaceStruct *Surface, 
			       long SurfaceIndex, long *HitFlag, ccsERROR *error)
{
    float RayPosition_LOCAL_2D[2];

    
    /* CASE 1: SURFACE TYPE = PLANE */
    /* ---------------------------- */
    if(strcmp(Surface[SurfaceIndex].Type, "Wall")==0)
    {
        TransformPoint_Global_xyz_To_Local_a1a2(Surface[SurfaceIndex].Axis1,
                                                Surface[SurfaceIndex].Axis2,
                                                Surface[SurfaceIndex].VertexPoint,
                                                RayPosition,
                                                RayPosition_LOCAL_2D);

        /* The 2 x 1 vector 'RayPosition_LOCAL_2D' now contains the {Axis1,Axis2}-
           coordinates of the ray position in the local coordinate system w.r.t.
           the vertex. Check if these coordinates lie in the valid range for a 'Wall'-
           type surface */
        if((RayPosition_LOCAL_2D[0]>=0)&&(RayPosition_LOCAL_2D[0]<=Surface[SurfaceIndex].Width)&&
           (RayPosition_LOCAL_2D[1]>=0)&&(RayPosition_LOCAL_2D[1]<=Surface[SurfaceIndex].Height))
        {
            *HitFlag = 1;
        }
        else
        {
            *HitFlag = 0;
        }
        return(SUCCESS);
    }

    /* CASE 2: SURFACE TYPE = CYLINDER */
    /* ------------------------------- */
    if(strcmp(Surface[SurfaceIndex].Type, "Cylinder")==0)
    {
        /* In case of a cylinder it has only to be checked if the w-coordinate
           is below 'Surface[SurfaceIndex].Height' */
        if(RayPosition[2]<=Surface[SurfaceIndex].Height)
        {
            *HitFlag = 1; /* the ray hits the cylinder in the given height range */
        }
        else
        {
            *HitFlag = 0; /* surface is hit above the height limit --> no real hit */
        }
        return(SUCCESS);
    }
    /* The program should NEVER reach this point */
    /*fprintf(stderr, "*** ERROR Message ***\n");
    fprintf(stderr, "Bad internal error in issshadow routine 'CheckSurfaceHit'.\n");
    fprintf(stderr, "Invalid surface type = %s\n", Surface[SurfaceIndex].Type);
    exit(0);*/
    errAdd(error, issshadowMOD, issshadowERR_CHECKSURFACEHIT, __FILE__, "", "");
    return(FAILURE);
}
