function lorentzian, xi, parms, DOUBLE=double
;+
; NAME:
;       GAUSSIAN
; PURPOSE:
;       Compute the 1-d Gaussian function and optionally the derivative
; EXPLANATION:
;       Compute the 1-D Gaussian function and optionally the derivative 
;       at an array of points.
;
; CALLING SEQUENCE:
;       y = lorentzian( xi, parms)
;
; INPUTS:
;       xi = array, independent variable of Gaussian function.
;
;       parms = parameters of Gaussian, 2, 3 or 4 element array:
;               parms[0] = maximum value (factor) of Gaussian,
;               parms[1] = mean value (center) of Gaussian,
;               parms[2] = standard deviation (sigma) of Gaussian.
;               (if parms has only 2 elements then sigma taken from previous
;               call to lorentzian(), which is stored in a  common block).
;               parms[3] = optional, constant offset added to Gaussian.
; OUTPUT:
;       y -  Function returns array of Gaussian evaluated at xi.    Values will
;            be floating pt. (even if xi is double) unless the /DOUBLE keyword
;            is set.
;
; OPTIONAL INPUT:
;       /DOUBLE - set this keyword to return double precision for both
;             the function values and (optionally) the partial derivatives.
;
; EXAMPLE:
;       Evaulate a Gaussian centered at x=0, with sigma=1, and a peak value
;       of 10 at the points 0.5 and 1.5.
;
;       IDL> f = lorentzian( [0.5,1.5], [10,0,1] )
;       ==> f= [8.825,3.25]. 
; 
; COMMON BLOCKS:
;       None
; HISTORY:
;       Written, Frank Varosi NASA/GSFC 1992.
;       Converted to IDL V5.0   W. Landsman   September 1997
;       Use machar() for machine precision, added /DOUBLE keyword,
;       add optional constant 4th parameter    W. Landsman   November 2001
;-
  On_error,2
  common lorentzian, sigma

  if N_params() LT 2 then begin
        print,'Syntax - y = GAUSSIAN( xi, parms,[ /DOUBLE ])'
        print,'         parms[0] = maximum value (factor) of Gaussian'
        print,'         parms[1] = mean value (center) of Gaussian'
        print,'         parms[2] = standard deviation (sigma) of Gaussian'
        print,'         parms[3] = optional constant to be added to Gaussian'
        return, -1
  endif

  common lorentzian, sigma

        Nparmg = N_elements( parms )
        npts = N_elements(xi) 
        ptype = size(parms,/type)
        if (ptype LE 3) or (ptype GE 12) then parms = float(parms)
        if (Nparmg GE 3) then sigma = parms[2]

        double = keyword_set(DOUBLE)
        if double then $       ;Double precision?
            lorentz = dblarr( npts ) else $
            lorentz = fltarr( npts )
 
        z = ( xi - parms[1] )/sigma
        zz = z*z

; Get smallest value expressible on computer.   Set lower values to 0 to avoid
; floating underflow
        minexp = alog((machar(DOUBLE=double)).xmin)     
 
        lorentz = (zz^2 + 1)

 if Nparmg LT 4 then return, parms[0] / lorentz else $
                     return, parms[0] / lorentz + parms[3]
 end
