function psf_lorentzian, parameters, NPIXEL=npixel, FWHM=fwhm,  $
                        DOUBLE = double, CENTROID=cntrd, ST_DEV=st_dev,  $
                        NORMALIZE=normalize
;+
; NAME:
;       PSF_LORENTZIAN
;
; PURPOSE:
;       Create a 1-d Lorentzian with specified FWHM, center 
; EXPLANATION:
;       Return a point spread function having Lorentzian profiles,
;       as either a 1D vector, a 2D image, or 3D volumetric-data.
;
; CALLING SEQUENCE:
;       psf = psf_Lorentzian( NPIXEL=, FWHM= , CENTROID = 
;                     [ /DOUBLE, /NORMALIZE, ST_DEV= ] ) 
; or:
;       psf = psf_Lorentzian( parameters, NPIXEL = )
;
; REQUIRED INPUT KEYWORD:
;       NPIXEL = number pixels for each dimension, specify as an array,
;               or just one number to make all sizes equal.
;
; OPTIONAL KEYWORDS:
;       CENTROID = floating scalar or vector giving position of  PSF center.    
;               default is exact center of requested vector/image/volume.
;               The number of elements in CENTROID should equal the number of
;               dimensions.    **The definition of Centroid was changed in
;               March 2002, and now an integer defines the center of a pixel.**
;
;       /DOUBLE  = If set, then the output array is computed in double precision
;               the default is to return a floating point array.
;
;       FWHM = the desired Full-Width Half-Max (pixels) in each dimension,
;               specify as an array, or single number to make all the same.
;
;       /NORMALIZE causes resulting PSF to be normalized so Total( psf ) = 1.
;
;       ST_DEV = optional way to specify width by standard deviation param.
;                Ignored if FWHM is specified.
;
; INPUTS (optional):
;
;       parameters = an 3 element vector giving:
;                       [ maxval, center, st_dev ],  overrides other keywords.
;
; EXAMPLE:
;
;       (1) Create a 50 pixel 1-d Lorentzian vector with a maximum of 12, 
;          centered at  pixel 23 with a sigma of 19.2
;
;       IDL> psf = psf_lorentzian([12,23,19.2],npixel=50)
; EXTERNAL CALLS:
;       function Lorentzian()
; NOTES:
;       To improve speed, floating underflow exceptions are suppressed (using 
;       the MASK=32  keyword of CHECK_MATH() rather than being flagged.
;
; HISTORY:
;       Written, Frank Varosi NASA/GSFC 1991.
;       Suppress underflow messages, add DOUBLE keyword. **Modified centroid
;       definition so integer position is pixel center** W. Landsman March 2002
;       Allow use of the ST_DEV (not STDEV) keyword W. Landsman Nov. 2002
;       Do not modify NPIXEL input keyword   W. Landsman  
;-
        On_error,2
	compile_opt idl2

        if (N_params() LT 1 ) and $
            ~(keyword_set( FWHM) || keyword_set(ST_DEV)) then begin
                print,'Syntax - psf = PSF_LORENTZIAN( parameters, NPIXEL = )'
                print, $
       'or       psf = PSF_LORENTZIAN( FWHM = ,ST_DEV = ,NPIXEL = ,[CENTROID = ])'
                return, -1
        endif

        sp = size( parameters )
	if sp[0] EQ 1 then begin
        	factor = parameters[0]
        	cntrd = parameters[1]
        	st_dev = parameters[2] 
	endif

        double = keyword_set(double)
        if double then idltype = 5 else idltype = 4

        if N_elements( npixel ) LE 0 then begin
                message,"must specify size of result with NPIX=",/INFO
                return,(-1)
          endif else begin 
	      npix = npixel
         endelse

        if N_elements( cntrd ) LE 0 then cntrd=(npix-1)/2. 
        if N_elements( fwhm ) GT 0 then begin 
               st_dev = fwhm/( 2.0d* sqrt( 2.0d* aLog(2.0d) ) )
               if not double then st_dev  = float(st_dev)
        endif 

        if N_elements( st_dev ) LE 0 then begin
                message,"must specify ST_DEV= or FWHM=",/INFO
                return,(-1)
          endif

        x = findgen( npix[0] ) - cntrd[0]
        psf = lorentzian( x, [1,0,st_dev] )

        if keyword_set( normalize ) then return, psf/total( psf )

        if N_elements( factor ) EQ 1 then begin
                if (factor NE 1) then return,factor*psf else return,psf
           endif else return, psf
end
