pro runlblrtm, wavestart, waveend, gdasdata, obsaltitude, wavetransmission, transmission, obselevation=obselevation, water=water, methane=methane, oxygen=oxygen, carbondioxide=carbondioxide, ammonia=ammonia, nitricoxide=nitricoxide, ozone=ozone, silent=silent, keep=keep

;+
; NAME:
;	RUNLBLRTM
;
; PURPOSE:
;	This procedure creates a TAPE5 using the GDAS data and the MIPAS model
;	atmosphere and runs LBLRTM to obtain a transmission spectrum in 
;	the specified wavelength range.
;
; CALLING SEQUENCE:
;	RUNLBLRTM, Wavestart, Waveend, Gdasdata, Obsaltitude, Wavetransmission, Transmission
;
; INPUTS:
;	Wavestart:   Start wavelength in AA (scalar float or double)
;	Waveend:     End wavelength in AA (scalar float or double)
;	Gdasdata:    Scalar string containing absolute path to GDAS file
;	Obsaltitude: Altitude angle in deg observation was taken at (scalar float or double)
;	
; KEYWORD PARAMETERS:
;	OBSELEVATION:   Elevation above sea level in km of observatory site, default is 2.648
;		        for VLT at Paranal
;	WATER:          Water abundance (scalar float or double) relative to model abundance
;	METHANE:        Methane abundance (scalar float or double) relative to model abundance
;	OXYGEN:         Oxygen abundance (scalar float or double) relative to model abundance
;	CARBONDIOXIDE:  Carbondioxide abundance (scalar float or double) relative to model abundance
;	AMMONIA:        Ammonia abundance (scalar float or double) relative to model abundance
;	NITRICOXIDE:    Nitricoxide abundance (scalar float or double) relative to model abundance
;	OZONE:          Ozone abundance (scalar float or double) relative to model abundance
;	SILENT:	        Set this keyword to suppress informational messages by LBLRTM
;	KEEP:		Set this keyword to keep the produced TAPE files.
;
; OUTPUTS:
;	Wavetransmission: Wavelength in vacuum (double array)
;	Transmission: Calculated transmission spectrum (double array)
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; RESTRICTIONS:
;	LBLRTM can only calculate a piece of less then 2020 cm^-1 in width.
;
; EXAMPLE:
;	A simple call looks like this:
;	RUNLBLRTM, 6000., 6500., '/here/is/the/GDAS/file', 79.8, wt, t
;
;	If you want to adjust the abundance of water and suppress informational
;	messages, use this:
;	RUNLBLRTM, 6000., 6500., '/here/is/the/GDAS/file', 79.8, wt, t, water=0.8, /silent
;
;	wt contains the wavelength array and t the transmission spectrum.
;	The abundance keywords work relative to the model abundandance, i.e., setting water=1.
;	means using the water abundance from the atmospheric model, setting water=0.8 means
;	lowering the abundance from the atmospheric model by 20 %. 
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, December 2014, Changed abundance keyword check to allow setting them to 0.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 6 then begin
    message,/info,'Syntax: runlblrtm, wavestart, waveend, gdasdata, obsaltitude, wavetransmission, transmission,'
    message,/info,'obselevation=obselevation, water=water, methane=methane, oxygen=oxygen, carbondioxide=carbondioxide, ammonia=ammonia, nitricoxide=nitricoxide, ozone=ozone, silent=silent'
    retall
endif

; check the keywords
if not keyword_set(obselevation) then obselevation = 2.648 else obselevation = obselevation ; elevation of the VLT above sea level [km] = 2.648
if n_elements(water) eq 0 then water = 1. else water = water
if water eq 0 then message, 'Warning! You set the water abundance to 0. This might cause LBLRTM to crash or produce strange results!', /informational
if n_elements(methane) eq 0 then methane = 1. else methane = methane
if n_elements(oxygen) eq 0 then oxygen = 1. else oxygen = oxygen
if n_elements(carbondioxide) eq 0 then carbondioxide = 1. else carbondioxide = carbondioxide
if n_elements(ammonia) eq 0 then ammonia = 1. else ammonia = ammonia
if n_elements(nitricoxide) eq 0 then nitricoxide = 1. else nitricoxide = nitricoxide
if n_elements(ozone) eq 0 then ozone = 1. else ozone = ozone

; read the model atmosphere
heightmodatm=readmodatm('HGT')
pressuremodatm=readmodatm('PRE')
temperaturemodatm=readmodatm('TEM')
o2modatm=readmodatm('O2')
co2modatm=readmodatm('CO2')
o3modatm=readmodatm('O3')
h2omodatm=readmodatm('H2O')
ch4modatm=readmodatm('CH4')
n2omodatm=readmodatm('N2O')
hno3modatm=readmodatm('HNO3')
comodatm=readmodatm('CO')
no2modatm=readmodatm('NO2')
nomodatm=readmodatm('NO')
nh3modatm=readmodatm('NH3')
ocsmodatm=readmodatm('OCS')
so2modatm=readmodatm('SO2')

; read the gdas data (only pressuregdas,heightgdas,temperaturegdas,dewpointgdas are used further)
readcol,gdasdata,pressuregdas,heightgdas,temperaturegdas,dewpointgdas,winddirgdas,windspeedgdas,skipline=6,/silent

; mix both atmospheric models
; how far goes GDAS
heightmaxgdas=max(heightgdas)/1000.  ; GDAS data in m, MIPAS profile in km
; where do we switch to MIPAS model
heightswitch=where(heightmodatm gt heightmaxgdas)
heightswitch=heightswitch[0]
; number of levels to build
levels=50
; create arrays for everyone
height=fltarr(levels)
pressure=fltarr(levels)
temperature=fltarr(levels)
o2=fltarr(levels)
co2=fltarr(levels)
o3=fltarr(levels)
h2o=fltarr(levels)
ch4=fltarr(levels)
n2o=fltarr(levels)
hno3=fltarr(levels)
co=fltarr(levels)
no2=fltarr(levels)
no=fltarr(levels)
nh3=fltarr(levels)
ocs=fltarr(levels)
so2=fltarr(levels)
; fill the arrays
; first use GDAS height, pressure, temperature and interpolate abundances of MIPAS to that grid
for i=0,n_elements(heightgdas)-1 do begin
    height[i]=heightgdas[i]/1000.  ; GDAS data in m, MIPAS profile in km
    pressure[i]=pressuregdas[i]
    temperature[i]=temperaturegdas[i]+273.15  ; GDAS in °C, LBLRTM expects K
    o2[i]=interpol(o2modatm,heightmodatm,height[i],/spline)
    co2[i]=interpol(co2modatm,heightmodatm,height[i],/spline)
    o3[i]=interpol(o3modatm,heightmodatm,height[i],/spline)
    h2o[i]=interpol(h2omodatm,heightmodatm,height[i],/spline)
    ch4[i]=interpol(ch4modatm,heightmodatm,height[i],/spline)
    n2o[i]=interpol(n2omodatm,heightmodatm,height[i],/spline)
    hno3[i]=interpol(hno3modatm,heightmodatm,height[i],/spline)
    co[i]=interpol(comodatm,heightmodatm,height[i],/spline)
    no2[i]=interpol(no2modatm,heightmodatm,height[i],/spline)
    no[i]=interpol(nomodatm,heightmodatm,height[i],/spline)
    nh3[i]=interpol(nh3modatm,heightmodatm,height[i],/spline)
    ocs[i]=interpol(ocsmodatm,heightmodatm,height[i],/spline)
    so2[i]=interpol(so2modatm,heightmodatm,height[i],/spline)
endfor
; next add the MIPAS model as is up to index 41, i.e. use 1km spacing
for i=n_elements(heightgdas),41 do begin
    j=heightswitch+i-n_elements(heightgdas)  ; index in MIPAS model, first is 27 km as GDAS usually ends somewhere between 26 and 27 km
    height[i]=heightmodatm[j]
    pressure[i]=pressuremodatm[j]
    temperature[i]=temperaturemodatm[j]
    o2[i]=o2modatm[j]
    co2[i]=co2modatm[j]
    o3[i]=o3modatm[j]
    h2o[i]=h2omodatm[j]
    ch4[i]=ch4modatm[j]
    n2o[i]=n2omodatm[j]
    hno3[i]=hno3modatm[j]
    co[i]=comodatm[j]
    no2[i]=no2modatm[j]
    no[i]=nomodatm[j]
    nh3[i]=nh3modatm[j]
    ocs[i]=ocsmodatm[j]
    so2[i]=so2modatm[j]
endfor
; next add levels in 5 km spacing up to 88 km
for i=42,49 do begin
    height[i]=heightmodatm[j+(i-41)*5]
    pressure[i]=pressuremodatm[j+(i-41)*5]
    temperature[i]=temperaturemodatm[j+(i-41)*5]
    o2[i]=o2modatm[j+(i-41)*5]
    co2[i]=co2modatm[j+(i-41)*5]
    o3[i]=o3modatm[j+(i-41)*5]
    h2o[i]=h2omodatm[j+(i-41)*5]
    ch4[i]=ch4modatm[j+(i-41)*5]
    n2o[i]=n2omodatm[j+(i-41)*5]
    hno3[i]=hno3modatm[j+(i-41)*5]
    co[i]=comodatm[j+(i-41)*5]
    no2[i]=no2modatm[j+(i-41)*5]
    no[i]=nomodatm[j+(i-41)*5]
    nh3[i]=nh3modatm[j+(i-41)*5]
    ocs[i]=ocsmodatm[j+(i-41)*5]
    so2[i]=so2modatm[j+(i-41)*5]
endfor
; scale abundances by factor
o2=o2*oxygen
co2=co2*carbondioxide
o3=o3*ozone
h2o=h2o*water
; if you change water abundance dewpoint temperature also changes
abundchange=0.4343*alog(water)
dewpointchange=0.
if abundchange ne 0 then dewpointchange=273.3/(8.286/abundchange-1.)  ; in °C
dewpointgdas=dewpointgdas+dewpointchange
ch4=ch4*methane
no=no*nitricoxide
nh3=nh3*ammonia

; set remaining parameters needed for TAPE5
wnstart=1.d8/waveend  ; start wavenumber in cm^-1, waveend in AA
wnend=1.d8/wavestart  ; end wavenumber in cm^-1, wavestart in AA
if wnend-wnstart ge 2018. then begin
    print,'ATTENTION: wnstart-wnend must be < 2020 cm^-1'
    retall
endif
zenithangle=90.-obsaltitude  ; in degree
null=0.

; create TAPE5
openw,unit,'TAPE5',/get_lun
; record 1.1
printf,unit,'$ TAPE5 created by runlblrtm.pro'
; record 1.2
printf,unit,'    1    1    1    0    1    0    0    0    0    1    0    0    0    0    5    5'
; record 1.3
printf,unit,format='(F10.3,F10.3,A)',wnstart-1,wnend+1,'     4.000 0.000E-00     0.000   0.0000 2.000E-04 1.000E-03'
; record 1.4
printf,unit,'     0.000     0.000     0.000     0.000     0.000     0.000     0.000'
; record 3.1
printf,unit,'    0    3    0    0    0   19    0          0.000    75.000     0.000   000.000'
; record 3.2
printf,unit,format='(F10.5,F10.5,F10.5,F10.5,F10.5,A)',obselevation,null,zenithangle,null,null,'    0'
; record 3.3A
printf,unit,'     2.000     5.000     8.000     0.000     0.000'
; user supplied atmospheric profile
; record 3.4
printf,unit,format='(I5)',levels
; use GDAS dewpoint for H2O first levels
for i=0,n_elements(heightgdas)-1 do begin
    ; record 3.5
    printf,unit,format='(E10.3,E10.3,E10.3,A)',height[i],pressure[i],temperature[i],'     AA   GAAAAAAAAAA'
    ; record 3.6.1
    printf,unit,format='(8E10.3)',dewpointgdas[i],co2[i],o3[i],n2o[i],co[i],ch4[i],o2[i],no[i]
    ; record 3.6.2
    printf,unit,format='(4E10.3)',so2[i],no2[i],nh3[i],hno3[i]
    ; record 3.6.3
    printf,unit,format='(3E10.3)',null,null,ocs[i]
endfor
; MIPAS model for the remaining ones
for i=n_elements(heightgdas),levels-1 do begin
    ; record 3.5 
    printf,unit,format='(E10.3,E10.3,E10.3,A)',height[i],pressure[i],temperature[i],'     AA   AAAAAAAAAAA'
    ; record 3.6.1
    printf,unit,format='(8E10.3)',h2o[i],co2[i],o3[i],n2o[i],co[i],ch4[i],o2[i],no[i]
    ; record 3.6.2
    printf,unit,format='(4E10.3)',so2[i],no2[i],nh3[i],hno3[i]
    ; record 3.6.3
    printf,unit,format='(3E10.3)',null,null,ocs[i]
endfor
; signal end of repitition
printf,unit,'-1.'
; make LBLRTM write ASCII of transmittance
; record 1.1
printf,unit,'$ Make ASCII'
; record 1.2 only activate plotting
printf,unit,' HI=0 F4=0 CN=0 AE=0 EM=0 SC=0 FI=0 PL=1 TS=0 AM=0 MG=0 LA=0 MS=0 XS=0    0    0'
; printf,unit,'    0    0    0    0    0    0    0    1    0    0    0    0    0    0    0    0'
; record 12.1
printf,unit,'# Plot title not used'
; record 12.2A
printf,unit,format='(F10.3,F10.3,A)',wnstart,wnend,'   10.2000  001.0000    1    0   12    0     1.000 0  0    0'
; record 12.3A
printf,unit,'    0.0000    1.2000    7.0200    0.2000    4    0    1    0    0    0 1    3 99'
; terminate plotting
printf,unit,'-1.'
; signal end of TAPE5 and terminate LBLRTM
printf,unit,'%'
; close TAPE5
free_lun,unit

; run LBLRTM, intercept messages by LBLRTM if asked to be silent
if not keyword_set(silent) then spawn,executeablename,result else spawn,executeablename,result,err

; read the transmittance
rdfloat,'TAPE99',wavenumber,transmittance,skipline=40,/double,/silent
wavetransmission=reverse(1.d8/wavenumber)
transmission=reverse(transmittance)

; clean up folder
if not keyword_set(keep) then begin
    spawn,'rm -f TAPE5'
    spawn,'rm -f TAPE6'
    spawn,'rm -f TAPE9'
    spawn,'rm -f TAPE10'
    spawn,'rm -f TAPE11'
    spawn,'rm -f TAPE12'
    spawn,'rm -f TAPE99'
endif else begin
;   message,/info,'TAPE files produced have not been removed! Remember to move them before next run or they will be overwritten!'
endelse
end
