function boxconv,x,y,width
;+
; NAME:
;           BOXCONV
;
; PURPOSE:
;           This function smooths an array by convolving with a Gaussian profile.
;
; CALLING SEQUENCE:
;           smoothed_y = BOXCONV(X, Y, Fwhm)
;
; INPUTS:
;           X:	   Array (double or float) of the values of the x-axis in ascending
;		   order.
;           Y:	   Array (double or float) of same size as x of the y-axis values.
;           Width: Width in units of the x-axis of the boxcar profile.
;
; OUTPUTS:
;           Array (double or float) of same size as x containing smoothed y.
;
; EXAMPLE:
;           flux_smoothed = BOXCONV(wavelength,flux,0.75)
;
; MODIFICATION HISTORY:
; 	Written by:	Christian Hummel, December 2018.
;-
; Copyright (C) 2018 Christian Hummel
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: smoothed_y = BOXCONV(X, Y, Fwhm)'
    retall
endif

return,smooth(y,nint(width/median(x-shift(x,1))))

end
function create_standard_parinfo
;+
; NAME:
;	CREATE_STANDARD_PARINFO
;
; PURPOSE:
;	This function creates a parameter info structure for use with fittell
;	containing appropriate values.
;
; CALLING SEQUENCE:
;	Result = CREATE_STANDARD_PARINFO()
;
; INPUTS:
;	None.
;
; OUTPUTS:
;	This function returns a parameter info structure for use with fittell
;	and in the format expected by MPFIT.
;
; EXAMPLE:
;       parinfo = CREATE_STANDARD_PARINFO()
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-

; assign parameters: p[0]=smooth, p[1]=v, p[2]=slope, p[3]=b, p[4]=wat, p[5]=meth, p[6]=co2
parinfo = replicate({value:0.D, fixed:0, limited:[0,0],limits:[0.D,0],step:0.d},8)

; smooth, i.e. width of boxcar
parinfo[0].value=8.0
parinfo[0].fixed=0
parinfo[0].limited[0]=1
parinfo[0].limited[1]=1
parinfo[0].limits[0]=4.0
parinfo[0].limits[1]=12.0
parinfo[0].step=0.5d
; smooth, i.e. width of lorentzian
parinfo[0].value=5.0
parinfo[0].fixed=0
parinfo[0].limited[0]=1
parinfo[0].limited[1]=1
parinfo[0].limits[0]=4.0
parinfo[0].limits[1]=8.0
parinfo[0].step=0.5d
; smooth, i.e. width of gaussian
parinfo[0].value=4.5
parinfo[0].fixed=0
parinfo[0].limited[0]=1
parinfo[0].limited[1]=1
parinfo[0].limits[0]=3.5
parinfo[0].limits[1]=5.5
parinfo[0].step=0.1d
; v, i.e. for shift in wavelength [km/s]
parinfo[1].value=-45.0	; corresponds to -3.6 AA
parinfo[1].fixed=0
parinfo[1].limited[0]=1
parinfo[1].limited[1]=1
parinfo[1].limits[0]=-55.d
parinfo[1].limits[1]=-35.d
parinfo[1].step=1.0
; y-intercept of line that is used to adapt observed spectrum
parinfo[2].value=1.d
parinfo[2].fixed=0
parinfo[2].limited[0]=1
parinfo[2].limited[1]=1
parinfo[2].limits[0]=0.2
parinfo[2].limits[1]=1.2
parinfo[2].step=0.02d
; slope of line that is used to adapt to observed spectrum
parinfo[3].value=0
parinfo[3].fixed=0
parinfo[3].limited=[1,1]
parinfo[3].limits=[-0.001,0.001]
parinfo[3].step=[0.0001]
; curvature of line that is used to adapt to observed spectrum
parinfo[4].value=0
parinfo[4].fixed=0
parinfo[4].limited=[1,1]
parinfo[4].limits=[-0.000005,0.000005]
parinfo[4].step=[0.000001]
; wat, i.e. abundance factor for water
parinfo[5].value=1.0d
parinfo[5].fixed=1
parinfo[5].limited=[1,1]
parinfo[5].limits[0]=0.4
parinfo[5].limits[1]=1.5
parinfo[5].step=0.1d
; meth, i.e. abundance factor for methane
parinfo[6].value=1.0d
parinfo[6].fixed=1
parinfo[6].limited=[1,1]
parinfo[6].limits[0]=0.4
parinfo[6].limits[1]=1.5
parinfo[6].step=0.1d
; co2, i.e. abundance factor for carbondioxide
parinfo[7].value=1.0d
parinfo[7].fixed=1
parinfo[7].limited=[1,1]
parinfo[7].limits[0]=0.4
parinfo[7].limits[1]=1.5
parinfo[7].step=0.1d

return,parinfo
end
function exclusion,wave,range,exclude

;+
; NAME:
;	EXCLUSION
;
; PURPOSE:
;	This function extracts from a wavelength array a certain range but without the
;	regions specified to exclude.
;
; CALLING SEQUENCE:
;	Result = EXCLUSION(Wavelength, Range, Exclude)
;
; INPUTS:
;	Wavelength:   Wavelength array from a range of which one wants to exclude certain
;		      regions.
;	Range:	      Wavelength range to extract from above array as two-element
;		      array in the form [start,end].
;	Exclude:      Regions to exclude from wavelength array in the range specified
;		      as array with up to 10 entries of the form [start1,end1,...,start5,end5].
;
; OUTPUTS:
;	This function returns an index array for the wavelength array that contains the
;	specified range but does not contain the regions stated in exclude.
;
; EXAMPLE:
;	indices = EXCLUSION(Wavelength, [6500.,6800.], [6558.,6568.])
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: induse=exclusion(wave,range,exclude)'
    retall
endif

; check that requested range lies within given wavelength array
if range[0] le min(wave) or range[0] ge max(wave) then begin
     print,'Requested range outside wavelength array'
     retall
endif
if range[1] le min(wave) or range[1] ge max(wave) then begin
     print,'Requested range outside wavelength array'
     retall
endif

; build a mask
fitmask=intarr(n_elements(wave))
; mark the elements that are in the range
ind=where(wave ge range[0] and wave le range[1])
fitmask[ind]=1
; check that the regions to exclude are in the range
for i=0,n_elements(exclude)-1 do begin
    if (exclude[i] le range[0] or exclude[i] ge range[1]) then begin
        print,'Borders chosen for exclusion not within chosen wavelength range'
        retall
    endif
endfor

case n_elements(exclude) of  ; how many exclusions are there
	2: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
	end
	4: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
	end
	6: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
		ind=where(wave ge exclude[4] and wave le exclude[5])
		fitmask[ind]=0
	end
	8: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
		ind=where(wave ge exclude[4] and wave le exclude[5])
		fitmask[ind]=0
		ind=where(wave ge exclude[6] and wave le exclude[7])
		fitmask[ind]=0
	end
	10: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
		ind=where(wave ge exclude[4] and wave le exclude[5])
		fitmask[ind]=0
		ind=where(wave ge exclude[6] and wave le exclude[7])
		fitmask[ind]=0
		ind=where(wave ge exclude[8] and wave le exclude[9])
		fitmask[ind]=0
	end
else: begin
	print,'Number of exclusions not supported!'
	retall
end
endcase

; determine which indices to return
fitind=where(fitmask eq 1)
return,fitind
end
pro fittell,wave,data,error,range,gdasdata,obsaltitude,params,paramserror,waveresult,result,cleaned,cleanederror,exclude=exclude,plot=plot,silent=silent,parinfo=parinfo,titleplot=titleplot
;+
; NAME:
;	FITTELL
;
; PURPOSE:
;	This procedure uses LBLRTM to fit a transmission model to the observed
;	spectrum and then uses this model to remove the telluric lines from
;	the spectrum.
;
; CALLING SEQUENCE:
;	FITTELL,Wave,Data,Error,Range,Gdasdata,Obsaltitude,Params,Paramserror,Waveresult,Result,Cleaned,Cleanederror
;
; INPUTS:
;	Wave:		Wavelength array of observed spectrum
;	Data:		Flux array of observed spectrum
;	Error:		Error of flux measurements
;	Range:		Wavelength range of fit as two-element array in the form [start,end]
;	Gdasdata:	Scalar string containing absolute path to GDAS file
;	Obsaltitude:	Altitude angle observation was taken at in deg
;	
; KEYWORD PARAMETERS:
;	EXCLUDE:	Regions to exclude from wavelength array in the range specified
;			as array with up to 10 entries of the form [start1,end1,...,start5,end5].
;	PLOT:		Set this keyword to get a plot of the result.
;	SILENT:		Set this keyword to suppress informational messages
;	PARINFO:	Parameter info structure to use in MPFIT. Default is a standard structure
;			created by CREATE_STANDARD_PARINFO.
;	TITLEPLOT:	Title of plot.
;
; OUTPUTS:
;	Params:		Fitted parameters ([smoothing_FWHM,velocity_shift,
;			slope_of_straight,intercept_of_straight,water_abundance,
;			methane_abundance,carbondioxide_abundance,oxygen_abundance])
;	Paramserror:	Formal 1-sigma errors of parameters
;	Waveresult:	Wavelength array of the fitted part
;	Result:		Model fitted with all parameters (to be compared to observation)
;	Cleaned:	Telluric line removed spectrum.
;	Cleanederror:	Error of flux measurements divided by transmission model.
;
; EXAMPLE:
;	FITTELL,Wave,Data,Error,[9000.,9300.],'/here/lies/the/Gdasdata',87.2,Params,Paramserror,Waveresult,Result,Cleaned,Cleanederror
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-

; check the input parameters & keywords
if n_params() lt 11 then begin
    message,/info,'Syntax: fittell,wave,data,error,range,gdasdata,obsaltitude,params,paramserror,waveresult,result,cleaned,cleanederror,exclude=exclude, plot=plot,silent=silent,parinfo=parinfo,titleplot=titleplot'
    retall
endif

if not keyword_set(parinfo) then parinfo=create_standard_parinfo() else parinfo=parinfo

anfang=systime(/julian) ; to measure execution time
if not keyword_set(silent) then begin   ; make sure quiet has a value
      quiet=0
      silent=0
endif else quiet=1 ; suppress output messages of mpfitfun

; define index of region to fit
if keyword_set(exclude) then induse=exclusion(wave,range,exclude) $
			else induse=where(wave ge range[0] and wave le range[1])
wavuse=wave[induse]
datuse=data[induse]
erruse=error[induse]

; actual fit using MPFIT
if keyword_set(silent) $
	then fctargs={gdasdata:gdasdata,obsaltitude:obsaltitude,silent:silent} $
	else fctargs={gdasdata:gdasdata,obsaltitude:obsaltitude}
params=mpfitfun('tell',wavuse,datuse,erruse,parinfo=parinfo,niter=n, $
	yfit=res,status=stat,errmsg=erm,bestnorm=chi,dof=dof,nfev=nfev, $
	quiet=quiet,functargs=fctargs,perror=perror)
; if there was an error give error message independent of setting of silent
if stat le 0 then print,stat,erm
paramserror=perror

; create informative messages if wanted
; if not keyword_set(silent) then begin
     ende=systime(/julian)
     dauer=(ende-anfang)*24*60
;    print,format='(A13,F6.2,A4)','Time needed: ',dauer,' min'
;    text=statusinterpreter(stat)
;    print,'Status: ',stat,' ',text
     if stat le 0 then print,erm
;    print,'Number of iterations: ',n
;    print,'Number of calls to tell: ',nfev
;    print,'Parameters of tell:'
     print,'      smooth,           Dv (km/s),   Y-intercept,     slope,         curvature,         water,         methane,        co2'
     print,params
;    print,perror
     print,'Bestnorm: ',chi
;    print,'DOF: ',dof
; endif

; return fit result of complete range if exclude was set
if keyword_set(exclude) then begin
     waveresult=wave[where(wave ge range[0] and wave le range[1])]
     result=tell(waveresult,params,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)
endif else begin
     waveresult=wavuse
     result=res
endelse

; recalculate the model without adjustment by straight line, i.e. get transmission spectrum convolved with Gaussian
modell=tellwithout(waveresult,params,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)

; remove the telluric lines 
if keyword_set(exclude) then begin
     cleaned=data[where(wave ge range[0] and wave le range[1])]/modell
     cleanederror=error[where(wave ge range[0] and wave le range[1])]/modell
endif else begin
    cleaned=datuse/modell
    cleanederror=erruse/modell
    ; CHU: should use best-fit model?
    cleaned=datuse/result
    cleanederror=erruse/result
endelse

; plot result
if keyword_set(plot) then begin
    if keyword_set(titleplot) then nameplot=titleplot else nameplot=''
    loadct,39,/silent
    !P.MULTI=[0,1,2]
    ; plot original data and fit
    plot,wave,data,xra=[range[0]-10.d,range[1]+10.d],/xsty,xtitle='Wavelength',ytitle='Flux',title=nameplot+' Original data (black) and fit (red)',xtickformat='(I5)'
    oplot,waveresult,result,color=235
    ; mark the regions that where excluded
    if keyword_set(exclude) then begin
        y=mean(result)
        for da=0,n_elements(exclude)-1 do plots,exclude[da],[y-0.1*y,y+0.1*y],col=191
    endif
    ; plot cleaned data and original data
    plot,waveresult,cleaned,xra=[range[0]-10.d,range[1]+10.d],/xsty,xtitle='Wavelength',ytitle='Flux',title=nameplot+' Cleaned data (black) and original data (red)',xtickformat='(I5)'
    oplot,wave,data,color=235
    !P.MULTI=0
endif
end
function gaussconv,x,y,fwhm
;+
; NAME:
;           GAUSSCONV
;
; PURPOSE:
;           This function smooths an array by convolving with a Gaussian profile.
;
; CALLING SEQUENCE:
;           smoothed_y = GAUSSCONV(X, Y, Fwhm)
;
; INPUTS:
;           X:	   Array (double or float) of the values of the x-axis in ascending
;		   order.
;           Y:	   Array (double or float) of same size as x of the y-axis values.
;           Fwhm:  FWHM in units of the x-axis of the Gaussian profile.
;
; OUTPUTS:
;           Array (double or float) of same size as x containing smoothed y.
;
; EXAMPLE:
;           flux_smoothed = GAUSSCONV(wavelength,flux,0.75)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: smoothed_y = GAUSSCONV(X, Y, Fwhm)'
    retall
endif

; start and end of x array
xmin=min(x)
xmax=max(x)

; oversampling, the profile must not have more pixels then the data array
fwhm_pix=10.d	; gauss will have fwhm_pix*fwhm_pix pixels
dx=fwhm/fwhm_pix
xint=xmin + dx*dindgen(long((xmax-xmin)/dx+1))
yint=interpol(y,x,xint)

; number of pixels for dimension -> npixel keyword of psf_gaussian
nopd=fix(fwhm_pix*fwhm_pix)

; 1D gaussian profile
gauss = psf_gaussian(npixel=nopd,fwhm=fwhm_pix,/normalize,ndimen=1)

; convolve y with the gaussian profile
yconv=convol(yint,gauss,/center,/edge_truncate)
ysmooth=interpol(yconv,xint,x)

return,ysmooth
end
pro getabundances,wavenir,specnir,errnir,gdasdata,obsaltitude, $
	ab_h2o,ab_co2,ab_ch4,ab_h2o_str,ab_co2_str,ab_ch4_str, $
	silent=silent,plot=plot
;+
; NAME:
;	GETABUNDANCES
;
; PURPOSE:
;	This procedure determines the abundances of the main molecules (H2O,
;	CO2, CH4) producing telluric lines in the GRAVITY spectral range.
;
; CALLING SEQUENCE:
;	GETABUNDANCES, Wavenir, Specnir, Errnir,
;	Gdasdata, Obsaltitude, Ab_h2o, Ab_co2, Ab_ch4, Ab_h2o_str,
;	Ab_co2_str, Ab_ch4_str
;
; INPUTS:
;	Wavenir:	Wavelength array of NIR data
;	Specnir:	Flux array of NIR data
;	Errnir:		Error array of NIR data
;	Gdasdata:	Scalar string containing absolute path to GDAS file
;	Obsaltitude:	Altitude angle observation was taken at in deg
;	
; KEYWORD PARAMETERS:
;	SILENT:		Set this keyword to suppress informational messages
;	PLOT:		Set this keyword to get a plot of the result.
;
; OUTPUTS:
;	Ab_h2o:		Abundance of H2O in this observation
;	Ab_co2:		Abundance of CO2 in this observation
;	Ab_ch4:		Abundance of CH4 in this observation
;	Ab_h2o_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of H20
;	Ab_co2_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of CO2
;	Ab_ch4_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of CH4
;
; EXAMPLE:
;	GETABUNDANCES, Wavenir, Specnir, Errnir,
;	'/here/lies/the/Gdasdata',87.2, Ab_h2o, Ab_co2, Ab_ch4,
;	Ab_h2o_str, Ab_co2_str, Ab_ch4_str
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, October 2014, Adapted parameter borders
;	N. Rudolf, November 2014, Added output to save the fit results of
;	the individual regions used to determine the abundances of the
;	main telluric contributors.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 5 then begin
    message,/info,'Syntax: GETABUNDANCES,wavenir,specnir,errnir,gdasdata,obsaltitude,ab_h2o,ab_co2,ab_ch4,ab_h2o_str,ab_co2_str,ab_ch4_str,silent=silent,plot=plot'
    retall
endif
if not keyword_set(plot) then plot=0

abund={range:dblarr(2),p:dblarr(8),pe:dblarr(8)}
ab_h2o_str=replicate(abund,1)
ab_co2_str=replicate(abund,2)
ab_ch4_str=replicate(abund,2)

if not keyword_set(silent) then print,'Fitting abundances:'

; set up parinfo structure for H2O
parw = create_standard_parinfo()
; wat, i.e. abundance factor for water
parw[5].fixed=0
if not keyword_set(silent) then print,'Fit H2O'
fittell,wavenir,specnir,errnir,[24100.,24500.],gdasdata,obsaltitude, $
	pw1,pew1,wrw1,rw1,clw1,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[0].range=[24100.,24500.]
ab_h2o_str[0].p=pw1
ab_h2o_str[0].pe=pew1

; abundance used is mean of all abundances
; ab_h2o=mean([pw1[4],pw2[4],pw3[4],pw4[4],pw5[4]])
ab_h2o=pw1[5]

; set up parinfo structure for CO2
parco2 = create_standard_parinfo()
; wat, i.e. abundance factor for water
parco2[5].value=ab_h2o
; co2, i.e. abundance factor for carbondioxide
parco2[7].fixed=0
if not keyword_set(silent) then print,'Fit CO2'
fittell,wavenir,specnir,errnir,[19950.,20300.],gdasdata,obsaltitude, $
	pco21,peco21,wrco21,rco21,clco21,parinfo=parco2,/silent, $
	plot=plot,titleplot='Fit CO2:'
ab_co2_str[0].range=[19950.,20300.]
ab_co2_str[0].p=pco21
ab_co2_str[0].pe=peco21

fittell,wavenir,specnir,errnir,[20450.,20800.],gdasdata,obsaltitude, $
	pco22,peco22,wrco22,rco22,clco22,parinfo=parco2,/silent, $
	plot=plot,titleplot='Fit CO2:'
ab_co2_str[1].range=[20450.,20800.]
ab_co2_str[1].p=pco22
ab_co2_str[1].pe=peco22

; abundance used is mean of all abundances
ab_co2=mean([pco21[7],pco22[7]])

; set up parinfo structure for CH4
parch4 = create_standard_parinfo()
; wat, i.e. abundance factor for water
parch4[5].value=ab_h2o
; meth, i.e. abundance factor for methane
parch4[6].fixed=0
; co2, i.e. abundance factor for carbondioxide
parch4[7].value=ab_co2
if not keyword_set(silent) then print,'Fit CH4'
fittell,wavenir,specnir,errnir,[22850.,23110.],gdasdata,obsaltitude, $
	pch41,pech41,wrch41,rch41,clch41,parinfo=parch4,/silent, $
	plot=plot,titleplot='Fit CH4:'
ab_ch4_str[0].range=[22850.,23110.]
ab_ch4_str[0].p=pch41
ab_ch4_str[0].pe=pech41

; fittell,wavenir,specnir,errnir,[23370.,23630.],gdasdata,obsaltitude, $
; 	pch42,pech42,wrch42,rch42,clch42,parinfo=parch4,/silent, $
; 	plot=plot,titleplot='Fit CH4:'
; ab_ch4_str[1].range=[23370.,23630.]
; ab_ch4_str[1].p=pch42
; ab_ch4_str[1].pe=pech42

fittell,wavenir,specnir,errnir,[23630.,23900.],gdasdata,obsaltitude, $
	pch43,pech43,wrch43,rch43,clch43,parinfo=parch4,/silent, $
	plot=plot,titleplot='Fit CH4:'
ab_ch4_str[1].range=[23630.,23900.]
ab_ch4_str[1].p=pch43
ab_ch4_str[1].pe=pech43

; abundance used is mean of all abundances
ab_ch4=mean([pch41[6],pch43[6]])

end
function getobjects
;+
; NAME:
;	GETOBJECTS
;
; PURPOSE:
;	This function extracts the names of the objects to use the TELLREM
;	package on.
;
; CALLING SEQUENCE:
;	Result = GETOBJECTS()
;
; INPUTS:
;	None.
;
; OUTPUTS:
;	This function returns a string array of the folder names located
;	in the path containing the reduced spectra specified in the
;	info file provide to LOADTELLREMINFO. It expects these to be the
;	object names which will be added as auxiliary information during
;	the further run of TELLREM.
;
; COMMON BLOCKS:
;       TELLREM_INFO: This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;       objects = GETOBJECTS()
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; extract folder names
objects=file_search(spectrafolder,'*.fits')
; just get the name itself without absolute path
for i=0,n_elements(objects)-1 do begin
	d=mrdfits(objects[i],0,h)
	objects[i]=fitshparse(h,'OBJECT')
;   obj=strsplit(objects[i],'/',/extract)
;   objects[i]=obj[n_elements(obj)-1]
endfor
; return them
return,objects
end
pro idlplottitlechanger,filename
;+
; NAME:
;	IDLPLOTTITLECHANGER
;
; PURPOSE:
;	This procedure replaces the IDL standard plot title "Graphics produced
;	by IDL" with the file name in the specified postscript file.
;
; CALLING SEQUENCE:
;	IDLPLOTTITLECHANGER, Filename
;
; INPUTS:
;	Filename:   Name of the postscript file to work on (scalar string).
;
; OUTPUTS:
;	None.
;
; EXAMPLE:
;	IDLPLOTTITLECHANGER,'plotfileXYZ.ps'
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: IDLPLOTTITLECHANGER,filename'
    retall
endif

spawn,'cat '+filename+'| sed "s|Graphics produced by IDL|'+filename+'|" >  idltemp.ps; mv idltemp.ps '+filename
end
pro loadtellreminfo,file
;+
; NAME:
;	LOADTELLREMINFO
;
; PURPOSE:
;	This procedure loads the information necessary to run the procedures
;	of the tellrem package and creates the common block TELLREM_INFO. An example file
;	called 'info_for_tellrem' you can adjust for your needs is
;	provided with the package. Run this procedure before any of the other
;	functions or procedure of the tellrem package.
;
; CALLING SEQUENCE:
;	LOADTELLREMINFO, filename
;
; INPUTS:
;	Filename:  Absolute path to file containing the inforamtion as string.
;
; OUTPUTS:
;	None. But creates the common block TELLREM_INFO. This common block
;	contains the variables Gdasfolder, Modelatmosphere, Executeablename,
;	Spectrafolder, and Obstype. All of them are scalar strings. Gdasfolder
;	contains the absolute path of the folder containing the GDAS files,
;	Modelatmosphere contains the absolute  path to the MIPAS model
;	atmosphere, Executeablename contains the name of LBLRTM executable,
;	Spectrafolder contains the absolute path to the folder containing the
;	reduced spectra (in individual subfolders therein), and Obstype contains
;	the prefix specifying the type of observation (i.e., SCI, FLUX, or TELL)
;	the X-Shooter pipeline assigns to the .fits files.
;
; COMMON BLOCKS:
;       TELLREM_INFO: This common block contains relevant folder names and strings
;	              for running tellrem. It is created by this procedure.
;
; EXAMPLE:
;	LOADTELLREMINFO,'/here/lies/info_for_tellrem'
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: LOADTELLREMINFO, filename'
    retall
endif

; open the file
openr,unit,file,/get_lun
; read the lines and fill the variables
n=numlines(file)
cont=strarr(n)
readf,unit,cont
gdasfolder=strsplit(cont[0],/extract)
gdasfolder=gdasfolder[0]
modelatmosphere=strsplit(cont[1],/extract)
modelatmosphere=modelatmosphere[0]
executeablename=strsplit(cont[2],/extract)
executeablename=executeablename[0]
spectrafolder=strsplit(cont[3],/extract)
spectrafolder=spectrafolder[0]
obstype=strsplit(cont[4],/extract)
obstype=obstype[0]
free_lun,unit
end
function lorentzconv,x,y,fwhm
;+
; NAME:
;           GAUSSCONV
;
; PURPOSE:
;           This function smooths an array by convolving with a Gaussian profile.
;
; CALLING SEQUENCE:
;           smoothed_y = GAUSSCONV(X, Y, Fwhm)
;
; INPUTS:
;           X:	   Array (double or float) of the values of the x-axis in ascending
;		   order.
;           Y:	   Array (double or float) of same size as x of the y-axis values.
;           Fwhm:  FWHM in units of the x-axis of the Gaussian profile.
;
; OUTPUTS:
;           Array (double or float) of same size as x containing smoothed y.
;
; EXAMPLE:
;           flux_smoothed = GAUSSCONV(wavelength,flux,0.75)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: smoothed_y = LORENTZCONV(X, Y, Fwhm)'
    retall
endif

; start and end of x array
xmin=min(x)
xmax=max(x)

; oversampling, the profile must not have more pixels then the data array
fwhm_pix=10.d	; lorentz will have fwhm_pix*fwhm_pix pixels
dx=fwhm/fwhm_pix
xint=xmin + dx*dindgen(long((xmax-xmin)/dx+1))
yint=interpol(y,x,xint)

; number of pixels for dimension -> npixel keyword of psf_lorentzian
nopd=fix(fwhm_pix*fwhm_pix)

; 1D lorentzian profile
lorentz = psf_lorentzian(npixel=nopd,fwhm=fwhm_pix,/normalize)

; convolve y with the lorentzian profile
yconv=convol(yint,lorentz,/center,/edge_truncate)
ysmooth=interpol(yconv,xint,x)

return,ysmooth
end
function numlines,file
;+
; NAME:
;     NUMLINES() 
; PURPOSE:
;     Return the number of lines in a file
;
;     This procedures became mostly obsolete in V5.6 with the introduction of
;     the FILE_LINES() procedure
; CALLING SEQUENCE:
;     nl = NUMLINES( filename )
; INPUT:
;     filename = name of file, scalar string
; OUTPUT:
;     nl = number of lines in the file, scalar longword
;          Set to -1 if the number of lines could not be determined
; METHOD:
;     If Unix then spawn to wc; otherwise read 1 line at a time and count
;
; PROCEDURE CALLS:
;     EXPAND_TILDE(), SPEC_DIR()
; MODIFICATION HISTORY:
;     W. Landsman                              February 1996
;     Use /bin/sh shell with wc under Unix     March 1997
;     Use EXPAND_TILDE() under Unix         September 1997
;     Converted to IDL V5.0   W. Landsman   September 1997
;     Call intrinsic FILE_LINES() if V5.6 or later   December 2002
;-
 On_error,2

 if N_params() EQ 0 then begin
        print,'Syntax - nl = NUMLINES( file)'
        return,-1
 endif

 if !VERSION.RELEASE GE '5.6' then return,file_lines(file)
  nl = -1L
 openr,lun,file,/get_lun, ERROR = err
 if err NE 0 then begin
        if !VERSION.OS eq "vms" then file = spec_dir(file,'DAT') else $
        file = spec_dir(file)
        message,'ERROR - Unable to open file '+ file,/CON
        return,-1
 endif

 if !VERSION.OS_FAMILY EQ 'unix' then begin
         free_lun,lun
         if strpos(file,'~') GE 0 then file = expand_tilde(file)
         spawn,'wc -l < '+file, result, /sh    
         return,long(result[0])
 endif else begin                 ;=====>> Loop through file counting lines  
        On_ioerror,NOASCII
        nl = 0l
        tmp = ' '
         while not eof(lun) do begin
           readf,lun,tmp
           nl = nl + 1
         endwhile
         free_lun,lun
         return,nl
 endelse

NOASCII:
  message,'Error reading file ' + string(file),/CON
  return,-1
 end
pro rddat,file,wave,spec,error,header,phoenix=phoenix
;+
; NAME:
;	RDDAT
;
; PURPOSE:
;	This procedure reads VLT/X-Shooter and Gravity spectra.
;
; CALLING SEQUENCE:
;	RDDAT,File,Wave,Spec,Error,Header
;
; INPUTS:
;	File:	Scalar string containing complete path to file
;	
; KEYWORD PARAMETERS:
;	PHOENIX:  Set this keyword if you want to read spectra from
;		  the Göttingen Spectral Library by PHOENIX
;		  (http://phoenix.astro.physik.uni-goettingen.de/)
;
; OUTPUTS:
;	Wave:	  Wavelength array in Angstrom
;	Spec:	  Flux array

;
; OPTIONAL OUTPUTS:
;	Error:	  Error array
;	Header:	  Header of the fits file
;
; EXAMPLE:
;	If you just want to have wavelength and flux use
;       rddat,'/here/is/the/data/SCI_SLIT_MERGE1D_VIS.fits',wave,spec
;
;	If you also want error and header use
;       rddat,'/here/is/the/data/SCI_SLIT_MERGE1D_VIS.fits',wave,spec,error,header
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: rddat,file,wave,spec,error,header  (file=string)'
    retall
endif

; read data and header
spec = mrdfits(file,0,/silent)
if keyword_set(phoenix) then spec=spec*1.d-8
head = headfits(file)

if n_params() ge 4 then begin
	error=mrdfits(file,1,/silent)
	qual=mrdfits(file,2,/silent)
	index=where(qual eq 1,count)
	if count gt 0 then error(index)=error(index)*10
;	error(*)=0.1
endif
if n_params() eq 5 then header=head

;start wavelength
startwave = sxpar(head,'CRVAL1')

;wavelength bin size
wavebin = sxpar(head,'CDELT1')

; wavelength array
wave = dblarr(n_elements(spec))
for i = 0L,n_elements(spec)-1 do wave[i] = startwave + double(i)*wavebin
if keyword_set(phoenix) then begin
    wa=exp(wave)
;   vactoair,wa
    wave=wa
endif else wave=wave*10.d
end
function readmodatm,parameter

;+
; NAME:
;	READMODATM
;
; PURPOSE:
;	This procedures reads the requested parameter (e.g. pressure, abundance)
;	from the MIPAS model atmosphere.
;
; CALLING SEQUENCE:
;	Result = READMODATM(Parameter)
;
; INPUTS:
;	Parameter:   Parameter to be read (scalar string) in format of MIPAS model
;		     atmosphere file.
;
; OUTPUTS:
;	This function returns the requested parameter in a float array.
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; RESTRICTIONS:
;	This function only works for the parameters available parameters in the
;	MIPAS model atmosphere. These are:
;	HGT, PRE, TEM, N2, O2, CO2, O3, H2O, CH4, N2O, HNO3, CO, NO2,
;	N2O5, ClO, HOCl, ClONO2, NO, HNO4, HCN, NH3, F11, F12, F14,
;	F22, CCl4, COF2, H2O2, C2H2, C2H6, OCS, SO2, SF6.
;
; EXAMPLE:
;	height = READMODATM('HGT')
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = READMODATM(Parameter)'
    retall
endif

; how many lines has the file
n=numlines(modelatmosphere)
cont=strarr(n)

; read it
openr,unit,modelatmosphere,/get_lun
readf,unit,cont

; find the number of profile levels
therepl=where(strmatch(cont,'*Profile Levels*') eq 1,count)
if count eq 0 then begin
    print,'Could not determine number of profile levels! Check file!'
    retall
endif
reads,cont[therepl],npl

; create array to take data
array=fltarr(npl)

; find where the individual data portions lie
inddat=where(strmid(cont,0,1) eq '*')
continddat=cont[inddat]

; find the parameter asked for
there=where(strmatch(continddat,'\*'+parameter+' *') eq 1,count)
if count eq 0 then begin
    print,'Could not find parameter asked for! Check input and/or file!'
    retall
endif
reads,cont[(inddat[there]+1):(inddat[there+1]-1)],array
close,unit
free_lun,unit
return,array
end
pro recrfittellres,wave,data,error,range,gdasdata,obsaltitude,params,waveresult,result,cleaned,cleanederror,silent=silent
;+
; NAME:
;	RECRFITTELLRES
;
; PURPOSE:
;	This procedure recreates the output of FITTELL for the parameters specified.
;
; CALLING SEQUENCE:
;	RECRFITTELLRES,Wave,Data,Error,Range,Gdasdata,Obsaltitude,Params,Waveresult,Result,Cleaned,Cleanederror
;
; INPUTS:
;	Wave:		Wavelength array of observed spectrum
;	Data:		Flux array of observed spectrum
;	Error:		Error of flux measurements
;	Range:		Wavelength range of fit as two-element array in the form [start,end]
;	Gdasdata:	Scalar string containing absolute path to GDAS file
;	Obsaltitude:	Altitude angle observation was taken at in deg
;	Params:		Fitted parameters ([smoothing_FWHM,velocity_shift,
;			slope_of_straight,intercept_of_straight,water_abundance,
;			methane_abundance,carbondioxide_abundance,oxygen_abundance])
;	
; KEYWORD PARAMETERS:
;	SILENT:		Set this keyword to suppress informational messages
;
; OUTPUTS:
;	Waveresult:	Wavelength array of the fitted part
;	Result:		Model fitted with all parameters (to be compared to observation)
;	Cleaned:	Telluric line removed spectrum.
;	Cleanederror:	Error of flux measurements divided by transmission model.
;
; EXAMPLE:
;       RECRFITTELLRES,Wave,Data,Error,[9000.,9300.],'/here/lies/the/Gdasdata',87.2,Params,Waveresult,Result,Cleaned,Cleanederror
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; define index of region of fit
induse=where(wave ge range[0] and wave le range[1])
wavuse=wave[induse]
datuse=data[induse]
erruse=error[induse]
datuse=subzero(datuse) ; remove subzero points (if any)

; remove telluric lines from observed spectrum and return telluric spectrum of complete range if exclude was set
waveresult=wavuse
result=tell(waveresult,params,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)
modell=tellwithout(waveresult,params,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)
cleaned=datuse/modell
cleanederror=erruse/modell
;
; CHU: should use best-fit model?
cleaned=datuse/result
cleanederror=erruse/result

end
pro runlblrtm, wavestart, waveend, gdasdata, obsaltitude, wavetransmission, transmission, obselevation=obselevation, water=water, methane=methane, oxygen=oxygen, carbondioxide=carbondioxide, ammonia=ammonia, nitricoxide=nitricoxide, ozone=ozone, silent=silent, keep=keep

;+
; NAME:
;	RUNLBLRTM
;
; PURPOSE:
;	This procedure creates a TAPE5 using the GDAS data and the MIPAS model
;	atmosphere and runs LBLRTM to obtain a transmission spectrum in 
;	the specified wavelength range.
;
; CALLING SEQUENCE:
;	RUNLBLRTM, Wavestart, Waveend, Gdasdata, Obsaltitude, Wavetransmission, Transmission
;
; INPUTS:
;	Wavestart:   Start wavelength in AA (scalar float or double)
;	Waveend:     End wavelength in AA (scalar float or double)
;	Gdasdata:    Scalar string containing absolute path to GDAS file
;	Obsaltitude: Altitude angle in deg observation was taken at (scalar float or double)
;	
; KEYWORD PARAMETERS:
;	OBSELEVATION:   Elevation above sea level in km of observatory site, default is 2.648
;		        for VLT at Paranal
;	WATER:          Water abundance (scalar float or double) relative to model abundance
;	METHANE:        Methane abundance (scalar float or double) relative to model abundance
;	OXYGEN:         Oxygen abundance (scalar float or double) relative to model abundance
;	CARBONDIOXIDE:  Carbondioxide abundance (scalar float or double) relative to model abundance
;	AMMONIA:        Ammonia abundance (scalar float or double) relative to model abundance
;	NITRICOXIDE:    Nitricoxide abundance (scalar float or double) relative to model abundance
;	OZONE:          Ozone abundance (scalar float or double) relative to model abundance
;	SILENT:	        Set this keyword to suppress informational messages by LBLRTM
;	KEEP:		Set this keyword to keep the produced TAPE files.
;
; OUTPUTS:
;	Wavetransmission: Wavelength in vacuum (double array)
;	Transmission: Calculated transmission spectrum (double array)
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; RESTRICTIONS:
;	LBLRTM can only calculate a piece of less then 2020 cm^-1 in width.
;
; EXAMPLE:
;	A simple call looks like this:
;	RUNLBLRTM, 6000., 6500., '/here/is/the/GDAS/file', 79.8, wt, t
;
;	If you want to adjust the abundance of water and suppress informational
;	messages, use this:
;	RUNLBLRTM, 6000., 6500., '/here/is/the/GDAS/file', 79.8, wt, t, water=0.8, /silent
;
;	wt contains the wavelength array and t the transmission spectrum.
;	The abundance keywords work relative to the model abundandance, i.e., setting water=1.
;	means using the water abundance from the atmospheric model, setting water=0.8 means
;	lowering the abundance from the atmospheric model by 20 %. 
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, December 2014, Changed abundance keyword check to allow setting them to 0.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 6 then begin
    message,/info,'Syntax: runlblrtm, wavestart, waveend, gdasdata, obsaltitude, wavetransmission, transmission,'
    message,/info,'obselevation=obselevation, water=water, methane=methane, oxygen=oxygen, carbondioxide=carbondioxide, ammonia=ammonia, nitricoxide=nitricoxide, ozone=ozone, silent=silent'
    retall
endif

; check the keywords
if not keyword_set(obselevation) then obselevation = 2.648 else obselevation = obselevation ; elevation of the VLT above sea level [km] = 2.648
if n_elements(water) eq 0 then water = 1. else water = water
if water eq 0 then message, 'Warning! You set the water abundance to 0. This might cause LBLRTM to crash or produce strange results!', /informational
if n_elements(methane) eq 0 then methane = 1. else methane = methane
if n_elements(oxygen) eq 0 then oxygen = 1. else oxygen = oxygen
if n_elements(carbondioxide) eq 0 then carbondioxide = 1. else carbondioxide = carbondioxide
if n_elements(ammonia) eq 0 then ammonia = 1. else ammonia = ammonia
if n_elements(nitricoxide) eq 0 then nitricoxide = 1. else nitricoxide = nitricoxide
if n_elements(ozone) eq 0 then ozone = 1. else ozone = ozone

; read the model atmosphere
heightmodatm=readmodatm('HGT')
pressuremodatm=readmodatm('PRE')
temperaturemodatm=readmodatm('TEM')
o2modatm=readmodatm('O2')
co2modatm=readmodatm('CO2')
o3modatm=readmodatm('O3')
h2omodatm=readmodatm('H2O')
ch4modatm=readmodatm('CH4')
n2omodatm=readmodatm('N2O')
hno3modatm=readmodatm('HNO3')
comodatm=readmodatm('CO')
no2modatm=readmodatm('NO2')
nomodatm=readmodatm('NO')
nh3modatm=readmodatm('NH3')
ocsmodatm=readmodatm('OCS')
so2modatm=readmodatm('SO2')

; read the gdas data (only pressuregdas,heightgdas,temperaturegdas,dewpointgdas are used further)
readcol,gdasdata,pressuregdas,heightgdas,temperaturegdas,dewpointgdas,winddirgdas,windspeedgdas,skipline=6,/silent

; mix both atmospheric models
; how far goes GDAS
heightmaxgdas=max(heightgdas)/1000.  ; GDAS data in m, MIPAS profile in km
; where do we switch to MIPAS model
heightswitch=where(heightmodatm gt heightmaxgdas)
heightswitch=heightswitch[0]
; number of levels to build
levels=50
; create arrays for everyone
height=fltarr(levels)
pressure=fltarr(levels)
temperature=fltarr(levels)
o2=fltarr(levels)
co2=fltarr(levels)
o3=fltarr(levels)
h2o=fltarr(levels)
ch4=fltarr(levels)
n2o=fltarr(levels)
hno3=fltarr(levels)
co=fltarr(levels)
no2=fltarr(levels)
no=fltarr(levels)
nh3=fltarr(levels)
ocs=fltarr(levels)
so2=fltarr(levels)
; fill the arrays
; first use GDAS height, pressure, temperature and interpolate abundances of MIPAS to that grid
for i=0,n_elements(heightgdas)-1 do begin
    height[i]=heightgdas[i]/1000.  ; GDAS data in m, MIPAS profile in km
    pressure[i]=pressuregdas[i]
    temperature[i]=temperaturegdas[i]+273.15  ; GDAS in °C, LBLRTM expects K
    o2[i]=interpol(o2modatm,heightmodatm,height[i],/spline)
    co2[i]=interpol(co2modatm,heightmodatm,height[i],/spline)
    o3[i]=interpol(o3modatm,heightmodatm,height[i],/spline)
    h2o[i]=interpol(h2omodatm,heightmodatm,height[i],/spline)
    ch4[i]=interpol(ch4modatm,heightmodatm,height[i],/spline)
    n2o[i]=interpol(n2omodatm,heightmodatm,height[i],/spline)
    hno3[i]=interpol(hno3modatm,heightmodatm,height[i],/spline)
    co[i]=interpol(comodatm,heightmodatm,height[i],/spline)
    no2[i]=interpol(no2modatm,heightmodatm,height[i],/spline)
    no[i]=interpol(nomodatm,heightmodatm,height[i],/spline)
    nh3[i]=interpol(nh3modatm,heightmodatm,height[i],/spline)
    ocs[i]=interpol(ocsmodatm,heightmodatm,height[i],/spline)
    so2[i]=interpol(so2modatm,heightmodatm,height[i],/spline)
endfor
; next add the MIPAS model as is up to index 41, i.e. use 1km spacing
for i=n_elements(heightgdas),41 do begin
    j=heightswitch+i-n_elements(heightgdas)  ; index in MIPAS model, first is 27 km as GDAS usually ends somewhere between 26 and 27 km
    height[i]=heightmodatm[j]
    pressure[i]=pressuremodatm[j]
    temperature[i]=temperaturemodatm[j]
    o2[i]=o2modatm[j]
    co2[i]=co2modatm[j]
    o3[i]=o3modatm[j]
    h2o[i]=h2omodatm[j]
    ch4[i]=ch4modatm[j]
    n2o[i]=n2omodatm[j]
    hno3[i]=hno3modatm[j]
    co[i]=comodatm[j]
    no2[i]=no2modatm[j]
    no[i]=nomodatm[j]
    nh3[i]=nh3modatm[j]
    ocs[i]=ocsmodatm[j]
    so2[i]=so2modatm[j]
endfor
; next add levels in 5 km spacing up to 88 km
for i=42,49 do begin
    height[i]=heightmodatm[j+(i-41)*5]
    pressure[i]=pressuremodatm[j+(i-41)*5]
    temperature[i]=temperaturemodatm[j+(i-41)*5]
    o2[i]=o2modatm[j+(i-41)*5]
    co2[i]=co2modatm[j+(i-41)*5]
    o3[i]=o3modatm[j+(i-41)*5]
    h2o[i]=h2omodatm[j+(i-41)*5]
    ch4[i]=ch4modatm[j+(i-41)*5]
    n2o[i]=n2omodatm[j+(i-41)*5]
    hno3[i]=hno3modatm[j+(i-41)*5]
    co[i]=comodatm[j+(i-41)*5]
    no2[i]=no2modatm[j+(i-41)*5]
    no[i]=nomodatm[j+(i-41)*5]
    nh3[i]=nh3modatm[j+(i-41)*5]
    ocs[i]=ocsmodatm[j+(i-41)*5]
    so2[i]=so2modatm[j+(i-41)*5]
endfor
; scale abundances by factor
o2=o2*oxygen
co2=co2*carbondioxide
o3=o3*ozone
h2o=h2o*water
; if you change water abundance dewpoint temperature also changes
abundchange=0.4343*alog(water)
dewpointchange=0.
if abundchange ne 0 then dewpointchange=273.3/(8.286/abundchange-1.)  ; in °C
dewpointgdas=dewpointgdas+dewpointchange
ch4=ch4*methane
no=no*nitricoxide
nh3=nh3*ammonia

; set remaining parameters needed for TAPE5
wnstart=1.d8/waveend  ; start wavenumber in cm^-1, waveend in AA
wnend=1.d8/wavestart  ; end wavenumber in cm^-1, wavestart in AA
if wnend-wnstart ge 2018. then begin
    print,'ATTENTION: wnstart-wnend must be < 2020 cm^-1'
    retall
endif
zenithangle=90.-obsaltitude  ; in degree
null=0.

; create TAPE5
openw,unit,'TAPE5',/get_lun
; record 1.1
printf,unit,'$ TAPE5 created by runlblrtm.pro'
; record 1.2
printf,unit,'    1    1    1    0    1    0    0    0    0    1    0    0    0    0    5    5'
; record 1.3
printf,unit,format='(F10.3,F10.3,A)',wnstart-1,wnend+1,'     4.000 0.000E-00     0.000   0.0000 2.000E-04 1.000E-03'
; record 1.4
printf,unit,'     0.000     0.000     0.000     0.000     0.000     0.000     0.000'
; record 3.1
printf,unit,'    0    3    0    0    0   19    0          0.000    75.000     0.000   000.000'
; record 3.2
printf,unit,format='(F10.5,F10.5,F10.5,F10.5,F10.5,A)',obselevation,null,zenithangle,null,null,'    0'
; record 3.3A
printf,unit,'     2.000     5.000     8.000     0.000     0.000'
; user supplied atmospheric profile
; record 3.4
printf,unit,format='(I5)',levels
; use GDAS dewpoint for H2O first levels
for i=0,n_elements(heightgdas)-1 do begin
    ; record 3.5
    printf,unit,format='(E10.3,E10.3,E10.3,A)',height[i],pressure[i],temperature[i],'     AA   GAAAAAAAAAA'
    ; record 3.6.1
    printf,unit,format='(8E10.3)',dewpointgdas[i],co2[i],o3[i],n2o[i],co[i],ch4[i],o2[i],no[i]
    ; record 3.6.2
    printf,unit,format='(4E10.3)',so2[i],no2[i],nh3[i],hno3[i]
    ; record 3.6.3
    printf,unit,format='(3E10.3)',null,null,ocs[i]
endfor
; MIPAS model for the remaining ones
for i=n_elements(heightgdas),levels-1 do begin
    ; record 3.5 
    printf,unit,format='(E10.3,E10.3,E10.3,A)',height[i],pressure[i],temperature[i],'     AA   AAAAAAAAAAA'
    ; record 3.6.1
    printf,unit,format='(8E10.3)',h2o[i],co2[i],o3[i],n2o[i],co[i],ch4[i],o2[i],no[i]
    ; record 3.6.2
    printf,unit,format='(4E10.3)',so2[i],no2[i],nh3[i],hno3[i]
    ; record 3.6.3
    printf,unit,format='(3E10.3)',null,null,ocs[i]
endfor
; signal end of repitition
printf,unit,'-1.'
; make LBLRTM write ASCII of transmittance
; record 1.1
printf,unit,'$ Make ASCII'
; record 1.2 only activate plotting
printf,unit,' HI=0 F4=0 CN=0 AE=0 EM=0 SC=0 FI=0 PL=1 TS=0 AM=0 MG=0 LA=0 MS=0 XS=0    0    0'
; printf,unit,'    0    0    0    0    0    0    0    1    0    0    0    0    0    0    0    0'
; record 12.1
printf,unit,'# Plot title not used'
; record 12.2A
printf,unit,format='(F10.3,F10.3,A)',wnstart,wnend,'   10.2000  001.0000    1    0   12    0     1.000 0  0    0'
; record 12.3A
printf,unit,'    0.0000    1.2000    7.0200    0.2000    4    0    1    0    0    0 1    3 99'
; terminate plotting
printf,unit,'-1.'
; signal end of TAPE5 and terminate LBLRTM
printf,unit,'%'
; close TAPE5
free_lun,unit

; run LBLRTM, intercept messages by LBLRTM if asked to be silent
if not keyword_set(silent) then spawn,executeablename,result else spawn,executeablename,result,err

; read the transmittance
rdfloat,'TAPE99',wavenumber,transmittance,skipline=40,/double,/silent
wavetransmission=reverse(1.d8/wavenumber)
transmission=reverse(transmittance)

; clean up folder
if not keyword_set(keep) then begin
    spawn,'rm -f TAPE5'
    spawn,'rm -f TAPE6'
    spawn,'rm -f TAPE9'
    spawn,'rm -f TAPE10'
    spawn,'rm -f TAPE11'
    spawn,'rm -f TAPE12'
    spawn,'rm -f TAPE99'
endif else begin
;   message,/info,'TAPE files produced have not been removed! Remember to move them before next run or they will be overwritten!'
endelse
end
function statusinterpreter,n

;+
; NAME:
;	STATUSINTERPRETER
;
; PURPOSE:
;	This function maps the status code of MPFITFUN to a text
;	explaining its meaning.
;
; CALLING SEQUENCE:
;	text = STATUSINTERPRETER(Status)
;
; INPUTS:
;	Status:	The integer status code returned by MPFITFUN.
;
; OUTPUTS:
;	This function returns a string explaining the meaning of the status
;	code MPFITFUN set.
;
; EXAMPLE:
;       textmeaning = STATUSINTERPRETER(1)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; map the interger status code of MPFITFUN to its text explanation
case n of
	0: return,'Failure: improper input parameters'
	1: return,'both actual and predicted relative reductions in the sum of squares are at most FTOL'
	2: return,'relative error between two consecutive iterates is at most XTOL'
	3: return,'conditions for STATUS = 1 and STATUS = 2 both hold, i.e. both actual and predicted relative reductions in the sum of squares are at most FTOL and relative error between two consecutive iterates is at most XTOL'
	4: return,'the cosine of the angle between fvec and any column of the jacobian is at most GTOL in absolute value.'
	5: return,'the maximum number of iterations has been reached'
	6: return,'FTOL is too small. no further reduction in the sum of squares is possible.'
	7: return,'XTOL is too small. no further improvement in the approximate solution x is possible.'
	8: return,'GTOL is too small. fvec is orthogonal to the columns of the jacobian to machine precision.'
else: return,'Unknown status condition'
endcase
end
function subzero,inputarray
;+
; NAME:
;	SUBZERO
;
; PURPOSE:
;	This function replaces zero or subzero elements of the input array
;	by the mean of the neighbouring points.
;
; CALLING SEQUENCE:
;	Result = SUBZERO(Inputarray)
;
; INPUTS:
;	Inputarray:   Array to be treated.
;
; OUTPUTS:
;	This function returns the array containing no zero or subzero elements.
;
; EXAMPLE:
;       outputarray = subzero(inputarray)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; does the array have (sub-)zero points
x=inputarray
ind=where(x le 0)
; if yes replace them by the mean of the neighbouring points
if not (n_elements(ind) eq 1 and ind[0] eq -1) then begin
    for i=0,n_elements(ind)-1 do begin
        ; point is first of array
        if ind[i] eq 0 then begin
	    x[ind[i]]=x[ind[i]+1]
	    continue
	endif
	; point is last of array
        if ind[i] eq (n_elements(inputarray)-1) then begin
	    x[ind[i]]=x[ind[i]-1]
	    continue
	endif
        x[ind[i]]=(x[ind[i]-1]+x[ind[i]+1])/2
    endfor
endif
return,x
end
function tell,wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent
;+
; NAME:
;	TELL
;
; PURPOSE:
;	This function calculates a model transmission spectrum with the abundances
;	stated and adjusts it by convolving with a Gaussian, shifting in
;	wavelength and multiplying by a straight line so that it can be compared
;	to an observed spectrum.
;
; CALLING SEQUENCE:
;	model = TELL(Wavelength,Parameters,Gdasdata=Gdasdata,Obsaltitude=Obsaltitude)
;
; INPUTS:
;	Wavelength:   Wavelength array in AA for which model is requested
;	Parameters:   Array of parameters of model ([smoothing_FWHM,velocity_shift,
;		      slope_of_straight,intercept_of_straight,water_abundance,
;		      methane_abundance,carbondioxide_abundance])
;
; REQUIRED KEYWORD PARAMETERS:
;	GDASDATA:     Scalar string containing absolute path to GDAS file
;	OBSALTITUDE:  Altitude angle observation was taken at in deg
;
; KEYWORD PARAMETERS:
;	SILENT:	      Set to 1 if you do not want to get informational messages.
;
; OUTPUTS:
;	This function returns an array of the same size as wavelength containing
;	the transmission spectrum modelled and adjusted according to the parameters.
;
;
; COMMON BLOCKS:
;       MERKEN:       This common block saves the parameters and results to avoid
;		      having to run LBLRTM to often.
;
; EXAMPLE:
;       mod = TELL(wave_array,[0.5,0.1,-5.8,2.,1.0,1.0,1.0],gdasdata='/here/lies/the/Gdasdata',
;	obsaltitude=82.2)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 2 then begin
    message,/info,'Syntax: model=tell(wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent)'
    retall
endif
if not keyword_set(gdasdata) then begin
    print,'Keyword GDASDATA not set. This has to be set!'
    retall
endif
if not keyword_set(obsaltitude) then begin
    print,'Keyword OBSALTITUDE not set. This has to be set!'
    retall
endif

; make sure silent keyword has a value
if not keyword_set(silent) then silent=0

;common block to cache lblrtm results
common merken,par_old,range_old,w_old,t_old


; parameter assignment (general spectrum adjustment, needed below)
width=p[0]
v=p[1]
b=p[2]
slope=p[3]	; parameters should be of equivalent magnitude (see MPFIT FAQ)
curve=p[4]

; determine range model is needed in
range=minmax(wave)
; convert to vacuum
; airtovac,range

; these are the abundances for this run
par_new= p[5:7]

; cache results of lblrtm
; if abundances did not change compared to last call, just return cached results
if ((n_elements(range_old) ne 0) and (n_elements(par_old) ne 0)) && (array_equal(range,range_old) and array_equal(par_new,par_old)) then begin
    w = w_old
    t = t_old
endif else begin
    ; parameter assignment (abundances)
    wat=p[5]
    meth=p[6]
    co2=p[7]
    
    ; run lblrtm for this set up (extend range by 10 AA to avoid edge effects)
    runlblrtm,range[0]-10.,range[1]+10.,gdasdata,obsaltitude,w,t,water=wat,methane=meth,carbondioxide=co2,silent=silent

    ; return wavelength array to air
    ; vactoair,w

    ; keep the result to make further calls faster
    w_old = w
    t_old = t
    range_old = range
    par_old = par_new
endelse

; convolve high res model spectrum with gaussian to get to instrument resolution
tsm=gaussconv(w,t,width)
;tsm=boxconv(w,t,width)
;tsm=lorentzconv(w,t,width)
; apply shift in wavelength
c=299792.458d
wm=w*(1.-v/c)
; rebin to wavelength scale of data
tsmrb=interpol(tsm,wm,wave)
; multiply by straight line to adjust to data
dwave=wave-avg(wave)
straight=curve*dwave^2+slope*dwave + b
ymod=tsmrb*straight
; return the model
return,ymod
end
function tellremparameters,objects,plotname=plotname,noplot=noplot,nosave=nosave,savname=savname,silent=silent
;+
; NAME:
;	TELLREMPARAMETERS
;
; PURPOSE:
;	This function determines the parameters needed to create a telluric
;	line model for the individual observations.
;
; CALLING SEQUENCE:
;	Result = TELLREMPARAMETERS(Objects)
;
; INPUTS:
;	Objects:   String array containing the object names that are to be
;		   treated, i.e. the names of the folders that contain the
;		   data of the individual objects. 
;	
; KEYWORD PARAMETERS:
;	PLOTNAME:  String containing the name of the file containing the plots.
;		   It is named plotnameobjectname.ps. Default is 'tellrem'.
;	NOPLOT:	   Set this keyword if you do not want the fit results to be
;		   plotted.
;	NOSAVE:	   Set this keyword if you do not want the parameters to be
;		   saved into an IDL .sav file.
;	SAVNAME:   String containing name of the .sav file. It will be called
;		   savname.sav, default is 'tellremparams'.
;	SILENT:	   Set this keyword if you do not want informational message
;		   about the progress to printed to the terminal.
;
; OUTPUTS:
;	This function returns an array of structures containing the parameters
;	fitted to the spectrum and additional information for each object.
;	For each object the structure consists of the tags 'object' containing
;	the object name in a string, 'obsdate' containing the JD observation
;	date as double scalar, 'obsaltitude' containing the altitude angle the
;	observation was taken at as double scalar, 'exptimev' containing the
;	exposure time in the VIS arm as double scalar, 'exptimen' containing the
;	exposure time in the NIR arm as double scalar as well as 'abundance_h2o', 
;	'abundance_co2', 'abundance_ch4', and 'abundance_o2', that are structures
;	containing the fit results of the individual regions used to determine the
;	abundances of the main telluric contributors H2O, CO2, CH4, and O2,
;	constructed like 'para' (see below).
;	Additionally, it contains the 55-element array of structures 'para'. 
;	The individual structures consist of the tags 'range' containing the start
;	and end of the region the parameters stored in 'p', an 8-element double
;	array, were determined for. 'pe', an 8-element double array too, contains
;	the errors of those. The parameters are [smoothing_FWHM,velocity_shift,
;	slope_of_straight, intercept_of_straight, water_abundance, 
;	methane_abundance, carbondioxide_abundance, oxygen_abundance] used in TELL.
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;	params=TELLREMPARAMETERS(['Object1','Object2','Object3']
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, August 2014, Changed extraction of keyword for altitude of
;	     observation
;	N. Rudolf, November 2014, Added tags to the tellremparameters structure to save
;	the fit results of the individual regions used to determine the abundances of 
;	the main telluric molecules
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = tellremparameters(objects)'
    retall
endif

; info message
if not keyword_set(silent) then begin
    print,'Started tellremparameters: ',systime()
    anfang=systime(/julian)
    silent=0
    verb=1
endif else verb=0

; structure to save parameters and additional information
abund={range:dblarr(2),p:dblarr(8),pe:dblarr(8)}
para={range:dblarr(2),p:dblarr(8),pe:dblarr(8)}
anz=15  ; number of wavelength segments
entry={object:'',obsdate:0.d,obsaltitude:0.d,exptimen:0.d, $
	abundance_h2o:replicate(abund,1), $
	abundance_co2:replicate(abund,2), $
	abundance_ch4:replicate(abund,3), $
	para:replicate(para,anz)}
tellremparams=replicate(entry,n_elements(objects))

; set up plotting
if not keyword_set(noplot) then begin
    set_plot,'ps'
    !P.MULTI=[0,1,2]
    plot=1
endif else plot=0
if keyword_set(plotname) then nameplot=plotname else nameplot='tellrem_'

; loop over each object
for i=0,n_elements(objects)-1 do begin
    ; info message
    if not keyword_set(silent) then print,objects[i]
    ; name of object
    tellremparams[i].object=objects[i]
    ; read data
    rddat,spectrafolder+'/'+obstype+'.fits',wn,sn,en,hn
    ; check that there are no subzero datapoints
    sn=subzero(sn)
    ; if the error of NIR is 0 somewhere use median error
    if n_elements(where(en eq 0)) ne 1 then en[where(en eq 0)]=median(en)
    ; extract and save exposure times
    exptimen=sxpar(hn,'EXPTIME')
    tellremparams[i].exptimen=exptimen
    ; extract and save obsaltitude
    a=stregex(hn,'HIERARCH ESO TEL ALT')   ; find keyword, sxpar does not work as it only compares the first 8 characters
    indaltkw=where(a ne -1)
    indaltitude=indaltkw[0]
    h_alt=hn[indaltitude]
    alt=strsplit(strmid(h_alt,strpos(h_alt,'=')+1),' ',/extract)
;   alt=sxpar(hn,'ALT')
    obsaltitude=round(double(alt[0])*10.)/10.
    tellremparams[i].obsaltitude=obsaltitude
    ; extract and save obsdate
    obsdate=sxpar(hn,'MJD-OBS')+2400000.5d
    tellremparams[i].obsdate=obsdate
    ; determine GDAS sounding file to use
    gdasdata=whichgdas(obsdate)
    
    ; open plot
    if not keyword_set(noplot) then begin
	filename=nameplot+objects[i]+'.ps'
	device,/color,filename=filename
    endif

    ; determine the abundances of H2O, CO2, and CH4 in this observation
    getabundances,wn,sn,en,gdasdata,obsaltitude,ab_h2o,ab_co2,ab_ch4, $
	    ab_h2o_str,ab_co2_str,ab_ch4_str,silent=silent,plot=plot
    tellremparams[i].abundance_h2o=ab_h2o_str
    tellremparams[i].abundance_co2=ab_co2_str
    tellremparams[i].abundance_ch4=ab_ch4_str
    
    ; Parinfo structure for the overall fit
    parstructure=create_standard_parinfo()
    ; wat, i.e. abundance factor for water
    parstructure[5].value=ab_h2o
    ; meth, i.e. abundance factor for methane
    parstructure[6].value=ab_ch4
    ; co2, i.e. abundance factor for carbondioxide
    parstructure[7].value=ab_co2
    
    ; fit of telluric lines
    if not keyword_set(silent) then print,'Begin actual fit:'
    for k=1,15 do begin
	  if not keyword_set(silent) then print,'Region '+strtrim(k,1)
	  anf=19900.d + (k-1)*300.d
	  ende=19900.d + k*300.d
	  fittell,wn,sn,en,[anf,ende],gdasdata,obsaltitude,p,pe,wr,r,cl, $
		  parinfo=parstructure,/silent,plot=plot, $
		  titleplot='Region '+strtrim(k,1)+':'
	  tellremparams[i].para[k-1].range=[anf,ende]
	  tellremparams[i].para[k-1].p=p
	  tellremparams[i].para[k-1].pe=pe
    endfor
    ; no 300 AA left before stop at 24500 AA so set region individually
    if 0 then begin
    if not keyword_set(silent) then print,'Region 16'
    anf=24400.d
    ende=24500.d
    fittell,wn,sn,en,[anf,ende],gdasdata,obsaltitude,p,pe,wr,r,cl, $
	    parinfo=parstructure,/silent,plot=plot
    tellremparams[i].para[15].range=[anf,ende]
    tellremparams[i].para[15].p=p
    tellremparams[i].para[15].pe=pe
    endif

    ; plotting things
    if not keyword_set(noplot) then begin
	device,/close
	idlplottitlechanger,filename
    endif
endfor

; plotting things
if not keyword_set(noplot) then begin
    !P.MULTI=0
    set_plot,'x'
endif

; save the parameters
if not keyword_set(nosave) then begin
    if keyword_set(savname) then fn=savname+'.sav' else fn='tellrem_params.sav'
    save,tellremparams,description='Parameters for telluric line removal',filename=fn,verbose=verb
endif

; info messages
if not keyword_set(silent) then begin
    print,'Done: ',systime()
    ende=systime(/julian)
    dauer=(ende-anfang)*24
    print,format='(A19,F5.2,A2)','Total time needed: ',dauer,' h'
endif
; return the result
return,tellremparams
end
pro tellrem,infofile,telluricremovedspectra,silent=silent,nosave=nosave,noplot=noplot
;+
; NAME:
;	TELLREM
;
; PURPOSE:
;	This procedures creates telluric line removed spectra using the
;	information specified by the info file.
;
; CALLING SEQUENCE:
;	TELLREM,Infofile,Telluricremovedspectra
;
; INPUTS:
;	Infofile:  Absolute path to file containing the necessary
;		  information to run the procedures of the
;		  tellrem package. An example file called
;		  'info_for_tellrem' you can adjust for
;		  your needs is provided with the package. The
;		  format must be adhered to.
;	
; KEYWORD PARAMETERS:
;	NOSAVE:	Set this keyword if you do not want the results to be saved
;		to an IDL .sav file.
;	NOPLOT:	Set this keyword if you do not want plots of the results to
;		be created.
;
; OUTPUTS:
;	This procedure returns an array of structures 'Telluricremovedspectra'
;	containing the telluric line removed spectra and additional information
;	for each object. For each object the structure consists of the tags
;	'object' containing the object name in a string, 'obsdate' containing
;	the JD observation date as double scalar, 'obsaltitude' containing the
;	altitude angle the observation was taken at as double scalar, 'exptimeu'
;	containing the exposure time in the UVB arm as double scalar, 'exptimev'
;	containing the exposure time in the VIS arm as double scalar, 'exptimen'
;	containing the exposure time in the NIR arm as double scalar, 'wclu'
;	containing the wavelength array for the UVB data, 'clu' containing the
;	flux array of UVB, 'eclu' the error array of the UVB flux, 'wclv'
;	containing the wavelength array for the VIS data, 'clv' containing the
;	flux array of VIS, 'eclv' the error array of the VIS flux, 'wcln1'
;	containing the wavelength array for the NIR data in region 1, 'cln1'
;	containing the flux array of NIR in region 1, 'ecln1' the error array of
;	the NIR flux in region 1, 'wcln2' containing the wavelength array for the
;	NIR data in region 2, 'cln2' containing the flux array of NIR in region 2,
;	'ecln2' the error array of the NIR flux in region 2, 'wcln3' containing
;	the wavelength array for the NIR data in region 3, 'cln3' containing the
;	flux array of NIR in region 3, 'ecln3' the error array of the NIR flux in
;	region 3. The NIR arm spectrum is divided into 3 parts omitting the heavily
;	absorbed parts.
;
; EXAMPLE:
;	TELLREM,'/here/lies/info_for_tellrem',Telluricremovedspectra
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
;if n_params() lt 2 then begin
;    message,/info,'Syntax: TELLREM,Infofile,Telluricremovedspectra'
;    retall
;endif
if n_elements(infofile) eq 0 then infofile='tellrem.info'
; check keywords
if not keyword_set(nosave) then nosave=0
if not keyword_set(noplot) then noplot=0
if not keyword_set(silent) then silent=0

; get the info needed to run tellrem, i.e. create the common block TELLREM_INFO
loadtellreminfo,infofile

; get the objects to work on
objects=getobjects()

; determine parameters needed for the removal of telluric lines
tellremparams=tellremparameters(objects,nosave=nosave,noplot=noplot,silent=silent)

; do telluric line removal and build cleaned spectra
telluricremovedspectra=tellremspectra(tellremparams,nosave=nosave,silent=silent)
end
function tellremspectra,tellremparameters,silent=silent,savname=savname,nosave=nosave
;+
; NAME:
;	TELLREMSPECTRA
;
; PURPOSE:
;	This function takes the parameters determined by TELLREMPARAMETERS
;	and uses the modelled telluric line spectrum to remove the telluric
;	lines from the observed spectrum.
;
; CALLING SEQUENCE:
;	Result = TELLREMSPECTRA(Tellremparameters)
;
; INPUTS:
;	Tellremparameters:   The array of structures returned by
;			     TELLREMPARAMETERS.
;	
; KEYWORD PARAMETERS:
;	SILENT:		     Set this keyword if you do not want
;			     informational message about the progress printed
;			     to the terminal.
;	SAVNAME:	     String containing name of the .sav file. It will
;			     be called savname.sav, default is 'spectra_tellrem'.
;	NOSAVE:		     Set this keyword if you do not want the resulting
;			     spectra to be saved to an IDL .sav file.
;
; OUTPUTS:
;	This function returns an array of structures containing the telluric
;	line removed spectra and additional information for each object.
;	For each object the structure consists of the tags 'object' containing
;	the object name in a string, 'obsdate' containing the JD observation
;	date as double scalar, 'obsaltitude' containing the altitude angle the
;	observation was taken at as double scalar, 'exptimeu' containing the
;	exposure time in the UVB arm as double scalar, 'exptimev' containing the
;	exposure time in the VIS arm as double scalar, 'exptimen' containing the
;	exposure time in the NIR arm as double scalar, 'wclu' containing the
;	wavelength array for the UVB data, 'clu' containing the flux array of
;	UVB, 'eclu' the error array of the UVB flux, 'wclv' containing the
;	wavelength array for the VIS data, 'clv' containing the flux array of
;	VIS, 'eclv' the error array of the VIS flux, 'wcln1' containing the
;	wavelength array for the NIR data in region 1, 'cln1' containing the
;	flux array of NIR in region 1, 'ecln1' the error array of the NIR flux
;	in region 1, 'wcln2' containing the wavelength array for the NIR data in
;	region 2, 'cln2' containing the flux array of NIR in region 2, 'ecln2' the
;	error array of the NIR flux in region 2, 'wcln3' containing the wavelength
;	array for the NIR data in region 3, 'cln3' containing the flux array of
;	NIR in region 3, 'ecln3' the error array of the NIR flux in region 3.
;	The NIR arm spectrum is divided into 3 parts omitting the heavily absorbed
;	parts.
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;       tellremspecs = TELLREMSPECTRA(Tellremparams)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = tellremspectra(tellremparameters)'
    retall
endif

; first run to get element numbers
; info messages
if not keyword_set(silent) then begin
    print,'Started tellremspectra...'
    print,tellremparameters[0].object
    print,systime()
endif

; read data
rddat,spectrafolder+'/'+obstype+'.fits',wn,sn,en,hn

; get GDAS profile to use
gdas=whichgdas(tellremparameters[0].obsdate)
exptimen=sxpar(hn,'EXPTIME')

; build telluric line removed spectra
;
; NIR
for k=0,14 do begin
    recrfittellres,wn,sn,en,tellremparameters[0].para[k].range,gdas,tellremparameters[0].obsaltitude,tellremparameters[0].para[k].p,wr,r,cl,ecl,/silent
    if k eq 0 then wcln3=wr else wcln3=[wcln3,wr]
    if k eq 0 then cln3=cl else cln3=[cln3,cl]
    if k eq 0 then ecln3=ecl else ecln3=[ecln3,ecl]
endfor

; build entry for structure
entry={object:'', obsdate:0.d, obsaltitude:0.d, exptimen:0.d, wcln3:dblarr(n_elements(wcln3)), cln3:dblarr(n_elements(cln3)), ecln3:dblarr(n_elements(ecln3))}

;build structure
tellremspecs=replicate(entry,n_elements(tellremparameters))

; fill in the 0th index
tellremspecs[0].object=tellremparameters[0].object
tellremspecs[0].obsdate=tellremparameters[0].obsdate
tellremspecs[0].obsaltitude=tellremparameters[0].obsaltitude
tellremspecs[0].exptimen=tellremparameters[0].exptimen
tellremspecs[0].wcln3=wcln3
tellremspecs[0].cln3=cln3
tellremspecs[0].ecln3=ecln3

; do for the rest
if n_elements(tellremparameters) gt 1 then begin
    for i=1,n_elements(tellremparameters)-1 do begin
	; info messages
	if not keyword_set(silent) then begin
	    print,tellremparameters[i].object
	    print,systime()
	endif
	; read data
;	rddat,spectrafolder+tellremparameters[i].object+'/'+obstype+'_SLIT_MERGE1D_NIR.fits',wn,sn,en
	rddat,spectrafolder+'/'+obstype+'.fits',wn,sn,en,hn
	; get GDAS profile to use
	gdas=whichgdas(tellremparameters[i].obsdate)
	exptimen=sxpar(hn,'EXPTIME')

	; build telluric line removed spectra
	for k=0,14 do begin
	    recrfittellres,wn,sn,en,tellremparameters[i].para[k].range,gdas,tellremparameters[i].obsaltitude, tellremparameters[i].para[k].p,wr,r,cl,ecl,/silent
	    if k eq 0 then wcln3=wr else wcln3=[wcln3,wr]
	    if k eq 0 then cln3=cl else cln3=[cln3,cl]
	    if k eq 0 then ecln3=ecl else ecln3=[ecln3,ecl]
	endfor

	; fill in the structure
	tellremspecs[i].object=tellremparameters[i].object
	tellremspecs[i].obsdate=tellremparameters[i].obsdate
	tellremspecs[i].obsaltitude=tellremparameters[i].obsaltitude
	tellremspecs[i].exptimen=tellremparameters[i].exptimen
	tellremspecs[i].wcln3=wcln3
	tellremspecs[i].cln3=cln3
	tellremspecs[i].ecln3=ecln3
    endfor
endif
if not keyword_set(nosave) then begin
    if keyword_set(savname) then namesav=savname else namesav='tellrem_spectra'
    if keyword_set(silent) then quiet=0 else quiet=1
    save,tellremspecs,description='Telluric line removed spectra',filename=namesav+'.sav',verbose=quiet
endif
if not keyword_set(silent) then begin
	print,'Finished: ',systime()
endif
return,tellremspecs
end
pro tellrem_test_run,packagefolder,executablename,mipasmodel
;+
; NAME:
;	TELLREM_TEST_RUN
;
; PURPOSE:
;	This procedure perfoms a test run of the tellrem package using
;	exemplary data provided in the package to check whether the 
;	installation of the related software was sucessful.
;
; CALLING SEQUENCE:
;	TELLREM_TEST_RUN, Packagefolder, Executablename, Mipasmodel
;
; INPUTS:
;	Packagefolder:	String containing the path to where the tellrempackage
;			folder is located.
;	Executablename:	String containing the name of LBLRTM executable.
;	Mipasmodel:	String containing the absolute path to the MIPAS model
;			atmosphere.
;
; OUTPUTS:
;	None. But creates plot file Tellrem_test_run_comparison.ps comparing the
;	freshly determined telluric line removed spectrum to the spectrum
;	provided in the package.
;
; EXAMPLE:
;	TELLREM_TEST_RUN, '/the/tellrem/package/is/here/', 'lblrtm_executable',
;	'/MIPAS/model/is/here/equ.atm'
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, December 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-

; print informational message
print,'This is a test run of the tellrem package.'
; check whether folder path has / at the end
; CHU if strmid(packagefolder,0,1,/reverse_offset) ne '/' then packagefolder=packagefolder+'/*'
; create info_for_tellrem file from input to run test
openw,unit,'test_info',/get_lun
; first line GDAS
printf,unit,packagefolder+'/example_files/Test_Run/GDAS/'
; second line MIPAS model
printf,unit,mipasmodel
; third line executable name
printf,unit,executablename
; fourth line folder containing data
printf,unit,packagefolder+'/example_files/Test_Run/Test_data/'
; fifth line obstype
printf,unit,'SCI'
; close info_for_tellrem file
free_lun,unit

; get the current folder
spawn,'pwd',currentfolder

; get the info needed to run tellrem, i.e. create the common block TELLREM_INFO
loadtellreminfo,currentfolder+'/test_info'

; get the objects to work on
objects=getobjects()
; determine parameters needed for the removal of telluric lines
tellremparams=tellremparameters(objects,plotname='tellrem_test_run_',savname='testrun_params')
; do telluric line removal and build cleaned spectra
testspec=tellremspectra(tellremparams,savname='testrun_spectra')

; create plots enabling comparison between provided result and freshly calculated result
; restore provided result (variable 'spec')
restore,packagefolder+'example_files/Test_Run/tellrem_test_run_spectra.sav',/verbose

; prepare plotting
set_plot,'ps'
filename='Tellrem_test_run_comparison.ps'
device,/color,filename=filename
; plot arms together and their difference
!p.multi=[0,1,2]
; UVB
plot,spec.wclu,spec.clu,title='Comparison UVB - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B)
oplot,testspec.wclu,testspec.clu,col=235
; check shape of difference to determine plot range
diffu=spec.clu-testspec.clu
minmaxu=minmax(diffu)
meanu=mean(diffu)
if meanu eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxu[0]-0.1*minmaxu[0]
    y2=minmaxu[1]+0.1*minmaxu[1]
endelse
plot,spec.wclu,diffu,title='Difference between provided and newly calculated UVB',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst

; VIS
plot,spec.wclv,spec.clv,title='Comparison VIS - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[5400,10300],/xst,xtickformat='(I5)'
oplot,testspec.wclv,testspec.clv,col=235
; check shape of difference to determine plot range
diffv=spec.clv-testspec.clv
minmaxv=minmax(diffv)
meanv=mean(diffv)
if meanv eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxv[0]-0.1*minmaxv[0]
    y2=minmaxv[1]+0.1*minmaxv[1]
endelse
plot,spec.wclv,diffv,title='Difference between provided and newly calculated VIS',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[5400,10300],/xst,xtickformat='(I5)'

; NIR1
plot,spec.wcln1,spec.cln1,title='Comparison NIR1 - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[9900,13500],/xst,xtickformat='(I5)'
oplot,testspec.wcln1,testspec.cln1,col=235
; check shape of difference to determine plot range
diffn1=spec.cln1-testspec.cln1
minmaxn1=minmax(diffn1)
meann1=mean(diffn1)
if meann1 eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxn1[0]-0.1*minmaxn1[0]
    y2=minmaxn1[1]+0.1*minmaxn1[1]
endelse
plot,spec.wcln1,diffn1,title='Difference between provided and newly calculated NIR1',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[9900,13500],/xst,xtickformat='(I5)'

; NIR2
plot,spec.wcln2,spec.cln2,title='Comparison NIR2 - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[14400,18200],/xst,xtickformat='(I5)'
oplot,testspec.wcln2,testspec.cln2,col=235
; check shape of difference to determine plot range
diffn2=spec.cln2-testspec.cln2
minmaxn2=minmax(diffn2)
meann2=mean(diffn2)
if meann2 eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxn2[0]-0.1*minmaxn2[0]
    y2=minmaxn2[1]+0.1*minmaxn2[1]
endelse
plot,spec.wcln2,diffn2,title='Difference between provided and newly calculated NIR2',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[14400,18200],/xst,xtickformat='(I5)'

; NIR3
plot,spec.wcln3,spec.cln3,title='Comparison NIR3 - black=provided, red=new calculated',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),xra=[19500,24100],/xst,xtickformat='(I5)'
oplot,testspec.wcln3,testspec.cln3,col=235
; check shape of difference to determine plot range
diffn3=spec.cln3-testspec.cln3
minmaxn3=minmax(diffn3)
meann3=mean(diffn3)
if meann3 eq 0 then begin
    y1=-0.2
    y2=0.2
endif else begin
    y1=minmaxn3[0]-0.1*minmaxn3[0]
    y2=minmaxn3[1]+0.1*minmaxn3[1]
endelse
plot,spec.wcln3,diffn3,title='Difference between provided and newly calculated NIR3',ytitle='Flux in ADU',xtitle='Wavelength in '+STRING("305B),yra=[y1,y2],/yst,xra=[19500,24100],/xst,xtickformat='(I5)'
!p.multi=0
device,/close
idlplottitlechanger,filename
set_plot,'x'

print,'Tellrem test run complete. Take a look at the file Tellrem_test_run_comparison.ps created in this folder to compare the telluric line removed spectrum provided in the package and the one just produced from the same data.'
end
function tellwithout,wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent,worig=worig,torig=torig
;+
; NAME:
;	TELLWITHOUT
;
; PURPOSE:
;	This function calculates a model transmission spectrum with the abundances
;	stated and adjusts it by convolving with a Gaussian and shifting in
;	wavelength to enable its use to remove telluric lines from an observed
;	spectrum.
;
; CALLING SEQUENCE:
;	model = TELLWITHOUT(Wavelength,Parameters,GDASDATA=gdasdata,OBSALTITUDE=obsaltitude)
;
; INPUTS:
;	Wavelength:   Wavelength array in AA for which model is requested
;	Parameters:   Array of parameters of model ([smoothing_FWHM,velocity_shift,
;		      slope_of_straight,intercept_of_straight,water_abundance,
;		      methane_abundance,carbondioxide_abundance,oxygen_abundance])
;
; REQUIRED KEYWORD PARAMETERS:
;	GDASDATA:     Scalar string containing absolute path to GDAS file
;	OBSALTITUDE:  Altitude angle observation was taken at in deg
;
; KEYWORD PARAMETERS:
;	SILENT:	      Set to 1 if you do not want to get informational messages
;	WORIG:	      Wavelength array of unconvolved model
;	TORIG:	      Unconvolved model
;
; OUTPUTS:
;	This function returns an array of the same size as wavelength containing
;	the transmission spectrum modelled and adjusted according to the parameters
;	but without using the straight line that TELL used to fit the model to the data.
;
; EXAMPLE:
;       mod = TELLWITHOUT(wave_array,[0.5,0.1,-5.8,2.,1.0,1.0,1.0,1.0],gdasdata='/here/lies/the/Gdasdata',
;	obsaltitude=82.2)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 2 then begin
    message,/info,'Syntax: model=tell(wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent,worig=worig,torig=torig)'
    retall
endif
if not keyword_set(gdasdata) then begin
    print,'Keyword GDASDATA not set. This has to be set!'
    retall
endif
if not keyword_set(obsaltitude) then begin
    print,'Keyword OBSALTITUDE not set. This has to be set!'
    retall
endif

; make sure silent keyword has a value
if not keyword_set(silent) then silent=0

; parameter assignment
width=p[0]
v=p[1]
wat=p[5]
meth=p[6]
co2=p[7]

; determine range model is needed in
range=minmax(wave)
; convert to vacuum
; airtovac,range

; run lblrtm for this setup (extend range by 10 AA to avoid edge effects)
runlblrtm,range[0]-10.,range[1]+10.,gdasdata,obsaltitude,w,t,water=wat,methane=meth,carbondioxide=co2,silent=silent
; return wavelength array to air
; vactoair,w

; also return the unconvolved spectrum if asked for
if keyword_set(worig) then begin
    worig=w*(1.-v/c)
    torig=t
endif

; convolve high res model spectrum with gaussian to get to instrument resolution
tsm=gaussconv(w,t,width)
;tsm=boxconv(w,t,width)
;tsm=lorentzconv(w,t,width)
; apply shift in wavelength
c=299792.458d
wm=w*(1.-v/c)
; rebin to wavelength scale of data
tsmrb=interpol(tsm,wm,wave)
return,tsmrb
end
function whichgdas,obsdate
;+
; NAME:
;	WHICHGDAS
;
; PURPOSE:
;	This function selects the GDAS sounding data file that lies closest to the observation date.
;
; CALLING SEQUENCE:
;	Result = WHICHGDAS(Obsdate)
;
; INPUTS:
;	Obsdate:      Julian date of the observation (scalar float or double)

; OUTPUTS:
;	Result:       Scalar string with absolute path of the closest GDAS sounding data file
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;	filename = WHICHGDAS(2456590.9276736d)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = WHICHGDAS(Obsdate)'
    retall
endif

; check whether folder path has / at the end
if strmid(gdasfolder,0,1,/reverse_offset) ne '/' then gdasf=gdasfolder+'/*' else gdasf=gdasfolder+'*'
; get content of folder
gdasfiles=file_search(gdasf)
; create array to take the available souding data dates
gdasdates=dblarr(n_elements(gdasfiles))
; extract the dates from the filenames
for i=0,n_elements(gdasfiles)-1 do begin
    ; get the filename
    split=strsplit(gdasfiles[i],'/',/extract)
    filename=split[n_elements(split)-1]
    ; check whether it adheres to naming convention GDASYYYY_MM_DD_HH
    if strpos(filename,'.') ne -1 || strmid(filename,0,4) ne 'GDAS' then begin
	print,'The specified folder for the GDAS sounding files contains files that do not adhere to the naming convention!'
	print,'Please check correctness of stated folder and its content!'
	stop
	retall
    endif
    ; extract the date
    date=strsplit(filename,'_',/extract)
    date[0]=strmid(date[0],4)
    ; transform date to julian date
    gdasdates[i]=julday(date[1],date[2],date[0],date[3])
endfor
; which gdas is the closest in time to obsdate
thisone=where(abs(gdasdates-obsdate) eq min(abs(gdasdates-obsdate)))
file=gdasfiles[thisone]
return,file[0]
end
