function tellremspectra,tellremparameters,silent=silent,savname=savname,nosave=nosave
;+
; NAME:
;	TELLREMSPECTRA
;
; PURPOSE:
;	This function takes the parameters determined by TELLREMPARAMETERS
;	and uses the modelled telluric line spectrum to remove the telluric
;	lines from the observed spectrum.
;
; CALLING SEQUENCE:
;	Result = TELLREMSPECTRA(Tellremparameters)
;
; INPUTS:
;	Tellremparameters:   The array of structures returned by
;			     TELLREMPARAMETERS.
;	
; KEYWORD PARAMETERS:
;	SILENT:		     Set this keyword if you do not want
;			     informational message about the progress printed
;			     to the terminal.
;	SAVNAME:	     String containing name of the .sav file. It will
;			     be called savname.sav, default is 'spectra_tellrem'.
;	NOSAVE:		     Set this keyword if you do not want the resulting
;			     spectra to be saved to an IDL .sav file.
;
; OUTPUTS:
;	This function returns an array of structures containing the telluric
;	line removed spectra and additional information for each object.
;	For each object the structure consists of the tags 'object' containing
;	the object name in a string, 'obsdate' containing the JD observation
;	date as double scalar, 'obsaltitude' containing the altitude angle the
;	observation was taken at as double scalar, 'exptimeu' containing the
;	exposure time in the UVB arm as double scalar, 'exptimev' containing the
;	exposure time in the VIS arm as double scalar, 'exptimen' containing the
;	exposure time in the NIR arm as double scalar, 'wclu' containing the
;	wavelength array for the UVB data, 'clu' containing the flux array of
;	UVB, 'eclu' the error array of the UVB flux, 'wclv' containing the
;	wavelength array for the VIS data, 'clv' containing the flux array of
;	VIS, 'eclv' the error array of the VIS flux, 'wcln1' containing the
;	wavelength array for the NIR data in region 1, 'cln1' containing the
;	flux array of NIR in region 1, 'ecln1' the error array of the NIR flux
;	in region 1, 'wcln2' containing the wavelength array for the NIR data in
;	region 2, 'cln2' containing the flux array of NIR in region 2, 'ecln2' the
;	error array of the NIR flux in region 2, 'wcln3' containing the wavelength
;	array for the NIR data in region 3, 'cln3' containing the flux array of
;	NIR in region 3, 'ecln3' the error array of the NIR flux in region 3.
;	The NIR arm spectrum is divided into 3 parts omitting the heavily absorbed
;	parts.
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;       tellremspecs = TELLREMSPECTRA(Tellremparams)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = tellremspectra(tellremparameters)'
    retall
endif

; first run to get element numbers
; info messages
if not keyword_set(silent) then begin
    print,'Started tellremspectra...'
    print,tellremparameters[0].object
    print,systime()
endif

; read data
rddat,spectrafolder+'/'+obstype+'.fits',wn,sn,en,hn

; get GDAS profile to use
gdas=whichgdas(tellremparameters[0].obsdate)
exptimen=sxpar(hn,'EXPTIME')

; build telluric line removed spectra
;
; NIR
for k=0,14 do begin
    recrfittellres,wn,sn,en,tellremparameters[0].para[k].range,gdas,tellremparameters[0].obsaltitude,tellremparameters[0].para[k].p,wr,r,cl,ecl,/silent
    if k eq 0 then wcln3=wr else wcln3=[wcln3,wr]
    if k eq 0 then cln3=cl else cln3=[cln3,cl]
    if k eq 0 then ecln3=ecl else ecln3=[ecln3,ecl]
endfor

; build entry for structure
entry={object:'', obsdate:0.d, obsaltitude:0.d, exptimen:0.d, wcln3:dblarr(n_elements(wcln3)), cln3:dblarr(n_elements(cln3)), ecln3:dblarr(n_elements(ecln3))}

;build structure
tellremspecs=replicate(entry,n_elements(tellremparameters))

; fill in the 0th index
tellremspecs[0].object=tellremparameters[0].object
tellremspecs[0].obsdate=tellremparameters[0].obsdate
tellremspecs[0].obsaltitude=tellremparameters[0].obsaltitude
tellremspecs[0].exptimen=tellremparameters[0].exptimen
tellremspecs[0].wcln3=wcln3
tellremspecs[0].cln3=cln3
tellremspecs[0].ecln3=ecln3

; do for the rest
if n_elements(tellremparameters) gt 1 then begin
    for i=1,n_elements(tellremparameters)-1 do begin
	; info messages
	if not keyword_set(silent) then begin
	    print,tellremparameters[i].object
	    print,systime()
	endif
	; read data
;	rddat,spectrafolder+tellremparameters[i].object+'/'+obstype+'_SLIT_MERGE1D_NIR.fits',wn,sn,en
	rddat,spectrafolder+'/'+obstype+'.fits',wn,sn,en,hn
	; get GDAS profile to use
	gdas=whichgdas(tellremparameters[i].obsdate)
	exptimen=sxpar(hn,'EXPTIME')

	; build telluric line removed spectra
	for k=0,14 do begin
	    recrfittellres,wn,sn,en,tellremparameters[i].para[k].range,gdas,tellremparameters[i].obsaltitude, tellremparameters[i].para[k].p,wr,r,cl,ecl,/silent
	    if k eq 0 then wcln3=wr else wcln3=[wcln3,wr]
	    if k eq 0 then cln3=cl else cln3=[cln3,cl]
	    if k eq 0 then ecln3=ecl else ecln3=[ecln3,ecl]
	endfor

	; fill in the structure
	tellremspecs[i].object=tellremparameters[i].object
	tellremspecs[i].obsdate=tellremparameters[i].obsdate
	tellremspecs[i].obsaltitude=tellremparameters[i].obsaltitude
	tellremspecs[i].exptimen=tellremparameters[i].exptimen
	tellremspecs[i].wcln3=wcln3
	tellremspecs[i].cln3=cln3
	tellremspecs[i].ecln3=ecln3
    endfor
endif
if not keyword_set(nosave) then begin
    if keyword_set(savname) then namesav=savname else namesav='tellrem_spectra'
    if keyword_set(silent) then quiet=0 else quiet=1
    save,tellremspecs,description='Telluric line removed spectra',filename=namesav+'.sav',verbose=quiet
endif
if not keyword_set(silent) then begin
	print,'Finished: ',systime()
endif
return,tellremspecs
end
