function exclusion,wave,range,exclude

;+
; NAME:
;	EXCLUSION
;
; PURPOSE:
;	This function extracts from a wavelength array a certain range but without the
;	regions specified to exclude.
;
; CALLING SEQUENCE:
;	Result = EXCLUSION(Wavelength, Range, Exclude)
;
; INPUTS:
;	Wavelength:   Wavelength array from a range of which one wants to exclude certain
;		      regions.
;	Range:	      Wavelength range to extract from above array as two-element
;		      array in the form [start,end].
;	Exclude:      Regions to exclude from wavelength array in the range specified
;		      as array with up to 10 entries of the form [start1,end1,...,start5,end5].
;
; OUTPUTS:
;	This function returns an index array for the wavelength array that contains the
;	specified range but does not contain the regions stated in exclude.
;
; EXAMPLE:
;	indices = EXCLUSION(Wavelength, [6500.,6800.], [6558.,6568.])
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: induse=exclusion(wave,range,exclude)'
    retall
endif

; check that requested range lies within given wavelength array
if range[0] le min(wave) or range[0] ge max(wave) then begin
     print,'Requested range outside wavelength array'
     retall
endif
if range[1] le min(wave) or range[1] ge max(wave) then begin
     print,'Requested range outside wavelength array'
     retall
endif

; build a mask
fitmask=intarr(n_elements(wave))
; mark the elements that are in the range
ind=where(wave ge range[0] and wave le range[1])
fitmask[ind]=1
; check that the regions to exclude are in the range
for i=0,n_elements(exclude)-1 do begin
    if (exclude[i] le range[0] or exclude[i] ge range[1]) then begin
        print,'Borders chosen for exclusion not within chosen wavelength range'
        retall
    endif
endfor

case n_elements(exclude) of  ; how many exclusions are there
	2: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
	end
	4: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
	end
	6: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
		ind=where(wave ge exclude[4] and wave le exclude[5])
		fitmask[ind]=0
	end
	8: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
		ind=where(wave ge exclude[4] and wave le exclude[5])
		fitmask[ind]=0
		ind=where(wave ge exclude[6] and wave le exclude[7])
		fitmask[ind]=0
	end
	10: begin
		; unmark the elements in the exclusion region
		ind=where(wave ge exclude[0] and wave le exclude[1])
		fitmask[ind]=0
		ind=where(wave ge exclude[2] and wave le exclude[3])
		fitmask[ind]=0
		ind=where(wave ge exclude[4] and wave le exclude[5])
		fitmask[ind]=0
		ind=where(wave ge exclude[6] and wave le exclude[7])
		fitmask[ind]=0
		ind=where(wave ge exclude[8] and wave le exclude[9])
		fitmask[ind]=0
	end
else: begin
	print,'Number of exclusions not supported!'
	retall
end
endcase

; determine which indices to return
fitind=where(fitmask eq 1)
return,fitind
end