pro getabundances,wavenir,specnir,errnir,gdasdata,obsaltitude, $
	ab_h2o,ab_co2,ab_ch4,ab_h2o_str,ab_co2_str,ab_ch4_str, $
	silent=silent,plot=plot
;+
; NAME:
;	GETABUNDANCES
;
; PURPOSE:
;	This procedure determines the abundances of the main molecules (H2O,
;	CO2, CH4) producing telluric lines in the GRAVITY spectral range.
;
; CALLING SEQUENCE:
;	GETABUNDANCES, Wavenir, Specnir, Errnir,
;	Gdasdata, Obsaltitude, Ab_h2o, Ab_co2, Ab_ch4, Ab_h2o_str,
;	Ab_co2_str, Ab_ch4_str
;
; INPUTS:
;	Wavenir:	Wavelength array of NIR data
;	Specnir:	Flux array of NIR data
;	Errnir:		Error array of NIR data
;	Gdasdata:	Scalar string containing absolute path to GDAS file
;	Obsaltitude:	Altitude angle observation was taken at in deg
;	
; KEYWORD PARAMETERS:
;	SILENT:		Set this keyword to suppress informational messages
;	PLOT:		Set this keyword to get a plot of the result.
;
; OUTPUTS:
;	Ab_h2o:		Abundance of H2O in this observation
;	Ab_co2:		Abundance of CO2 in this observation
;	Ab_ch4:		Abundance of CH4 in this observation
;	Ab_h2o_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of H20
;	Ab_co2_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of CO2
;	Ab_ch4_str:	Structure containing the fit results for the
;			individual regions used to determine the abundance
;			of CH4
;
; EXAMPLE:
;	GETABUNDANCES, Wavenir, Specnir, Errnir,
;	'/here/lies/the/Gdasdata',87.2, Ab_h2o, Ab_co2, Ab_ch4,
;	Ab_h2o_str, Ab_co2_str, Ab_ch4_str
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, October 2014, Adapted parameter borders
;	N. Rudolf, November 2014, Added output to save the fit results of
;	the individual regions used to determine the abundances of the
;	main telluric contributors.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 5 then begin
    message,/info,'Syntax: GETABUNDANCES,wavenir,specnir,errnir,gdasdata,obsaltitude,ab_h2o,ab_co2,ab_ch4,ab_h2o_str,ab_co2_str,ab_ch4_str,silent=silent,plot=plot'
    retall
endif
if not keyword_set(plot) then plot=0

abund={range:dblarr(2),p:dblarr(7),pe:dblarr(7)}
ab_h2o_str=replicate(abund,1)
ab_co2_str=replicate(abund,2)
ab_ch4_str=replicate(abund,3)

if not keyword_set(silent) then print,'Fitting abundances:'

; set up parinfo structure for H2O
parw = create_standard_parinfo()
; wat, i.e. abundance factor for water
parw[4].fixed=0
if not keyword_set(silent) then print,'Fit H2O'
fittell,wavenir,specnir,errnir,[24100.,24500.],gdasdata,obsaltitude, $
	pw1,pew1,wrw1,rw1,clw1,parinfo=parw,/silent,plot=plot,titleplot='Fit H2O:'
ab_h2o_str[0].range=[24100.,24500.]
ab_h2o_str[0].p=pw1
ab_h2o_str[0].pe=pew1

; abundance used is mean of all abundances
; ab_h2o=mean([pw1[4],pw2[4],pw3[4],pw4[4],pw5[4]])
ab_h2o=pw1[4]

; set up parinfo structure for CO2
parco2 = create_standard_parinfo()
; wat, i.e. abundance factor for water
parco2[4].value=ab_h2o
; co2, i.e. abundance factor for carbondioxide
parco2[6].fixed=0
if not keyword_set(silent) then print,'Fit CO2'
fittell,wavenir,specnir,errnir,[19950.,20300.],gdasdata,obsaltitude, $
	pco21,peco21,wrco21,rco21,clco21,parinfo=parco2,/silent, $
	plot=plot,titleplot='Fit CO2:'
ab_co2_str[0].range=[19950.,20300.]
ab_co2_str[0].p=pco21
ab_co2_str[0].pe=peco21

fittell,wavenir,specnir,errnir,[20450.,20800.],gdasdata,obsaltitude, $
	pco22,peco22,wrco22,rco22,clco22,parinfo=parco2,/silent, $
	plot=plot,titleplot='Fit CO2:'
ab_co2_str[1].range=[20450.,20800.]
ab_co2_str[1].p=pco22
ab_co2_str[1].pe=peco22

; abundance used is mean of all abundances
ab_co2=mean([pco21[6],pco22[6]])

; set up parinfo structure for CH4
parch4 = create_standard_parinfo()
; wat, i.e. abundance factor for water
parch4[4].value=ab_h2o
; meth, i.e. abundance factor for methane
parch4[5].fixed=0
; co2, i.e. abundance factor for carbondioxide
parch4[6].value=ab_co2
if not keyword_set(silent) then print,'Fit CH4'
fittell,wavenir,specnir,errnir,[22850.,23110.],gdasdata,obsaltitude, $
	pch41,pech41,wrch41,rch41,clch41,parinfo=parch4,/silent, $
	plot=plot,titleplot='Fit CH4:'
ab_ch4_str[0].range=[22850.,23110.]
ab_ch4_str[0].p=pch41
ab_ch4_str[0].pe=pech41

fittell,wavenir,specnir,errnir,[23370.,23630.],gdasdata,obsaltitude, $
	pch42,pech42,wrch42,rch42,clch42,parinfo=parch4,/silent, $
	plot=plot,titleplot='Fit CH4:'
ab_ch4_str[1].range=[23370.,23630.]
ab_ch4_str[1].p=pch42
ab_ch4_str[1].pe=pech42

fittell,wavenir,specnir,errnir,[23630.,23900.],gdasdata,obsaltitude, $
	pch43,pech43,wrch43,rch43,clch43,parinfo=parch4,/silent, $
	plot=plot,titleplot='Fit CH4:'
ab_ch4_str[2].range=[23630.,23900.]
ab_ch4_str[2].p=pch43
ab_ch4_str[2].pe=pech43

; abundance used is mean of all abundances
ab_ch4=mean([pch41[5],pch42[5],pch43[5]])

end
