function gaussconv,x,y,fwhm
;+
; NAME:
;           GAUSSCONV
;
; PURPOSE:
;           This function smooths an array by convolving with a Gaussian profile.
;
; CALLING SEQUENCE:
;           smoothed_y = GAUSSCONV(X, Y, Fwhm)
;
; INPUTS:
;           X:	   Array (double or float) of the values of the x-axis in ascending
;		   order.
;           Y:	   Array (double or float) of same size as x of the y-axis values.
;           Fwhm:  FWHM in units of the x-axis of the Gaussian profile.
;
; OUTPUTS:
;           Array (double or float) of same size as x containing smoothed y.
;
; EXAMPLE:
;           flux_smoothed = GAUSSCONV(wavelength,flux,0.75)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 3 then begin
    message,/info,'Syntax: smoothed_y = GAUSSCONV(X, Y, Fwhm)'
    retall
endif

; start and end of x array
xmin=min(x)
xmax=max(x)

; oversampling
dx=fwhm/17.d
dx=fwhm/10.d
xint=xmin + dx*dindgen(long((xmax-xmin)/dx+1))
yint=interpol(y,x,xint)

; number of pixels for dimension -> npixel keyword of psf_gaussian
fwhm_pix=fwhm/dx
nopd=fix(17*fwhm_pix)
nopd=fix(10*fwhm_pix)

; 1D gaussian profile
gauss = psf_gaussian(npixel=nopd,fwhm=fwhm_pix,/normalize,ndimen=1,/double)

; convolve y with the gaussian profile
yconv=convol(yint,gauss,/center,/edge_truncate)
ysmooth=interpol(yconv,xint,x)

return,ysmooth
end
