function tellremparameters,objects,plotname=plotname,noplot=noplot,nosave=nosave,savname=savname,silent=silent
;+
; NAME:
;	TELLREMPARAMETERS
;
; PURPOSE:
;	This function determines the parameters needed to create a telluric
;	line model for the individual observations.
;
; CALLING SEQUENCE:
;	Result = TELLREMPARAMETERS(Objects)
;
; INPUTS:
;	Objects:   String array containing the object names that are to be
;		   treated, i.e. the names of the folders that contain the
;		   data of the individual objects. 
;	
; KEYWORD PARAMETERS:
;	PLOTNAME:  String containing the name of the file containing the plots.
;		   It is named plotnameobjectname.ps. Default is 'tellrem'.
;	NOPLOT:	   Set this keyword if you do not want the fit results to be
;		   plotted.
;	NOSAVE:	   Set this keyword if you do not want the parameters to be
;		   saved into an IDL .sav file.
;	SAVNAME:   String containing name of the .sav file. It will be called
;		   savname.sav, default is 'tellremparams'.
;	SILENT:	   Set this keyword if you do not want informational message
;		   about the progress to printed to the terminal.
;
; OUTPUTS:
;	This function returns an array of structures containing the parameters
;	fitted to the spectrum and additional information for each object.
;	For each object the structure consists of the tags 'object' containing
;	the object name in a string, 'obsdate' containing the JD observation
;	date as double scalar, 'obsaltitude' containing the altitude angle the
;	observation was taken at as double scalar, 'exptimev' containing the
;	exposure time in the VIS arm as double scalar, 'exptimen' containing the
;	exposure time in the NIR arm as double scalar as well as 'abundance_h2o', 
;	'abundance_co2', 'abundance_ch4', and 'abundance_o2', that are structures
;	containing the fit results of the individual regions used to determine the
;	abundances of the main telluric contributors H2O, CO2, CH4, and O2,
;	constructed like 'para' (see below).
;	Additionally, it contains the 55-element array of structures 'para'. 
;	The individual structures consist of the tags 'range' containing the start
;	and end of the region the parameters stored in 'p', an 8-element double
;	array, were determined for. 'pe', an 8-element double array too, contains
;	the errors of those. The parameters are [smoothing_FWHM,velocity_shift,
;	slope_of_straight, intercept_of_straight, water_abundance, 
;	methane_abundance, carbondioxide_abundance, oxygen_abundance] used in TELL.
;
; COMMON BLOCKS:
;       TELLREM_INFO:  This common block contains relevant folder names and strings
;	              for running tellrem. It has to be initialised by running
;		      LOADTELLREMINFO.
;
; EXAMPLE:
;	params=TELLREMPARAMETERS(['Object1','Object2','Object3']
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;	N. Rudolf, August 2014, Changed extraction of keyword for altitude of
;	     observation
;	N. Rudolf, November 2014, Added tags to the tellremparameters structure to save
;	the fit results of the individual regions used to determine the abundances of 
;	the main telluric molecules
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
common tellrem_info,gdasfolder,modelatmosphere,executeablename,spectrafolder,obstype

; check the input parameters
if n_params() lt 1 then begin
    message,/info,'Syntax: Result = tellremparameters(objects)'
    retall
endif

; info message
if not keyword_set(silent) then begin
    print,'Started tellremparameters: ',systime()
    anfang=systime(/julian)
    silent=0
    verb=1
endif else verb=0

; structure to save parameters and additional information
abund={range:dblarr(2),p:dblarr(5),pe:dblarr(5)}
para={range:dblarr(2),p:dblarr(5),pe:dblarr(5)}
anz=2  ; number of wavelength segments
entry={object:'',obsdate:0.d,obsaltitude:0.d,exptimen:0.d, $
	abundance_h2o:replicate(abund,2), $
	para:replicate(para,anz)}
tellremparams=replicate(entry,n_elements(objects))

; set up plotting
if not keyword_set(noplot) then begin
    set_plot,'ps'
    !P.MULTI=[0,1,2]
    plot=1
endif else plot=0
if keyword_set(plotname) then nameplot=plotname else nameplot='tellrem'

; loop over each object
for i=0,n_elements(objects)-1 do begin
    ; info message
    if not keyword_set(silent) then print,objects[i]
    ; name of object
    tellremparams[i].object=objects[i]
    ; read data
    rddat,spectrafolder+'/'+obstype+'.fits',wn,sn,en,hn
    ; check that there are no subzero datapoints
    sn=subzero(sn)
    ; if the error of NIR is 0 somewhere use median error
    if n_elements(where(en eq 0)) ne 1 then en[where(en eq 0)]=median(en)
    ; extract and save exposure times
    exptimen=sxpar(hn,'EXPTIME')
    tellremparams[i].exptimen=exptimen
    ; extract and save obsaltitude
    a=stregex(hn,'HIERARCH ESO TEL ALT')   ; find keyword, sxpar does not work as it only compares the first 8 characters
    indaltkw=where(a ne -1)
    indaltitude=indaltkw[0]
    h_alt=hn[indaltitude]
    alt=strsplit(strmid(h_alt,strpos(h_alt,'=')+1),' ',/extract)
;   alt=sxpar(hn,'ALT')
    obsaltitude=round(double(alt[0])*10.)/10.
    tellremparams[i].obsaltitude=obsaltitude
    ; extract and save obsdate
    obsdate=sxpar(hn,'MJD-OBS')+2400000.5d
    tellremparams[i].obsdate=obsdate
    ; determine GDAS sounding file to use
    gdasdata=whichgdas(obsdate)
    
    ; open plot
    if not keyword_set(noplot) then begin
	filename=nameplot+objects[i]+'.ps'
	device,/color,filename=filename
    endif

    ; determine the abundances of H2O, CO2, and CH4 in this observation
    getabundances,wn,sn,en,gdasdata,obsaltitude,ab_h2o, $
	    ab_h2o_str,silent=silent,plot=plot
    tellremparams[i].abundance_h2o=ab_h2o_str
    help,ab_h2o
    print,ab_h2o
    
    ; Parinfo structure for the overall fit
    parstructure = replicate({value:0.D, fixed:0, limited:[0,0], $
	    limits:[0.D,0],step:0.d},5)
    ; smooth, i.e. width of gaussian
    parstructure[0].value=5.d
    parstructure[0].fixed=0
    parstructure[0].limited[0]=1
    parstructure[0].limited[1]=1
    parstructure[0].limits[0]=5.
    parstructure[0].limits[1]=20.d  
    parstructure[0].step=0.1d
    ; v, i.e. shift in wavelength
    parstructure[1].value=0
    parstructure[1].fixed=0
    parstructure[1].limited[0]=1
    parstructure[1].limited[1]=1
    parstructure[1].limits[0]=-90.d
    parstructure[1].limits[1]=50.d 
    parstructure[1].step=1.d
    ; slope, i.e. slope of straigth line that is used to adapt to slope of observed spectrum
    parstructure[2].value=0
    parstructure[2].fixed=1
    ; b, i.e. y-intercept of straigth line that is used to adapt to slope of observed spectrum
    parstructure[3].value=1.d
    parstructure[3].fixed=0
    parstructure[3].limited[0]=1
    parstructure[3].limited[1]=1
    parstructure[3].limits[0]=0.8
    parstructure[3].limits[1]=1.2
    parstructure[3].step=0.05d
    ; wat, i.e. abundance factor for water
    parstructure[4].value=ab_h2o
    parstructure[4].fixed=1
    
    ; fit of telluric lines
    if not keyword_set(silent) then print,'Begin actual fit:'
    for k=1,1 do begin
	  if not keyword_set(silent) then print,'Region '+strtrim(k,1)
	  anf=24100.d + (k-1)*300.d
	  ende=24100.d + k*300.d
	  fittell,wn,sn,en,[anf,ende],gdasdata,obsaltitude,p,pe,wr,r,cl, $
		  parinfo=parstructure,/silent,plot=plot
	  tellremparams[i].para[k-1].range=[anf,ende]
	  tellremparams[i].para[k-1].p=p
	  tellremparams[i].para[k-1].pe=pe
    endfor
    ; no 300 AA left before stop at 24500 AA so set region individually
    if not keyword_set(silent) then print,'Region 2'
    anf=24400.d
    ende=24500.d
    fittell,wn,sn,en,[anf,ende],gdasdata,obsaltitude,p,pe,wr,r,cl, $
	    parinfo=parstructure,/silent,plot=plot
    tellremparams[i].para[1].range=[anf,ende]
    tellremparams[i].para[1].p=p
    tellremparams[i].para[1].pe=pe

    ; plotting things
    if not keyword_set(noplot) then begin
	device,/close
	idlplottitlechanger,filename
    endif
endfor

; plotting things
if not keyword_set(noplot) then begin
    !P.MULTI=0
    set_plot,'x'
endif

; save the parameters
if not keyword_set(nosave) then begin
    if keyword_set(savname) then fn=savname+'.sav' else fn='tellremparams.sav'
    save,tellremparams,description='Parameters for telluric line removal',filename=fn,verbose=verb
endif

; info messages
if not keyword_set(silent) then begin
    print,'Done: ',systime()
    ende=systime(/julian)
    dauer=(ende-anfang)*24
    print,format='(A19,F5.2,A2)','Total time needed: ',dauer,' h'
endif
; return the result
return,tellremparams
end
