function tellwithout,wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent,worig=worig,torig=torig
;+
; NAME:
;	TELLWITHOUT
;
; PURPOSE:
;	This function calculates a model transmission spectrum with the abundances
;	stated and adjusts it by convolving with a Gaussian and shifting in
;	wavelength to enable its use to remove telluric lines from an observed
;	spectrum.
;
; CALLING SEQUENCE:
;	model = TELLWITHOUT(Wavelength,Parameters,GDASDATA=gdasdata,OBSALTITUDE=obsaltitude)
;
; INPUTS:
;	Wavelength:   Wavelength array in AA for which model is requested
;	Parameters:   Array of parameters of model ([smoothing_FWHM,velocity_shift,
;		      slope_of_straight,intercept_of_straight,water_abundance,
;		      methane_abundance,carbondioxide_abundance,oxygen_abundance])
;
; REQUIRED KEYWORD PARAMETERS:
;	GDASDATA:     Scalar string containing absolute path to GDAS file
;	OBSALTITUDE:  Altitude angle observation was taken at in deg
;
; KEYWORD PARAMETERS:
;	SILENT:	      Set to 1 if you do not want to get informational messages
;	WORIG:	      Wavelength array of unconvolved model
;	TORIG:	      Unconvolved model
;
; OUTPUTS:
;	This function returns an array of the same size as wavelength containing
;	the transmission spectrum modelled and adjusted according to the parameters
;	but without using the straight line that TELL used to fit the model to the data.
;
; EXAMPLE:
;       mod = TELLWITHOUT(wave_array,[0.5,0.1,-5.8,2.,1.0,1.0,1.0,1.0],gdasdata='/here/lies/the/Gdasdata',
;	obsaltitude=82.2)
;
; MODIFICATION HISTORY:
; 	Written by:	Natascha Rudolf, October 2013.
;-
; Copyright (C) 2013 Natascha Rudolf
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the "Software"), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
; The above copyright notice and this permission notice shall be included in all
; copies or substantial portions of the Software.
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.
;-
; check the input parameters
if n_params() lt 2 then begin
    message,/info,'Syntax: model=tell(wave,p,gdasdata=gdasdata,obsaltitude=obsaltitude,silent=silent,worig=worig,torig=torig)'
    retall
endif
if not keyword_set(gdasdata) then begin
    print,'Keyword GDASDATA not set. This has to be set!'
    retall
endif
if not keyword_set(obsaltitude) then begin
    print,'Keyword OBSALTITUDE not set. This has to be set!'
    retall
endif

; make sure silent keyword has a value
if not keyword_set(silent) then silent=0

; parameter assignment
smooth=p[0]
v=p[1]
wat=p[4]

; determine range model is needed in
range=minmax(wave)
; convert to vacuum
; airtovac,range

; run lblrtm for this setup (extend range by 10 AA to avoid edge effects)
runlblrtm,range[0]-10.,range[1]+10.,gdasdata,obsaltitude,w,t,water=wat,silent=silent
; return wavelength array to air
; vactoair,w

; also return the unconvolved spectrum if asked for
if keyword_set(worig) then begin
    worig=w*(1.-v/c)
    torig=t
endif

; convolve high res model spectrum with gaussian to get to instrument resolution
tsm=gaussconv(w,t,smooth)
; apply shift in wavelength
c=299792.458d
wm=w*(1.-v/c)
; rebin to wavelength scale of data
tsmrb=interpol(tsm,wm,wave)
return,tsmrb
end
