/* 
 * This sets up all the ancillary stuff that needs be done only once for
 * the plots.  Specifically, it:
 *
 *   1) Determines whether you want to plot versus wavelength or channel
 *   number and sets up the abcissa.
 *
 *   2) Asks whether you want screen shots or a hardcopy plot and if the
 *   latter sets up the base name to store the files as a function of
 *   inclination.
 *
 *   3) Does the spade work for the UV coverage illustrations, and
 *
 *   4) Lays out the outline of the model for PA=0-deg so that the
 *   geometry of the UV coverage versus the PA of the object can be
 *   illustrated.
 */
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "roche.h"
#include "cpgplot.h"

/*
int InitPlot(Roche * R, Observatory * O, LimbDark * LD, Scans * S, 
	     Plot * P, int IfWave) {
*/
int InitPlot(Roche * R, Observatory * O, LimbDark * LD, Scans * S, 
	     Plot * P) {

  int i, j, k, iYN, iHA; 
  int iiPAnot=0, iaPAnot=0, iiPAdel=0, ifPAdel=0, iIfWave=0; 
  int iIfPlot=0, IfWave; 
  char YN[]=" ", sString[40], line[81];
  FILE *fp; 

  /* Calculate Outline */
  float thp, th, ph, rout, sth, cth, sph, cph, ci, si;

  /* See if a window or a plot is desired */
  iYN=0;
  while(iYN == 0) {
    iYN=1;
    printf("Want to (S)ee or (P)lot this: ");
    scanf("%s",YN);
    getchar();
    if (toupper(YN[0]) == 'P') {
      P->IfScreen=0; // File, not screen
      cpgldev(); // list the devices
      printf("Base File Name (ex: plot.ps/CPS): ");
      scanf("%s",sString);
      getchar();
      //printf("Got: %s\n",sString);
    }
    else if (toupper(YN[0]) == 'S') P->IfScreen=1; //plot in window
    else {
      printf("Nope, choices are (S)ee or (P)lot/n");
      iYN=0;
    }
    if (P->IfScreen) {
      strcpy(P->sBaseFile,""); 
      strcpy(P->sDevice,""); 
      strcpy(P->sExtension,""); 
    }
    if (!P->IfScreen) {
      if (strrchr(sString,'/') != NULL) {
	strcpy(P->sDevice,strrchr(sString,'/')); // Device Spec
	//printf("Device: %s\n",P->sDevice);
	strcpy(strrchr(sString,'/'),"\0"); // move string termination up
      }
      //printf("Device: %s  After removal: %s\n",P->sDevice,sString);
      if (strrchr(sString,'.') != NULL) {
	strcpy(P->sExtension,strrchr(sString,'.')); //  File Extension
	strcpy(strrchr(sString,'.'),"\0"); // move string termination up
      }
      strcpy(P->sBaseFile,sString); // rest in base file name
    }
    /*
    printf("Device=%s Extension=%s BaseFile=%s\n",P->sDevice,
	   P->sExtension,P->sBaseFile); // check it out
    */
  }

  /* Set up for plotting the UV coverage */
  for (iHA = 0; iHA < S->iNScans; iHA++) {
    if (GetUVW(O, R, S->afHA[iHA]) == EXIT_FAILURE) return EXIT_FAILURE;
    if (iHA==0) {
      P->fUVmax[0]=O->fUVW[0][0]; // initial window limits [0]=U
      P->fUVmin[0]=O->fUVW[0][0];
      P->fUVmax[1]=O->fUVW[0][1]; //[1]=V
      P->fUVmin[1]=O->fUVW[0][1];
    }
    for (j = 0; j < O->iBaselines; j++) {
      for (k = 0; k < 2; k++) {
	P->fSaveUV[iHA][j][k] = O->fUVW[j][k];
	if (O->fUVW[j][k] > P->fUVmax[k]) P->fUVmax[k] = O->fUVW[j][k];
	if (O->fUVW[j][k] < P->fUVmin[k]) P->fUVmin[k] = O->fUVW[j][k];
      }
    }
  }
  /*
    printf("Umax=%5.1f Umin=%5.1f Vmax=%5.1f Vmin=%5.1f\n",P->fUVmax[0],
    P->fUVmin[0],P->fUVmax[1],P->fUVmin[1]);
  */
  
  for (i = 0; i < 2; i++) { // make sure you get the extremes
    if (-P->fUVmin[i] > P->fUVmax[i]) P->fUVmax[i] = -P->fUVmin[i];
    if (-P->fUVmax[i] < P->fUVmin[i]) P->fUVmin[i] = -P->fUVmax[i];
  }
  
  /* Get model outline for PA=0. */
  ci = cos(R->fIncl);
  si = sin(R->fIncl);
  P->iOutline = 73; // every 5 degrees
  for (i = 0; i < P->iOutline-1; i++) {
    //printf("Picture: i=%1i thp=%1.6g\n", i, 5.*i);
    //thp=5.*i*(float)RADDEG;
    thp=5.*i*(float)RADDEG;
    if (Edge(R, thp, &th, &ph, &rout) == EXIT_FAILURE)
      return EXIT_FAILURE;
    //printf("Th=%1.6g Phi=%1.6g Rout=%1.6g\n",(float)DEGRAD * th, 
    //(float)DEGRAD * ph, rout);
    sth = sin(th);
    cth = cos(th);
    sph = sin(ph);
    cph = cos(ph);
    P->afxol[i] = rout*(sth*cph*ci-cth*si);
    P->afyol[i] = rout*sth*sph;
  }
  P->afxol[P->iOutline-1] = P->afxol[0];
  P->afyol[P->iOutline-1] = P->afyol[0];

  /* Read in the plotting parameters */

  /* See if plot file available */
  if ((fp = fopen("plot", "r")) != NULL) {
    while (fgets(line, 81, fp) != NULL) {
      //printf("line: %s",line);
      if (strstr(line,"iPAnot") != NULL) {
	strtok(line," ");
	P->iPAnot=atoi(strtok(NULL," "));
	iiPAnot=1; // got it
	//printf("sName = %s\n",sName);
      }
      if (strstr(line,"afPAnot") != NULL) {
	strtok(line," ");
	for (i=0; i<P->iPAnot; i++) {
	  P->afPAnot[i]=atof(strtok(NULL," "));
	}
	iaPAnot=1; // got it
	//printf("sName = %s\n",sName);
      }
      if (strstr(line,"iPAdel") != NULL) {
	strtok(line," ");
	P->iPAdel=atoi(strtok(NULL," "));
	iiPAdel=1; // got it
	//printf("sName = %s\n",sName);
      }
      if (strstr(line,"fPAdel") != NULL) {
	strtok(line," ");
	P->fPAdel=atof(strtok(NULL," "));
	ifPAdel=1; // got it
	//printf("sName = %s\n",sName);
      }
      if (strstr(line,"IfPlot") != NULL) {
	strtok(line," ");
	P->IfPlot=atoi(strtok(NULL," "));
	iIfPlot=1; // got it
	//printf("sName = %s\n",sName);
      }
      if (strstr(line,"IfWave") != NULL) {
	strtok(line," ");
	IfWave=atoi(strtok(NULL," "));
	iIfWave=1; // got it
	//printf("sName = %s\n",sName);
      }
    }
    if (!(iiPAnot*iaPAnot*iiPAdel*ifPAdel*iIfPlot*iIfWave)) {
      printf("All the plot variable not properly set:\n");
      printf("iiPAnot=%1i iPAnot=%1i\n",iiPAnot,P->iPAnot);
      printf("iaPAnot=%1i iIfWave=%1i\n",iaPAnot,iIfWave);
      for (i=0; i<P->iPAnot; i++) {
	printf("afPAnot[%1i]=%5g ",i,P->afPAnot[i]);
      }
      printf("\n");
      printf("iiPAdel=%1i iPAdel=%1i\n",iiPAdel,P->iPAdel);
      printf("ifPAnot=%1i fPAdel=%5g\n",ifPAdel,P->fPAdel);
      printf("iIfPlot=%1i IfPlot=%1i\n",iIfPlot,P->IfPlot);
      return EXIT_FAILURE;
    }

    /* Set up the plot abcissa, will only use j1->j2 of it */
    for (j = 0; j < LD->indexWaveMax; j++) {
      if (IfWave) {
	P->afAbcissa[j]=LD->afWaves[j]; // for wavelengths
	strcpy(P->sAbLabel,"\\gl(nm)");
      }
      if (!IfWave) {
	P->afAbcissa[j]=j; // for channels
	strcpy(P->sAbLabel,"Channels");
      }
    }
    P->fAbcmin = P->afAbcissa[P->j1Wave];
    P->fAbcmax = P->afAbcissa[P->j2Wave-1];
    if (P->fAbcmin > P->fAbcmax) {
      P->fAbcmax = P->afAbcissa[P->j1Wave];
      P->fAbcmin = P->afAbcissa[P->j2Wave-1];
    }

    iIfPlot = P->IfPlot;
    P->iPlotSum = iIfPlot - 2*(iIfPlot/2);
    iIfPlot = iIfPlot/2;
    P->iPlotClose = iIfPlot - 2*(iIfPlot/2);
    iIfPlot = iIfPlot/2;
    P->iPlotV2 = iIfPlot - 2*(iIfPlot/2);
    iIfPlot = iIfPlot/2;
    P->iPlotPhi = iIfPlot - 2*(iIfPlot/2);
    /*
      printf("iPlotSum=%i iPlotClose=%i iPlotV2=%i iPlotPhi=%i\n",
      iPlotSum,iPlotClose,iPlotV2,iPlotPhi);
    */

  }
  else {
    printf("Couldn't find plot parameter file!");
    return EXIT_FAILURE;
  }

  return EXIT_SUCCESS;
}
