/***********************************************************************
 * InitRoche: Brings in model parameters, sets up the wavelengths, and
 * loads the (model atmosphere) limbdarkening grid file.
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "roche.h"

int InitRoche(Roche * R, LimbDark * LD, char * name, int iargc, 
	      char * acargv[], int * j1Wave, int * j2Wave) {

  int indexWave, j, i;
  //int indexWaveMax=32;
  char cModel[9], sModelfile[20];

  /* Command line arguments:
     0: is the program name
     1: Model data set
     2: IndexWave (see description above)
  */

  /* Keep this for the moment.  The values get overridden in ReadLimbDark */
  float knot=.0011764706, dk=.000033733924;
  LD->indexWaveMax=32; 
  for (j = 0; j < LD->indexWaveMax; j++) {
    LD->afWaves[j] = 1./(knot + dk*j); // nm
  }  

  if (iargc > 1 ) {
    if (iargc > 1) strcpy(cModel,acargv[1]);
    if (iargc > 2) indexWave = atoi(acargv[2]) ;
  }
  if (iargc < 2) {
    printf("Model data file (leave the '.dat' off):  ");
    scanf("%s", cModel);
    /* And kill the next character, particularly if it is a newline */
    //getchar();
  }
  if (iargc < 3) {
    printf("Wavelength index(-), range(+), 500nm(0) or Wave(nm) if > %1i:  ",
	   LD->indexWaveMax);
    scanf("%i", &indexWave);
    /* And kill the next character, particularly if it is a newline */
    getchar();
  }
  strcpy(sModelfile, cModel);
  strcat(sModelfile, ".dat");
  printf("Data file for model: %s\n", sModelfile);
  
  
  /* The model file needs to set things next */
  if (ReadModel(R, LD, name, sModelfile) == EXIT_FAILURE) return EXIT_FAILURE;
  
  /* Set the wavelength indices */
  R->fWave=0.;
  *j1Wave=0;
  LD->iWave=0;
  if (abs(indexWave) <= LD->indexWaveMax) {
    /* Normal coding */ 
    if (indexWave > 0) {
      *j2Wave=indexWave;
    }
    else if (indexWave < 0) {
      *j1Wave=-indexWave;
      *j2Wave=*j1Wave+1;
    }
    else if (indexWave == 0) {
      /* Will set indices, wavelength, below */
      R->fWave = 500.; // 500nm
      strcpy(LD->sWaveset,"Mon"); // Not using an Instrument wavelength
    }
  }
  else {
    /* Assume the index is actually a wavelength (in nm).  Set indices 
       below. */
    R->fWave = fabs(indexWave);
    if (R->fWave <= 300. || R->fWave > 10000.) { // out of range, complain
      printf ("Monochromatic wavelength set limited to 300nm <= lambda");
      printf(" <= 10mu, sorry\n");
      return EXIT_FAILURE;
    }
    strcpy(LD->sWaveset,"Mon"); //  Not using an NPOI wavelength
    //printf("indexWave=%3i R->Wave=%5.0f\n",indexWave,R->fWave);
  }

  /* Load Limb Darkening file */
  if (ReadLimbDark(LD) == EXIT_FAILURE) return EXIT_FAILURE; 
  if (indexWave < 0) {
    R->fWave = LD->afWaves[-indexWave];
    LD->iWave = -indexWave;
  }
  /* Make certain a lower Max Wave index doesn't screw up things */
  if (*j1Wave == 0 && *j2Wave > LD->indexWaveMax) *j2Wave=LD->indexWaveMax;

  /* If monochromatic, set up wavelength interpolation */
  if (!strcmp(LD->sWaveset,"Mon")) {
    /*
      printf("sWaveset:%s  indexWave:%2i  fWave:%5.0f  indexWaveMax:%2i\n",
	 LD->sWaveset,indexWave,R->fWave,LD->indexWaveMax);
    */
    i=-1;
    while (R->fWave > LD->afWaves[++i] && i < LD->indexWaveMax-1) {
      /*
      printf("i:%2i  afWaves[i]:%4.0f  afWaves[i+1]:%4.0f  fWave:%4.0f\n",
	     i,LD->afWaves[i],LD->afWaves[i+1],R->fWave);
      */
      LD->fWaveWtMinus = (R->fWave - LD->afWaves[i+1]) / (LD->afWaves[i] - 
						       LD->afWaves[i+1]);
      *j1Wave=i;
      *j2Wave=i+1;
      LD->iWave=i;
      //i++;
    }
    /*
    printf("Wave_1=%5.0f Wt=%4.2f Wave_2=%5.0f WT=%4.2f Wave=%6.0f",
	   LD->afWaves[*j1Wave],LD->fWaveWtMinus,
	   LD->afWaves[*j2Wave],1.-LD->fWaveWtMinus,
	   R->fWave);
    printf(" j1Wave=%2i j2Wave=%2i i=%2i\n",*j1Wave,*j2Wave,i);
    return EXIT_FAILURE;
    */
  }

  return EXIT_SUCCESS;
}
