/***********************************************************************
 * This draws lines of constant co-latitude (theta) and constant
 * longitude (phi) on the spheroid.  Assumes PGPLOT has already been
 * opened. 
 *
 * This now adds an offset, dPh, that can be used to make the star
 * appear to rotate...
 *
 * This also adds a Position Angle offset so that it can be canted, as
 * seen projected on the sky.  The plot scale (mas/polar breakup radius)
 * is provided to plot on a physical scale.
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>

#include "roche.h"
#include "cpgplot.h"

int LatLong(Roche * R, float dPh, float ThPA, float rscale) {

  int i, j, ic, ii, il1, il2, ilim1, ilim2, iphilim;
  float thlim1, thlim2, philim, r0, r180, dth, dcph, dphi;
  float th, sth, sth2, cth, cth2, phi, sph, cph, r, rc, rs, a, w=R->fwOmega; 
  float xp[400], yp[400], w2=w*w, incl=R->fIncl, ci=cos(incl), si=sin(incl); 
  float sPA = sin(ThPA), cPA = cos(ThPA), x, y;

  /* i=0 */
  /* There is no Position Angle offset in this case */
  if (incl == 0.) { 
    /* Loci of Theta=const first. */
    for (i = 1; i < 9; i++) {
      th=10.*i*(float)RADDEG;
      sth=sin(th);
      sth2 = sth*sth;
      r=FractRadius(w2, sth2);
      cth=cos(th);
      cth2=cth*cth;
      rs=r*sth;
      rc=r*cth;
      /* Doesn't need transforming of Theta offsets (or anything). */
      for (j = 0; j < 361; j++) {
	phi=(float)RADDEG*j;
	xp[j] = -(rs*ci*cos(phi)-rc*si)*rscale;
	yp[j] = -rs*sin(phi)*rscale;
      }
      cpgline(361, yp, xp);
    }
    /* Loci for Phi=const next. */
    for (i = 0; i < 37; i++) {
      phi=10.*i*(float)RADDEG-dPh;
      cph = cos(phi);
      sph = sin(phi);
      for (j = 0; j < 91; j++) {
	th=j*(float)RADDEG;
	sth = sin(th);
	sth2 = sth*sth;
	r=FractRadius(w2, sth2);
	cth = cos(th);
	cth2 = cth*cth;
	rs=r*sth;
	rc=r*cth;
	xp[j] = -(rs*ci*cph-rc*si)*rscale;
	yp[j] = -rs*sph*rscale;
      }
      cpgline(91, yp, xp);
    }
  }

  /* i=90 */
  else if (fabs(incl-(float)PI2) <=1.e-5) { 
    /* Loci of Theta=const first. */
    for (i = 1; i < 18; i++) {
      th=10.*i*(float)RADDEG;
      sth=sin(th);
      sth2 = sth*sth;
      r=FractRadius(w2, sth2);
      cth=cos(th);
      rs=r*sth;
      rc=r*cth;
      for (j = 0; j < 181; j++) {
	phi=(float)RADDEG*(j-90);
	x = -(rs*ci*cos(phi)-rc*si);
	y = -rs*sin(phi);
	xp[j] = (cPA*x-sPA*y)*rscale;
	yp[j] = (sPA*x+cPA*y)*rscale;
      }
      cpgline(181, yp, xp); 
    }
    /* Loci for Phi=const next. */
    for (i = 0; i < 37; i++) {
      phi=10.*(i-9)*(float)RADDEG-dPh;
      while (phi < -(float)TWOPI) {
	phi += (float)TWOPI;
      }
      while (phi > (float)TWOPI) {
	phi -= (float)TWOPI;
      }
      if (phi > -(float)PI2 && phi <= (float)PI2) {
	cph = cos(phi);
	sph = sin(phi);
	for (j = 0; j < 181; j++) {
	  th=j*(float)RADDEG;
	  sth = sin(th);
	  sth2 = sth*sth;
	  r=FractRadius(w2, sth2);
	  cth = cos(th);
	  rs=r*sth;
	  rc=r*cth;
	  x = -(rs*ci*cph-rc*si);
	  y = -rs*sph;
	  xp[j] = (cPA*x-sPA*y)*rscale;
	  yp[j] = (sPA*x+cPA*y)*rscale;
	}
      cpgline(181, yp, xp); 
      }
    }
  } // finish i=pi/2

  /* up to thlim1 you have full co-latitude circles.  After thlim2 there
     are no segements even of co-latitude circles. */

  else {
    if (Edge(R, (float)PI, &thlim1, &phi, &r180) == EXIT_FAILURE)
      return EXIT_FAILURE;
    if (Edge(R, 0., &thlim2, &phi, &r0) == EXIT_FAILURE)
      return EXIT_FAILURE;
    il1 = abs((int)(thlim1/10.*(float)DEGRAD));
    il2 = abs((int)(thlim2/10.*(float)DEGRAD));
    ilim1=il1;
    ilim2=il2;
    if (il1 > il2) {
      ilim1=il2;
      ilim2=il1;
    }
    //printf("ilim1=%i ilim2=%i\n",ilim1,ilim2);

    if (ilim1 > 0) { 
      /* Some full circles */
      for (i = 1; i <= ilim1; i++) {
	th=10.*i*(float)RADDEG;
	sth=sin(th);
	sth2 = sth*sth;
	r=FractRadius(w2, sth2);
	cth=cos(th);
	rs=r*sth;
	rc=r*cth;
	for (j = 0; j < 361; j++) {
	  phi=(float)RADDEG*(j-90);
	  x = -(rs*ci*cos(phi)-rc*si);
	  y = -rs*sin(phi);
	  if (R->fIncl < 0.) y = -y;
	  xp[j] = (cPA*x-sPA*y)*rscale;
	  yp[j] = (sPA*x+cPA*y)*rscale;
	}
	cpgline(361, yp, xp); 
      }
    }
    if (ilim2 > ilim1) {
      /* Some partial co-latitude circles */
      for (i = ilim1+1; i <= ilim2; i++) {
	//for (i = ilim2+1; i <= ilim1; i++) {
	th=10.*i*(float)RADDEG;
	sth=sin(th);
	sth2 = sth*sth;
	r=FractRadius(w2, sth2);
	cth=cos(th);
	rs=r*sth;
	rc=r*cth;
	a=8./27.*w2*r*r*r;
	philim=fabs(acos(cth/sth*ci/si/(a-1.))*(float)DEGRAD);
	iphilim=(int)philim;
	dphi=philim/iphilim*(float)RADDEG;
	for (j = -iphilim; j <= iphilim; j++) {
	  phi=dphi*j;
	  x = -(rs*ci*cos(phi)-rc*si);
	  if (R->fIncl < 0.) x = -x;
	  y = -rs*sin(phi);
	  xp[j+iphilim] = (cPA*x-sPA*y)*rscale;
	  yp[j+iphilim] = (sPA*x+cPA*y)*rscale;
	}
	//cout << "i=" << i << " iphilim=" << iphilim << endl;
	cpgline(iphilim*2+1, yp, xp); 
      }
      //cout << endl;
    }


    for (i = -18; i < 18; i++) {
      phi=10.*i*(float)RADDEG-dPh;
      //cout << "phi=" << phi*(float)DEGRAD;
      cph = cos(phi);
      sph = sin(phi);

      /* iterate for point at edge */
      r=FractRadius(w2, 1.); // first guess
      a=8./27.*w2*r*r*r; // ditto
      ii=0;
      dcph=1.;
      while (fabs(dcph) > 1.e-5 && ii < 100) {
	th=atan(1./(cph*si/ci*(a-1.)));
	sth=sin(th);
	sth2=sth*sth;
	r=FractRadius(w2, sth2);
	a=8./27.*w2*r*r*r;
	cth = cos(th);
	dcph=cph-(1./(tan(th)*si/ci*(a-1.)));
	ii++;
      }
      if (th < 0.) th += (float)PI;
      //printf(" ii=%1i a=%1.6g", ii, a);

      ic=(int)(th*(float)DEGRAD);
      dth=th/(ic+1);
      //printf(" ic=%1i thc=\n", ic, th*(float)DEGRAD);
      for (j = 0; j < ic; j++) {
	th=(j+1)*(float)RADDEG;
	sth = sin(th);
	sth2 = sth*sth;
	r=FractRadius(w2, sth2);
	a=8./27.*w2*r*r*r;
	cth = cos(th);
	rs=r*sth;
	rc=r*cth;
	x = -(rs*ci*cph-rc*si);
	y = -rs*sph;
	xp[j] = (cPA*x-sPA*y)*rscale;
	yp[j] = (sPA*x+cPA*y)*rscale;
      }
      cpgline(ic, yp, xp); // rotate 90

    }
  }
  return EXIT_SUCCESS;
}
