/* 
 * This reads in the Van Hamme (AJ 1993) tables and generates
 * Monochromatic Limb Darkening coefficients for Kurucz, Solar
 * composition models.  The Monochromatic Wavelength Set is used.
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

int main(int iargc, char *acargv[]) {

  float afwaves[32], affilternorm[32],oneow2, A;
  char swavesetname[30];
  int iwaves=24, i, j, aiwaves[32];
  char line[81];
  float teff, glog; float waveb, wavee;
  float wavem, linear, xlog, ylog, xsq, ysq, I1;
  float wavemb, linearb, xlogb, ylogb, xsqb, ysqb, I1b;
  float swavem, slinear, sxlog, sylog, sxsq, sysq, sI1, fI1; int sicount;
  int imodel, iread, jmodel, kmodel;
  int iwm, iwmb, iwme, iw, iwb, iwe; 
  float wtm, wtp, Beff, wave, weff, IoB;
  FILE *fp, *fp1, *fp3c, *fpoutln, *fpoutlg, *fpoutsq;

  /* These are for purely monochromatic wavelengths.  At the present we
   * use a set of 24, but the limitation is 32.
   */

  if ((fp = fopen("monochromatic.dat", "r")) != NULL) {
    if (fgets(swavesetname, 30, fp) == NULL) {
      printf("Waveset File Empty!  Bailing out.\n");
      return EXIT_FAILURE;
    }
    j=strcspn(swavesetname,"\n");
    swavesetname[j]='\0';
    printf("Waveset: %30s\n", swavesetname);

    for (i = 0; i < iwaves; i++) {
      fscanf(fp, "%3i%9f", &j, &afwaves[i]);
      //printf("i=%3i j=%3i wave=%6.1f\n", i, j, afwaves[i]);
      if (i != j-1) {
	printf("wavelengths out of order!\n");
	return EXIT_FAILURE;
      }
      aiwaves[i] = (int)(afwaves[i]*10.+.5);
    }
  }
  else {
    printf("Can't open %20s! Bailing out.\n", swavesetname);
    return EXIT_FAILURE;
  }
  fclose(fp);


  /* Bring in the Van Hamme tables, one model atmosphere at a time. */
  if ((fp1 = fopen("/data/Catalogs/LimbDarkening/VanHamme/table1", "r")) 
      == NULL) {
    printf("Can't open Table1! Bailing out.\n");
    return EXIT_FAILURE;
  }

  if ((fp3c = fopen("/data/Catalogs/LimbDarkening/VanHamme/table3c", "r")) 
      == NULL) {
    printf("Can't open Table3c! Bailing out.\n");
    return EXIT_FAILURE;
  }

  /* Open output files for results */
  if ((fpoutln = fopen("NPOI_Lin.dat", "w")) == NULL) {
    printf("Can't open NPOI_Lin.dat for output! Bailing out.\n");
    return EXIT_FAILURE;
  }
  fprintf(fpoutln, "Coefficients: Linear, Waveset: %s\n",swavesetname);

  if ((fpoutlg = fopen("NPOI_Log.dat", "w")) == NULL) {
    printf("Can't open NPOI_Log.dat for output! Bailing out.\n");
    return EXIT_FAILURE;
  }
  fprintf(fpoutlg, "Coefficients: Log, Waveset: %s\n",swavesetname);

  if ((fpoutsq = fopen("NPOI_Sqrt.dat", "w")) == NULL) {
    printf("Can't open NPOI_Sqrt.dat for output! Bailing out.\n");
    return EXIT_FAILURE;
  }
  fprintf(fpoutsq, "Coefficients: Square root, Waveset: %s\n",swavesetname);


  /* Initial line from table3c */
  if (fscanf(fp3c, "%80c", line) == EOF) {
    printf("Failed reading first line from Table3c! Bailing out.\n");
    return EXIT_FAILURE;
  }
  line[79]='\0';
  //printf("%s\n", line);
  iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
	       &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);



  /* And (Teff, glog) info from Table1, indexed by model number */
  for (imodel = 0; imodel < 410; imodel++) {
    if(fscanf(fp1, "%80c", line) == EOF) 
      printf("Failed reading Table1! Continuing.");
    line[79]='\0';
    //printf("%s\n", line);
    iread=sscanf(line, "%3i%6f%5f", &jmodel, &teff, &glog);
    /*
    printf("Num read=%2i Model# %3i Teff =%6.0f log g =%4.1f\n", 
	   iread, jmodel, teff, glog);
    */
    if (kmodel != jmodel) {
      printf("Model numbers don't match! table3c:%3i table1:%3i\n",
	     kmodel, jmodel);
      return EXIT_FAILURE;
    }
    fprintf(fpoutln, "Teff %5.0f  glog %4.1f  waves %2i\n",teff, glog, iwaves);
    fprintf(fpoutlg, "Teff %5.0f  glog %4.1f  waves %2i\n",teff, glog, iwaves);
    fprintf(fpoutsq, "Teff %5.0f  glog %4.1f  waves %2i\n",teff, glog, iwaves);
    fprintf(fpoutln, "Idx   wave   xlin I(1)/B\n");
    fprintf(fpoutlg, "Idx   wave   xlog    ylog I(1)/B\n");
    fprintf(fpoutsq, "Idx   wave   xsqr    ysqr I(1)/B\n");


    /* New model, start through the filters */
    wavemb=0.;

    /* Do the wavelength averages for this model. */
    for (i = 0; i < iwaves; i++){
      /* clear accumulators */
      swavem=0.; slinear=0.; sxlog=0.; sylog=0.; sxsq=0.; sysq=0.; 
      sI1=0.; //sicount=0;
      
      /* Find initial wavelength from model calculations */
      while (wavem < afwaves[i]) {
      
	/* save previous line */
	wavemb=wavem; linearb=linear; xlogb=xlog; ylogb=ylog; xsqb=xsq; 
	ysqb=ysq; I1b=I1;
	
	/* Next line from table3c */
	if (fscanf(fp3c, "%80c", line) == EOF) {
	  printf("Failed reading line from Table3c! Bailing out.\n");
	  return EXIT_FAILURE;
	}
	line[79]='\0';
	//printf("%s\n", line);
	iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
		     &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);
      }
      
      while (wavemb < afwaves[i]) {
	if (wavem >= afwaves[i]) {
	  wtm=(afwaves[i]-wavem)/(wavemb-wavem);
	  wtp=1.-wtm;
	  slinear=(wtm*linearb+wtp*linear);
	  sxlog=(wtm*xlogb+wtp*xlog);
	  sylog=(wtm*ylogb+wtp*ylog);
	  sxsq=(wtm*xsqb+wtp*xsq);
	  sysq=(wtm*ysqb+wtp*ysq);
	  sI1=(wtm*I1b+wtp*I1);
	}
	/* save previous line */
	wavemb=wavem; linearb=linear; xlogb=xlog; ylogb=ylog; xsqb=xsq; 
	ysqb=ysq; I1b=I1;
	
	//printf("%s\n", line);
	/* Next line from table3c */
	if (fscanf(fp3c, "%80c", line) == EOF) {
	  printf("Failed reading line from Table3c! Bailing out.\n");
	  return EXIT_FAILURE;
	}
	line[79]='\0';
	//printf("%s\n", line);
	iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
		     &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);
      }
      
      
      weff=afwaves[i];
      Beff=1.191062e23/(weff*weff*weff*weff*weff)/
	(exp(1.4388e7/(weff*teff))-1.);
      IoB=sI1/Beff;
      if (kmodel != 299) {
	/*
	  fprintf(fpout, "%6.0f%5.1f%8.1f%6.3f%6.3f%7.3f%7.3f%7.3f%7.3f\n",
	  teff, glog, weff,slinear/sicount,sxlog/sicount,
	  sylog/sicount,sxsq/sicount,sysq/sicount,IoB);
	*/
	fprintf(fpoutln, "%3i %6.1f %6.3f %6.4f\n",
		i+1,afwaves[i],slinear,IoB);
	fprintf(fpoutlg, "%3i %6.1f %6.3f %7.3f %6.4f\n",
		i+1,afwaves[i],sxlog,sylog,IoB);
	fprintf(fpoutsq, "%3i %6.1f %6.3f %7.3f %6.4f\n",
		i+1,afwaves[i],sxsq,sysq,IoB);
      }
    }

    /* Clean up the remaining wavelengths from this model.  Exits when
       it finds the first (shortest) wavelength for the next model. */
    while (wavem > afwaves[iwaves-1]) {
      /* save previous line */
      wavemb=wavem; linearb=linear; xlogb=xlog; ylogb=ylog; xsqb=xsq; 
      ysqb=ysq; I1b=I1;
	
      /* Next line from table3c */
      if (fscanf(fp3c, "%80c", line) == EOF) {
	printf("End of Table3c! That's normal.\n");
	fclose(fpoutln);
	fclose(fpoutlg);
	fclose(fpoutsq);
	return EXIT_FAILURE;
      }
      line[79]='\0';
      //printf("%s\n", line);
      iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
		   &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);
    }
    
  } // imodel < 410 loop
  fclose(fpoutln);
  fclose(fpoutlg);
  fclose(fpoutsq);

  return EXIT_SUCCESS ;
}
