/* 
 * This reads in the Van Hamme (AJ 1993) tables and generates Limb
 * Darkening coefficients for Kurucz, Solar composition models.
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

int main(int iargc, char *acargv[]) {

  float afwavemid[32], afwaves[33], affilternorm[32],oneow2, A;
  char swavefilename[20], swavesetname[20], save[20];
  int iwaves=33, iwavesmid=32, i, j, aiwaves[33];
  char line[81];
  float teff, glog; float waveb, wavee;
  float wavem, linear, xlog, ylog, xsq, ysq, I1;
  float wavemb, linearb, xlogb, ylogb, xsqb, ysqb, I1b;
  float swavem, slinear, sxlog, sylog, sxsq, sysq, sI1, fI1; int sicount;
  int imodel, iread, jmodel, kmodel;
  int iwm, iwmb, iwme, iw, iwb, iwe; 
  float wtm, wtp, Beff, wave, weff, IoB;
  FILE *fp, *fp1, *fp3c, *fpoutln, *fpoutlg, *fpoutsq;

  /* These are the NPOI channel parameters, roughly evenly spaced in
   * inverse wavelength.  Central wavelengths are given.  We calculate
   * beginning and end wavelengths by finding the midpoints in
   * inverse wavelengths.  The inner points are used to extrapolate to
   * the two outer end points.  It counts from the long wavelength.
   */

  /* Command line arguments:
     0: is the program name
     1: Wavelength File (contains waveset name and 32 wavelengths)
  */

  if (iargc > 1 ) {
    if (iargc > 1) strncpy(swavefilename,acargv[1],19);
  }

  /*  Or read in the Waveset name and wavelengths from a file. */
  else {
    printf("Wavelength set filename: ");
    scanf("%19s", swavefilename);
    /* And kill the next character, particularly if it is a newline */
    getchar();
  }

  if ((fp = fopen(swavefilename, "r")) != NULL) {
    if (fgets(swavesetname, 20, fp) == NULL) {
      printf("Waveset File Empty!  Bailing out.\n");
      return EXIT_FAILURE;
    }
    j=strcspn(swavesetname,"\n");
    swavesetname[j]='\0';
    printf("Waveset: %20s\n", swavesetname);

    for (i = 0; i < iwavesmid; i++) {
      fscanf(fp, "%3i%9f", &j, &afwavemid[i]);
      //printf("i=%3i j=%3i wave=%6.1f\n", i, j, afwavemid[i]);
      if (i != j-1) {
	printf("wavelengths out of order!\n");
	return EXIT_FAILURE;
      }
    }
  }
  else {
    printf("Can't open %20s! Bailing out.\n", swavesetname);
    return EXIT_FAILURE;
  }
  fclose(fp);

  /* Define the bandpass endpoints */
  //printf("Define bandpasses\n I EndPts Midpts\n");
  /* 
   * Assumes the endpoints are midway between the midpoints, in
   * wavenumbers.
   *
   * Note: aiwaves[] are integers in angstroms.
   */
  for (i = 0; i < iwaves; i++) {
    if (i == 0) {
      afwaves[i] = 1./(1.5/afwavemid[i]-.5/afwavemid[i+1]) ;
      aiwaves[i] = (int)(afwaves[i]*10.+.5);
    }
    else if (i == iwavesmid) {
      afwaves[i] = 1./(1.5/afwavemid[i-1]-0.5/afwavemid[i-2]);
      aiwaves[i] = (int)(afwaves[i]*10.+.5);
    }
    else {
      afwaves[i] = 2./(1./afwavemid[i-1]+1./afwavemid[i]);
      aiwaves[i] = (int)(afwaves[i]*10.+.5);
    }
    //printf("i=%2i%7.1f\n", i, afwaves[i]);
    //if (i < iwavesmid) printf("         %7.1f\n",afwavemid[i]);
  }
  //printf("\n");

  /* 
   *  We will be working in wavelength rather than wavenumber space and
   *  the NPOI prism disperses roughly uniformily in wavenumber.  Thus,
   *  to take weighted means using equal spacing in wavelengths we need
   *  to multiply each term in the mean by a factor A/wave^2 to account
   *  for the translation (really affects significantly only the long
   *  wavelength bands).  "A", roughly w_end*w_beg/(w_end-w_beg) for
   *  each channel, needs to include any normalization offsets arising
   *  from our specific choice of integration procedures.  We calculate
   *  the A's (affilternorm[]) next.
   */
  for (i = 0; i < iwavesmid; i++) {
    affilternorm[i]=0.;
    A=0.;
    iwb=aiwaves[i+1];
    iwe=aiwaves[i];
    /*
    if (i >= 0){
      printf("i=32: iwb=%5i waveb=%5i iwe=%5i wavee=%5i\n", 
	     iwb, aiwaves[i+1], iwe, aiwaves[i]) ;
    }
    */
    for (iw=iwb; iw<=iwe; iw++) {
      oneow2 = 1./((float) (iw*iw));
      //oneow2=1.;
      wave = iw/10.;
      A += wave*oneow2;
      affilternorm[i] += oneow2;
      //printf("%5i  1/w^2%10.4g  sum%10.4g\n", j, oneow2, affilternorm[i]);
    }
    /*
    if (i >= 0) {
      printf("iwb%5i  iwe%5i  A/Anal%7.3f  w-eff%6.1f  w-mid%6.1f\n", 
	     iwb, iwe, affilternorm[i]*((float)(iwb*iwe))/((float)(iwe-iwb)),
	     A/affilternorm[i], afwavemid[i]);
    }
    */
  }
  //return EXIT_SUCCESS;

  /* Bring in the Van Hamme tables, one model atmosphere at a time. */
  if ((fp1 = fopen("/data/Catalogs/LimbDarkening/VanHamme/table1", "r")) 
      == NULL) {
    printf("Can't open Table1! Bailing out.\n");
    return EXIT_FAILURE;
  }

  if ((fp3c = fopen("/data/Catalogs/LimbDarkening/VanHamme/table3c", "r")) 
      == NULL) {
    printf("Can't open Table3c! Bailing out.\n");
    return EXIT_FAILURE;
  }

  /* Open output files for results */
  if ((fpoutln = fopen("NPOI_Lin.dat", "w")) == NULL) {
    printf("Can't open NPOI_Lin.dat for output! Bailing out.\n");
    return EXIT_FAILURE;
  }
  fprintf(fpoutln, "Coefficients: Linear, Waveset: %s\n",swavesetname);

  if ((fpoutlg = fopen("NPOI_Log.dat", "w")) == NULL) {
    printf("Can't open NPOI_Log.dat for output! Bailing out.\n");
    return EXIT_FAILURE;
  }
  fprintf(fpoutlg, "Coefficients: Log, Waveset: %s\n",swavesetname);

  if ((fpoutsq = fopen("NPOI_Sqrt.dat", "w")) == NULL) {
    printf("Can't open NPOI_Sqrt.dat for output! Bailing out.\n");
    return EXIT_FAILURE;
  }
  fprintf(fpoutsq, "Coefficients: Square root, Waveset: %s\n",swavesetname);


  /* Initial line from table3c */
  if (fscanf(fp3c, "%80c", line) == EOF) {
    printf("Failed reading first line from Table3c! Bailing out.\n");
    return EXIT_FAILURE;
  }
  line[79]='\0';
  //printf("%s\n", line);
  iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
	       &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);

  /* And (Teff, glog) info from Table1, indexed by model number */
  for (imodel = 0; imodel < 410; imodel++) {
    if(fscanf(fp1, "%80c", line) == EOF) 
      printf("Failed reading Table1! Continuing.");
    line[79]='\0';
    //printf("%s\n", line);
    iread=sscanf(line, "%3i%6f%5f", &jmodel, &teff, &glog);
    /*
    printf("Num read=%2i Model# %3i Teff =%6.0f log g =%4.1f\n", 
	   iread, jmodel, teff, glog);
    */
    if (kmodel != jmodel) {
      printf("Model numbers don't match! table3c:%3i table1:%3i\n",
	     kmodel, jmodel);
      return EXIT_FAILURE;
    }
    fprintf(fpoutln, "Teff %5.0f  glog %4.1f  waves 32\n",teff, glog);
    fprintf(fpoutlg, "Teff %5.0f  glog %4.1f  waves 32\n",teff, glog);
    fprintf(fpoutsq, "Teff %5.0f  glog %4.1f  waves 32\n",teff, glog);
    fprintf(fpoutln, "Idx w-mid w-eff xlin I(1)/B\n");
    fprintf(fpoutlg, "Idx w-mid w-eff  xlog  ylog I(1)/B\n");
    fprintf(fpoutsq, "Idx w-mid w-eff  xsqr  ysqr I(1)/B\n");


    /* New model, start through the filters */
    wavemb=0.;

    /* Do the wavelength averages for this model. */
    for (i = iwavesmid; i > 0; i--){
      wavee=afwaves[i-1];
      waveb=afwaves[i];
      iwe = aiwaves[i-1];
      iwb = aiwaves[i];

      /* clear accumulators */
      swavem=0.; slinear=0.; sxlog=0.; sylog=0.; sxsq=0.; sysq=0.; 
      sI1=0.; //sicount=0;

    /* Find initial wavelength from model calculations */
      while (wavem < waveb) {
      
	/* save previous line */
	wavemb=wavem; linearb=linear; xlogb=xlog; ylogb=ylog; xsqb=xsq; 
	ysqb=ysq; I1b=I1;
	
	/* Next line from table3c */
	if (fscanf(fp3c, "%80c", line) == EOF) {
	  printf("Failed reading line from Table3c! Bailing out.\n");
	  return EXIT_FAILURE;
	}
	line[79]='\0';
	//printf("%s\n", line);
	iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
		     &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);
      }
      
      while (wavem < wavee) {
	iwmb=(int)(wavemb*10.+.5);
	iwme=(int)(wavem*10.+.5);
	if (iwmb >= iwb) iwmb += 1; // keep from doing twice
	//for (iwm = iwmb; iwm <= iwme; iwm++) {
	for (iwm = iwmb; iwm <= iwme; iwm++) {
	  wtm=((float)(iwm-iwme))/((float)(iwmb-iwme));
	  wtp=1.-wtm;
	  oneow2=1./((float)iwm*iwm);
	  //oneow2=1.;
	  wave = iwm/10.;
	  if (iwm >= iwb && iwm <= iwe) {
	    fI1=(wtm*I1b+wtp*I1);
	    swavem+=wave*fI1*oneow2;
	    slinear+=(wtm*linearb+wtp*linear)*fI1*oneow2;
	    sxlog+=(wtm*xlogb+wtp*xlog)*fI1*oneow2;
	    sylog+=(wtm*ylogb+wtp*ylog)*fI1*oneow2;
	    sxsq+=(wtm*xsqb+wtp*xsq)*fI1*oneow2;
	    sysq+=(wtm*ysqb+wtp*ysq)*fI1*oneow2;
	    sI1+=fI1*oneow2;
	  }
	}
	/* save previous line */
	wavemb=wavem; linearb=linear; xlogb=xlog; ylogb=ylog; xsqb=xsq; 
	ysqb=ysq; I1b=I1;
	
	/* Next line from table3c */
	if (fscanf(fp3c, "%80c", line) == EOF) {
	  printf("Failed reading line from Table3c! Bailing out.\n");
	  return EXIT_FAILURE;
	}
	line[79]='\0';
	//printf("%s\n", line);
	iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
		     &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);
      }
      
      /* catch the last partial interval */
      if (wavemb < wavee && wavem >= wavee) {
	iwmb=(int)(wavemb*10.+.5);
	iwme=(int)(wavem*10.+.5);
	if (iwmb >= iwb) iwmb += 1; // keep from doing twice
	for (iwm=iwmb; iwm <= iwme; iwm++) {
	  wtm=((float)(iwm-iwme))/((float)(iwmb-iwme));
	  wtp=1.-wtm;
	  oneow2=1./((float)iwm*iwm);
	  //oneow2=1.;
	  wave = iwm/10.;
	  if (iwm >= iwb && iwm <= iwe) {
	    fI1=(wtm*I1b+wtp*I1);
	    swavem+=wave*fI1*oneow2;
	    slinear+=(wtm*linearb+wtp*linear)*fI1*oneow2;
	    sxlog+=(wtm*xlogb+wtp*xlog)*fI1*oneow2;
	    sylog+=(wtm*ylogb+wtp*ylog)*fI1*oneow2;
	    sxsq+=(wtm*xsqb+wtp*xsq)*fI1*oneow2;
	    sysq+=(wtm*ysqb+wtp*ysq)*fI1*oneow2;
	    sI1+=fI1*oneow2;
	  }
	}
      }
      
      //weff=swavem/affilternorm[i-1];
      weff=swavem/sI1;
      Beff=1.191062e23/(weff*weff*weff*weff*weff)/
	(exp(1.4388e7/(weff*teff))-1.);
      IoB=sI1/Beff/affilternorm[i-1];
      if (kmodel != 299) {
	/*
	  fprintf(fpout, "%6.0f%5.1f%8.1f%6.3f%6.3f%7.3f%7.3f%7.3f%7.3f\n",
	  teff, glog, weff,slinear/sicount,sxlog/sicount,
	  sylog/sicount,sxsq/sicount,sysq/sicount,IoB);
	*/
	fprintf(fpoutln, "%3i%6.1f%6.1f%6.3f%6.3f\n",
		i,afwavemid[i-1],weff,slinear/sI1,IoB);
	fprintf(fpoutlg, "%3i%6.1f%6.1f%6.3f%7.3f%6.3f\n",
		i,afwavemid[i-1],weff,sxlog/sI1,
		sylog/sI1,IoB);
	fprintf(fpoutsq, "%3i%6.1f%6.1f%6.3f%7.3f%6.3f\n",
		i,afwavemid[i-1],weff,sxsq/sI1,
		sysq/sI1,IoB);
      }
    }

    /* Clean up the remaining wavelengths from this model.  Exits when
       it finds the first (shortest) wavelength for the next model. */
    while (wavem > afwaves[iwavesmid]) {
      /* save previous line */
      wavemb=wavem; linearb=linear; xlogb=xlog; ylogb=ylog; xsqb=xsq; 
      ysqb=ysq; I1b=I1;
	
      /* Next line from table3c */
      if (fscanf(fp3c, "%80c", line) == EOF) {
	printf("End of Table3c! That's normal.\n");
	fclose(fpoutln);
	fclose(fpoutlg);
	fclose(fpoutsq);
	return EXIT_FAILURE;
      }
      line[79]='\0';
      //printf("%s\n", line);
      iread=sscanf(line, "%3i%12f%6f%*7s%6f%7f%*7s%7f%7f%*7s%10g", 
		   &kmodel, &wavem, &linear, &xlog, &ylog, &xsq, &ysq, &I1);
    }

  } // imodel < 410 loop
  fclose(fpoutln);
  fclose(fpoutlg);
  fclose(fpoutsq);

  return EXIT_SUCCESS ;
}
