/**********************************************************************
 * Calculates the main physical variables for this model.  Assumes the
 * critical fractional rotation, w, has been defined and that the mass
 * and non-rotating radius, fMass and fRo, are already in.
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "roche.h"

int MainVar(Roche * R, LimbDark * LD, int IfWrite) {

  float w, w2, re, rp, gpb, Ms, Ro;
  /* ttc = (2/3)^3, tts = (2/3)^6 */
  float ttc = 0.296296296, tts = 0.087791495, fact;
  float hck = 1.43878e7; // units: nm*K
  //float scalepole = 0.1087 ; // Approximately keeps the model at const lum.
  float scalepole = 0.; // no shrinking of pole allowed
  float gEffp, gEffeq, tEffp, tEffeq, hclktp, hclkteq, eMissp, eMisseq;
  float lam=R->fWave, lam5 = lam*lam*lam*lam*lam;
  float thc2ol5 = 1.191062e23/lam5; // 2hc^2/lam^5
  float cth2, sth2, rout, rout2;
  float SurfBp, SurfB;
  float th, phi, cth, sth, cphi, fXp, ci = cos(R->fIncl), 
    si = sin(R->fIncl);

  w = R->fwOmega;
  w2 = w*w;
  Ms = R->fMass;
  Ro = R->fRNot;

  /* Main Variables */
  R->fRp = Ro*(1.-scalepole*w2);
  R->fRpB = (1.-scalepole)*Ro;
  R->fReB = 1.5*R->fRpB;

  /* Equatorial and Polar Radii */
  re = FractRadius(w2, 1.);
  rp = FractRadius(w2, 0.);

  //if (IfWrite) printf("\n");

  R->fRe = re*R->fRpB;
  if (IfWrite) {
    printf("Radii: Rp=%1.6g Re=%1.6g RpB=%1.6g ReB=%1.6g\n", 
	   R->fRp, R->fRe, R->fRpB, R->fReB); 
    printf("       rp=%1.6g re=%1.6g D=%1.6g\n", rp, re, rp/re); 
  }

  /* Projected Outline */
  if (Edge(R, 0., &th, &phi, &rout) == EXIT_FAILURE) 
    return EXIT_FAILURE;
  sth=sin(th);
  cth=cos(th);
  cphi=cos(phi);
  fXp=rout*(sth*cphi*ci-cth*si);
  R->fAngDiam_Xp = R->fAngDiam_pB * fXp;
  R->fAngDiam_Yp = R->fAngDiam_pB * re;
  if (IfWrite) {
    printf("Diameters(mas): Minor=%1.6g Major=%1.6g b/a=%1.6g\n",
	   R->fAngDiam_Xp, R->fAngDiam_Yp, fXp/re);
  }

  R->fOmegaB = sqrt(GRAV*Ms*MSUN/R->fReB/R->fReB/R->fReB/RSUN/RSUN/RSUN);
  R->fVeqB = R->fOmegaB*R->fReB*RSUN;
  R->fOmega = w*R->fOmegaB;
  R->fVeq = R->fOmega*R->fRe*RSUN;

  if (IfWrite) {
    printf("Vel: OmegaB=%1.6g Omega=%1.6g\n", R->fOmegaB, R->fOmega);
    printf("     VeqB=%1.6g Veq=%1.6g Vsini=%1.6g\n", 
	   R->fVeqB/1.e5, R->fVeq/1.e5, R->fVeq/1.e5*sin(R->fIncl));
  }
  gpb= GRAV*R->fMass*MSUN/R->fRpB/R->fRpB/RSUN/RSUN;

  fact = fabs(ttc*w2*re-1./re/re);
  R->flogGp = log10(R->fMass/R->fRp/R->fRp)+4.43772;
  R->flogGpB = log10(gpb);
  R->flogGe = log10(gpb*fact);
  //printf("fact=%1.6g log10(gpb)=%1.6g\n",fact, log10(gpb)); 

  if (IfWrite) {
  printf("log g's: Pole=%1.6g P-Breakup=%1.6g Eff-Equatorial=%1.6g\n", 
	 R->flogGp, R->flogGpB, R->flogGe);
  }

  /* Range of Planck Function Brightness */
  /* Polar region */
  sth2=0.;
  cth2=1.;
  rout=rp;
  rout2 = rp*rp;
  fact = (ttc*w2*rout*sth2-1/rout2);
  gEffp = sqrt(fact*fact + tts*w2*w2*rout2*sth2*cth2); // scaled by gpb
  tEffp = R->fTeffpB*sqrt(sqrt(gEffp));
  R->fTeffp = tEffp; // store it
  hclktp = hck/lam/tEffp;
  if (SurfBright(LD, tEffp, R->flogGp, 1., &SurfBp) == EXIT_FAILURE)
    return EXIT_FAILURE;
  /* The real normalization */
  eMissp = thc2ol5/(exp(hclktp)-1.)*SurfBp; 
  /* Equatorial region */
  sth2=1.;
  cth2=0.;
  rout=re;
  rout2 = re*re;
  fact = (ttc*w2*rout*sth2-1/rout2);
  gEffeq = sqrt(fact*fact + tts*w2*w2*rout2*sth2*cth2); // scaled by gpb
  tEffeq = R->fTeffpB*sqrt(sqrt(gEffeq));
  R->fTeffe = tEffeq; // store it
  hclkteq = hck/lam/tEffeq;
  /* Compare polar surface brightness, seen normally, to equator, grazing */
  if (SurfBright(LD, tEffeq, R->flogGe, 0., &SurfB) == EXIT_FAILURE)
    return EXIT_FAILURE;
  eMisseq = thc2ol5/(exp(hclkteq)-1.)*SurfB; 
  if (IfWrite) {
    printf("Teff's: Pole=%1.6g Equator=%1.6g Imin/IMax =%1.6g ", 
	   tEffp, tEffeq, eMisseq/eMissp);
    printf("at %1.6g(nm)\n", lam);
  }


  if (Volume(R) == EXIT_FAILURE) return(EXIT_FAILURE) ;

  if (IfWrite) {
    printf("Vol/(4PI/3 Rp^2 Re)=%1.6g",
	   R->fVol/(4.*PI/3.*R->fRp*R->fRp*R->fRe));
    //	   R->fVol/(4.*PI/3.*R->fRp*R->fRe*R->fRe));
    printf(",  Area(equator-on)/(PI Rp Re)=%1.6g\n\n",
	   R->feqArea/(PI*R->fRp*R->fRe));
  }
  return EXIT_SUCCESS;
}
