/***********************************************************************
 * This plots a color representation of the spheroid.  Needs
 * inclination, fractional angular velocity, Teff at polar breakup, and
 * the wavelength.  Otherwise assumes T = g^1/4.
 *
 * Have added a Position Angle offset to the object in Ver 1_2.  Also
 * the scales are now in mas and the sense of rotation is reversed (to
 * make it right handed).
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <string.h>

#include "roche.h"
#include "cpgplot.h"

int Picture(Roche * R, LimbDark * LD, float ThPA, char * name, int imagedev, 
	    int IfRotate) {

  /* Get outline every 5 degrees */
  const int iol=73;
  float xol[100],yol[100], thol[100], phiol[100], rol[100];
  float xolp[100],yolp[100];  // To offset the pole in Position Angle

  float si, ci, thp, th, ph, sth, cth, sph, cph, rout; 
  int i, j, k;

  /* For the image */
  const int ndim=256;
  int inmax, ismax, jymax, nd2=ndim/2, ifoutside, icolor0;
  float red, green, blue, eMissMin;
  float xs, xn, rp, re, dd, fimage[ndim*ndim], yolmean, 
    rscale = R->fAngDiam_pB/2;
  float xmax=1.75*rscale, ymax=1.75*rscale, 
    tr[6] = {0.,0.,1.,0.,-1.,0.}, xi, yj;
  int irotate=1;
  char sRomanName[30], sIncl[20];

  /* For gravity/temperature calculation */
  float gpb= GRAV*R->fMass*MSUN/R->fRpB/R->fRpB/RSUN/RSUN, dlrdth;
  float ttc = 0.2962963, tts = 0.087791495, fact, w = R->fwOmega, w2=w*w;
  float lam=R->fWave, lam5=lam*lam*lam*lam*lam; 
  float hck = 1.43878e7; // units: nm*K
  float thc2ol5 = 1.191062e23/lam5; // 2hc^2/lam^5
  float gEff, tEff, hcklt, eMisspB=thc2ol5/(exp(hck/lam/R->fTeffpB) - 1.);
  float sth2, cth2, rout2, eMiss;

  float SurfBp, SurfB, mu;

  float sPA=sin(ThPA), cPA= cos(ThPA);

  if (SurfBright(LD, R->fTeffp, R->flogGp, 1., &SurfBp) == EXIT_FAILURE)
    return EXIT_FAILURE;
  //printf("Polar Surface Brightness:%6.3f\n",SurfBp);
  eMisspB = eMisspB*SurfBp; // SurfBright is scaled by the Planck Fn

  /* set transform to image */
  dd=3./ndim;
  //tr[2]=-dd;
  tr[2]=dd;
  tr[4]=-dd;
  //tr[0]=dd*(nd2+.5);
  tr[0]=-dd*(nd2+.5);
  tr[3]=dd*(nd2+.5);
  //tr[0]=-dd*(nd2);
  //tr[3]=-dd*(nd2);

  /* Try rotating it a bit */
  tr[0] = -(-sPA+cPA)*dd*(nd2+.5)*rscale;
  tr[1] = -sPA*dd*rscale;
  tr[2] = cPA*dd*rscale;
  tr[3] = (cPA+sPA)*dd*(nd2+.5)*rscale;
  tr[4] = -cPA*dd*rscale;
  tr[5] = -sPA*dd*rscale;

  /* Initialize image */
  for (i=0; i<ndim; i++) {
    for (j=0; j<ndim; j++) {
      fimage[i*ndim+j] = 0.;
    }
  }

  si=sin(R->fIncl);
  ci=cos(R->fIncl);

  /* First get the dimensions of the actual figure */
  re=FractRadius(w2, 1.); // equatorial
  rp=FractRadius(w2, 0.); // polar
  //ymax = re;

  /* South side */
  Edge(R, 0., &th, &ph, &rout); // 
  sth = sin(th);
  cth = cos(th);
  cph = cos(ph);
  xs = rout*(sth*cph*ci-cth*si);
  ismax = nd2 - (int)(xs/dd+.5); // farthest picture element on south side

  Edge(R, (float) PI, &th, &ph, &rout); // 
  sth = sin(th);
  cth = cos(th);
  cph = cos(ph);
  xn = rout*(sth*cph*ci-cth*si);
  inmax = nd2 - (int)(xn/dd-.5); // farthest picture element on north side

  strcpy(sRomanName, "\\fr");
  strcat(sRomanName, name);
  sprintf(sIncl,"  \\fii=%1.4g",R->fIncl*(float)DEGRAD);
  strcat(sRomanName, sIncl);
  cpgslw(6);
  cpgsch(1.5);
  cpgenv(xmax, -xmax, -ymax, ymax, 1, 0);
  cpglab("RA (mas)", "Dec (mas)", sRomanName);
  /* Get Outline */
  for (i = 0; i < iol-1; i++) {
    //printf("Picture: i=%1i thp=%1.6g\n", i, 5.*i);
    //thp=5.*i*(float)RADDEG;
    thp=5.*i*(float)RADDEG;
    if (Edge(R, thp, &th, &ph, &rout) == EXIT_FAILURE)
      return EXIT_FAILURE;
    //printf("Th=%1.6g Phi=%1.6g Rout=%1.6g\n",(float)DEGRAD * th, 
    //(float)DEGRAD * ph, rout);
    sth = sin(th);
    cth = cos(th);
    sph = sin(ph);
    cph = cos(ph);
    xol[i] = rout*(sth*cph*ci-cth*si);
    yol[i] = rout*sth*sph;
    xolp[i] = (cPA*xol[i]-sPA*yol[i])*rscale;
    yolp[i] = (sPA*xol[i]+cPA*yol[i])*rscale;
    rol[i] = rout;
    thol[i] = th;
    phiol[i] = ph;
  }  
  xol[iol-1] = xol[0];
  yol[iol-1] = yol[0];
  xolp[iol-1] = xolp[0];
  yolp[iol-1] = yolp[0];
  rol[iol-1] = rol[0];
  thol[iol-1] = thol[0];
  phiol[iol-1] = phiol[0];
  //return EXIT_FAILURE; // !

  eMissMin=1.;
  /* Now fill in the picture.  (x,y) are in units of polar breakup. */
  for (i = 0; i < ndim; i++) {
    ifoutside=1;
    if (i >= ismax && i < inmax) {
      ifoutside = 0; // There will be j's with image elements
      jymax=-1;
      k=0;
      xi=(nd2-i-0.5)*dd;
      while (k < iol/2-1 && jymax < 0) {
	if ((xol[k] < xi && xol[k+1] > xi) || (xol[k] > xi && xol[k+1] < xi)) {
	  yolmean=(yol[k]*(xol[k+1]-xi)+yol[k+1]*(xi-xol[k]))
	    /(xol[k+1]-xol[k]);
	  rout=(rol[k]*(xol[k+1]-xi)+rol[k+1]*(xi-xol[k]))/(xol[k+1]-xol[k]);
	  jymax = (int)(yolmean/dd+0.5);
	}
	k++;
      }	
    }
    for (j = 0; j < nd2; j++) {
      k=ndim-1-i+ndim*j;
      if (ifoutside) {
	fimage[k] = 0.;
      }
      else if (nd2-j > jymax) {
	fimage[k] = 0.;
	fimage[ndim-1-i+ndim*(ndim-j-1)] = 0.; // E-W symmetry
      }
      else {
	yj = (nd2-j-0.5)*dd;
	/*
	if (RThetaPhifromXY(xi, yj, R, &rout, &th, &ph, 0) == EXIT_FAILURE) 
	  return EXIT_FAILURE;
	*/
	if (RThetaPhifromXY(xi, yj, R, &rout, &th, &ph) == EXIT_FAILURE) 
	  return EXIT_FAILURE;
	sth = sin(th);
	sth2 = sth*sth;
	cth = cos(th);
	cth2 = cth*cth;
	cph=cos(ph);
	rout2=rout*rout;
	dlrdth=(ttc*w2*rout2*rout*sth*cth)/(1.-ttc*w2*rout2*rout*sth2);
	mu=(dlrdth*(sth*ci-cth*cph*si)+cth*ci+sth*cph*si)/
	  sqrt(dlrdth*dlrdth+1.);
	if (mu < 0.) mu=0.; // fix any noise in the numerical procedures
	fact = (ttc*w2*rout*sth2-1/rout2);
	gEff = sqrt(fact*fact + tts*w2*w2*rout2*sth2*cth2); // scaled by gpb
	tEff = R->fTeffpB*sqrt(sqrt(gEff));
	hcklt = hck/lam/tEff;
	if (SurfBright(LD, tEff, log10(gEff*gpb), mu, &SurfB) 
	    == EXIT_FAILURE) return EXIT_FAILURE;
	/* scaled to polar breakup at mu=1*/
	eMiss = thc2ol5*SurfB/(eMisspB*(exp(hck/lam/tEff)-1.)); 
	if (eMiss < .01) eMiss = .01; // set minimum
	if (eMiss < eMissMin) eMissMin=eMiss;
	fimage[k] = eMiss; 
	fimage[ndim-1-i+ndim*(ndim-j-1)] = fimage[k]; // symmetry
	//if (fabs(ph-(float)PI) < 0.1) { // ~+/-5def
	/*
	if (xi < -1.07) { // pick off the top
	  printf("Picture: xi=%1.6g yj=%1.6g r=%1.6g", xi, yj, rout); 
	  printf(" th=%1.6g ph=%1.6g eMiss=%1.6g\n", (float)DEGRAD * th, 
	         (float)DEGRAD * ph, eMiss);
	}
	*/
	//return EXIT_FAILURE;
      }
    }
  }
  
  /* output the whole image */
  /*
  printf("\nfimage:\n");
  for (i = 0; i < ndim; i++) {
    for (j = 0; j < nd2; j++) {
      k=i+ndim*j;
      printf("%4.2f ",fimage[k]);
    }
    printf("\n");
  }
  printf("\n");
  */

  /* set various indices */
  /* 99 is the max number of colors on this system */
  //cpgqcir(&i, &j); // what is default?
  //printf("ICILO=%2i ICIHI=%2i\n", i, j);
  icolor0=17;
  for (i=icolor0; i<100; i++) {
    red = (99.-i)/82.;
    green = .0;
    blue = 1.-red;
    cpgscr(i, red, green, blue); // blue to red
  }
  cpgscr(16, 1., 1., 1.); // back-ground
  cpgscir(16,99);
  jymax = (int)(re/dd+0.5);
  //cpggray(fimage,ndim,ndim,ismax+1,inmax,nd2-jymax+1,nd2+jymax,0.,256.,tr);
  //cpgimag(fimage,ndim,ndim,ismax+1,inmax,nd2-jymax+1,nd2+jymax,eMissMin-.01,1.,tr);

  //printf("Spin it\n");
  /* Spin it */
  irotate = 1;
  if (IfRotate) irotate = 101;
  if (imagedev != 0) irotate = 1;  // Only rotate it on the screen
  for (i=0; i<irotate; i++) {

    /* longitude offset, first number sets how coarse it is */
    th = -1.*(float)RADDEG*(float)i; 

    /* Refresh the image */
    cpgimag(fimage,ndim,ndim,ismax+1,inmax,nd2-jymax+1,nd2+jymax,0,1.,tr);
    
    /* Thicken up the outline to hide the warts */
    cpgslw(7);
    cpgline(iol, yolp, xolp);
    cpgslw(1);
    
    /* Draw latitude and longitude lines with increasing offsets*/
    if (LatLong(R, th, ThPA, rscale) == EXIT_FAILURE) return EXIT_FAILURE;

    /* SubSolar point */
    cpgslw(6);
    cpgsch(1.3);
    cpgpt1(0., 0., 2);
    cpgsch(1.0);
    cpgslw(1);
  }

  /* Labels */
  cpgslw(6);
  cpgsch(1.3);
  cpgptxt(1.5*rscale,-1.*rscale,90.,0.5,"East");
  cpgptxt(1.*rscale,1.5*rscale,0.,0.5,"North");
  cpgslw(1);
  cpgsch(1.);

  /* Get pole */
  //rpole = FractRadius(w2, 0.);
  //xp=rpole*si;
  //yp=0.;
  //cpgpt1(yp, xp, 9);
  
  printf("\n");

  return EXIT_SUCCESS;
}
