/*
 * This plots out the observables for arrays operated with closure
 * phase/amplitudes.  In addition to other things, for each baseline
 * triangle it plots three rows of squared Visibilities and phases, one
 * for each baseline.  The last column has the triple amplitude and
 * triple phase.  If less than three baselines are available, then the
 * triple quantities are not plotted and the above plots are enlarged.
 *
 * In either case, two additional plots are added to each page: the
 * total UV coverage with that particular HA highlighted and the outline
 * of the model of the object, with lines showing the Position Angles
 * drawn and illustrating the plotting format (solid, dots, dash-dot,
 * etc).
 *
 * A new plot sequence is initiated for each inclination. (That is: each
 * inclination forces a new plot file identified by the inclination.
 * Within a file are all the HA's, triangles, etc, page after page).
 * Each sequence includes all Hour Angles and for each Hour Angle, one
 * to six groups of Position Angles with up to six PAs in each group.
 * You interact with the plot only once to tell it whether you want to
 * see it or print it and for the latter the base name of the plot file.
 *
 * This routine was initialy written for Spot. DMP 10/04/03
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "cpgplot.h"
#include "roche.h"

int PlotTriangle(Roche *R, LimbDark *LD, Observatory *O, Plot *P, Scans *S) {

  /* From Spot */
  float vertDTh=.32;
  float fBmax, fmaxx, fminx, fmaxy, fminy;
  float vismin[3], vismax[3], phimin[3], phimax[3]; 
  float vis3min, vis3max, phi3min, phi3max;
  int icols=4, iDim=4, ier, iIncl; 
  char sBName[3][20], sPhiName[3][20], sBnum[2], sParam[30];
  char sModel[50], sBigParam[70],csPA[10], sBaseLine[30], sPhases[30];
  int i, j, iB, ib, iHA, iStoreFG;
  float fUVmin[2], fUVmax[2], fULambda[32], fVLambda[32];
  float PAnot, cPA, sPA, rscale = R->fAngDiam_pB/2, xolp[73], yolp[73], HA;
  float x[2], y[2], th;
  int iWaves, iPA, iuHA, idHA;
  //int iPlotSum, iPlotClose, iPlotV2, iPlotPhi, iIfPlot; // decode P->IfPlot
  float fPlotZero[] = {0.,0.};
  float fWaveZero[] = {LD->afWaves[P->j1Wave], LD->afWaves[P->j2Wave-1]};
  //fWaveZero[0] = LD->afWaves[P->j1Wave];
  //fWaveZero[1] = LD->afWaves[P->j2Wave-1];


  iStoreFG = 1; // store default color index for plotting
  iWaves=P->j2Wave-P->j1Wave;

  if (P->iPlotSum) {
    /* Set up for plotting the UV coverage */
    cpgpage(); // darn well better be open already!
    cpgsci(iStoreFG); // force default foreground
    cpgslw(3);
    cpgsvp(0.08, 0.26, 0.70, 0.95); // xmin, xmax, ymin, ymax
    cpgmtxt("B", 2.2, .5, .5, "V (M\\gl)");
    cpgmtxt("L", 1.7, .5, .5, "U (M\\gl)");
    cpgmtxt("T", .5, .5, .5, "UV Coverage");
    cpgmtxt("T", -1.4, .15, .5, "\\m30 N");
    cpgmtxt("B", -2.0, .065, 0., "E");
    cpgmtxt("B", -1.0, .065, 0., "\\m28");
    
    fBmax=0.;
    for (i=0; i<2; i++) { // make sure you get the extremes
      fUVmin[i] = P->fUVmin[i]/LD->afWaves[P->j2Wave-1]*1.E3;
      //if (-fUVmin[i] > fBmax) fBmax = -fUVmin[i];
      fUVmax[i] = P->fUVmax[i]/LD->afWaves[P->j2Wave-1]*1.E3;
      if (fUVmax[i] > fBmax) fBmax = fUVmax[i];
    }
    fBmax *= 1.3;
    //printf("fBmax=%2gMwaves\n",fBmax);
    
    /*
      printf("Umax=%5.1f Umin=%5.1f Vmax=%5.1f Vmin=%5.1f\n",fUVmax[0],
      fUVmin[0],fUVmax[1],fUVmin[1]);
      printf("PUmax=%5.1f PUmin=%5.1f PVmax=%5.1f PVmin=%5.1f\n",P->fUVmax[0],
      P->fUVmin[0],P->fUVmax[1],P->fUVmin[1]);
      printf("afWaves[j2]=%5.1f\n",LD->afWaves[P->j2Wave-1]);
    */
    
    //cpgenv(fUVmax[0], fUVmin[0], fUVmin[1], fUVmax[1], 1, 0); // initialize
    cpgwnad(fBmax, -fBmax, -fBmax, fBmax); // East points left, North up
    cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
    //cpglab("U (+->East) M\\gl", "V (+->North) M\\gl", "UV Coverage");
    //printf("Default ci=%i\n",iStoreFG);
    for (iHA = 0; iHA < S->iNScans; iHA++) {
      for (i = 0; i < O->iBaselines; i++) {
	
	/* identify baselines relevant to this plot */
	cpgsci(iStoreFG); // force default
	for (iB=0; iB<3; iB++) {
	  if (P->aiB1[iB] == i) cpgsci(2); // try plot in red
	}
	  
	for (j = P->j1Wave; j < P->j2Wave; j++) {
	  fULambda[j-P->j1Wave] = P->fSaveUV[iHA][i][0]/LD->afWaves[j]*1.E3;
	  fVLambda[j-P->j1Wave] = P->fSaveUV[iHA][i][1]/LD->afWaves[j]*1.E3;
	}
	cpgpt(iWaves,fULambda,fVLambda,1);
	for (j = P->j1Wave; j < P->j2Wave; j++) {
	  fULambda[j-P->j1Wave] = -fULambda[j-P->j1Wave];
	  fVLambda[j-P->j1Wave] = -fVLambda[j-P->j1Wave];
	}
	cpgpt(iWaves,fULambda,fVLambda,1);
      }
    }
    cpgsci(iStoreFG); // default
    
    /* 
     * Calculate the outline for the initial PA of this group.  The
     * outline for PA=0. is already available in the PLOT structure.
     */
    PAnot = P->afPAnot[P->iPAg] * (float)RADDEG;
    sPA = sin(PAnot);
    cPA = cos(PAnot);
    fmaxx=0.;
    fmaxy=0.;
    fminx=0.;
    fminy=0.;
    for (i = 0; i < P->iOutline; i++) {
      xolp[i] = (cPA*P->afxol[i]-sPA*P->afyol[i])*R->fAngDiam_pB/2;
      if (xolp[i] > fmaxx) fmaxx = xolp[i];
      if (xolp[i] < fminx) fminx = xolp[i];
      yolp[i] = (sPA*P->afxol[i]+cPA*P->afyol[i])*R->fAngDiam_pB/2;
      if (yolp[i] > fmaxy) fmaxy = yolp[i];
      if (yolp[i] < fminy) fminy = yolp[i];
    }
    cpgsvp(0.08, 0.26, 0.34, 0.59); // xmin, xmax, ymin, ymax
    cpgwnad(1.3*fmaxx, 1.2*fminx, 1.2*fminy, 1.3*fmaxy);
    cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
    cpgmtxt("B", 2.0, .5, 0.5, "RA cos\\gd (mas)");
    cpgmtxt("L", 1.7, .5, 0.5, "Dec (mas)");
    cpgmtxt("T", .5, .5, .5, "Model and PA's plotted");
    cpgmtxt("T", -1.4, .15, .5, "\\m30 N");
    cpgmtxt("B", -2.0, .065, 0., "E");
    cpgmtxt("B", -1.0, .065, 0., "\\m28");
    cpgsci(7); // yellow
    cpgsfs(1); // solid- default
    cpgpoly(P->iOutline,xolp,yolp); // draw and color it
    cpgsci(iStoreFG); // restore usual foreground = black
    
    /* 
     * Then draw in the various PA angles displaying the line styles used
     * in the rest of the plot.  Assuming inclination > 0.
     */
    if (R->fIncl != 0.) {
      x[0]=0.;
      y[0]=0.;
      for (iPA=0; iPA<P->iPAdel; iPA++) {
	if (iPA < 5) {
	  cpgsls(iPA+1);
	  cpgsci(1); // black
	}
	else {
	  cpgsls(iPA-4);
	  cpgsci(2); // red
	}
	th = (P->afPAnot[P->iPAg]+iPA*P->fPAdel)*(float)RADDEG; 
	x[1] = 1.2*fmaxx*sin(th);
	y[1] = 1.2*fmaxx*cos(th);
	cpgline(2,x,y); // and draw the line
      }
      cpgsls(1); // restore
      cpgsci(1); // black
    } // finishes the model/PA representation
  } // P->iPlotSum

  /* Name, Date */
  HA = fabs(S->afHA[P->iHA]); // remove sign
  iuHA = (int)HA; // capture units
  idHA = (int)(((HA-(float)iuHA)+.005)*100.); // 2 decimal places
  if (S->afHA[P->iHA] >= 0.) 
    sprintf(sModel,"\\fr%s %s %01i\\fn.\\b\\uW\\d\\fr%02i",
	    R->sName,S->sID[P->iHA],iuHA,idHA);
  if (S->afHA[P->iHA] < 0.) 
    sprintf(sModel,"\\fr%s %s %01i\\fn.\\b\\uE\\d\\fr%02i",
	    R->sName,S->sID[P->iHA],iuHA,idHA);
  //sprintf(sModel,"%s %s",R->sName,S->sID[P->iHA],iuHA,idHA);
  //printf("sModel: %s\n",sModel);

  if (P->iPlotSum){
    cpgmtxt("B", 3.5, -.3, 0., sModel);
    
    /* Model Parameters */
    cpgmtxt("B", 4.8, -.3, 0., "Model Parameters:");
    sprintf(sParam,"\\fii \\fn=%5.2f \\gw = %4.2f",
	    (float)DEGRAD*R->fIncl,R->fwOmega);
    cpgmtxt("B", 6.0, -.3, 0., sParam);
    if (R->fAngDiam_Yp >= 10.) 
      sprintf(sParam,"\\gh\\dX,Y\\u = %4.1f, %4.1fmas",R->fAngDiam_Xp,
	      R->fAngDiam_Yp);
    if (R->fAngDiam_Yp < 10.) 
      sprintf(sParam,"\\gh\\dX,Y\\u = %4.2f, %4.2fmas",R->fAngDiam_Xp,
	      R->fAngDiam_Yp);
    cpgmtxt("B", 7.0, -.3, 0., sParam);
    sprintf(sParam,"T\\dp,eq\\u =%6.0f,%6.0f K",R->fTeffp,R->fTeffe);
    cpgmtxt("B", 8.2, -.3, 0., sParam);
    sprintf(sParam,"Mass = %5.2f",R->fMass);
    cpgmtxt("B", 9.5, -.3, 0., sParam);
    
    /* Array Parameters */
    cpgmtxt("B", 10.9, -.3, 0., "Baseline Triangle:");
    sprintf(sParam,"%s",O->sTriangleNames[P->iTri]);
    cpgmtxt("B", 12.1, -.3, 0., sParam);
  } // P->iPlotSum

  /* Do V^2's and phases for each baseline */
  for (iB = 0; iB < O->iBaselines; iB++) {
    
    sprintf(sBnum,"%1i",iB+1);

    strcpy(sBName[iB],"V\\d");
    strcat(sBName[iB],sBnum);
    strcat(sBName[iB],"\\u\\u2\\d");
    
    strcpy(sPhiName[iB],"\\gf\\d");
    strcat(sPhiName[iB],sBnum);
    strcat(sPhiName[iB],"\\u");
    //printf("sBnum=%s sPhiName=%s\n",sBName[iB], sPhiName[iB]);
    
    /* Get vismin, vismax, phimin, phimax */
    vismin[iB]=1.;
    vismax[iB]=0.;
    phimin[iB]=P->afphase[0][iB][0];
    phimax[iB]=P->afphase[0][iB][0];

    for (iPA=0; iPA < P->iPAdel; iPA++) {
      for (i=P->j1Wave; i<P->j2Wave; i++) {
	if (P->afV2[iPA][iB][i]>vismax[iB]) vismax[iB]=P->afV2[iPA][iB][i];
	if (P->afV2[iPA][iB][i]<vismin[iB]) vismin[iB]=P->afV2[iPA][iB][i];
	if (P->afphase[iPA][iB][i]>phimax[iB]) 
	  phimax[iB]=P->afphase[iPA][iB][i];
	if (P->afphase[iPA][iB][i]<phimin[iB]) 
	  phimin[iB]=P->afphase[iPA][iB][i];
      }
    }
    /* 
    printf("iB=%i vismax=%6.4f vismin=%6.4f phimax=%6.1f phimin=%6.1f\n",
	   iB,vismax[iB],vismin[iB],phimax[iB],phimin[iB]);
    */

    if (phimax[iB] > 0.) {
      if (phimax[iB] > 260.) phimax[iB]=300.;
      else phimax[iB] *= 1.05;
    }
    if (phimax[iB] <= 10.) {
      if (phimax[iB] >= 0.) phimax[iB] = 10.;
      else if (phimax[iB] >= -10.) phimax[iB] = 1.;
      else phimax[iB] *= 0.95;
    }
    if (phimin[iB] < 0.) {
      if (phimin[iB] < -80.) phimin[iB]=-100.;
      else phimin[iB] *= 1.05;
    }
    if (phimin[iB] > -10.) {
      if (phimin[iB] <= 0.) phimin[iB] = -10.;
      else if (phimin[iB] <= 10.) phimin[iB] = -1.;
      else phimin[iB] *= 0.95;
    }
    /*
    printf("iB=%i vismax=%6.4f vismin=%6.4f phimax=%6.1f phimin=%6.1f\n",
	   iB,vismax[iB],vismin[iB],phimax[iB],phimin[iB]);
    */
    

    if (P->iPlotSum){
      /* V^2 */
      cpgsci(1);
      cpgsvp(0.32, 0.50, 0.70-vertDTh*iB, 0.95-vertDTh*iB);
      cpgswin(P->fAbcmin, P->fAbcmax, vismin[iB],vismax[iB]);
      cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
      cpgmtxt("B", 2.0, .5, 0.5, P->sAbLabel);
      cpgmtxt("L", 1.8, .5, 0.5, sBName[iB]);
      if (iB == 0)cpgmtxt("T", .3, .5, 0.5, "Visibility\\u2\\u");
      for (iPA=0; iPA < P->iPAdel; iPA++) {
	if (iPA < 5) {
	  cpgsls(iPA+1);
	  cpgsci(1);
	}
	else {
	  cpgsls(iPA-4);
	  cpgsci(2);
	}	
	//cpgline(iWaves, &LD->afWaves[P->j1Wave], &P->afV2[iPA][iB][P->j1Wave]);
	cpgline(iWaves,&P->afAbcissa[P->j1Wave],&P->afV2[iPA][iB][P->j1Wave]);
      }
      //printf("V^2(%i) ",iB);
      
      /* phi */
      cpgsls(1);
      cpgsci(1);
      cpgsvp(0.56, 0.74, 0.70-vertDTh*iB, 0.95-vertDTh*iB);
      cpgswin(P->fAbcmin, P->fAbcmax, phimin[iB],phimax[iB]);
      cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
      cpgmtxt("B", 2.0, .5, 0.5, P->sAbLabel);
      cpgmtxt("L", 1.8, .5, 0.5, sPhiName[iB]);
      if (iB == 0)cpgmtxt("T", .3, .5, 0.5, "Phases");
      for (iPA=0; iPA < P->iPAdel; iPA++) {
	if (iPA < 5) {
	  cpgsls(iPA+1);
	  cpgsci(1);
	}
	else {
	  cpgsls(iPA-4);
	  cpgsci(2);
	}	
	cpgline(iWaves,&P->afAbcissa[P->j1Wave],
		&P->afphase[iPA][iB][P->j1Wave]);
      }
      //printf("Phase(%i)\n",iB);
    }
  }
    
  /* Get 3-vismin, 3-vismax, 3-phimin, 3-phimax */
  vis3min=1.;
  vis3max=0.;
  phi3min=P->afCphase[0][0];
  phi3max=P->afCphase[0][0];
  for (iPA=0; iPA < P->iPAdel; iPA++) {
    for (i=0; i<iWaves; i++) {
      if (P->afCamp[iPA][i]>vis3max) vis3max=P->afCamp[iPA][i];
      if (P->afCamp[iPA][i]<vis3min) vis3min=P->afCamp[iPA][i];
      if (P->afCphase[iPA][i]>phi3max) phi3max=P->afCphase[iPA][i];
      if (P->afCphase[iPA][i]<phi3min) phi3min=P->afCphase[iPA][i];
    }
  }
  /*
    printf("3vismax=%6.4f 3vismin=%6.4f 3phimax=%6.2f 3phimin=%6.2f\n",
    vis3max,vis3min,phi3max,phi3min);
  */
  
  if (phi3max > 0.) {
    if (phi3max > 260.) phi3max=300.;
    else phi3max *= 1.05;
  }
  if (phi3max <= 10.) {
    if (phi3max >= 0.) phi3max = 10.;
    else if (phi3max >= -10.) phi3max = 1.;
    else phi3max *= 0.95;
    }

  if (phi3min < 0.) {
    if (phi3min < -90.) phi3min=-100.;
    else phi3min *= 1.05;
  }
  if (phi3min > -10.) {
    if (phi3min <= 0.) phi3min = -10.;
    else if (phi3min <= 10.) phi3min = -1.;
    else phi3min *= 0.95;
  }

  /*
    printf("3vismax=%6.4f 3vismin=%6.4f 3phimax=%6.2f 3phimin=%6.2f\n",
    vis3max,vis3min,phi3max,phi3min);
  */
  
  if (P->iPlotSum){
    /* Triple Phase */
    cpgsls(1);
    cpgsci(1);
    cpgsvp(0.80, 0.99, 0.70, 0.95);
    cpgswin(P->fAbcmin, P->fAbcmax, phi3min, phi3max);
    cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
    cpgmtxt("B", 2., .5, .5, P->sAbLabel);
    cpgmtxt("L", 2., .5, .5,"\\gh\\d1\\u+\\gh\\d2\\u+\\gh\\d3\\u (Deg)");
    cpgmtxt("T", .3, .5, 0.5, "Closure Phase" );
    cpgsls(1);
    for (iPA=0; iPA < P->iPAdel; iPA++) {
      if (iPA < 5) {
	cpgsls(iPA+1);
	cpgsci(1);
      }
      else {
	cpgsls(iPA-4);
	cpgsci(2);
      }	
      cpgline(iWaves,&P->afAbcissa[P->j1Wave],&P->afCphase[iPA][P->j1Wave]);
      //if (phi3max > 0. && phi3min < 0.) cpgline(2,fWaveZero,fPlotZero);
    }
    //printf("Triple Phase ");
    
    /* Triple Amplitude */
    cpgsls(1);
    cpgsci(1);
    cpgsvp(0.80, 0.99, 0.34, 0.59);
    cpgswin(P->fAbcmin, P->fAbcmax, vis3min, vis3max);
    cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
    cpgmtxt("B", 2., .5, .5, P->sAbLabel);
    cpgmtxt("L", 2., .5, .5,"V\\d1\\uV\\d2\\uV\\d3\\u");
    cpgmtxt("T", .3, .5, 0.5, "Triple Amplitude");
    cpgsls(1);
    for (iPA=0; iPA < P->iPAdel; iPA++) {
      if (iPA < 5) {
	cpgsls(iPA+1);
	cpgsci(1);
      }
      else {
	cpgsls(iPA-4);
	cpgsci(2);
      }	
      cpgline(iWaves,&P->afAbcissa[P->j1Wave],&P->afCamp[iPA][P->j1Wave]);
    }
  }
  //printf("Triple Amplitude\n");
  
  sprintf(sBigParam,"%s  Incl=%g  PA= ",sModel,(float)DEGRAD*R->fIncl);
  for (iPA=0; iPA < P->iPAdel; iPA++) {
    sprintf(csPA,"%g ",P->afPAnot[P->iPAg]+iPA*P->fPAdel);
    strcat(sBigParam,csPA);
  }

  if (P->iPlotClose) {
    //printf("PlotClose\n");
    /* Triple Phase */
    cpgpage(); // darn well better be open already!
    cpgsci(iStoreFG); // force default foreground
    cpgslw(3);
    cpgsls(1);
    //cpgsvp(0.80, 0.99, 0.70, 0.95);
    cpgvstd();
    cpgswin(P->fAbcmin, P->fAbcmax, phi3min, phi3max);
    cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
    cpgmtxt("B", 2., .5, .5, P->sAbLabel);
    cpgmtxt("L", 2., .5, .5,"\\gh\\d1\\u+\\gh\\d2\\u+\\gh\\d3\\u (Deg)");
    cpgmtxt("T", .4, .1, 0.5, "Closure Phase" );
    cpgmtxt("T", .4, .6, 0.5, sBigParam );
    cpgsls(1);
    for (iPA=0; iPA < P->iPAdel; iPA++) {
      if (iPA < 5) {
	cpgsls(iPA+1);
	cpgsci(1);
      }
      else {
	cpgsls(iPA-4);
	cpgsci(2);
      }	
      cpgline(iWaves,&P->afAbcissa[P->j1Wave],&P->afCphase[iPA][P->j1Wave]);
      cpgpt(iWaves,&P->afAbcissa[P->j1Wave],&P->afCphase[iPA][P->j1Wave],0);
    }
    if (phi3max > 0. && phi3min < 0.) { 
      cpgsci(6);
      cpgsls(1);
      cpgline(2,fWaveZero,fPlotZero);
      cpgsci(iStoreFG);
    }
    //printf("Triple Phase ");
    
    /* Triple Amplitude */
    cpgpage(); // darn well better be open already!
    cpgsls(1);
    cpgsci(iStoreFG);
    //cpgsvp(0.80, 0.99, 0.34, 0.59);
    cpgvstd();
    cpgswin(P->fAbcmin, P->fAbcmax, vis3min, vis3max);
    cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
    cpgmtxt("B", 2., .5, .5, P->sAbLabel);
    cpgmtxt("L", 2., .5, .5,"V\\d1\\uV\\d2\\uV\\d3\\u");
    cpgmtxt("T", .4, .1, 0.5, "Triple Amplitude");
    cpgmtxt("T", .4, .6, 0.5, sBigParam );
    cpgsls(1);
    for (iPA=0; iPA < P->iPAdel; iPA++) {
      if (iPA < 5) {
	cpgsls(iPA+1);
	cpgsci(1);
      }
      else {
	cpgsls(iPA-4);
	cpgsci(2);
      }	
      cpgline(iWaves,&P->afAbcissa[P->j1Wave],&P->afCamp[iPA][P->j1Wave]);
    }
  } // P->iPlotClose

  if (P->iPlotV2) {
    //printf("PlotV2\n");
    for (iB = 0; iB < O->iBaselines; iB++) {
      ib = O->iTriangleBaselines[P->iTri][iB];
      if (ib > 0) sprintf(sBaseLine,"Visibility\\u2\\d %s", 
	      O->sBaselineNames[ib-1][0]);
      if (ib < 0) sprintf(sBaseLine,"Visibility\\u2\\d %s", 
	      O->sBaselineNames[-ib-1][1]);
      /* V^2 */
      cpgpage(); // darn well better be open already!
      cpgsci(iStoreFG); // force default foreground
      //cpgsvp(0.32, 0.50, 0.70-vertDTh*iB, 0.95-vertDTh*iB);
      cpgvstd();
      cpgswin(P->fAbcmin, P->fAbcmax, vismin[iB],vismax[iB]);
      cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
      cpgmtxt("B", 2.0, .5, 0.5, P->sAbLabel);
      cpgmtxt("L", 1.8, .5, 0.5, sBName[iB]);
      cpgmtxt("T", .4, .1, 0.5, sBaseLine );
      cpgmtxt("T", .4, .6, 0.5, sBigParam );
      for (iPA=0; iPA < P->iPAdel; iPA++) {
	if (iPA < 5) {
	  cpgsls(iPA+1);
	  cpgsci(1);
	}
	else {
	  cpgsls(iPA-4);
	  cpgsci(2);
	}	
	cpgline(iWaves, &P->afAbcissa[P->j1Wave], &P->afV2[iPA][iB][P->j1Wave]);
      }
      //printf("V^2(%i) ",iB);
    }
  } 

  if (P->iPlotPhi) {
    //printf("PlotPhi\n");
    for (iB = 0; iB < O->iBaselines; iB++) {
      ib = O->iTriangleBaselines[P->iTri][iB];
      if (ib > 0) sprintf(sPhases,"Phases %s", 
	      O->sBaselineNames[ib-1][0]);
      if (ib < 0) sprintf(sPhases,"Phases %s", 
	      O->sBaselineNames[-ib-1][1]);
      /* phi */
      cpgpage(); // darn well better be open already!
      cpgsci(iStoreFG); // force default foreground
      cpgsls(1);
      //cpgsvp(0.56, 0.74, 0.70-vertDTh*iB, 0.95-vertDTh*iB);
      cpgvstd();
      cpgswin(P->fAbcmin, P->fAbcmax, phimin[iB],phimax[iB]);
      cpgbox("BCTN", 0.0, 0, "BCTN", 0.0, 0);
      cpgmtxt("B", 2.0, .5, 0.5, P->sAbLabel);
      cpgmtxt("L", 1.8, .5, 0.5, sPhiName[iB]);
      cpgmtxt("T", .4, .1, 0.5, sPhases);
      cpgmtxt("T", .4, .6, 0.5, sBigParam );
      for (iPA=0; iPA < P->iPAdel; iPA++) {
	if (iPA < 5) {
	  cpgsls(iPA+1);
	  cpgsci(1);
	}
	else {
	  cpgsls(iPA-4);
	  cpgsci(2);
	}	
	cpgline(iWaves,&P->afAbcissa[P->j1Wave],
		&P->afphase[iPA][iB][P->j1Wave]);
	cpgpt(iWaves,&P->afAbcissa[P->j1Wave],
		&P->afphase[iPA][iB][P->j1Wave],0);
	if (phimax[iB]>0. && phimin[iB]<0.) {
	  cpgsci(6);
	  cpgsls(1);
	  cpgline(2,fWaveZero,fPlotZero);
	  cpgsci(iStoreFG);
	}
      }
      //printf("Phase(%i)\n",iB);
    }
  }

  return EXIT_SUCCESS;

}
