/***********************************************************************
 * ReadBaselines: Brings in the physical layout of the interferometer
 * and Observatory parameters. No longer asks for that information if
 * the file "baselines" is not found, it just flops over.
 *
 * Revised to use only the Triangle provided, ala NPOI. 8/29/03
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "roche.h"

int ReadBaselines(Observatory * O, Roche * R) {

  int i, j, k, l, l1, iCheck ;
  int iLat=0, iLong=0, iAlt=0, iAp=0, iName=0, iApn[IAPMAX]={0}, 
    iT[ITRIMAX]={0};
  int iApertures, iB, indexTriangle[ITRIMAX][3];
  char line[81], sApName[IAPMAX][4], sName[30];
  float fAxyz[IAPMAX][3];
  FILE *fp;

  /* Initialize */
  O->iTriangles=ITRIMAX;
  O->iBaselines=IBMAX;

  /* See if there's a file specifying the Observatory & baselines. */

  if ((fp = fopen("baselines", "r")) != NULL) {
    while (fgets(line, 81, fp) != NULL) {
      //printf("line: %s",line);
      if (strstr(line,"sInstName") != NULL) {
	strtok(line,"\"");
	strcpy(sName,strtok(NULL,"\""));
	iName=1; // got it
	//printf("sName = %s\n",sName);
      }

      if (strstr(line,"fLat") != NULL) {
	strtok(line,"=");
	O->fLat = (float)RADDEG*atof(strtok(NULL,";"));
	iLat=1; // got it
	//printf("fLat = %f\n",(float)DEGRAD*O->fLat);
      }

      if (strstr(line,"fLong") != NULL) {
	strtok(line,"=");
	O->fLong = (float)RADDEG*atof(strtok(NULL,";"));
	iLong=1; // got it
	//printf("fLong = %f\n",(float)DEGRAD*O->fLong);
      }

      if (strstr(line,"fAlt") != NULL) {
	strtok(line,"=");
	O->fAlt = atof(strtok(NULL,";"));
	iAlt=1; // got it
	//printf("fAlt = %f\n",O->fAlt);
      }

      if (strstr(line,"iApertures") != NULL) {
	strtok(line,"=");
	iApertures = atoi(strtok(NULL,";"));
	iAp=1; // got it
	if (iApertures > IAPMAX) {
	  printf("ReadBaselines: Set for %i apertures, max.  Ignoring rest",
		 IAPMAX);
	  iApertures = IAPMAX;
	}
	O->iTriangles = (iApertures-1)*(iApertures-2)/2;
	/*
	printf("iApertures = %i, iTriangles = %i\n",iApertures,O->iTriangles);
	*/
      }

      if (strstr(line,"iApn") != NULL) {
	strtok(line,"n");
	i = atoi(strtok(NULL,"s"));
	if (i < iApertures) {
	  strtok(strstr(line,"sApName"),"\"");
	  strcpy(sApName[i],strtok(NULL,"\""));
	  strtok(NULL,"x");
	  fAxyz[i][0]=atof(strtok(NULL,","));
	  strtok(NULL,"y");
	  fAxyz[i][1]=atof(strtok(NULL,","));
	  strtok(NULL,"z");
	  fAxyz[i][2]=atof(strtok(NULL,"\0"));
	  iApn[i]=1; // got it
	}
	/*
	printf("iApn %i Name \"%3s\", x %7.3f, y %7.3f, z %#5.0f\n",
	       i,sApName[i],fAxyz[i][0],fAxyz[i][1],fAxyz[i][2]);
	*/
      }
      if (strstr(line,"indexTriangle") != NULL) {
	strtok(line," ");
	i = atoi(strtok(NULL,"A"));
	if (i < O->iTriangles) {
	  strtok(strstr(line,"Aps")," ");
	  indexTriangle[i][0] = atoi(strtok(NULL," "));
	  indexTriangle[i][1] = atoi(strtok(NULL," "));
	  indexTriangle[i][2] = atoi(strtok(NULL," "));
	  iT[i]=1;
	}
	/*
	printf("indexTriangle[%i][i]: %i %i %i\n",i,indexTriangle[i][0],
		 indexTriangle[i][1],indexTriangle[i][2]);
	*/
      }
    }

    /* Make sure everything is there and do the initializations */
    iCheck=1;
    if(!iLat) {
      printf("Obs Latitude not set!\n");
      iCheck=0;
    }
    if(!iLong) {
      printf("Obs Longitude not set!\n");
      iCheck=0;
    }
    /* not that critical
       if(!iAlt) {
       printf("Obs Altitude not set!\n");
       iCheck=0;
       }
    */
    if(!iAp) {
      printf("Number of Apertures not set!\n");
      iCheck=0;
    }
    if(!iName) {
      printf("Name of Observatory/Aperture set not found!\n");
      iCheck=0;
    }
    for (i=0; i<iApertures; i++) {
      if(!iApn[i]) {
	printf("Position of Aperture %i not set!\n",i);
	iCheck=0;
      }
    }
    for (i=0; i<O->iTriangles; i++) {
      //if(iT[i]) iTmax = i;
      if(!iT[i]) {
	printf("Baselines for Triangle %i not set!\n", i);
	iCheck=0;
      }
    }
    /*
    if (iTmax < 0) iCheck=0;
    O->iTriangles = iTmax+1;
    */
    
    /* Leave, if necessary */
    if (!iCheck) {
      printf("ReadBaselines: Sorry, that's a \"NO\".\n");
      return EXIT_FAILURE;
    }
    
    /* Define rotation matrix that project (Bx,By,Bz) to (X,Y,Z). */
    O->afRotateBase[0][0]= 0.;
    O->afRotateBase[0][1]=-sin(O->fLat);
    O->afRotateBase[0][2]= cos(O->fLat);
    O->afRotateBase[1][0]= 1.;
    O->afRotateBase[1][1]= 0.;
    O->afRotateBase[1][2]= 0.;
    O->afRotateBase[2][0]= 0.;
    O->afRotateBase[2][1]= cos(O->fLat);
    O->afRotateBase[2][2]= sin(O->fLat);
    /*
    printf("\nafRotateBaseline:\n        0       1        2\n");
    for (i=0; i<3; i++) printf("%i: %8.4f%8.4f%8.4f\n",i,O->afRotateBase[i][0],
			       O->afRotateBase[i][1],O->afRotateBase[i][2]);
    */
    
    /* Next, define the baselines */
    O->iBaselines = iApertures*(iApertures-1)/2;
    iB=-1;
    for (i=0; i<iApertures; i++) {
      for (j=i+1; j<iApertures; j++) {
	iB++;
	strcpy(O->sBaselineNames[iB][0],sApName[j]);
	strcat(O->sBaselineNames[iB][0],">");
	strcat(O->sBaselineNames[iB][0],sApName[i]);
	/* [1] when it goes backwards */
	strcpy(O->sBaselineNames[iB][1],sApName[i]);
	strcat(O->sBaselineNames[iB][1],">");
	strcat(O->sBaselineNames[iB][1],sApName[j]);
	O->iBaselineIndex[iB][0]=j;
	O->iBaselineIndex[iB][1]=i;
	O->fBAzi[iB][0] = 0.;
	for (k=0; k<3; k++) {
	  O->fBxyz[iB][k] = fAxyz[i][k]-fAxyz[j][k]; // Ap[1]-Ap[0]
	  O->fBAzi[iB][0] += O->fBxyz[iB][k]*O->fBxyz[iB][k];
	}
	O->fBAzi[iB][0] = sqrt(O->fBAzi[iB][0]); // |B|
	O->fBAzi[iB][1] = PI2-atan2(O->fBxyz[iB][1],O->fBxyz[iB][0]);// Azimuth
	/* and rotate it into (X,Y,Z) [(X,Y) in the equatorial plane]*/
	for (k=0; k<3; k++) {
	  O->fXYZ[iB][k]=0;
	  for (l=0; l<3; l++) {
	    O->fXYZ[iB][k] += O->afRotateBase[k][l]*O->fBxyz[iB][l];
	  }
	}
      }
    }
    
    /*
    printf("\nBaselines: \n n    Name   -Name  i  j    B    Azimuth    Bx");
    printf("      By      Bz       X       Y       Z\n");
    for (i=0; i<O->iBaselines; i++) {
      printf("%2i%8s%8s%3i%3i%8.3f%8.2f%8.3f%8.3f%8.3f%8.3f%8.3f%8.3f\n",
	     i,O->sBaselineNames[i][0],O->sBaselineNames[i][1],
	     O->iBaselineIndex[i][0],O->iBaselineIndex[i][1],
	     O->fBAzi[i][0],(float)DEGRAD*O->fBAzi[i][1],
	     O->fBxyz[i][0],O->fBxyz[i][1],O->fBxyz[i][2],
	     O->fXYZ[i][0],O->fXYZ[i][1],O->fXYZ[i][2]);
    }
    */
    
    /* Find which baselines are for each Triangle */
    for (k=0; k<O->iTriangles; k++) {
      for (l=1; l<=O->iBaselines; l++) {
	l1 = l-1;
	if (indexTriangle[k][0] == O->iBaselineIndex[l1][0] 
	    && indexTriangle[k][1] == O->iBaselineIndex[l1][1]) 
	  O->iTriangleBaselines[k][0] = l;
	if (indexTriangle[k][0] == O->iBaselineIndex[l1][1] 
	    && indexTriangle[k][1] == O->iBaselineIndex[l1][0]) 
	  O->iTriangleBaselines[k][0] = -l;
	if (indexTriangle[k][1] == O->iBaselineIndex[l1][0] 
	    && indexTriangle[k][2] == O->iBaselineIndex[l1][1]) 
	  O->iTriangleBaselines[k][1] = l;
	if (indexTriangle[k][1] == O->iBaselineIndex[l1][1] 
	    && indexTriangle[k][2] == O->iBaselineIndex[l1][0]) 
	  O->iTriangleBaselines[k][1] = -l;
	if (indexTriangle[k][2] == O->iBaselineIndex[l1][0] 
	    && indexTriangle[k][0] == O->iBaselineIndex[l1][1]) 
	  O->iTriangleBaselines[k][2] = l;
	if (indexTriangle[k][2] == O->iBaselineIndex[l1][1] 
	    && indexTriangle[k][0] == O->iBaselineIndex[l1][0]) 
	  O->iTriangleBaselines[k][2] = -l;
      }
      O->sTriangleNames[k][0] = '\0';
      for (i=0; i<3; i++) {
	j=O->iTriangleBaselines[k][i];
	if (j > 0) {
	  strcat(O->sTriangleNames[k],O->sBaselineNames[j-1][0]);
	}
	if (j < 0) {
	  j = -j;
	  strcat(O->sTriangleNames[k],O->sBaselineNames[j-1][1]);
	}
	if (i < 2) strcat(O->sTriangleNames[k],"+");
      }
    }
    
    /*
    printf("\nBaselines used on the triangles:\n i   0   1   2  Name\n");
    for (i=0; i<O->iTriangles; i++) {
      printf(" %i",i);
      for (k=0; k<3; k++) {
	j = O->iTriangleBaselines[i][k];
	//printf("(j=%i) ",j);
	ssign[0]=' ';
	if (j < 0) ssign[0]='-';
	j = abs(j)-1;
	if (j < 10) printf(" ");
	printf(" %s%i",ssign,j);
      }
      printf("  %s\n",O->sTriangleNames[i]);
    }
    */
  }
  
  else {
    printf("ReadBaselines: No baselines file, not a chance!\n");
    return EXIT_FAILURE;
  }
  
  printf("Baselines used:\n");
  for (i = 0; i < O->iBaselines; i++) {
    printf("  i=%2i Name:%8s  Theta= %7.2f  B[i]= %7.3f\n", i, 
	   O->sBaselineNames[i][0], 
	   O->fBAzi[i][1]*(float)DEGRAD, O->fBAzi[i][0]);
  }

  if (O->iTriangles > 0) {
    printf("Triangles formed:\n");
    for (j=0; j < O->iTriangles; j++) {
      printf("  j=%2i Name:%28s\n",j,O->sTriangleNames[j]);
    }
  }
  else printf("No Triangles!\n");

  //printf("\n");

  //return EXIT_FAILURE;

  return EXIT_SUCCESS;
}
