/* This fits Roche models to interferometric measurements of stars.  It
 * is built on the Roche simulation program, Ver 1.4.
 * 
 * For the moment it simply calculates Chi^2 for the data set using a
 * range of values for the parameters.
 *
 * Version 2.0: This adds the capability to do baseline amplitudes
 * separately, rather than just triangles.  This is to accomodate PTI
 * Vega data, immediately, but eventually we may want to do triangle
 * baselines independently rather than as triple amplitudes.  We now
 * index the Scan, LimbDark and Observatory structures to accomodate
 * data from more than one site.
 *
 * 2.1: Implement a "no-triangles" mode.  Aimed at PTI data, but fairly
 * general.  This will require ReadScansChi2 to look at the DS.iFormat
 * parameter to choose an input subroutine.
 *
 * Rework the StripBrightness calculation to make it a 1st order (ie,
 * linear) approximation of the intensity distributions of the disk.
 * This becomes a simple Simpson's rule for integration perpendicular
 * to the baseline, and the complex Visibility is now the FT of a
 * linear function.  This should reduce dramatically the number of
 * points that need to be evaluated through the Roche calculation.
 * We've also created a separate routine for the Visibility, both to
 * make the logic clearer and to make the implementation in Oyster more
 * transparent.  3/8/04
 */

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "cpgplot.h"
#include "rochechi2.h"

Roche R; // Model and derived parameters
LimbDark LD[ILIMBDARKSETS]; // Limb Darkening data
Observatory O[IOBSERVATORIES]; // Observatory parameters
Disk D[IBMAX]; // Stellar disks, for each (baseline,PA)
Scans S[IDATASETS]; // Storage for the data
DataSets DS; // Master List - What's used, needed, etc.

int main(int iargc, char *acargv[]) {

  float Incl[100], W[100], DpBmas[100], PA[100]; 
  //float Bright[ISTRIPMAX],X[ISTRIPMAX],Uniform[ISTRIPMAX];
  float Vis[IBMAX][32], phase[IBMAX][32], Vis2[IBMAX][32]; 
  float ReV[IBMAX][32], ImV[IBMAX][32], phiClose, amplitudeClose;
  int IfWrite, iBaseline, iIncl, iInclmax=100, inl;
  int iWmax=100, iW, iw, iDpBmax=100, iDpB, idp; 
  int iPAmax=100, iPA, ipa;
  int i, icp, j, j1Wave, j2Wave, k, isc, ib, iLD, iBL, iDS;

  /* To calculate Visibilities */
  //float cmasnmm=1.e9/(3.6e6*(float)DEGRAD); // cycles/mas*nm/m
  float Re=0., Im=0., ReC, ImC;
  //float sum=0., sumu=0., u, xi, delta, dsincu;

  int tri, itri, ibln, bln, iB1, iB2;

  /* Will control the Chi2 index myself */
  float Chi2[10000], dampC, dphiC, dampV2; 
  static int iChi2Max=10000; int iDOF;

  FILE *fp; char sDataFile[] = "ChiSq.dat";

  /* Bring in master "scans" file */
  if (ReadDataSets(&DS) == EXIT_FAILURE) return EXIT_FAILURE;
  
  /* The model file needs to set things next */
  if (ReadModelChi2(&R, &DS) == EXIT_FAILURE) return EXIT_FAILURE;
  
  printf("\nMs=%1.6g Ro=%1.6g Teff(pB)=%1.6g\n", R.fMass, R.fRNot, 
	 R.fTeffpB);
    
  /* Read in the parameter values that will be used */
  //printf("Do ReadParam\n");
  if (ReadParameters(iInclmax, Incl, &iIncl, iWmax, W, &iW, iDpBmax, 
		     DpBmas, &iDpB, iPAmax, PA, &iPA) == EXIT_FAILURE) 
    return EXIT_FAILURE;
  if (iIncl*iW*iDpB*iPA > iChi2Max) {
    printf("In main: iIncl*iW*iDpB*iPA=%i > iChi2Max=%i!\n",iIncl*iW*iDpB*iPA,
	   iChi2Max);
    return EXIT_FAILURE;
  }
    
  //printf("Entering ReadLimbDark\n");
  for (i=0; i<DS.iNLimbDarkFiles; i++) {
    strcpy(LD[i].sLDLaw, DS.sLDLaw[i]);
    strcpy(LD[i].sInstrument, DS.sInstrument[i]);
    strcpy(LD[i].sWaveset, DS.sWaveset[i]);
    strcpy(LD[i].sPath, DS.sLimbDarkPath[i]);
    if (ReadLimbDarkChi2(&LD[i]) == EXIT_FAILURE) return EXIT_FAILURE; 
  }

  //printf("Entering ReadBaseLines\n");
  for (i=0; i<DS.iNBaselineFiles; i++) {
    strcpy(O[i].sObsFileName,DS.sBaselinesFile[i]);
    if (ReadBaselinesChi2(&O[i]) == EXIT_FAILURE) 
      return EXIT_FAILURE;
  }

  /* Read in the observations (Scans). Get the wavelength index range.*/
  //printf("Entering ReadScans\n");
  for (iDS=0; iDS<DS.iNDataSets; iDS++){
    iLD = DS.iIndexLD[i];
    strcpy(S[iDS].sPathFileName,DS.sScanFile[iDS]);
    S[iDS].iIfTri = DS.iIfTri[iDS];
    if (ReadScansChi2(&LD[iLD], &S[iDS], DS.iFormat[iDS]) == EXIT_FAILURE) 
      return EXIT_FAILURE;
  }

  //return EXIT_FAILURE;

  //printf("Do OpenChiSq\n");
  /* Open file to output Chi^2.  Use "a" option in case you forgot to
     save the previous one. */
  if ((fp = fopen(sDataFile, "a")) == NULL) {
    printf("Can't open %s\n",sDataFile);
    return EXIT_FAILURE;
  }
  /* Put header into the file */
  fprintf(fp,"%s Mass=%g Radius(p)=%g Teff(p)=%g\n",
	  R.sName,R.fMass,R.fRNot,R.fTeffpB);
  fprintf(fp,"iIncl %i Incl:",iIncl);
  for (inl = 0; inl < iIncl; inl++) fprintf(fp," %5.2f",Incl[inl]);
  fprintf(fp,"\niW %i W:",iW);
  for (iw=0; iw<iW; iw++) fprintf(fp," %5.3f",W[iw]);
  fprintf(fp,"\niDpB %i Dp:",iDpB);
  for (idp=0; idp<iDpB; idp++) fprintf(fp," %5.3f",DpBmas[idp]);
  fprintf(fp,"\niPA %i PA:",iPA);
  for (ipa = 0; ipa < iPA; ipa++) fprintf(fp," %5.1f",PA[ipa]);
  fprintf(fp,"\n");


  //printf("Starting main loop\n");
  /* Do over (inclination, omega, angDiam, PosAngle) */
  for (inl = 0; inl < iIncl; inl++) {
    R.fIncl = Incl[inl]*(float)RADDEG;
    for (iw=0; iw<iW; iw++) {
      R.fwOmega = W[iw];
      for (idp=0; idp<iDpB; idp++) {
	R.fAngDiam_pB = DpBmas[idp];

	LD[0].iWave = 0; // fake for MainVar
	R.fWave = LD[0].afWaves[0]; // fake
	MainVar(&R, &LD[0], 0); // calculate all the physical variables
	fprintf(fp,"%3i %3i %3i ",inl,iw,idp); // record (i,w,d) indices
	//printf("\nDp=%g w=%g Incl=%g PA: ",DpBmas[idp],W[iw],Incl[inl]);

	for (ipa = 0; ipa < iPA; ipa++) {
	  
	  /* iChi2 = ipa + iPA * (idp + iDpB * (iw + iW * inl)); */
	  Chi2[ipa] = 0.;
	  iDOF = 0;

	  /* Over Data Sets */
	  for (iDS = 0; iDS<DS.iNDataSets; iDS++) {
	    iLD = DS.iIndexLD[iDS];
	    iBL = DS.iIndexBL[iDS];
	    iBaseline=O[iBL].iBaselines;

	    j1Wave=S[iDS].ajnWave[0];
	    j2Wave=S[iDS].ajnWave[1];

	    /* Over Observations (Scans = Hour Angles) */
	    //printf("Start HA loop. iNScans=%i\n",S[iDS].iNScans);
	    for (isc = 0; isc < S[iDS].iNScans; isc++) {
	    
	      /* Transform to Baseline projected on the stellar disk */
	      //printf("isc=%i Call GetUVW\n",isc);
	      if (GetUVW(&O[iBL], &R, S[iDS].afHA[isc]) == EXIT_FAILURE) 
		return EXIT_FAILURE;
	      /*
		printf("iBaselines=%i\n",O[iBL].iBaselines);
		for (i=0; i<O[iBL].iBaselines; i++) {
		printf("isc=%i iBl=%i U=%g Th=%g\n",
		isc,i,O[iBL].fUTheta[i][0],(float)DEGRAD*O[iBL].fUTheta[i][1]);
		}
		printf("\n");
	      */

	      //printf("\n**********ipa=%i inl=%i isc=%i\n", ipa, inl, isc);
	      /* Set up Disk map for the baselines at these HA's*/
	      for (ib=0; ib<O[iBL].iBaselines; ib++) {
		//printf("ib=%i iBaselines=%i\n",ib,O[iBL].iBaselines);
		D[ib].iB = ISTRIPMAX;
		/*
		  if (ib == 2 && ipa == 5 && inl == 0) {
		  D[ib].iB = -ISTRIPMAX;
		  printf("Baseline: %s ib=%i\n",
		  O[iBL].sBaselineNames[ib][0],ib);
		  }
		*/
		//printf("@Surface, Baseline: %s ib=%i\n",
		//     O[iBL].sBaselineNames[ib][0],ib);
	    
		if (Surface(&R, &D[ib], O[iBL].fUTheta[ib][1]-PA[ipa]*(float)RADDEG) 
		    == EXIT_FAILURE) return EXIT_FAILURE;
		if (ib > 0 && D[ib].iB < 0) D[ib].iB = -D[ib].iB; 
		if (D[ib].iB < 0) D[ib].iB = -D[ib].iB; 
		//printf("ib=%i - Exit Surface\n",ib);
	    
		/* Do over wavelengths */
		for (j=j1Wave; j<j2Wave; j++) {
		  R.fWave=LD[iLD].afWaves[j];
		  LD[iLD].iWave = j; // correct bug found by Christian!!

		  /* Get Strip Brightness Distribution at the wavelength. */
		  /* Setting D.iB < 0 causes a lot of printing
		     if (ib == 2 && inl == 0 && ipa == 5) 
		     D[ib].iB = -ISTRIPMAX;
		     else D[ib].iB = ISTRIPMAX;
		  */
		
		  //if (ib == 0 && j == j1Wave) IfWrite=1;
		
		  IfWrite=0;
		  /* complex Visibility */
		  if (Visibility(&R, &LD[iLD], &D[ib], O[iBL].fUTheta[ib][0], 
				 &Re, &Im, IfWrite) 
		      == EXIT_FAILURE) return EXIT_FAILURE;
		  //printf("Out of Visibility\n");
		  IfWrite = 0;
		
		  Vis2[ib][j] = Re*Re+Im*Im;
		  Vis[ib][j] = sqrt(Re*Re+Im*Im);
		  phase[ib][j] = atan2(Im, Re) * (float)DEGRAD;
		  ReV[ib][j] = Re;
		  ImV[ib][j] = Im;

		  /*
		    printf("j=%i Wave=%g Re=%g Im=%g phi=%g\n",
		    j,R.fWave,Re,Im,phase[ib][j]);
		    return EXIT_SUCCESS;
		  */
		
		  /* remove any 360-degree discontinuities */
		  /*
		  if (j > j1Wave) {
		    if (fabs(fabs(phase[ib][j]-phase[ib][j-1])-360.)
			< 1.) {
		      if (phase[ib][j-1] > 0.) phase[ib][j] +=360.;
		      if (phase[ib][j-1] < 0.) phase[ib][j] -=360.;
		    }		     
		  }
		   */
		  /*
		    if (phase[ib][j] < -180.1) phase[ib][j] += 360.;
		    else if (phase[ib][j] > 180.1)
		    phase[ib][j] -= 360.;
		  */
		
		  /*
		  //if (ipa == 0 && isc == 0 && inl == 0) {
		  if (j == j1Wave && ipa == 0 && isc == 0 && inl == 0) {
		  //if ((j == j1Wave || j == j2Wave-1) && ipa == 0) {
		  printf("j=%2i Wave=%1.6g Visibility=%1.6g Phase=%1.6g\n", 
		  j,LD[iLD].afWaves[j],Vis[ib][j],phase[ib][j]);
		  printf("     Re=%1.6g Im=%1.6g\n", Re, Im);
		  }
		  */
		} // jWaves
	      } // iBaselines
	    
	      /* Now calculate the Chi^2!! */
	      /* Is this a triangle or separate baselines */
	      if (IfWrite) printf("\nisc=%i HA=%5.2f\n",isc,S[iDS].afHA[isc]);
	      if (S[iDS].iIfTri) {

		/* Do Triangles */
		for (tri=0; tri<S[iDS].iNTri; tri++) {
		  //for (tri=0; tri<1; tri++) { // for 1 triangle of several
		  itri = S[iDS].aiTri[tri]; // index into the triangles
	      
		  for (j = j1Wave; j < j2Wave; j++) {
		    //printf("j(wave)=%i\n",j);
		    ReC=1.;
		    ImC=0.;
		    phiClose=0.;
		    amplitudeClose=1.;
		    //printf("j=%i Wave=%g\n",j,LD[iLD].afWaves[j]);
		  
		    /* Do around the triangle for triple product */
		    for (icp = 0; icp < 3; icp++) {
		      iB1=O[iBL].iTriangleBaselines[itri][icp];
		      /* iB2 indicates the direction along that baseline.  + for
		       * the direction assumed in the baselines, - for the other
		       * way.  Need this for triangle closure.  The sense is
		       * encoded in the sign of iTriangleBaselines.  Therefore
		       * you can't use index=0, see ReadBaselines.
		       */
		      iB2=1;
		      if (iB1 < 0) {
			iB1 = -iB1;
			iB2 = -1;
		      }
		      iB1--;
		      /* A bug!
		      Re = (ReC*ReV[iB1][j]-ImC*ImV[iB1][j])*iB2;
		      Im = (ReC*ImV[iB1][j]+ImC*ReV[iB1][j])*iB2;
		      */
		      Re = (ReC*ReV[iB1][j]-ImC*ImV[iB1][j]*iB2);
		      Im = (ReC*ImV[iB1][j]*iB2+ImC*ReV[iB1][j]);
		      ReC = Re;
		      ImC = Im;
		    }
		
		    phiClose = atan2(ImC,ReC);
		    amplitudeClose = sqrt(ReC*ReC+ImC*ImC);
		    if (isc == 18) IfWrite=0;

		    if (IfWrite) printf("j=%2i w=%5.1f phiC=%7.4f ampC=%6.4f ",j,
					LD[iLD].afWaves[j],phiClose,amplitudeClose);
		    //if (IfWrite) printf("\n");

		    /* Do Deltas only if this wavelength has an observation */
		    for (k=0; k<S[iDS].aiNWaves[isc]; k++) {
		      if (S[iDS].aiWaveIndex[isc][k] == j) {
			if (IfWrite) printf("phiM=%7.4f ampM=%6.4f ",
					    S[iDS].afTPh[isc][k][itri],
					    S[iDS].afTAmp[isc][k][itri]);

			if (j < j2Wave-1) { // bluest point out
			  dphiC = S[iDS].afTPh[isc][k][itri] - phiClose;
			  /*
			    if (IfWrite) printf("k=%i S[iDS].index=%i ",k,S[iDS].
			    aiWaveIndex[isc][k]);
			    if (IfWrite) printf("S[iDS].Ph=%7.4f dphiC0=%7.4f ",
			    S[iDS].afTPh[isc][k],dphiC);
			  */
			  /*
			   */
			  if (fabs(dphiC+(float)TWOPI) < fabs(dphiC)) 
			    dphiC += (float)TWOPI;
			  //if (IfWrite) printf("dphiC1=%7.4f ",dphiC);
			  if (fabs(dphiC-(float)TWOPI) < fabs(dphiC)) 
			    dphiC -= (float)TWOPI;
			  //if (IfWrite) printf("dphiCf=%7.4f S[iDS].PhErr=%7.4f\n",
			  //     dphiC,S[iDS].afTPhErr[isc][k]);
			  dphiC /= S[iDS].afTPhErr[isc][k][itri];
			  if (IfWrite) printf("dphiCf/err=%7.4f ",dphiC);
			  
			  /* Keeps bad data out of the calculation */
			  if (!S[iDS].aiTPFlag[isc][k][itri]) { 
			    Chi2[ipa]+=dphiC*dphiC;
			    iDOF++;
			  }
			} // j<j2Wave-1!

			/* Do Amplitudes only on IfTri=1 */
			if (S[iDS].iIfTri != 2) {
			  dampC = (S[iDS].afTAmp[isc][k][itri] - amplitudeClose);
			  //if (IfWrite) printf("S[iDS].Amp=%g dampC=%6.5f S[iDS].AErr=%6.5f ",
			  //     S[iDS].afTAmp[isc][k],dampC,S[iDS].afTAmpErr[isc][k]);
			  dampC /= S[iDS].afTAmpErr[isc][k][itri];
			  if (IfWrite) printf("dampC/err=%7.4f",dampC);
			  
			  /* Keeps bad data out of the calculation */
			  if (!S[iDS].aiTAFlag[isc][k][itri]) { 
			    Chi2[ipa] += dampC*dampC;
			    iDOF++;
			  }
			  
			  /*      isc tri jw  dA/der dP/der */
			  /*
			    if (fabs(dampC) > 8. || fabs(dphiC) > 8.) {
			    printf("%2i %2i %2i %6.2f %6.2f\n",
			    isc,itri,j,dampC,dphiC);
			    }
			  */
			} // if (IfTri = 1)

		      } // if k(wave)==j(wave)
		    } // k(wave)
		    if (IfWrite) printf("\n");
		  } // jwave
		} // itri
		/*
		  printf("Triangle %i of Scan %i Chi2=%g DOF=%i\n",itri,isc,
		  Chi2[ipa],iDOF);
		*/
	      } // if S[iDS].iIfTri=1
	    
		/* Separate Baselines */
	      if (S[iDS].iIfTri != 1) {
		//printf("NBaselines=%i\n",S[iDS].iNBaselines);
		for (bln=0; bln<S[iDS].iNBaselines; bln++) {
		  ibln = abs(S[iDS].aiBln[bln]); // baseline index
		    
		  //printf("scan=%i HA=%g\n",isc,S[iDS].afHA[isc]);
		  for (j = j1Wave; j < j2Wave; j++) {
		    //printf("j(wave)=%i\n",j);
		      
		    /* Do Deltas only if this wavelength has an observation */
		    for (k=0; k<S[iDS].aiNWaves[isc]; k++) {
		      if (S[iDS].aiWaveIndex[isc][k] == j) {
			  
			/*
			  if (isc == 0) {
			  printf("j=%i k=%i V2obs=%g V2cal=%g ",j,k,
			  S[iDS].afBVissq[isc][k][bln],Vis2[ibln][j]);
			  }
			*/
			  
			dampV2=S[iDS].afBVissq[isc][k][bln]-Vis2[ibln][j];
			dampV2 /= S[iDS].afBVerr[isc][k][bln];
			  
			/*
			  if (isc == 0) {
			  printf("V2err=%g ",S[iDS].afBVerr[isc][k][bln]);
			  printf("dampV2=%g\n",dampV2);
			  }
			*/
			  
			/* Keeps bad data out of the calculation */
			if (!S[iDS].aiBFlag[isc][k][bln]) { 
			  Chi2[ipa] += dampV2*dampV2;
			  iDOF++;
			}
			  
		      } // if k == j
		    } // kWave
		  } // jWave
		} // baselines
	      } // S[iDS].iIfTri=0 or 2
	    
	      IfWrite=0;
	    } // nScans
	  } // DataSets
	    //return EXIT_FAILURE;

	  /*
	  printf("iChi2=%i ipa=%i idp=%i iw=%i inl=%i Chi^2=%g DoF=%i\n",
		 iChi2,ipa,idp,iw,inl,Chi2[iChi2],iDOF);
	  */
	  Chi2[ipa] /= iDOF;
	  fprintf(fp," %7.4f",Chi2[ipa]);
	  //printf("%i %g ",ipa,PA[ipa]);
	} // iPA
	fprintf(fp,"\n");
      } // iDpB
    } // iW
  } // Incl

  fclose(fp);

  return EXIT_SUCCESS;
}
