/***********************************************************************
 * SurfBright: This takes values for Lambda, Teff, log_g and mu (the
 * cosine of the limb angle) and, given a Limb Darkening grid, returns
 * the surface brightness in units of the corresponding Planck Function.
 *
 * Fixed two bugs found by Christian.  On came at high Teff when the
 * fact that there was only one log_g model per Teff was not treated
 * well (line 80 and following).  The other was the inconsistent use of
 * LD->iTeffMax (see ReadLimbDarkening.c), which was corrected at line
 * 38. 28/02/04
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "roche.h"

int SurfBright(LimbDark * LD, float Teff, float glog, float mu, 
	       float * SBright) {

  int iTeffm, iglogm[2], iWaveMon=1, i, j, k, ktop, l;
  float wtTm, wtTp, wtgm[2], wtgp, fBrightWm, fBright;
  float fXT[2], fYT[2], fIT[2], fX, fY, fI, A, B;

  /* Find Teff range */
  //printf("in SB Teff=%5.0f  log g=%5.2f  mu=%5.3f\n", Teff,glog,mu);

  /* If off the cool end, clip it to Teff(min) */
  if (Teff <= LD->afTeff[0]) {
    iTeffm=0;
    wtTm=1.;
  }

  /* Likewise, if off the hot end, clip it to Teff(max) */
  else if (Teff >= LD->afTeff[LD->iTeffMax-1]) {
    iTeffm=LD->iTeffMax-2;
    wtTm=0.;
  }

  else{
    j=-1;
    while (LD->afTeff[++j]<Teff && j < LD->iTeffMax-1) {
      iTeffm=j;
      wtTm=(Teff-LD->afTeff[j+1]) / (LD->afTeff[j]-LD->afTeff[j+1]);
    }
  }
  //printf("Done with Teff iTeffm=%3i\n",iTeffm);

  /* Within Teff, find log_g range */
  for (i=0; i<2; i++) {
    k=iTeffm+i;
    ktop=LD->aiLogG[k]-1;
    /*
    printf("Teff[%1i]=%6.0f  ktop=%2i loggMax=%4.1f\n",i,LD->afTeff[k],
	   ktop,LD->afLogG[ktop]);
    */

    /* If off the high end, clip it to log_g(max) */
    if (glog >= 5.0) {
      iglogm[i]=0;
      wtgm[i]=1.;
    }
    
    /* Likewise, if off the low end, clip it to log_g(min) */
    else if (glog <= LD->afLogG[ktop]) {
      iglogm[i]=ktop-1;
      wtgm[i]=0.;
    }
    
    else{
      j=-1;
      while (LD->afLogG[++j] > glog  &&  j < ktop) {
	iglogm[i]=j;
	wtgm[i]=(glog-LD->afLogG[j+1]) / (LD->afLogG[j]-LD->afLogG[j+1]);
      }
    }
    /* Fix problem that Christian found */
    if (LD->aiLogG[k] == 1) {
      iglogm[i]=0;
      wtgm[i]=1.;
    }
    
    /*
      printf("Done with log g(%1i) iglogm=%2i wtgm[i]=%6.3f log_gm=%4.1f\n",
	 i+1,iglogm[i],wtgm[i],LD->afLogG[iglogm[i]]);
    */
  }

  /* If a "monochromatic" calculation, need to interpolate in lambda too. */
  iWaveMon=1;
  if (!strcmp(LD->sWaveset,"Mon")) iWaveMon=2;
  //if (iWaveMon == 2) printf("Monochromatic case LD->iWave=%2i\n",LD->iWave);

  for (i=0; i<iWaveMon; i++) {
    //printf("Wave=%5.0f\n",LD->afWaves[LD->iWave+i]);

    /* Over Teff */
    for (j=0; j<2; j++) {
      k=iTeffm+j;
      l=iglogm[j];
      //printf("k=%2i l=%2i\n",k,l);
      
      /* Over log_g */
      wtgp=1.-wtgm[j];
      fXT[j] = wtgm[j]*LD->afLD[0][k][l][LD->iWave+i]
	+wtgp*LD->afLD[0][k][l+1][LD->iWave+i];
      fYT[j] = wtgm[j]*LD->afLD[1][k][l][LD->iWave+i]
	+wtgp*LD->afLD[1][k][l+1][LD->iWave+i];
      fIT[j] = wtgm[j]*LD->afLD[2][k][l][LD->iWave+i]
	+wtgp*LD->afLD[2][k][l+1][LD->iWave+i];
      //printf("j=%1i XT=%6.3f YT=%6.3f IT=%6.3f\n",j,fXT[j],fYT[j],fIT[j]);
    }

    wtTp=1.-wtTm;
    fX=wtTm*fXT[0]+wtTp*fXT[1];
    fY=wtTm*fYT[0]+wtTp*fYT[1];
    fI=wtTm*fIT[0]+wtTp*fIT[1];

    /* Evaluate limbdarkening */
    A=(1.-mu);
    if (!strcmp(LD->sLDLaw,"Lin")) B=0.;
    else if (!strcmp(LD->sLDLaw,"Sqrt")) B=1.-sqrt(mu);
    else if (!strcmp(LD->sLDLaw,"Log")) {
      if (mu > 0.) B=mu*log(mu);
      if (mu == 0.) B=0.;
    }
    else printf("LDLaw: %s not Lin, Log, or Sqrt!  Use linear\n",LD->sLDLaw);
    fBright = fI*(1.-A*fX-B*fY);
    //printf("X=%6.3f Y=%6.3f I=%7.4f A=%7.4f B=%7.4f\n",fX,fY,fI,A,B);
    if (i==0) fBrightWm=fBright;
  }

  if (iWaveMon>1) {
    /*
    printf("fBm=%7.4f wtm=%7.4f  fB=%7.4f wtp=%7.4f\n",fBrightWm,
	   LD->fWaveWtMinus,fBright,(1.-LD->fWaveWtMinus));
    */
    fBright=LD->fWaveWtMinus*fBrightWm 
		    + (1.-LD->fWaveWtMinus)*fBright;
  }
  *SBright=fBright;

  /*
  printf("in SB Teff=%5.0f  log g=%5.2f  mu=%5.3f  SurfB=%6.3f\n",
	 Teff,glog,mu,fBright);
  */

  return EXIT_SUCCESS;
}
