; Routines  manipulate VINCI Batch (like VINCIbatch) structures.
; Note: the structures are only VINCIbatch line since the fringe_a
; tabular data is of a size that is only determined at run time so 
; anomyous structures are actually used.

FUNCTION vinci_batch_select, batch
;********************************************************************
;#function#
; vinci_batch_select
;#call#
; vinci_batch_select( batch )
;#description#
; Returns a vector of indices in the batch that are selected
;#inputs#
;\anArg{ batch  }{ VINCIbatch-like  }{ Batch structure }
;#return#
; intarr = array of selected scans in the batch
;#end_function#
;********************************************************************
; which ones have good weights?
  index = where((*batch.scan_data).weight GT 0.0, count)
  return, index
END   ; end of vinci_batch_select

PRO vinci_batch_avg, batch, revision
;********************************************************************
;#procedure#
; vinci_batch_avg
;#call#
; vinci_batch_avg, batch, revision
;#description#
; Average the mu12 and mu22 values and diagnostice and error estimates
; of selected scans in the batch.\\
; Note: passing an array element will have no effect, only passing
; a simple variable will have the desired effect.
;#inputs#
;\anArg{ batch    }{ VINCIbatch-like  }{ Batch structure }
;\anArg{ revision }{ integer  }{ data format revision number }
;#end_procedure#
;********************************************************************
  if (batch.nscans LE 0) then RETURN
; which ones selected?
  index = vinci_batch_select( batch )
  mu12 = 0.0
  mu22 = 0.0
  mu122 = 0.0
  mu222 = 0.0

; accumulate sums if some selected
  if (index[0] GE 0) then begin
    temp = (*batch.scan_data).mu12
    temp = temp[index]
    mu12 = total (temp)
    temp = temp*temp
    mu122 = total (temp)
    temp = (*batch.scan_data).mu22
    temp = temp[index]
    mu22 = total (temp)
    temp = temp*temp
    mu222 = total (temp)
  endif

; convert
  count = size(index)
  count = count[1]
  if (count GT 3) then begin
     mu12 = mu12 / count
     mu22 = mu22 / count
     mu122 = mu122 / count
     mu222 = mu222 / count
     mu122 = sqrt ((mu122-mu12*mu12)/(count-1))
     mu222 = sqrt ((mu222-mu22*mu22)/(count-1))
  endif else begin
    mu12 = 0.0
    mu22 = 0.0
    mu122 = 0.0
    mu222 = 0.0
    batch.batch_data.flag = 1
  endelse

; replace coherence factors
  batch.nscans = count
  batch.batch_data.nscan = count
  batch.batch_data.mu12 = mu12
  batch.batch_data.mu22 = mu22
  batch.batch_data.emu12 = mu122
  batch.batch_data.emu22 = mu222

; get std deviation of opd
  batch.batch_data.sdopd = 0.0
  temp = (*batch.scan_data).opd
  temp = temp[index]
  index2 = where (temp GT -999.0, count2)
  if (count2 GT 0) THEN begin
    temp = temp[index2]
    if (count2 LT 1) then count2 = 1
    sumOpd = total (temp) / count2
    temp = temp*temp
    sumOpd2 = total (temp) / count2
    if (count2 GT 3) then begin
       batch.batch_data.sdopd = sqrt ((sumOpd2-sumOpd*sumOpd))
    endif
  endif

; get std deviation of opd1
  batch.batch_data.sdopd1 = 0.0
  temp = (*batch.scan_data).opd1
  temp = temp[index]
  index2 = where (temp GT -999.0, count2)
  if (count2 GT 0) THEN begin
    temp = temp[index2]
    if (count2 LT 1) then count2 = 1
    sumOpd = total (temp) / count2
    temp = temp*temp
    sumOpd2 = total (temp) / count2
    if (count2 GT 3) then begin
       batch.batch_data.sdopd1 = sqrt ((sumOpd2-sumOpd*sumOpd))
    endif
  endif

; done with revision 1
  if (revision LE 1) then return

; get std deviation of photometry
  for i=0,1 do begin
    temp = (*batch.scan_data).photom[i]
    temp = temp[index]
    sum = total (temp) / count
    temp = temp*temp
    sum2 = total (temp) / count
    if (count GT 3) then begin
       batch.batch_data.photom[i] = sum 
       batch.batch_data.photomsd[i] = sqrt ((sum2-sum*sum))
    endif else begin
       batch.batch_data.photom[i] = 0.0
       batch.batch_data.photomsd[i] = 0.0
    endelse
  endfor

; get std deviation of correlated power moments
  for i=0,2 do begin
    temp = (*batch.scan_data).corpower[i]
    temp = temp[index]
    sum = total (temp) / count
    temp = temp*temp
    sum2 = total (temp) / count
    if (count GT 3) then begin
       batch.batch_data.corpower[i] = sum 
       batch.batch_data.corpowsd[i] = sqrt ((sum2-sum*sum))
    endif else begin
       batch.batch_data.corpower[i] = 0.0
       batch.batch_data.corpowsd[i] = 0.0
    endelse
  endfor
END   ; end of vinci_batch_avg

PRO vinci_batch_edit, batch, revision, minWt=minWt, minSNRP=minSNRP, $
    maxOPDVel=maxOPDVel, maxMom2=maxMom2, sigEdit=sigEdit, clear=clear
;********************************************************************
;#procedure#
; vinci_batch_edit
;#call#
; vinci_batch_edit, batch, revision minWt=minWt, minSNRP=minSNRP, $
;       maxOPDVel=maxOPDVel, maxMom2=maxMom2, /clear
;#description#
; Deselect scans in a batch based on one or more selection criteria.
; Deselection is done by making the weights of the associated scans
; negative.\\
; Note: passing an array element for batch will have no effect, 
; only passing a simple variable will have the desired effect.
;#inputs#
;\anArg{ batch  }{ VINCIbatch-like  }{ Batch structure }
;\anArg{ revision}{ integer }{ data format revision }
;\anArg{ minWt }{ float  }{ Minimum value of the weight}
;\anArg{ minSNRP }{ float  }{ Minimum product of photometric SNRs}
;\anArg{ maxOPDvel }{ float  }{ Maximum abs OPD velocity (OPD1) mu/s}
;\anArg{ maxMom2 }{ float  }{ Maximum 2nd moment of corr. power (mu) }
;\anArg{ clear     }{        }{ if present, clear existing flags }
;#end_procedure#
;********************************************************************
; if clear specified - change all weights to abs value first
  if (KEYWORD_SET(clear)) then $
    (*batch.scan_data).weight = abs ((*batch.scan_data).weight)
          
; save weight array
  tempWt = (*batch.scan_data).weight

; weight selection
  if (KEYWORD_SET(minWt)) then begin
    index = where(tempWt LT minWt, count)
;print," edit by wt ", minWt, count
    if (count GT 0) then begin
      tempWt[index] = -abs(tempWt[index])
    endif
  endif

; photometric snr selection
  if (KEYWORD_SET(minSNRP)) then begin
    temp = (*batch.scan_data).snrp1 * (*batch.scan_data).snrp2
    index = where(temp LT minSNRP, count)
;print," edit by minSNRP ", minSNRP, count
    if (count GT 0) then begin
      tempWt[index] = -abs(tempWt[index])
    endif
  endif

    ; correlated power second moment
    if ((KEYWORD_SET(maxMom2)) AND (revision GT 1)) $
      then begin
        temp = (*batch.scan_data).corpower[2]
        index = where(abs(temp) GT maxMom2*1.0E-6, count)
;print," edit by mom2 ", maxMom2*1.0E-6, count
        if (count GT 0) then begin
          tempWt[index] = -abs(tempWt[index])
        endif
    endif

; automatic editing of values outside sigedit sigma
 if (KEYWORD_SET(sigEdit)) then begin 
   temp = (*batch.scan_data).mu12
   index = where(tempwt GT 0,count)
   if (count GT 1) then begin
     mu12avg = avg(temp(index))
     mu12dev = stddev(temp(index))
     index = where ( abs(temp-mu12avg) GT sigEdit*mu12dev,count)
   if (count GT 0) then tempWt[index] = -abs(tempWt[index])
   endif
   ;
   temp = (*batch.scan_data).mu22
   index = where(tempwt GT 0)
   if (count GT 1) then begin
     mu22avg = avg(temp(index))
     mu22dev = stddev(temp(index))
     index = where ( abs(temp-mu22avg) GT sigEdit*mu22dev,count)
   if (count GT 0) then tempWt[index] = -abs(tempWt[index])
   endif
   ;
 endif

; OPD velocity selection
  if (KEYWORD_SET(maxOPDvel)) then begin
    temp = (*batch.scan_data).opd1
    index = where(abs(temp) GT maxOPDVel*1.0E-6, count)
    if (count GT 0) then begin
;print," edit by OPDvel ", maxOPDvel*1.0E-6, count
      tempWt[index] = -abs(tempWt[index])
    endif
  endif

; save weight array
  (*batch.scan_data).weight = tempWt

; count survivors
   temp = (*batch.scan_data).weight
   index = where(temp GT 0, count)
   batch.batch_data.nscan = count
   batch.nscans = count
;print,"surviving batches ",count

END   ; end of vinci_batch_edit
