FUNCTION vinci_OF_get_UVW, ObsFile, time, source, baseline
;********************************************************************
;#function#
; vinci_OF_get_UVW
;#call#
; vinci_batch_OF_get_UVW, ObsFile, time, source, baseline
;#description#
; Compute the u,v, and w coordinates for a specified time, source 
; and baseline.
; Note: this is not extremely accurate.
;#inputs#
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;\anArg{ time }{ double  }{ MJD for u,v,w }
;\anArg{ source }{ long  }{ Source table index }
;\anArg{ baseline }{ int[2]  }{ telescope indices of the two telescopes }
;#return#
; uvw = double[3] containing u, v, w in m
;#end_function#
;********************************************************************
;  create output
uvw = dblarr(3)

; latitude and long of Paranal (radians)
  lat_vlti  = -0.429838786
  long_vlti = 1.228795511

; get info from tables
; source RA, dec
  index = where (ObsFile.source_data.source_id EQ source)
  ra = ObsFile.source_data[index[0]].raepp
  dec = ObsFile.source_data[index[0]].decepp
  ; these really should be precessed to date.
;print,"source ",ObsFile.source_data[index[0]].source
; baseline vector, values in the geometry table already 
; rotated to correct n-s
  index = where(ObsFile.geom_data.index EQ baseline[0])
  xyz1 = ObsFile.geom_data[index[0]].staxyz
  index = where(ObsFile.geom_data.index EQ baseline[1])
  xyz2 = ObsFile.geom_data[index[0]].staxyz
  xyz = xyz1-xyz2

; compute LST in radians (from Astronomical Almanac)
  mjd = long (time)
  ut = time - mjd
  tu = (mjd+0.5-51545.0) / 36525.0
  ; GMST at ut=0 in sec.
;print,"mjd tu ",mjd,tu
  gmst0ut = 24110.54841 + 8640184.812866*tu + $
     0.093104*tu*tu - 6.2e-6*tu*tu*tu
  gmst0ut = gmst0ut / 86400.0
  itemp = long(gmst0ut)
  gmst0ut = (gmst0ut - itemp) * 6.28315308
;print,"gmst0ut ", gmst0ut*57.29577951/15.0
  lstr = gmst0ut + (ut*6.283185308 - long_vlti) * 0.99726956633
;print,"ut lst ra dec ",ut*24.,lstr* 57.29577951/15.0, ra/15.0, dec

; get Interferometer Hour angle and declination in radians
  ihar = lstr - (ra / 57.29577951)
  decr = dec / 57.29577951
;print, "iha ",ihar*57.29577951/15.0

; compute u,v,w
  cha = cos(ihar)
  sha = sin(ihar)
  cdec = cos (decr)
  sdec = sin (decr)
  clat = cos(lat_vlti)
  slat = sin(lat_vlti)
  ; from Markus Schoeller:
  uvw[0] = xyz[0]*cha - xyz[1]*slat*sha + xyz[2]*clat*sha
  uvw[1] = xyz[0]*sha*sdec + xyz[1]*(slat*cha*sdec + clat*cdec) - $
           xyz[2]*(clat*cha*sdec - slat*cdec) 
  uvw[2] = -xyz[0]*sha*cdec - xyz[1]*(slat*cha*cdec - clat*sdec) + $
           xyz[2]*(clat*cha*cdec + slat*sdec)
;print,"GET uvw ", uvw, (uvw/ 2.128e-6) / 206265.
 
return, uvw
END   ; end of vinci_OF_get_UVW

PRO vinci_OF_cal_vis, batch, ObsFile
;********************************************************************
;#procedure#
; vinci_OF_cal_vis
;#call#
; vinci_OF_cal_vis, batch, ObsFile
;#description#
; Calculate calibrator visibility and transfer function in batch
;#inputs#
;\anArg{ batch  }{ VINCIbatch-like  }{ Batch structure }
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;#end_procedure#
;********************************************************************
; sanity check - don't use calibrators with really low or negative
; coherence factors
  if ((batch.batch_data.mu12 LE 0.01) OR $
      (batch.batch_data.mu22 LE 0.01)) then begin
     batch.batch_data.t12 = -999.99
     batch.batch_data.t22 = -999.99
     batch.batch_data.et12 = -999.99
     batch.batch_data.et22 = -999.99
     return
  endif

; get diameter and error
  index = where (ObsFile.source_data.source_id EQ $
         batch.batch_data.source)
  diam = ObsFile.source_data[index[0]].diameter
  ediam = ObsFile.source_data[index[0]].diam_err
;print,"source ",ObsFile.source_data[index[0]].source

  ; get observing wavelength
  lambda = 2.997924562e8 / ObsFile.source_data[index[0]].freq[0]

  ; spatial frequency (m)
   sf = batch.batch_data.uvw[0]*batch.batch_data.uvw[0] + $
        batch.batch_data.uvw[1]*batch.batch_data.uvw[1]
   sf = sqrt (sf)
  ; convert to cycles per asec 
   sf = (sf / lambda) / 206265.

   ; calibrator visibilities
   arg = 3.141592654 * diam * sf
   vis = 2.0*beselJ(arg,1) / arg
;print,"uvw ", batch.batch_data.uvw, (batch.batch_data.uvw/ 2.128e-6) / 206265.
;print,"arg, diam, sf, vis",arg,diam,sf,vis

; error in visibility squared, use numerical partial derivative
; use standard error propagation
   arg = 3.141592654 * (diam+ediam) * sf
   vis2 = 2.0*beselJ(arg,1) / arg
   partial = 0.0
   if (ediam GT 0.0) then partial = (vis*vis-vis2*vis2) / ediam
   evis = partial * ediam
   if (evis LT 0.0) then evis = 0.0
   batch.batch_data.v12 = vis*vis
   batch.batch_data.v22 = vis*vis
   batch.batch_data.ev12 = evis
   batch.batch_data.ev22 = evis

   ; shape factor
   shape =  batch.batch_data.shape/13.00

   ; calibrator cotransfer functions
   mu12 = batch.batch_data.mu12
   emu12 = batch.batch_data.emu12
   mu22 = batch.batch_data.mu22
   emu22 = batch.batch_data.emu22
   t1 = (batch.batch_data.v12*shape) / mu12
   t2 = (batch.batch_data.v22*shape) / mu22 
;print,"mu12 v12 t12 shape",batch.batch_data.mu12,batch.batch_data.v12,t1,shape

   ; standard error propagation, 
   ; assume error in model and measurements independent
   et1 = t1 * (((evis*evis) / vis) + (emu12*emu12 / mu12))
   et2 = t2 * (((evis*evis) / vis) + (emu22*emu22 / mu22))
   et1 = sqrt (et1)
   et2 = sqrt (et2)

   batch.batch_data.t12 = t1
   batch.batch_data.t22 = t2
   batch.batch_data.et12 = et1
   batch.batch_data.et22 = et2
END   ; end of vinci_OF_cal_vis

FUNCTION vinci_OF_get_transfn, ObsFile, time
;********************************************************************
;#function#
; vinci_OF_get_transfn
;#call#
; vinci_OF_get_transfn(ObsFile, time)
;#description#
; Interpolate the transfer function for a given time.
; Method taken from a memo by Guy Perrin dated July 31, 2000 
;#inputs#
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;\anArg{ time  }{ double  }{ Time at which transfer fn is desired }
;#return#
; transfer function as {t1, et1, t2, et2}
; t1 = -999.99 means it was not possible to determine the results
;#end_function#
;********************************************************************
  out = fltarr(4)  ; output array

; get list of calibrators with valid transfer functions
  index = where ((ObsFile.batches.isCal EQ 1) AND $
                 (ObsFile.batches.batch_data.flag EQ 0) AND $
                 (ObsFile.batches.batch_data.t12 GT 0.0) AND $
                 (ObsFile.batches.batch_data.t22 GT 0.0), ncal)

; make sure there are some
  if (ncal LE 0) then begin
    print,"ERROR: NO valid calibrator transfer functions available"
    out[0] = -999.99
    return, out
  endif

; get calibrator data arrays
  caltime = ObsFile.batches[index].batch_data.time
  calsour = ObsFile.batches[index].batch_data.source
  calt12  = ObsFile.batches[index].batch_data.t12
  calet12 = ObsFile.batches[index].batch_data.et12
  calt22  = ObsFile.batches[index].batch_data.t22
  calet22 = ObsFile.batches[index].batch_data.et22
  calmu12 = ObsFile.batches[index].batch_data.mu12
  calev12 = ObsFile.batches[index].batch_data.ev12
  calmu22 = ObsFile.batches[index].batch_data.mu22
  calev22 = ObsFile.batches[index].batch_data.ev22
; debug
;print, "time ",time
;print, "caltime ",caltime
;print, "calsour ",calsour
;print, "calt12 ",calt12
;print, "calet12 ",calet12
;print, "calt22 ",calt22
;print, "calet22 ",calet22

; find closest entry before time
  dtime = caltime - time
  kndex = where (dtime LE 0, kcount)
  jndex = where (dtime GT 0, jcount)
  if (jcount GT 0) then dtime[jndex] = -999. ; hide later times
  before = -1
  if (kcount GT 0) then begin
    temp = max(dtime, before)
  endif
;print,"before,kcount,kndex ",before,kcount,kndex

; find closest entry after time
  dtime = time - caltime
  kndex = where (dtime LE 0, kcount)
  jndex = where (dtime GT 0, jcount)
  if (jcount GT 0) then dtime[jndex] = -999. ; hide earlier times
  after = -1
  if (kcount GT 0) then begin
    temp = max(dtime, after)
  endif
;print,"after,kcount,kndex ",after,kcount,kndex

;print,"count ncal ",count,ncal
;print, "before after indices ", before,after,dtime,index
; interpolation weights
  if ((before GE 0) AND (after GE 0)) then begin
    wta = abs (time - caltime[before]) / $
          (caltime[after] - caltime[before])
    wtb = 1.0 - wta
  endif
  if ((before GE 0) AND (after LT 0)) then begin
    wtb = 1.0
    wta = 0.0
  endif
  if ((before LT 0) AND (after GE 0)) then begin
    wtb = 0.0
    wta = 1.0
  endif
  if ((before LT 0) AND (after LT 0)) then begin
    ; trouble - but should ever get here
    print,"ERROR2: NO valid calibrator transfer functions available"
    out[0] = -999.99
    return, out
  endif
  if (before LT 0) then before = 0
  if (after LT 0) then after = 0

; interpolate
   t12 = wtb*calt12[before] + wta*calt12[after]
   t22 = wtb*calt22[before] + wta*calt22[after]
   et12 = wtb*wtb*calet12[before]*calet12[before] + $
          wta*wta*calet12[after]*calet12[after]
   et22 = wtb*wtb*calet22[before]*calet22[before] + $
          wta*wta*calet22[after]*calet22[after]

; need a correction to the variance if the two calibrators are the same
   if (calsour[before] EQ calsour[after]) then begin
	mu12b = calmu12[before]
	mu12a = calmu12[after]
        ev12  = calev12[after]
        et12  = et12 + 2.0 * wtb*wta*ev12*ev12 / (mu12b*mu12a)

	mu22b = calmu22[before]
	mu22a = calmu22[after]
        ev22  = calev22[after]
        et22  = et22 + 2.0 * wtb*wta*ev22*ev22 / (mu22b*mu22a)
   endif
  ; convert to standard deviation
  et12 = sqrt(et12)
  et22 = sqrt(et22)
  caltime = ObsFile.batches[index].batch_data.time

;print,"apply t1 t2",t12,t22,et12,et22

  ; save values
   out[0] = t12
   out[1] = et12
   out[2] = t22
   out[3] = et22
          
RETURN, out
END   ; end of vinci_OF_get_transfn

PRO vinci_OF_cal_target, batch, ObsFile
;********************************************************************
;#procedure#
; vinci_OF_cal_target
;#call#
; vinci_OF_cal_target, batch, ObsFile
;#description#
; interpolate calibrator transfer functions and apply to data
;#inputs#
;\anArg{ batch  }  { VINCIbatch-like  }{ Batch structure }
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;#end_procedure#
;********************************************************************
;index = where (ObsFile.source_data.source_id EQ batch.batch_data.source)
;print,"source ",ObsFile.source_data[index[0]].source
   ; get transfer fn
   transfn = vinci_OF_get_transfn(ObsFile, batch.batch_data.time)
   t12  = transfn[0]
   et12 = transfn[1]
   t22  = transfn[2]
   et22 = transfn[3]

; can this be calibrated?
   if (t12 LT -100.0) then begin
     batch.batch_data.v12 = -1.0
     batch.batch_data.v22 = -1.0
     batch.batch_data.ev12 = -1.0
     batch.batch_data.ev22 = -1.0
   endif else begin

   ; target cotransfer functions
     batch.batch_data.t12 = t12
     batch.batch_data.t22 = t22
     batch.batch_data.et12 = et12
     batch.batch_data.et22 = et22

   ; shape factor
     shape =  batch.batch_data.shape/13.00

   ; target visibilities
     mu12 = batch.batch_data.mu12
     mu22 = batch.batch_data.mu22

     emu12 = batch.batch_data.emu12
     emu22 = batch.batch_data.emu22
     v12 = mu12 * t12 / shape
     v22 = mu22 * t22 / shape
     batch.batch_data.v12 = v12
     batch.batch_data.v22 = v22

; standard error analysis assuming uncorrelated errors in mu?? and t??
     ev12 = v12 * ((emu12*emu12/mu12) + (et12*et12/t12))
     ev22 = v22 * ((emu22*emu22/mu22) + (et22*et22/t22))
     batch.batch_data.ev12 = sqrt(ev12)
     batch.batch_data.ev22 = sqrt(ev22)
   endelse
END   ; end of vinci_OF_cal_target

PRO vinci_OF_flag_batch, batch, ObsFile, clear=clear
;********************************************************************
;#procedure#
; vinci_OF_flag_batch
;#call#
; vinci_OF_flag_batch, batch, ObsFile, /clear
;#description#
; flag or unflag a given batch in an OF.
;#inputs#
;\anArg{ batch  }  { int  }{ Batch number }
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;\anArg{ clear     }{        }{ if present, clear flags }
;#end_procedure#
;********************************************************************
  if ((batch LT 0) OR (batch GE ObsFile.nbatch)) then return
  if (KEYWORD_SET(clear)) then begin $
   ; clear flag
    ObsFile.batches[batch].batch_data.flag = 0
    ObsFile.batches[batch].batch_data.reason = " ";
  endif else begin
    ; flag it
    ObsFile.batches[batch].batch_data.flag = 1
    ObsFile.batches[batch].batch_data.reason =ObsFile.batches[batch].batch_data.reason $
      + "User"
  endelse
END   ; end of vinci_OF_flag_batch

PRO vinci_OF_process_batch, ObsFile, minWt=minWt, minSNRP=minSNRP, $
    maxOPDVel=maxOPDVel, maxMom2=maxMom2, sigEdit=sigEdit, clear=clear
;********************************************************************
;#procedure#
; vinci_OF_process_batch
;#call#
; vinci_OF_process_batch, ObsFile, minWt=minWt, minSNRP=minSNRP, $
;    maxOPDVel=maxOPDVel, maxMom2=maxMom2, /clear
;#description#
; Process all the batches of a VINCI Observing file.
; 1) Edit data by selection criteria
; 2) average mu squares and the error estimates
; 3) calculate u,v,w
; 4) calculate calibrator visibilities and transfer function
; Note: pass a simple variable as ObsFile or idl will not do 
; what you expect
;#inputs#
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;\anArg{ minWt }{ float  }{ Minimum value of the weight}
;\anArg{ minSNRP }{ float  }{ Minimum product of photometric SNRs}
;\anArg{ maxOPDvel }{ float  }{ Maximum abs OPD velocity (OPD1)}
;\anArg{ maxMom2 }{ float  }{ Maximum 2nd moment of corr. power (mu) }
;\anArg{ clear     }{        }{ if present, clear existing flags }
;#end_procedure#
;********************************************************************
; loop over batches processing
  for i = 0, ObsFile.nbatch-1 do begin
    ; get scalar version of batch to keep idl from screwing up
    batch = ObsFile.batches[i]
    noClear = 0

    ; editing
    ; weight selection
    if (KEYWORD_SET(minWt)) then begin
      if (KEYWORD_SET(clear)) then begin $
        noClear = 1;
        vinci_batch_edit, batch, ObsFile.revision, minWt=minWt, /clear
      endif else vinci_batch_edit, batch, ObsFile.revision, minWt=minWt
    endif

    ; photometric snr selection
    if (KEYWORD_SET(minSNRP)) then begin
      if (KEYWORD_SET(clear) AND (noClear EQ 0)) then begin $
        noClear = 1;
        vinci_batch_edit, batch, ObsFile.revision, minSNRP=minSNRP, /clear
        endif else vinci_batch_edit, batch, ObsFile.revision, minSNRP=minSNRP
    endif

    ; correlated power second moment
    if ((KEYWORD_SET(maxMom2)) AND (ObsFile.revision GT 1)) $
      then begin
        if (KEYWORD_SET(clear) AND (noClear EQ 0)) then begin $
        noClear = 1;
          vinci_batch_edit, batch, ObsFile.revision, maxMom2=maxMom2, /clear
          endif else vinci_batch_edit, batch, ObsFile.revision, maxMom2=maxMom2
    endif

    ; OPD velocity selection
    if (KEYWORD_SET(maxOPDVel)) then begin
      if (KEYWORD_SET(clear) AND (noClear EQ 0)) then begin $
        noClear = 1;
        vinci_batch_edit, batch, ObsFile.revision, maxOPDVel=maxOPDVel, /clear
        endif else vinci_batch_edit, batch, ObsFile.revision, maxOPDVel=maxOPDVel
    endif

    ; auto edit 
    if (KEYWORD_SET(sigEdit)) then begin
      if (KEYWORD_SET(clear) AND (noClear EQ 0)) then begin $
        noClear = 1;
        vinci_batch_edit, batch, ObsFile.revision, sigEdit=sigEdit, /clear
        endif else vinci_batch_edit, batch, ObsFile.revision, sigEdit=sigEdit 
    endif

    ; average the results
    vinci_batch_avg, batch, ObsFile.revision

; count survivors
    temp = (*batch.scan_data).weight
    index = where(temp GT 0, count)
    batch.batch_data.nscan = count
    batch.nscans = count
;print,"surviving scans ",count, i

; get baseline from the batch or optical_train table as appropriate
   telescope = intarr(2)
   ; if ObsFile.ot_table is a valid object then the
   ; optical train table exists 
   if (OBJ_VALID(ObsFile.ot_table)) then begin
     ; new form, data has reference to optical train.
     ; look up in optical train table
     j = batch.batch_data.opt_train[0]-1
     telescope[0] = ObsFile.ot_data[j].telescope
     j = batch.batch_data.opt_train[1]-1
     telescope[1] = ObsFile.ot_data[j].telescope
   endif else begin
     ; old form, data table has references to telescopes
     telescope = batch.batch_data.telescope
   endelse

    ; set u,v,w if needbe
    if (abs(batch.batch_data.uvw[0])+abs(batch.batch_data.uvw[1])+ $
	abs(batch.batch_data.uvw[2]) LE 0.01) then begin
        uvw = vinci_OF_get_UVW(ObsFile, $
           batch.batch_data.time, batch.batch_data.source, $
           telescope)
        batch.batch_data.uvw[0] = uvw[0]
        batch.batch_data.uvw[1] = uvw[1]
        batch.batch_data.uvw[2] = uvw[2]
    endif

    ; default shapefactor if none given
    if (batch.batch_data.shape LT 0.1) then begin
      batch.batch_data.shape = 13.00
    endif

    ; is this a calibrator (nonblank cal code)?
    index = where (ObsFile.source_data.source_id EQ $
         batch.batch_data.source)
    if (ObsFile.source_data[index[0]].calcode NE "    ") then begin
      batch.isCal = 1
    endif

    ; compute calibrator visibilities, transfer fn
    if ((batch.isCal GT 0)  AND $
      (batch.batch_data.flag EQ 0)) then begin
	vinci_OF_cal_vis, batch, ObsFile
    endif

    ; save modified batch
    ObsFile.batches[i] = batch
  endfor ; end loop over batches

END   ; end of vinci_OF_process_batch

PRO vinci_OF_edit, ObsFile, maxChi2=maxChi2, minSNR=minSNR, $
   maxOPDvsd=maxOPDvsd, minScan=minScan, clear=clear
;********************************************************************
;#procedure#
; vinci_OF_edit
;#call#
; vinci_OF_edit, ObsFile, maxChi2=maxChi2, minSNR=minSNR, $
;    maxOPDvsd=maxOPDvsd, minScan=minScan, /clear
;#description#
; Flag batches based on a given set of criteria:
; 1) chi squared of I1 and I2 visibility estimates
;   maxChi2 [ default 5]
; 2) SNRs = ratioes of mu^2 estimates to estimated errors.
;   minSNR [default 5]
; 3) OPD velocity dispersion (mu/s)
;  maxOPDvsd [default 15]
; 4) minScan = minimum number of scans in a batch [def. 10]
; If a batch is violates any of the conditions, the member 
; ObsFile.batches[?].batch_data.flag is set to 1 and
; ObsFile.batches[?].batch_data.reason is modifies to indicate which
; limits were violated.
; Note: pass a simple variable as ObsFile or idl will not do 
; what you expect
;#inputs#
;\anArg{ ObsFile  }{ VINCIObsFile  }{ Observing File }
;\anArg{ maxChi2 }{ float  }{ Chi square of visibility estimate}
;\anArg{ minSNR }{ float  }{ Minimum visibility SNR }
;\anArg{ maxOPDvsd }{ float  }{ Maximum abs OPD velocity dispersion}
;\anArg{ minScan }{ int  }{ Minimum number of scans in a batch}
;\anArg{ clear     }{        }{ if present, clear existing flags }
;#end_procedure#
;********************************************************************
; apply current calibration for the Chi^2 test to work
  vinci_OF_apply_cal, ObsFile

; set limits
  mxX2   = 5.0
  if (KEYWORD_SET(maxChi2)) then mxX2 = maxChi2
  mnSNR  = 5.0
  if (KEYWORD_SET(minSNR)) then mnSNR = minSNR
  mxOPDv = 15.0e-6
  if (KEYWORD_SET(maxOPDvsd)) then mxOPDv = 1.0e-6*maxOPDvsd
  mnScan = 10
  if (KEYWORD_SET(minScan)) then mnScan = minScan
  flag_count=0
; loop over batches editing
  for i = 0, ObsFile.nbatch-1 do begin
    reason = ""

   ; clear existing flags if requested
   if (KEYWORD_SET(clear)) then begin
      ObsFile.batches[i].batch_data.flag = 0
      ObsFile.batches[i].batch_data.reason = "    "
   endif

; Only do batches not already flagged
    if (ObsFile.batches[i].batch_data.flag EQ 0) then begin
      ; get scalar version of batch to keep idl from screwing up
      batch = ObsFile.batches[i]

      ; tests
      ; Chi squared
      v = 0.5*(batch.batch_data.v12+batch.batch_data.v22)
      Chi2 = 0.0
      if ((batch.batch_data.ev12 GT 0.0) AND (batch.batch_data.ev22 GT 0.0)) then $
        Chi2 = (((v-batch.batch_data.v12)/batch.batch_data.ev12)^2) + $
               (((v-batch.batch_data.v22)/batch.batch_data.ev22)^2)
      if (Chi2 GE mxX2) then begin
        batch.batch_data.flag = 1
        reason = reason + "Chi2 "
        flag_count = flag_count + 1
      endif

      ; min SNR
      SNR1 = 0.0
      SNR2 = 0.0
      if (batch.batch_data.emu12 GT 0.00001) then $
        SNR1 = batch.batch_data.mu12/batch.batch_data.emu12
      if (batch.batch_data.emu22 GT 0.00001) then $
        SNR2 = batch.batch_data.mu22/batch.batch_data.emu22
      if ((SNR1 LT mnSNR) OR (SNR2 LT mnSNR)) then begin
        batch.batch_data.flag = 1
        reason = reason + "SNR "
        flag_count = flag_count + 1
      endif

      ; max OPD velocity
      if (batch.batch_data.sdopd1 GT mxOPDv) then begin
        batch.batch_data.flag = 1
        reason = reason + "OPDvsd "
        flag_count = flag_count + 1
      endif

      ; min number of scans
      if (batch.batch_data.nscan LT mnScan) then begin
;print,batch.batch_data.nscan,i
        batch.batch_data.flag = 1
        reason = reason + "minScan "
        flag_count = flag_count + 1
      endif

      ; save reason string
      ;reason = reason + "                                "
      ;STRPUT, ObsFile.batches[i].batch_data.reason, reason
      batch.batch_data.reason = reason

      ; save modified batch
      if (batch.batch_data.flag EQ 1) then $
          ObsFile.batches[i] = batch

;print," edit batch reason ",i,reason
    endif
  endfor ; end loop over batches
  print,"Flagged ",flag_count," times"
END   ; end of vinci_OF_edit

PRO vinci_OF_edit_scans, ObsFile, batch, first, last
;********************************************************************
;#procedure#
; vinci_OF_edit_scans
;#call#
; vinci_OF_edit_scans, ObsFile,  batch, first, last
;#description#
; Flag a range of scans is a given batch.
;#inputs#
;\anArg{ ObsFile }{ VINCIObsFile  }{ Observing File }
;\anArg{ batch }{ int  }{ which batch (0-rel)}
;\anArg{ first }{ int  }{ first (0-rel) scan to flag }
;\anArg{ last  }{ int  }{ highest (0-rel) scan to flag }
;#end_procedure#
;********************************************************************
;  select batch
  batch = ObsFile.batches[batch]

  for i = first,last do begin
    (*batch.scan_data)[i].weight = -abs((*batch.scan_data)[i].weight)
  endfor
END   ; end of vinci_OF_edit

PRO vinci_OF_apply_cal, ObsFile
;********************************************************************
;#procedure#
; vinci_OF_apply_cal
;#call#
; vinci_OF_apply_cal, ObsFile
;#description#
; Apply calibration to target sources.  
; This is done by interpolating adjacent calibrators
; Note: pass a simple variable as ObsFile or idl will not do 
; what you expect
;#inputs#
;\anArg{ ObsFile  }{ VINCIObsFile-like  }{ Observing File }
;#end_procedure#
;********************************************************************
; another loop over batches calibrating
  for i = 0, ObsFile.nbatch-1 do begin
    if ((ObsFile.batches[i].isCal EQ 0) AND $
      (ObsFile.batches[i].batch_data.flag EQ 0)) then begin
      ; get scalar version of batch to keep idl from screwing up
      batch = ObsFile.batches[i]
      vinci_OF_cal_target, batch, ObsFile
      ; save modified batch
      ObsFile.batches[i] = batch
      endif
    endfor
END   ; end of vinci_OF_apply_cal

PRO vinci_OF_extract_cals, ObsFile, iErr=iErr
;********************************************************************
;#procedure#
; vinci_OF_extract_cals
;#call#
; vinci_OF_extract_cals, ObsFile, iErr=iErr
;#description#
; Make a copy of ObsFile containing only the calibrators and 
; writing this into a file specified by the user.
;#inputs#
;\anArg{ ObsFile }{ VINCIObsFile-like }{ memory resident version }
;#outputs#
;\anArg{ iErr }{ int }{ Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
  ; which ones are calibrators?
  index = where ((ObsFile.batches[*].isCal GT 0), nbatch)
  if (nbatch LE 0) then begin
    print,"No calibrators in data set"
    return
  endif

; create output structure (like VINCIObsFile)
; Note: this should have the same structure as the similar
; definition in vinci_OF_read
   out = {infile:ObsFile.infile, outfile:ObsFile.outfile, rawdata:"   ",             $
          revision:ObsFile.revision, nbatch:nbatch, priHead:ObsFile.priHead,         $
          source_table:ObsFile.source_table, source_data:ObsFile.source_data,        $
          ot_table:ObsFile.ot_table, ot_data:ObsFile.ot_data,                        $
          geom_table:ObsFile.geom_table, geom_data:ObsFile.geom_data,                $
          batch_table:ObsFile.batch_table,                                           $
          fringe_a_table:ObsFile.fringe_a_table,  batches:ObsFile.batches[index]} 

  ; reset batch numbers
  for i = 0, nbatch-1 do begin
     out.batches[i].batch_data.batch_id = i+1
     (*out.batches[i].scan_data).batch = i+1
  endfor

  ; where do you want it to go?
  ; default in input filename with "Cal" prepended.
  inname= ObsFile.infile
  last = strpos(inname, '/', /REVERSE_SEARCH)
  base = strmid(inname,last+1)
  directory = strmid(inname,0,last+1)
  outfile = dialog_pickfile(file=directory+"Cal"+base,$
     title="Select calibrator file name")

; write it if the name was given
  if (STRLEN(outfile[0]) GT 0) then vinci_OF_write, outfile[0], out
END  ; end of vinci_OF_extract_cals
