; Routines to plot VINCI ObsFile structures.

PRO vinci_Plot_Batch, ObsFile, source=source, isValid=isValid,          $
   over=over, number=number, cal=cal, nocal=nocal,                      $
   mu12=mu12, mu22=mu22, t12=t12, t22=t22, v12=v12, v22=v22,            $
   sigOPD=sigOPD, sigRateOPD=sigRateOPD,                                $
   corPow=corPow, mom1=mom1, mom2=mom2, pa=pa, pb=pb,                   $
   sigCorPow=sigCorPow, sigMom1=sigMom1, sigMom2=sigMom2,               $
   sigPa=sigPa, sigPb=sigPb
;********************************************************************
;#procedure#
; vinci_Plot_Batch
;#call#
; vinci_Plot_Batch, ObsFile, source=source, isValid=isValid,      $
;   /over, /number, /cal, /nocal,                                 $
;   /mu12, /mu22, /t12, /t22, /v12, /v22, /sigOPD, /sigOPDrate,   $
;   /corPow, /mom1, /mom2, /pa, /pb,                              $
;   /sigCorPow, /sigMom1, /sigMom2, /sigPa, /sigPb
;#description#
; Plots selected batch average data vs. time or batch number.
; Only one plot may be requested but overplotting is allowed using the 
; /over switch.
; Only unflagged values are plotted
;#inputs#
;\anArg{ ObsFile }{ VINCIObsFile-like }{ Observing file structure }
;\anArg{source      }{string }{Specified source name, may be "All"}
;\anArg{over      }{ }{if present means overplot the previous plot}
;\anArg{number    }{ }{if present means use batch number rather than time}
;\anArg{cal       }{ }{if present means only plot calibrators}
;\anArg{nocal     }{ }{if present means only plot noncalibrators}
;\anArg{mu12      }{ }{if present means plot mu12}
;\anArg{mu22      }{ }{if present means plot mu22}
;\anArg{t12       }{ }{if present means plot t12}
;\anArg{t22       }{ }{if present means plot t22}
;\anArg{v12       }{ }{if present means plot v12}
;\anArg{v22       }{ }{if present means plot v22}
;\anArg{sigOPD    }{ }{if present means plot std. deviation of OPD}
;\anArg{sigRateOPD}{ }{if present means plot std. deviation of OPD rate}
;\anArg{corPow    }{ }{if present means plot correlated power}
;\anArg{mom1      }{ }{plot 1st moment of correlated power spectrum}
;\anArg{mom2      }{ }{plot 2nd moment of correlated power spectrum}
;\anArg{pa        }{ }{if present means plot Photometric A}
;\anArg{pb        }{ }{if present means plot Photometric b}
;\anArg{sigCorPow }{ }{if present means plot sd of correlated power}
;\anArg{sigMom1   }{ }{plot sd of 1st moment of correlated power spectrum}
;\anArg{sigMom2   }{ }{plot sd of 2nd moment of correlated power spectrum}
;\anArg{sigPa     }{ }{if present means plot sd of Photometric A}
;\anArg{sigPb     }{ }{if present means plotsd of  Photometric b}
;#outputs#
;\anArg{isValid }{ integer }{ 1 if request valid, 0 invalid }
;#end_procedure#
;********************************************************************
; get calibrator flag array
  cals = ObsFile.batches.isCal

; find source number if selected
  source_id = -1
  if (KEYWORD_SET(source)) then begin
    if (source EQ "All") then begin
        source_id = -1
    endif else begin
    ; find selected source number
      indx = where (ObsFile.source_data.source EQ source)
      source_id = ObsFile.source_data[indx[0]].source_id
    endelse
  endif

; set X axis
  if (KEYWORD_SET(number)) then begin
    ; use batch number
    xaxislabel = "batch number"
    x = indgen(ObsFile.nbatch)
  endif else begin
    ; use time
    x = ObsFile.batches.batch_data.time
    iday = long(min(x))
    ntime = N_ELEMENTS(x)
    xaxislabel = " UT days since MJD "+STRING(iday)
    x = (x-iday)
  ; convert times to UT hours if less than 2 days
    if (x[ntime-1] LT 2.0) then begin
      xaxislabel = " UT time (hrs)"
      x = x * 24.0
    endif
  endelse

; set Y axis
  yaxislabel = "no valid data specified"
  valid = 0
  y = x
  e = x * (-1)
  if (KEYWORD_SET(mu12)) then begin
    valid = 1
    yaxislabel = "mu12"
    y = ObsFile.batches.batch_data.mu12
    e = ObsFile.batches.batch_data.emu12
  endif
  if (KEYWORD_SET(mu22)) then begin
    valid = 1
    yaxislabel = "mu22"
    y = ObsFile.batches.batch_data.mu22
    e = ObsFile.batches.batch_data.emu22
  endif
  if (KEYWORD_SET(t12)) then begin
    valid = 1
    yaxislabel = "t12"
    y = ObsFile.batches.batch_data.t12
    e = ObsFile.batches.batch_data.et12
  endif
  if (KEYWORD_SET(t22)) then begin
    valid = 1
    yaxislabel = "t22"
    y = ObsFile.batches.batch_data.t22
    e = ObsFile.batches.batch_data.et22
  endif
  if (KEYWORD_SET(v12)) then begin
    valid = 1
    yaxislabel = "v12"
    y = ObsFile.batches.batch_data.v12
    e = ObsFile.batches.batch_data.ev12
  endif
  if (KEYWORD_SET(v22)) then begin
    valid = 1
    yaxislabel = "v22"
    y = ObsFile.batches.batch_data.v22
    e = ObsFile.batches.batch_data.ev22
  endif
  if (KEYWORD_SET(sigOPD)) then begin
    valid = 1
    yaxislabel = "std. dev. OPD (mu)"
    y = 1.0e6*ObsFile.batches.batch_data.sdopd
  endif
  if (KEYWORD_SET(sigRateOPD)) then begin
    valid = 1
    yaxislabel = "std. dev. OPD rate (mu/s)"
    y = 1.0e6*ObsFile.batches.batch_data.sdopd1
  endif
  if (KEYWORD_SET(corPow) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Correlated power"
      y = ObsFile.batches.batch_data.corpower[0]
  endif
  if (KEYWORD_SET(mom1) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "First moment of correlated power spectrum (mu)"
      y = 1.0e6*ObsFile.batches.batch_data.corpower[1]
    endif
  if (KEYWORD_SET(mom2) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Second  moment of correlated power spectrum (mu)"
      y = 1.0e6*ObsFile.batches.batch_data.corpower[2]
  endif
  if (KEYWORD_SET(pa) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Photometric channel A"
      y = ObsFile.batches.batch_data.photom[0]
  endif
  if (KEYWORD_SET(pb) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Photometric channel B"
      y = ObsFile.batches.batch_data.photom[1]
  endif
  if (KEYWORD_SET(sigCorPow) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Std. Dev. Correlated power"
      y = ObsFile.batches.batch_data.corpowsd[0]
  endif
  if (KEYWORD_SET(sigMom1) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Std. Dev. First moment of corr. power spectrum (mu)"
      y = 1.0e6*ObsFile.batches.batch_data.corpowsd[1]
    endif
  if (KEYWORD_SET(sigMom2) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Std. Dev. Second  moment of corr power spectrum (mu)"
      y = 1.0e6*ObsFile.batches.batch_data.corpowsd[2]
  endif
  if (KEYWORD_SET(sigPa) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Std. Dev. Photometric channel A"
      y = ObsFile.batches.batch_data.photomsd[0]
  endif
  if (KEYWORD_SET(sigPb) AND (ObsFile.revision GT 1)) $
    then begin
      valid = 1
      yaxislabel = "Std. Dev. Photometric channel B"
      y = ObsFile.batches.batch_data.photomsd[1]
  endif

; check for valid data
  if (valid LE 0) then begin
    print,"No valid data selected"
    return;
  endif

  if (KEYWORD_SET(isValid)) then isValid = valid

   blist = indgen(ObsFile.nbatch)
; selection, first only unflagged values
  index = where((ObsFile.batches[blist].batch_data.flag EQ 0) AND $
    (y GT -999.),count)
  if (count GE 1) then begin
    x = x[index]
    y = y[index]
    cals = cals[index]
    title = "Batch average values all sources"
    blist = blist[index]

; only calibrators?
    if (KEYWORD_SET(cal) AND (source_id LT 0)) then begin
      index = where(cals EQ 1, ncount)
      if (ncount LE 0) then RETURN
      x = x[index]
      y = y[index]
      cals = cals[index]
      blist = blist[index]
      title = "Batch average values calibrators only"
    endif

; only noncalibrators?
    if (KEYWORD_SET(nocal) AND (source_id LT 0)) then begin
      index = where(cals EQ 0, ncount)
      if (ncount LE 0) then RETURN
      x = x[index]
      y = y[index]
      cals = cals[index]
      blist = blist[index]
      title = "Batch average values noncalibrators only"
    endif

; specific source
    if (source_id GE 0) then begin
      index = where (ObsFile.batches[blist].batch_data.source EQ source_id, ncount)
      if (ncount LE 0) then RETURN
      x = x[index]
      y = y[index]
      cals = cals[index]
      title = "Batch average values for "+source
    endif

; plot or overplot
    if (KEYWORD_SET(over)) then begin
      ; error bar or not
      if e[0] lt 0 then oplot,x,y,psym=2 else oploterr,x,y,e,2
    endif else begin
      if e[0] lt 0 then plot,x,y,xtitle=xaxislabel,ytitle=yaxislabel, $
        title=title, psym=1 $
	else begin
	plot,x,y,xtitle=xaxislabel,ytitle=yaxislabel, $
	title=title, psym=1,/nodata
	oploterr,x,y,e,1
	endelse
    endelse
  endif
END ; end vinci_Plot_Batch

PRO vinci_Plot_Scans, ObsFile, batch, source=source, over=over,     $
   cal=cal, nocal=nocal,                             $
   mu12=mu12, mu22=mu22, snrp1=snrp1, snrp2=snrp2,   $
   opd=opd, rateopd=rateopd, weight=weight,          $
   corPow=corPow, mom1=mom1, mom2=mom2, pa=pa, pb=pb
;********************************************************************
;#procedure#
; vinci_Plot_Scans
;#call#
; vinci_Plot_Scans, ObsFile, batch, source=source, /over, /cal, /nocal, $
;   /mu12, /mu22, /snrp1, /snrp2, /opd, /rateOpd, /weight, $
;   /corPow, /mom1, /mom2, /pa, /pb
;#description#
; Plots selected scan values vs. time.
; Only one plot may be requested but overplotting is allowed using the 
; /over switch.
; Only unflagged values (positive weight) are plotted.
;#inputs#
;\anArg{ ObsFile }{ VINCIObsFile-like }{ Observing file structure }
;\anArg{batch }{intarr }{index numbers of the selected batches, -1=> all}
;\anArg{source      }{string }{Specified source name, may be "All"}
;\anArg{over      }{ }{if present means overplot the previous plot}
;\anArg{cal       }{ }{if present means only plot calibrators}
;\anArg{nocal     }{ }{if present means only plot noncalibrators}
;\anArg{mu12      }{ }{if present means plot mu12}
;\anArg{mu22      }{ }{if present means plot mu22}
;\anArg{snrp1     }{ }{if present means plot snr of photometry channel 1}
;\anArg{snrp2     }{ }{if present means plot snr of photometry channel 2}
;\anArg{opd       }{ }{if present means plot OPD residual}
;\anArg{rateOPD   }{ }{if present means plot OPD rate residual}
;\anArg{weight    }{ }{if present means plot data weight}
;\anArg{sigOPD    }{ }{if present means plot std. deviation of OPD}
;\anArg{sigRateOPD}{ }{if present means plot std. deviation of OPD rate}
;\anArg{corPow    }{ }{if present means plot correlated power}
;\anArg{mom1      }{ }{plot 1st moment of correlated power spectrum}
;\anArg{mom2      }{ }{plot 2nd moment of correlated power spectrum}
;\anArg{pa        }{ }{if present means plot Photometric A}
;\anArg{pb        }{ }{if present means plot Photometric b}
;#end_procedure#
;********************************************************************
; must accumulate arrays by concatenation, loop over batches
  if (batch[0] LT 0) then begin
    ; all batches
    blist = indgen(ObsFile.nbatch)
  endif else begin
    blist = batch
  endelse
; find source number if selected
  source_id = -1
  if (KEYWORD_SET(source)) then begin
    if (source EQ "All") then begin
        source_id = -1
    endif else begin
    ; find selected source number
      indx = where (ObsFile.source_data.source EQ source)
      source_id = ObsFile.source_data[indx[0]].source_id
    endelse
  endif


; labeling
  title = "Scan values all sources"
; only calibrators?
  if (KEYWORD_SET(cal)) then title = "Scan values calibrators only"
; only noncalibrators?
  if (KEYWORD_SET(nocal)) then title = "Scan values noncalibrators only"
; specific source
  if (source_id GE 0) then title = "Scan values for "+source

; sort out the various plotting possibilities
  type = 0
  yaxislabel = "no valid data specified"
  if (KEYWORD_SET(mu12)) then begin
    yaxislabel = "mu12"
    type = 1
  endif
  if (KEYWORD_SET(mu22)) then begin
    yaxislabel = "mu22"
    type = 2
  endif
  if (KEYWORD_SET(snrp1)) then begin
    yaxislabel = "SNR P1"
    type = 3
  endif
  if (KEYWORD_SET(snrp2)) then begin
    yaxislabel = "SNR P2"
    type = 4
  endif
  if (KEYWORD_SET(opd)) then begin
    yaxislabel = "opd (mu)"
    type = 5
  endif
  if (KEYWORD_SET(rateopd)) then begin
    yaxislabel = "OPD rate (mu/s)"
    type = 6
  endif
  if (KEYWORD_SET(weight)) then begin
    yaxislabel = "data weight"
    type = 7
  endif
  if (KEYWORD_SET(corPow) AND (ObsFile.revision GT 1)) $
    then begin
      type = 8
      yaxislabel = "Correlated power"
  endif
  if (KEYWORD_SET(mom1) AND (ObsFile.revision GT 1)) $
    then begin
      type = 9
      yaxislabel = "First moment of correlated power spectrum (mu)"
    endif
  if (KEYWORD_SET(mom2) AND (ObsFile.revision GT 1)) $
    then begin
      type = 10
      yaxislabel = "Second  moment of correlated power spectrum (mu)"
  endif
  if (KEYWORD_SET(pa) AND (ObsFile.revision GT 1)) $
    then begin
      type = 11
      yaxislabel = "Photometric channel A"
  endif
  if (KEYWORD_SET(pb) AND (ObsFile.revision GT 1)) $
    then begin
      type = 12
      yaxislabel = "Photometric channel B"
  endif

; check for valid data
  if (type LE 0) then begin
    print,"No valid data selected"
    return;
  endif

;  loop over batches
  first = 1
  for j=0,N_ELEMENTS(blist)-1 do begin
    i = blist[j]; which batch?

    ; do calibrator selection here
    doit = 1
    if (KEYWORD_SET(cal) AND (ObsFile.batches[i].isCal NE 1)) $
      then doit = 0
    if (KEYWORD_SET(nocal) AND (ObsFile.batches[i].isCal NE 0)) $
      then doit = 0
    if (source_id GT 0) then begin
      doit = 0
      if (source_id EQ ObsFile.batches[i].batch_data.source) then doit = 1
    endif

    ; get list of valid (positive weight) data
    good = where((*ObsFile.batches[i].scan_data).weight GT 0.0, goodcnt)
    print,goodcnt
    if (type EQ 5) then $
      good = where(((*ObsFile.batches[i].scan_data).weight GT 0.0) AND $
        ((*ObsFile.batches[i].scan_data).opd GT -999.9), goodcnt)
    if (type EQ 6) then $
      good = where(((*ObsFile.batches[i].scan_data).weight GT 0.0) AND $
        ((*ObsFile.batches[i].scan_data).opd1 GT -999.9), goodcnt)
    if (goodcnt LT 2) then doit = 0

    if (doit EQ 1) then begin
      ; first time is different - initialize the arrays
      if (first EQ 1) then begin
        first = 0
	x = (*ObsFile.batches[i].scan_data)[good].time
        case type of
          1: y = (*ObsFile.batches[i].scan_data)[good].mu12
          2: y = (*ObsFile.batches[i].scan_data)[good].mu22
          3: y = (*ObsFile.batches[i].scan_data)[good].snrp1
          4: y = (*ObsFile.batches[i].scan_data)[good].snrp2
          5: y = 1.0e6*(*ObsFile.batches[i].scan_data)[good].opd
          6: y = 1.0e6*(*ObsFile.batches[i].scan_data)[good].opd1
          7: y = (*ObsFile.batches[i].scan_data)[good].weight
          8: y = (*ObsFile.batches[i].scan_data)[good].corpower[0]
          9: y = 1.0e6*(*ObsFile.batches[i].scan_data)[good].corpower[1]
          10: y = 1.0e6*(*ObsFile.batches[i].scan_data)[good].corpower[2]
          11: y = (*ObsFile.batches[i].scan_data)[good].photom[0]
          12: y = (*ObsFile.batches[i].scan_data)[good].photom[1]
        else: y = x ; a user error
        endcase
      endif else begin
        ; after the first concatenate
	x = [x, (*ObsFile.batches[i].scan_data)[good].time]
        case type of
          1: y = [y, (*ObsFile.batches[i].scan_data)[good].mu12]
          2: y = [y, (*ObsFile.batches[i].scan_data)[good].mu22]
          3: y = [y, (*ObsFile.batches[i].scan_data)[good].snrp1]
          4: y = [y, (*ObsFile.batches[i].scan_data)[good].snrp2]
          5: y = [y, 1.0e6*(*ObsFile.batches[i].scan_data)[good].opd]
          6: y = [y, 1.0e6*(*ObsFile.batches[i].scan_data)[good].opd1]
          7: y = [y, (*ObsFile.batches[i].scan_data)[good].weight]
          8: y = [y, (*ObsFile.batches[i].scan_data)[good].corpower[0]]
          9: y = [y, 1.0e6*(*ObsFile.batches[i].scan_data)[good].corpower[1]]
          10: y = [y, 1.0e6*(*ObsFile.batches[i].scan_data)[good].corpower[2]]
          11: y = [y, (*ObsFile.batches[i].scan_data)[good].photom[0]]
          12: y = [y, (*ObsFile.batches[i].scan_data)[good].photom[1]]
        else: y = [y, x] ; a user error
        endcase
      endelse
    endif
  end ; end loop over batches

; convert times to UT hours if less than 5 days
  iday = long(min(x))
  ntime = N_ELEMENTS(x)
  x = x - iday
  xaxislabel = " UT days since MJD "+STRING(iday)
  ; convert times to UT hours if less than 2 days
    if (x[ntime-1] LT 2.0) then begin
      x = x * 24.0
      xaxislabel = "UT time (hrs)"
    endif

; plot or overplot
  if (KEYWORD_SET(over)) then begin
    oplot,x,y,psym=2
  endif else begin
    plot,x,y,xtitle=xaxislabel,ytitle=yaxislabel, $
      title=title, psym=1
  endelse
END ; end vinci_Plot_Scans
