;************************************************************
;#class#
;VINCI ews utilities
;#description#
;a variety of programs dealing with (mostly raw) VINCI datasets
;#end_class#
;**************************************************************

FUNCTION vinciPowerSpec, fileSpec, freq, lightFreq, $
   microns, noaverage=noaverage
;****************************************************************************
;#function#
;vinciPowerSpec
;#description#
;calculate (average) power spectra from any number of VINCI batch files 
;as long as the batch format (scans,frames) is the same
;for all files.
;output is in the form of fltarr(nFrames/2, 5, nFiles) unless /noaverage
;is specified.  Then it is fltarr(nFrames/2, 5, nFiles)
;The 5 channels are those specified in vinciRawScans.
;If /noaverage is specified, each row is the abs(FFT)^2 of the input scan
;with zero frequency at the origin
;Only one half of the spectrum is return because input is real.
;By default, if /noaverage is not specified, the powerspectra for
;each whole batch are averaged.
;#call#
;powerSpectra = vinciPowerSpec(files, freq, lightFreq, microns, noaverage=noaverage)
;#inputs#
;\anArg{fileSpec}{file specs}{can be one or an array of file names or }
;\anArg{-}{-}{a fitsFileList Objects}
;#outputs#
;\anArg{freq}{fltarr(nFrame/2}{optional:independent variable:Freq (Hz) corresponding to }
;\anArg{-}{-}{points on power spectrum}
;\anArg{lightFreq}{fltarr}{same as freq but converted to frequency of light}
;\anArg{-}{-}{by multiplying by speed of light/scan speed}
;\anArg{-}{-}{assuming no atmospheric piston rate}
;\anArg{microns}{fltarr}{same as lightfreq but converted wavelength(microns)}
;#return#
;\anArg{-}{fltarr}{set of power spectra as defined in description}
;#end_function#
;****************************************************************************
   nAv = KEYWORD_SET(noaverage)
;figure out input type
   iErr = 1
   fileList = ''
   inType = SIZE(fileSpec,/type)
   if (7 EQ inType) then fileList = fileSpec $
   else if (11 eq inType) then begin 
         if ('FITSFILELIST' EQ OBJ_CLASS(fileSpec)) then $
            fileList = fileSpec->files() 
      endif
   if (fileList[0] EQ '') then begin
      print,' input must be a file, list of files or non-empty'
      print,'fileList object'
      RETURN,0
   endif
   nFiles = N_ELEMENTS(fileList)
   if (nFiles LE 0) then RETURN, 0
;open 1st file and get formatting stuff
   dataTable = OBJ_NEW('fitsTable', fileList[0], $
      extName='IMAGING_DATA', iErr=iErr)
   if (iErr NE 0) then begin
      print,'cant open input file ',fileList[0]
      RETURN,0
   endif
   priHead = dataTable->priHead()
   nScan = priHead->getPar('NC NSCAN')
   nFrames = priHead->getPar('RM NFRAM')
   halfSize = nFrames/2 -1
   baseFreq = priHead->getPar('QL FRAMERATE')
;frequency after FFT in Hz
   freq = baseFreq * findgen(nFrames/2)/nFrames
   scanSpeed = priHead->getPar('QL FRINGESPD')
   c = 2.9979e8 ; speed of light m/s
   lightFreq = c * baseFreq * findgen(nFrames/2) /(nFrames * scanSpeed)
   microns = c * 1.e6 / (lightFreq>lightFreq(1)) ; convert meters to microns
   dataTable->close
   OBJ_DESTROY,dataTable
   OBJ_DESTROY, priHead
;create output data arrays
   if (nAv) then power = fltarr(nFrames/2, nScan, 5, nFiles) $
      else power = fltarr(nFrames/2,5, nFiles)
   power = reform(power)
;loop through files
   for iFile = 0, nFiles -1 do begin
      dataTable = OBJ_NEW('fitsTable',fileList[iFile], extName='IMAGING_DATA', iErr=iErr)
      if (iErr NE 0) then begin
         print,'Error opening data table in file ',fileList[iFile]
         RETURN, 0
      endif
;check compatibility
      priHead = dataTable->priHead()
      newScan = priHead->getPar('NC NSCAN')
;if we are taking batch averages it doesnt matter if the number of scans varies
      fileOK = (nScan EQ newScan) OR (NOT nAv)
      if(fileOK) then fileOK = nFrames EQ priHead->getPar('RM NFRAM')
      if(fileOK) then fileOK = baseFreq EQ priHead->getPar('QL FRAMERATE')
      if(fileOK) then fileOK = scanSpeed EQ priHead->getPar('QL FRINGESPD')
      if (NOT fileOK) then begin
	print, 'format of file:',fileList[iFile],' different from ',files[0]
	print, 'scans ', nScan, newScan
	print,'frames ', nFrames, priHead->getPar('RM NFRAM')
	print, 'framerate ', baseFreq, priHead->getPar('QL FRAMERATE')
	print,'scanSpeed ', priHead->getPar('QL FRINGESPD')
	RETURN,0
      endif
      data=dataTable->readRows(iErr=iErr)
      if (iErr NE 0) then begin
         print,'Error reading data table in file ',onFile
         RETURN,0
      endif
      for chan = 0,4 do begin
         ps = fltarr(nFrames)
         dataC = REFORM(data.(chan+1),nFrames, newScan)
         if (nAv) then begin 
	    for iScan = 0, nScan-1 do begin 
	       ps = fft(dataC(*,iScan),1)
	       power(*, iScan, chan, iFile) = $
		  (float(ps(0:halfSize)))^2 + (imaginary(ps(0:halfSize)))^2
            endfor
         endif else begin
            for iScan = 0, newScan -1 do begin
	       p = fft(dataC(*,iScan),1)
	       ps = ps + (float(p(0:halfSize)))^2+(imaginary(p(0:halfSize)))^2
            endfor
            power(*,chan, iFile) = ps/newScan
         endelse
      endfor
      dataTable->close
      OBJ_DESTROY,dataTable
      OBJ_DESTROY,priHead 
   endfor
RETURN, power
END

FUNCTION vinciHalfPower, fileSpec, freq, lightFreq, microns, noaverage=noaverage
;*****************************************************************************
;#function#
;vinciHalfPower
;#call#
;PowerArray = vinciHalfPower(files, freq, lightFreq, microns, noaverage=noaverage)
;#description#
;Identical to vinciPowerSpec but subtracts powerspectrum derived from ends of scans
;from that calculated in middle.  This eliminates low frequency noise but
;increases high frequency noise.  
;#inputs#
;see vinciPowerSpec
;#return#
;see vinciPowerSpec
;#end_function#
;*****************************************************************************
   nAv = KEYWORD_SET(noaverage)
;figure out input type
   iErr = 1
   fileList = ''
   inType = SIZE(fileSpec,/type)
   if (7 EQ inType) then fileList = fileSpec $
   else if (11 eq inType) then begin 
         if ('FITSFILELIST' EQ OBJ_CLASS(fileSpec)) then $
            fileList = fileSpec->files() 
      endif
   if (fileList[0] EQ '') then begin
      print,' input must be a file, list of files or non-empty'
      print,'fileList object'
      RETURN,0
   endif
   nFiles = N_ELEMENTS(fileList)
   if (nFiles LE 0) then RETURN, 0
;open 1st file and get formatting stuff
   dataTable = OBJ_NEW('fitsTable', fileList[0], $
      extName='IMAGING_DATA', iErr=iErr)
   if (iErr NE 0) then begin
      print,'cant open input file ',fileList[0]
      RETURN,0
   endif
   priHead = dataTable->priHead()
   nScan = priHead->getPar('NC NSCAN')
   nFrames = priHead->getPar('RM NFRAM')
   halfSize = nFrames/2 -1
   baseFreq = priHead->getPar('QL FRAMERATE')
;frequency after FFT in Hz
   freq = baseFreq * findgen(nFrames/2)/nFrames
   scanSpeed = priHead->getPar('QL FRINGESPD')
   c = 2.9979e8 ; speed of light m/s
   lightFreq = c * baseFreq * findgen(nFrames/2) /(nFrames * scanSpeed)
   microns = c * 1.e6 / (lightFreq>lightFreq(1)) ; convert meters to microns
   dataTable->close
   OBJ_DESTROY,dataTable
   OBJ_DESTROY, priHead
;create output data arrays
   if (nAv) then power = fltarr(nFrames/2, nScan, 5, nFiles) $
      else power = fltarr(nFrames/2,5, nFiles)
   power = reform(power)
   timeFilter = replicate(0, nFrames)
   timeFilter((nFrames/4):(3*nFrames/4)) = 1
;loop through files
   for iFile = 0, nFiles -1 do begin
      dataTable = OBJ_NEW('fitsTable',fileList[iFile], extName='IMAGING_DATA', iErr=iErr)
      if (iErr NE 0) then begin
         print,'Error opening data table in file ',fileList[iFile]
         RETURN, 0
      endif
;check compatibility
      priHead = dataTable->priHead()
      newScan = priHead->getPar('NC NSCAN')
;if we are taking batch averages it doesnt matter if the number of scans varies
      fileOK = (nScan EQ newScan) OR (NOT nAv)
      if(fileOK) then fileOK = nFrames EQ priHead->getPar('RM NFRAM')
      if(fileOK) then fileOK = baseFreq EQ priHead->getPar('QL FRAMERATE')
      if(fileOK) then fileOK = scanSpeed EQ priHead->getPar('QL FRINGESPD')
      if (NOT fileOK) then begin
	print, 'format of file:',fileList[iFile],' different from ',files[0]
	print, 'scans ', nScan, newScan
	print,'frames ', nFrames, priHead->getPar('RM NFRAM')
	print, 'framerate ', baseFreq, priHead->getPar('QL FRAMERATE')
	print,'scanSpeed ', priHead->getPar('QL FRINGESPD')
	RETURN,0
      endif
      data=dataTable->readRows(iErr=iErr)
      if (iErr NE 0) then begin
         print,'Error reading data table in file ',onFile
         RETURN,0
      endif
      for chan = 0,4 do begin
         ps = fltarr(nFrames)
         dataC = REFORM(data.(chan+1),nFrames, newScan)
         if (nAv) then begin 
	    for iScan = 0, nScan-1 do begin 
               ps = fft(timefilter*dataC(*,iScan),1)
	       power(*, iScan, chan, iFile) = $
                  (float(ps(0:halfSize)))^2 + (imaginary(ps(0:halfSize)))^2
               ps = fft((1-timefilter)*dataC(*,iScan),1)
	       power(*, iScan, chan, iFile) = $
	       power(*, iScan, chan, iFile) - $
                  (float(ps(0:halfSize)))^2 + (imaginary(ps(0:halfSize)))^2
            endfor
         endif else begin
            for iScan = 0, newScan -1 do begin
	       p = fft(timeFilter*dataC(*,iScan),1)
	       ps = ps + (float(p(0:halfSize)))^2+(imaginary(p(0:halfSize)))^2
	       p = fft((1-timeFilter)*dataC(*,iScan),1)
	       ps = ps - (float(p(0:halfSize)))^2+(imaginary(p(0:halfSize)))^2
            endfor
            power(*,chan, iFile) = ps/newScan
         endelse
      endfor
      dataTable->close
      OBJ_DESTROY,dataTable
      OBJ_DESTROY,priHead 
   endfor
RETURN, power
END
