; routines to view raw VINCI data
; these use vndrs_Waterpower and fv

FUNCTION vinci_filename, inname
;********************************************************************
;#function#
; vinci_filename
;#call#
; vinci_filename(filename)
;#description#
; Checks to see if the file can be opened by inname and if so returns
; inname.  
; If this fails try prepending "$vinciRawData/" to base file name.
; If this fails then return filename with path stripped off (Unix only)
;#inputs#
;\anArg{ inname  }{ string  }{ possible full name of file }
;#return#
; string = only the filename with the path stripped.
;#end_function#
;********************************************************************
; is input name good enough?
  good=findfile(inname)
  if (STRLEN(good[0]) GT 0) then return, inname

; try prepending $vinciRawData/ to base filename
  last = strpos(inname, '/', /REVERSE_SEARCH)
  base = strmid(inname,last+1)
  try = "$vinciRawData/"+base
  good=findfile(try)
  if (STRLEN(good[0]) GT 0) then return, try

; return base as final
  return, base
END; end vinci_filename

PRO vinci_view_power, batch, ObsFile
;********************************************************************
;#procedure#
;  vinci_view_power
;#call#
;  vinci_view_power, batch, ObsFile
;#description#
; Conjure a display in fv of the power spectrum of the raw data associated 
; with a given batch. 
; The raw data should be either:
; 0) if ObsFile.rawdata contains a string, it is used as the directory,
;    (should end in '/') otherwise:
; 1) in it's original location,
; 2) in directory given by environment variable $vinciRawData, or
; 3) in the current working directory
; The top 10 rows of each image are the average spectrum.
; Uses the following script ($vincidldrs/bin/FV_Power.sh):
;#!/bin/sh
;# Script run waterpower program and display in fv
;# usage example:
;# >  FV_Power.sh input.fits
;~/bin/vndrs_Waterpower -in $1 -out temp.fits
;fv -cmap 2 temp.fits
;rm temp.fits
;#inputs#
;\anArg{ batch  }{ int  }{ 0-rel batch number}
;\anArg{ ObsFile }{ VINCIObsFile-like }{ VINCI observation file }
;#end_procedure#
;********************************************************************
  file_on = ObsFile.batches[batch].batch_data.file_on
  if (STRLEN (ObsFile.rawdata) GT 0) then begin
    last = strpos(file_on, '/', /REVERSE_SEARCH)
    base = strmid(file_on,last+1)
    filename = ObsFile.rawdata+base
  endif else begin
    ; try original full path or standard variations
    filename = vinci_filename(file_on)
  endelse

  ; check file
  good=findfile(filename)
  if (STRLEN(good[0]) GT 0) then begin
    spawn, "$vincidldrs/bin/FV_Power.sh "+filename
  endif else begin
    print,"Cannot open raw data file "+filename
  endelse
END ; end vinci_view_power

PRO vinci_view_raw, batch, ObsFile
;********************************************************************
;#procedure#
;  vinci_view_raw
;#call#
;  vinci_view_raw, batch, ObsFile
;#description#
; Conjure a display in fv of the raw data associated 
; with a given batch. 
; The raw data should be either:
; 0) if ObsFile.rawdata contains a string, it is used as the directory,
;    (should end in '/') otherwise:
; 1) in it's original location,
; 2) in directory given by environment variable $vinciRawData, or
; 3) in the current working directory
; Uses the following script ($vincidldrs/bin/FV_Waterfall.sh):
;#!/bin/sh
;# Script run waterfall program and display in fv
;# usage example:
;# >  FV_Waterfall.sh input.fits
;~/bin/vndrs_Waterfall -in $1 -out temp.fits
;fv -cmap 2 temp.fits
;rm temp.fits
;#inputs#
;\anArg{ batch  }{ int  }{ 0-rel batch number}
;\anArg{ ObsFile }{ VINCIObsFile-like }{ VINCI observation file }
;#end_procedure#
;********************************************************************
  file_on = ObsFile.batches[batch].batch_data.file_on
  if (STRLEN (ObsFile.rawdata) GT 0) then begin
    last = strpos(file_on, '/', /REVERSE_SEARCH)
    base = strmid(file_on,last+1)
    filename = ObsFile.rawdata+base
  endif else begin
    ; try original full path or standard variations
    filename = vinci_filename(file_on)
  endelse

  ; check file
  good=findfile(filename)
  if (STRLEN(good[0]) GT 0) then begin
    spawn, "$vincidldrs/bin/FV_Waterfall.sh "+filename
  endif else begin
    print,"Cannot open raw data file "+filename
  endelse
END ; end vinci_view_raw
