; Routines to write reports for VINCI data

FUNCTION vinci_julday, mjd
;********************************************************************
;#function#
;  vinci_julday
;#call#
;  vinci_index(mjd)
;#description#
; Create an index of a dataset as a list of character strings, one per batch. 
;#inputs#
;\anArg{mjd }{ double }{ modified Julian date }
;#return#
; Time and date as 'yyyy-mm-dd hh:mm:ss '
;#end_function#
;********************************************************************
; convert to full julian date
  jd = mjd + 2400000.5
  caldat, jd,month,day,year,hour,min,sec
  timedate = string (format='(I4,"-",I2.2,"-",I2.2,1X,I2.2,":",I2.2,":",I2.2,1X)',$
             year,month,day,hour,min,sec)
return,timedate
END ; end vinci_julday

PRO vinci_report, outfile, ObsFile
;********************************************************************
;#procedure#
;  vinci_report
;#call#
;  vinci_report, outfile, ObsFile
;#description#
; Write a report summarizing the VINCI data in ObsFile
;#inputs#
;\anArg{ outfile  }{ string  }{ name of output report file}
;\anArg{ ObsFile }{ VINCIObsFile-like }{ VINCI observation file }
;#end_procedure#
;********************************************************************
; apply current calibration to be sure
  vinci_OF_apply_cal, ObsFile

; open output
  OPENW,UNIT,outfile,/GET_LUN
; header
 printf,unit,format='("VINCI/VLTI Interferometer")'
 printf,unit,format='("Report for observation file ",A//)', $
   ObsFile.infile
 printf,unit,format='("Science targets")'
 printf,unit,format='("Name                  Date    time    Sp. freq  az lambda    V^2      eV^2   Chi^2")'
 printf,unit,format='("                                      asec^-1 deg ")'


; loop over target batches 
  for i = 0, ObsFile.nbatch-1 do begin
    index_source=ObsFile.batches[i].batch_data.source
;print,i,ObsFile.batches[i].isCal,ObsFile.batches[i].batch_data.source $
;    ,ObsFile.source_data[index_source-1].source
    if ((ObsFile.batches[i].isCal EQ 0) AND $
       (ObsFile.batches[i].batch_data.flag EQ 0)) then begin
      batch = ObsFile.batches[i]
      index = where (ObsFile.source_data.source_id EQ $
        batch.batch_data.source)
      ; get observing wavelength
      lambda = 2.997924562e8 / ObsFile.source_data[index[0]].freq[0]
      v = 0.5*(batch.batch_data.v12+batch.batch_data.v22)
      t1=sqrt(1/(batch.batch_data.t12/batch.batch_data.shape/13.00))
      t2=sqrt(1/(batch.batch_data.t22/batch.batch_data.shape/13.00))
      t=(0.5*(t1+t2))
      ev1=batch.batch_data.ev12
      ev2=batch.batch_data.ev22
      v1=batch.batch_data.v12
      v2=batch.batch_data.v22

;  Make sure values are OK
      if ((v1 LT -0.5) OR (v2 LT -0.5)) then begin
      ; bad values
        v = -1.0
        ev = -1.0
        Chi2 = 0.0
      endif else begin
; print,"v1,v2",v1,v2,ev1,ev2
        v=(1/(ev1+ev2-(2*t*ev1*ev2)))*(v1*(ev2-(t*ev1*ev2))+ $
    	 v2*(ev1-(t*ev1*ev2)))
        ev=((1-t)*ev1*ev2)/(ev1+ev2-(2*t*ev1*ev2))
        ; assuming correlation is zero
        v = (v1*ev2+v2*ev1)/(ev1+ev2)
;        ev = sqrt (ev1*ev2/(ev1+ev2))
	ev=  sqrt ( 1./(1./(ev1*ev1) + 1./(ev2*ev2)) )
        Chi2 = (((v-v1)/ev1)^2) +  (((v-v2)/ev2)^2)
      endelse
      sf = batch.batch_data.uvw[0]*batch.batch_data.uvw[0] + $
        batch.batch_data.uvw[1]*batch.batch_data.uvw[1]
      sf = sqrt (sf)
     ; convert to cycles per asec
      sf = (sf / lambda) / 206265.
      name = ObsFile.source_data[index_source-1].source
      mjd = vinci_julday(batch.batch_data.time)
      az = (57.296 * atan (batch.batch_data.uvw[1], batch.batch_data.uvw[0]))
      printf,unit,format='(a16,a20,f7.2,f9.2,f7.3,2f9.3,f6.2)', $
        name, mjd, sf, az, lambda*1e6, v, ev, Chi2
    endif
  endfor ; end loop over target batches 

 printf,unit,format='(//"all OBs observed coherence factors")'
 printf,unit,format='("  Name             cal  Date       time      mu1  emu1   mu2  emu2 ")'
; loop over all batches 
  for i = 0, ObsFile.nbatch-1 do begin
    index_source=ObsFile.batches[i].batch_data.source
    if (ObsFile.batches[i].batch_data.flag EQ 0) then begin
      batch = ObsFile.batches[i]
      index = where (ObsFile.source_data.source_id EQ $
        batch.batch_data.source)
      if (batch.isCal EQ 0) then cal=" "
      if (batch.isCal NE 0) then cal="C"
      mu1=batch.batch_data.mu12
      mu2=batch.batch_data.mu22
      emu1=batch.batch_data.emu12
      emu2=batch.batch_data.emu22
      name = ObsFile.source_data[index_source-1].source
      mjd = vinci_julday(batch.batch_data.time)
      printf,unit,format='(i3,1x,a16,a1,a20,1x,4f6.3)', $
        i,name, cal, mjd, mu1,emu1,mu2,emu2
    endif
  endfor ; end loop over all batches 

 printf,unit,format='(//"all OBs derived values")'
 printf,unit,format='("  Name             cal   v1    ev1    v2   ev2   t1    et1    t2   et2 ")'
; loop over all batches 
  for i = 0, ObsFile.nbatch-1 do begin
    index_source=ObsFile.batches[i].batch_data.source
    if (ObsFile.batches[i].batch_data.flag EQ 0) then begin
      batch = ObsFile.batches[i]
      index = where (ObsFile.source_data.source_id EQ $
        batch.batch_data.source)
      if (batch.isCal EQ 0) then cal=" "
      if (batch.isCal NE 0) then cal="C"
      t1=batch.batch_data.t12
      t2=batch.batch_data.t22
      et1=batch.batch_data.et12
      et2=batch.batch_data.et22
      v1=batch.batch_data.v12
      v2=batch.batch_data.v22
      ev1=batch.batch_data.ev12
      ev2=batch.batch_data.ev22
      name = ObsFile.source_data[index_source-1].source
      printf,unit,format='(i3,1x,a16,a1,1x,4f6.3,1x,4f6.3)', $
        i,name, cal, v1,ev1,v2,ev2,t1,et1,t2,et2
    endif
  endfor ; end loop over all batches 
; close output
  FREE_LUN,UNIT
END ; end vinci_report

FUNCTION vinci_index, ObsFile
;********************************************************************
;#function#
;  vinci_index
;#call#
;  vinci_index( ObsFile)
;#description#
; Create an index of a dataset as a list of character strings, one per batch. 
;#inputs#
;\anArg{ ObsFile }{ VINCIObsFile-like }{ VINCI observation file }
;#return#
; an array of character strings, one per batch
;#end_function#
;********************************************************************
; create output array
  OFindex = strarr( ObsFile.nbatch+2)

; get fringe rate (there are always 5 frames per fringe)
  framerate = ObsFile.priHead->getPar("QL FRAMERATE")
  fringerate = framerate / 5.0
  fringerate = round (fringerate)

; header
  OFindex[0] = "Index for observation file "+ObsFile.infile
  OFindex[1] = STRING(FORMAT='("Fringe rate =",I5," Hz")',fringerate)

; loop over target batches 
  for i = 0, ObsFile.nbatch-1 do begin
      batch = ObsFile.batches[i]
; index number as string
   number = string (format='(I4,2X)',i)

; source name
    index_source = batch.batch_data.source
    source_name = ObsFile.source_data[index_source-1].source

; time
    time = vinci_julday(batch.batch_data.time)

; calibrator
    calibrator = "     "
    if (batch.isCal NE 0) then begin
      calibrator = " Cal "
    endif

; flagged?
    flagged = "     "
    if (batch.batch_data.flag NE 0) then begin
      flagged = " Flag "+batch.batch_data.reason
    endif

;  put it together
    OFindex[i+2] = number[0]+" "+source_name+"  "+time[0]+calibrator+flagged
  endfor ; end loop over batches 

   return,OFindex
END ; end vinci_index
