;********************************************************
;#class#
;fbtFitsFile
;#description#
;Basic routines to open/initialize any kind of FITS file
;regardless of its specific contents.
;Also the parent of the Fits Image Class
;#end_class#
;*******************************************************

PRO FITSFILE__DEFINE
;define structure associated with FitsFile class
a1 = {FitsFile,     $  ;#class members#
filename:'',        $  ;string & fits file name
unit:0L,            $  ;long   & logical unit no.
openStatus:'',      $  ;string & READ/WRITE status at opening
state:0L,           $  ;long   & current file status (open/closed..)
priHead:OBJ_NEW(),  $  ;object & file primary header object
eof:0L,             $  ;long   & position of file EOF in bytes
isExt:0B,           $  ;bool   & are there extensions
numExt:0L,          $  ;long   & number of known extensions
lastExt:OBJ_NEW(),  $  ;object & last known extensions
extObjects:PTR_NEW()$  ;ptr to objarr & known extensions pointing to this file
}
END

FUNCTION FITSFILE::INIT, filename, status, priHead=priHead, noSelf=noSelf, iErr=iErr
;   Constructor for FITSFILE class
;   This is a pass through call to "open" with the IDL
;   constructor error return added (i.e. returns 1 if
;   successful and 0 if not)
;
;   INPUTS:  see FITSFILE::open
;   noSelf  : if true do not record "self" as reference object
;             use this if external object is initial reference
;   check input
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN, 0
   endif         ; handle actual error
   if (N_PARAMS() lt 1) then $
      midiSetError,'Need at least a file name to create fitsfile object'
   if (N_PARAMS() eq 1) then status = 'READ'
   self->open, filename, status, priHead=priHead, iErr=iErr
   if (iErr NE 0) then begin
      midiSetError,'',/notInitial
      RETURN,0
   endif
   if (NOT KEYWORD_SET(noSelf)) then self.extObjects = PTR_NEW(self)
RETURN,1
END

PRO FITSFILE::open, filename, status, priHead=priHead, iErr=iErr
;
;Populate a FITSFILE object by opening a disk fitsfile
;

;INPUTS:
;  filename   String    name of disk fits file
;  status     String    user's intended file status
;                       can be 'READ' (readonly) (default if not given)
;                              'WRITE' (new file)
;                              'APPEND' (add onto end of existing file)
;                              'UPDATE' (alter contents of existing file)
;                        the last two are not yet fully implemented
;OUTPUTS:
;  iErr       Int        0 is OK, 1 is bad
;INPUT/OUTPUT
;  priHead    *object    on input: if specified for 'WRITE', this
;                        primary header will be written to disk
;                        instead of the minimal header
;                        on output: if specified, a pointer to
;                        the disk primary header object will be here.
;
;  error handling setup
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;   check input
;Check if filename is a nonempty string and whether status is legitimate
   sFile = SIZE(filename)
   fileOK = (sFile(0) EQ 0) AND (sFile(1) EQ 7)
   if (fileOK) then fileOK = strLen(filename) GT 0
   s = strUpCase(StrTrim(status,2))
   old = (s EQ 'READ') or (s EQ 'UPDATE') or (s EQ 'APPEND')
   statusOK = old OR (s EQ 'WRITE')
   if (not fileOK) OR (not statusOK) then $
      midiSetError, 'FITSFILE:open filename or status invalid'
; Open the file in the appropriate mode and store file parameters
   self.filename = filename
   self.openStatus = STRUPCASE(STRTRIM(status,2))
   if (s EQ 'READ') then begin
      OPENR, unit, filename, /BLOCK, /GET_LUN
      self.unit = unit
      self.state = 1
      self.eof   = -1
   endif
   if (s EQ 'UPDATE' OR (s EQ 'APPEND')) then begin
      OPENU, unit, filename, /BLOCK, /GET_LUN
      self.unit = unit
      self.state = 1
      self.eof   = -1
   endif
   if (s EQ 'WRITE') then begin
      OPENW, unit, filename, /BLOCK, /GET_LUN
      self.unit = unit
      self.state = -1
      self.eof   = 0
;Create primary header or minimal header
      if (KEYWORD_SET(priHead)) then begin
;check if this really is a header object
         isObj = SIZE(priHead)
    isObj = isObj(1+isObj(0))
    objOK = (isObj eq 11)
    if (objOK) then objOK = (OBJ_CLASS(priHead) EQ 'FITSHEADER')
    if (objOK) then objOK = priHead->isPriHead()
    if (objOK) then self.priHead = priHead else $
       midiSetError, 'priHead parameter is not a primary header object'
      endif else self.priHead = OBJ_NEW('FITSHEADER')
   endif     ; new file
;If this is an existing file, read the primary header
   if (old) then begin
      self.priHead = OBJ_NEW('FITSHEADER', self.unit)
;If reading the header failed, error stack should
;have been set up by FITSHEADER::
      if (NOT OBJ_VALID(self.priHead)) then midiSetError,'',/notInitial
      self.isExt =  self.priHead->isPriHead()
      if (KEYWORD_SET(priHead)) then priHead = self.priHead
   endif     ; old file
   iErr = 0
RETURN
END

PRO FITSFILE::cleanup
;  FITSFILE class destructor
;
;close file
   if (self.unit NE 0) then self->close
   OBJ_DESTROY,self.priHead
   PTR_FREE, self.extObjects
   RETURN
END

PRO fitsFile::addExt, extObj
;add an object that refers to this file to list
   if (NOT OBJ_VALID(extObj)) then RETURN
;no registered objects yet
   if (NOT PTR_VALID(self.extObjects)) then self.extObjects = PTR_NEW(extObj) $
;is this object already registered
   else begin
      if (total(extObj EQ *self.extObjects) GT 0) then RETURN
      objList = [*self.extObjects, extObj]
      PTR_FREE, self.extObjects
      self.extObjects = PTR_NEW(objList)
   endelse
RETURN
END

FUNCTION fitsFile::removeExt, extObj
;remove an object from internal list of references and return
;number of remaining references
   IF (NOT PTR_VALID(self.extObjects)) then RETURN,0
   nObj = N_ELEMENTS(*self.extObjects)
   nValid = 0
   for iObj = 0, nObj-1 do begin
      if (OBJ_VALID(extObj)) then $
      if ((*self.extObjects)[iObj] EQ extObj) then (*self.extObjects)[iObj]=OBJ_NEW()
      if (OBJ_VALID((*self.extObjects)[iObj])) then nValid=nValid+1
   endfor
RETURN, nValid
END

FUNCTION fitsFile::extObjs
;
   if (PTR_VALID(self.extObjects)) then RETURN,*self.extObjects else RETURN,OBJ_NEW()
END

PRO FITSFILE::put_pHead, pHead
;  Insert a Header Object as primary header into this object
;  INPUT  pHead    FITSHEADER
;
;  side effects: clears pointer to, but does not destroy
;  any existing primary header
;
   PTR_FREE, self.priHead
   self.priHead = PTR_NEW(pHead)
   RETURN
END

PRO FITSFILE::close, iErr=iErr
;
;  Close the disk file associated with this object
;
;  OUTPUTS: iErr  0 = OK
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   if (self.unit GT 0) then FREE_LUN, self.unit
   self.unit = 0
   self.state = 0
   iErr = 0
   RETURN
END

FUNCTION FITSFILE::filename
;  RETURNS   String        internal file name associated with this object
   RETURN, self.filename
END

FUNCTION FITSFILE::unit
;  RETURNS  Int           internal logical unit number of open fits file
   RETURN, self.unit
END

FUNCTION FITSFILE::priHead
;  RETURNS pointer to FITSHEADER object; internally stored primary header
   RETURN, self.priHead
END

FUNCTION FITSFILE::Head
;  RETURNS pointer to FITSHEADER object; internally stored primary header
;  alias for priHead
   RETURN, self.priHead
END

FUNCTION FITSFILE::stHead
;  RETURNS contents of fitsheader object
   RETURN,(self.priHead)->stHead()
END

FUNCTION FITSFILE::eof
;   RETURNS  Long    Byte offset of end-of-file from beginning of file
;                    (if known, -1 otherwise)
   RETURN, self.eof
END

FUNCTION FITSFILE::eof
;   RETURNS  Long    Byte offset of end-of-file from beginning of file
;                    (if known, -1 otherwise)
   RETURN, self.eof
END

FUNCTION FITSFILE::numExt
;   RETURNS  Int     number of (known) extensions in this file
   RETURN, self.numExt
END

FUNCTION FITSFILE::lastExt
;   RETURNS  FITSEXTENSION object     Last known extension in file
   RETURN, self.lastExt
END

FUNCTION FITSFILE::openStatus
;   RETURNS  String  File state given when opening file ('READ', etc.)
   RETURN, self.openStatus
END

FUNCTION FITSFILE::state
;   RETURNS  Int    File status (0=not open, 1 = open)
   RETURN, self.state
END

PRO FITSFILE::setEof, eof
;   Sets the internal pointer to the end of file (bytes)
;   INPUTS:   eof    Long
   self.eof = eof
RETURN
END

PRO FITSFILE::maxEof, eof
;   Sets the internal pointer to the max of current end and new (possible) eof
;   INPUTS:   eof    Long
   self.eof = eof > self.eof
RETURN
END

PRO FITSFILE::setState, state
;   Sets internal state number
;   INPUTS:   state  Int
   self.state = state
RETURN
END

PRO FITSFILE::setNumExt, numExt
;   Sets number of active extension
;   INPUTS:   numExt   Int
   self.numExt = numExt
RETURN
END

PRO FITSFILE::setUnit, unit
;   Set file logical unit number
;   INPUTS:   unit   Int
   self.unit = unit
RETURN
END

PRO FITSFILE::setLastExt, LastExt
;   Set number of last known extension
;   INPUTS:   LastExt  Int
   self.LastExt = LastExt
RETURN
END

PRO FITSFILE::pad2880, rest, iErr=iErr
; pad output with ascii blanks to a multiple of 2880 bytes
;
; check here whether file is really open etc.
;
; OUTPUT:
;   rest  long   number of bytes padded onto end
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   ierr = 0
   IF (self.openStatus EQ 'READ') then RETURN
;  Round off extension to 2880 byte blocks
   bytes = self.eof
   rest  = (bytes MOD 2880L)
   IF (rest EQ 0) then RETURN
   rest = 2880 - rest
   point_lun, self.unit, bytes
   WRITEU, self.unit, replicate(0B, rest)
   self.eof = self.eof + rest
RETURN
END

FUNCTION FITSFILE::whichExt, byteSize, iErr=iErr
;assume file is positioned at beginning of extension header
;return the extension time and the size of the data section
;in bytes.  Exit with file repositioned at beginning of header
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
      if (eof(self.unit)) then iErr = 4
RETURN, 0
   endif         ; handle actual error
;is file open
   if ((self.Openstatus EQ '') OR (self.unit LE 0)) then $
      midiSetError, module+' file not open'
;get current position
   POINT_LUN, -self.unit, pointLun
;read in header
   hTemp = obj_new('fitsHeader', self.unit)
   if (1 EQ hTemp->getPar('SIMPLE')) then $
      extName = 'PRIMARY' else begin
;is this an extension header
      type = hTemp->getPar('XTENSION')
      if ((type NE 'BINTABLE') AND (type NE 'A3DTABLE')) then $
         midiSetError, module+' this is not a legitimate extension' $
      else extName = strTrim(strUpCase(hTemp->getPar('extName')),2)
   endelse ;             not a primary header
   Axis = hTemp->getPar('NAXIS')
   if (Axis EQ 0) then byteSize = 0 else begin
      nAxis = hTemp->getPar('NAXIS*')
      byteSize = abs(hTemp->getPar('BITPIX'))/8
      for iAxis = 0, Axis-1 do byteSize = byteSize * nAxis[iAxis]
   endelse
   POINT_LUN, self.unit, pointLun
   OBJ_DESTROY, hTemp
   iErr = 0
RETURN, extName
END

PRO FITSFILE::skipExt, byteSize, iErr=iErr, noEof=noEof
;skip over the current extension
;will also skip over primary header and data if you rewind first
;INPUT
;byteSize  int  size of the data section of this extension if given
;                       if not, it will be determined by program
;noEof            if set: if eof occurs on read, just return error code
;                 no further error handling
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      iErr = 1
      if (eof(self.unit)) then begin
        iErr = 4
        if (KEYWORD_SET(noEof)) then RETURN
      endif
      midiCatchError
      RETURN
   endif         ; handle actual error
;is file open
   if ((self.Openstatus EQ '') OR (self.unit LE 0)) then $
      midiSetError, module+' file not open'
;is byteSize given?
   if (N_PARAMS() LT 1) then begin
      extName = self->whichExt(byteSize, iErr=iErr)
      if (iErr NE 0) then midiSetError,'',/notInitial
   endif
;skip header
   input = REPLICATE(STRING(REPLICATE(32B,80)),36)
   done = 0
   while (NOT done) do begin
      READU, self.unit, input
      done = FIX(TOTAL((strPos(input, 'END     ') EQ 0) GT 0))
   endwhile
;skip data
   POINT_LUN, -self.unit, pointLun
   rest = long(byteSize) mod 2880L
   blocks = long(byteSize)/2880L
   if (rest EQ 0) then blocks = byteSize else blocks = (blocks+1)*2880L
   POINT_LUN, self.unit, pointLun + blocks
   iErr = 0
RETURN
END


PRO FITSFILE::copyExt, outFile, byteSize, iErr=iErr, buffSize=buffSize, $
  noEof=noEof
;copy the current extension to output file
;NOTE this can also copy primary header and data if you rewind
;file to head first
;INPUTS
;   outFile    fitsfile object for output file
;   byteSize   int  size of the data section of this extension if given
;                       if not, it will be determined by program
;   buffSize   int  size in bytes of buffer for copying.  Default
;                   is 288000 if not specified.  Larger number may
;                   speed copy but may blow core
;   noEof           if set; if an end of file is encountered, return
;                   ierr = 4 (eof) but clear error stack.  In
;                   this case a copy to end of file has been achieved.
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;is file open
   if ((self.Openstatus EQ '') OR (self.unit LE 0)) then $
      midiSetError, module+' file not open'
;is outFile legitimate?
   sout = size(outFile)
   sout = sout[sout[0]+1]
;is this an object
   outOK = sout EQ 11
   if (outOK) then outOK = (OBJ_CLASS(outfile) EQ 'FITSFILE')
   if (NOT outOK) then $
      midiSetError, module+' output argument is not a FITSFILE object' $
   else begin
      outUnit = outFile->unit()
      if ((outUnit LE 0) OR ('WRITE' NE outFile->openStatus()) OR $
         ((-1) NE outFile->state())) then $
    midiSetError, module + ' output file '+outFile->filename()+' not open'
   endelse
;is byteSize given?
   if (N_PARAMS() LT 2) then begin
      extName = self->whichExt(byteSize, iErr=iErr)
;error was eof, but that's OK.
      if ((iErr EQ 4) AND (KEYWORD_SET(noEOF))) then begin
         midiClearErrMsg
    RETURN
      endif
      if (iErr NE 0) then midiSetError,'',/notInitial
   endif
;copy header
   input = REPLICATE(STRING(REPLICATE(32B,80)),36)
   done = 0
   while (NOT done) do begin
      READU, self.unit, input
      WRITEU, outUnit, input
      done = FIX(TOTAL((strPos(input, 'END     ') EQ 0) GT 0))
   endwhile
;create copy buffer
   if (KEYWORD_SET(buffSize)) then bSize = buffSize else bSize = 288000L
   input = bytarr(bSize)
;copy data
;round data size to blocks of 2880 bytes
   rest = long(byteSize) mod 2880L
   blocks = long(byteSize)/2880L
   if (rest EQ 0) then rest = byteSize else rest = (blocks+1)*2880L
;copy big chunks
   while (rest GT bSize) do begin
      READU, self.unit, input
      WRITEU, outUnit, input
      rest = rest - bSize
   endwhile
;finish last part
   if (rest GT 0) then begin
      input = input(0:rest-1)
      READU, self.unit, input
      WRITEU, outUnit, input
   endif
;set output eof
   POINT_LUN, -outUnit, pointLun
   outfile->setEof,pointLun
   iErr = 0
RETURN
END

PRO FITSFILE::rewind, iErr = iErr
;set file at beginning
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;is file open
   if ((self.Openstatus EQ '') OR (self.unit LE 0)) then $
      midiSetError, module+' file not open'
    POINT_LUN, self.unit, 0
    iErr = 0
RETURN
END

PRO FITSFILE::delete, iErr = iErr
;if file is open, close it
;in any case delete it
   if (self.state NE 0) then close, self.unit
   openw, self.unit, self.filename, /delete
   FREE_LUN, self.unit
   self.state = 0
   self.openstatus=''
   self.filename=''
   iErr = 0
RETURN
END

PRO FITSFILE::skipToExt, extName=extName, extNumber=extNumber, $
   rewind=rewind, iErr = iErr,noEof=noEof
;skip through file until the extension defined by extName/extNumber
;is found (see FITSEXTENSION::init for defaults)
;
;if  rewind is set, rewind to top of file before doing this
;
;if successful, iErr = 0 and file is positioned at beginning of header
;if extension is not found, iErr = 4  In this case, if noEof is
;  set, no error messages will be printed.
;if you were texting whether extension exists, check on iErr and
;then clear error stack to get rid of generated error messages.
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;is file open
   if ((self.Openstatus EQ '') OR (self.unit LE 0)) then $
      midiSetError, errMsg = module+' file not open'
;rewind if requested
   if (KEYWORD_SET(rewind)) then self->rewind, iErr=iErr
   if (iErr NE 0) then midiSetError,'',/notInitial
;check defaults
   if (NOT KEYWORD_SET(extName)) then extName=''
   if (NOT KEYWORD_SET(extNumber)) then extNumber = 1
;standardize form of input name
   eName = STRUPCASE (STRTRIM(extName,2))
   done = 0
   extFound = 0
;cook until done
   while (NOT done) do begin
      name = self->whichExt (byteSize, iErr = iErr)
      if ((iErr EQ 4) AND (KEYWORD_SET(noEof))) then RETURN
      if (iErr NE 0) then midiSetError,'',/notInitial
      if ((name EQ eName) OR (eName EQ '')) then extFound = extFound + 1
      done = (extFound EQ extNumber)
      if (NOT done) then self->skipExt, byteSize, iErr = iErr
      if (iErr NE 0) then midiSetError,'',/notInitial
   endwhile          ; done loop
   iErr = 0
RETURN
END

PRO FITSFILE::copyToExt, outfile, extName=extName, extNumber=extNumber, $
   rewind=rewind, buffSize=buffSize, iErr = iErr, noEof = noEof
;copy all extension from self to outfile
;until the extension defined by extName/extNumber
;is found (see FITSEXTENSION::init for defaults)
;
;if  rewind is set, rewind to top of file before doing this
;
;if extname etc is found, iErr = 0 and file is positioned at beginning of header
;if extension is not found, iErr = 4
;you can use this to copy a whole file, unseen, to output file
;
;if buffSize is set, this will be used as buffer size for copying
;if not set default of copyExt will be used
;  noEof if set means on reading an EOF, return iErr = 4, but supress
;  error messages etc.
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;is file open?
   if ((self.Openstatus EQ '') OR (self.unit LE 0)) then $
      midiSetError, module+' file not open'
;is outFile legitimate?
   sout = size(outFile)
   sout = sout[sout[0]+1]
;is this an object
   outOK = sout EQ 11
   if (outOK) then outOK = (OBJ_CLASS(outfile) EQ 'FITSFILE')
   if (NOT outOK) then $
      midiSetError,  module+' output argument is not a FITSFILE object' $
   else begin
      outUnit = outFile->unit()
      if ((outUnit LE 0) OR ('WRITE' NE outFile->openStatus()) OR $
         ((-1) NE outFile->state())) then $
    midiSetError,module + ' output file '+outFile->filename()+' not open'
   endelse
;rewind if requested
   if (KEYWORD_SET(rewind)) then begin
      self->rewind, iErr=iErr
      if (iErr NE 0) then midiSetError,'',/notInitial
   endif
;check defaults
   if (NOT KEYWORD_SET(extName)) then extName=''
   if (NOT KEYWORD_SET(extNumber)) then extNumber = 1
;standardize form of input name
   eName = STRUPCASE (STRTRIM(extName,2))
   done = 0
   extFound = 0
;cook until done
   while (NOT done) do begin
      name = self->whichExt (byteSize, iErr = iErr)
      if ((iErr EQ 4) AND (KEYWORD_SET(noEof))) then RETURN
      if (iErr NE 0) then midiSetError,'',/notInitial
      if ((name EQ eName) OR (eName EQ '')) then extFound = extFound + 1
      done = (extFound EQ extNumber)
      if (NOT done) then self->copyExt, outfile, byteSize, $
         buffSize=buffSize, iErr = iErr, noEof = noEof
      if (iErr NE 0) then midiSetError,'',/notInitial
   endwhile          ; done loop
   iErr = 0
RETURN
END

FUNCTION fitsFile::getPar, key, matches
;pass through to get header keywords
RETURN,(self.priHead)->getPar(key, matches)
END

FUNCTION fitsFile::listExtensions, iErr=iErr
;list all extensions in this file
;
   self->rewind
   iErr = 0
   extNames=''
   while (iErr EQ 0) do begin
      name = self->whichExt(iErr=iErr)
      if (iErr EQ 0) then begin
    extNames=[extNames,name]
    self->skipExt,iErr=iErr,/noEof
      endif
   endwhile
   if (N_ELEMENTS(extNames) GT 1) then extNames=extNames[1:*]
RETURN, extNames
END


PRO fitsFile::addPar, name, value, iErr=iErr, comment=COMMENT, $
   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;pass through to set primary header keywords
   (self.priHead)->addPar,name, value, iErr=iErr, comment=comment,$
      BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
END
