;**********************************************************
;#class#
;fitsHeader
; none
;#description#
; object dealing with fits Headers (primary and extension)
;#end_class#
;*********************************************************

PRO FitsHeader__DEFINE
;*********************************************************
;#structure#
;fitsHeader
;#inheritance#
;none
;#description#
;structure holding information on a fits Header
;#structure_text#
;define structure associated with FitsHeader Class
a2 = {FitsHeader, $  ;#class members#
type:'',         $   ;string & header type:PRIMARY or EXTENSION
extName:'',      $   ;string & if extension which type
lines:0L,        $   ;long   & number of lines in header
endPos:0L,       $   ;long   & positions of header end (bytes)
keyWords:PTR_NEW(),$ ;ptr->string array & list of header keywords
head:PTR_NEW()   $   ;ptr->string array& list of header records
}
;#end_structure#
;*********************************************************
END

PRO fitsHeader::cleanup
;********************************************************************
;#procedure#
; cleanup
;#call#
; fitsHeader->cleanup
;#description#
; Class destructor
;#end_procedure#
;*******************************************************************
   PTR_FREE,self.head
   PTR_FREE,self.keyWords
   RETURN
END

FUNCTION fitsHeader::init, input, iErr=iErr, binTable=binTable, $
   extName=extName
;*******************************************************************
;#function#
;init
;#description#
;  FITSHEADER Constructor with polymorph inputs
;#call#
;headerObj = OBJ_NEW('fitsHeader')
;headerObj = OBJ_NEW('fitsHeader',/binTable,extName=extName)
;headerObj = OBJ_NEW('fitsHeader', LUN) 
;headerObj = OBJ_NEW('fitsHeader', headerLines) 
;#inputs#
;\anArg{input}{poly}{Possible Inputs are:}
;\anArg{}{}{1. None.  In this case a minimal primary header is constructed}
;\anArg{}{}{or if binTable is set, a minimum bintable header is constructed}
;\anArg{}{}{2.  A integer (int, long, byte).  This is assumed to be an}
;\anArg{}{}{open LUN pointing to the beginning of a disk header.  We}
;\anArg{}{}{read in the header and leave the file pointing at the}
;\anArg{}{}{beginning of the data block}
;\anArg{}{}{3.  a String array  This is assumed to be an existing header}
;\anArg{}{}{and we copy it into our own data array}
;\anArg{binTable}{ bool}{if set in case (1) construct a binTable minimal header
;\anArg{}{}{rather than primary header (which is default)}
;\anArg{extName}{String}{ Extension name for binTable header}
;#return#
;\anArg{-}{header object}{}
;#end_function#
;*******************************************************************
;    Error handling setup
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
      if (N_ELEMENTS(oldInput) GE 0) then $
	 if(eof(oldInput)) then iErr = 4
RETURN,0
   endif         ; handle actual error
;
;    Default on no input: minimal primary or bintab header
;
   if (N_PARAMS() EQ 0) then begin
      if (not KEYWORD_SET(binTable)) then begin 
         self.type = 'PRIMARY' 
         FXHMAKE, header, /EXTEND, /DATE
      endif else begin
         self.type = 'BINTABLE'
	 if (KEYWORD_SET(extName)) then $
	    FXBHMAKE, HEADER, 0, extName $
	 else FXBHMAKE, header, 0
      endelse
      self.lines   = N_ELEMENTS(header)
;   Fill out to 2880 bytes
      rest = (self.lines MOD 36)
      if (rest NE 0) then begin
         fil = STRING(REPLICATE(32B, 80))
         header = [header, REPLICATE(fil,36-rest)]
      endif
      self.lines   = N_ELEMENTS(header)
      self.head  = PTR_NEW(header)
      self->fill
      self.keyWords = PTR_NEW(STRMID(header,0,8))
      self.endPos  = min(where(STRPOS(header, 'END     ') EQ 0))
      iErr = 0
      RETURN,1
   endif           ;  default input, no parameters
;    
;    There is some input, let check what it is
;
   iSize = SIZE(input)
   iSize = iSize(1+iSize[0])
   isString = iSize EQ 7
   isStruct = iSize EQ 8
   isFix     = (iSize GE 1) AND (iSize LE 3)
;
;    Input is a number, assume this is an LUN and
;    that LUN is positioned at the beginning of a header
;
   if (isFix) then begin
;    Save input
      errMsg = ''  
      oldInput = input
      nb = 0
      hBlock = REPLICATE(STRING(REPLICATE(32B,80)),36)
      done = 0
      while (done EQ 0) do begin
         READU, oldinput, hBlock
	 if (nb EQ 0) then begin 
	    input = hBlock 
	 endif else input = [input, hBlock]
	 nb = nb + 1
;   Check for END record
	 done = FIX(TOTAL((strPos(hBlock, 'END     ') EQ 0) GT 0))
      endwhile
   endif
   if (isString OR isFix) then begin
;
;
;    Check if this is a legitimate header
;
      type = strMid(input(0), 0, 8) 
      if (type EQ 'SIMPLE  ') then self.type = 'PRIMARY' $
      else if (type EQ 'XTENSION') then  begin  
         if ((strPos(input(0), 'BINTABLE') GE 0) OR $
	     (strPos(input(0), 'A3DTABLE') GE 0)) then self.type = 'BINTABLE'
	 if (self.type EQ 'BINTABLE') then $
	    self.extName = strTrim(strUpCase(FXPAR(input, 'extName')),2) $
	 else midiSetError,'Fits extension of type '+type+' not BINTABLE'
      endif else $
         midiSetError, ' type '+type+' header not legitimate fits'
;  Find the first END statement
      self.endPos = min(where(strPos(input, 'END     ') EQ 0))
      if (self.endPos LT 0) then midiSetError, ' header has no END statement'
      self.lines = N_ELEMENTS(input)
;  Fill out to 2880 byte blocks if necessary
      rest = (self.lines MOD 36)
      if (rest EQ 0) then begin 
         self.head = PTR_NEW(input) 
	 self.keyWords = PTR_NEW(STRMID(input,0,8))
      endif else begin
	    rest = 36 - rest
	    self.lines = self.lines + rest
            self.head = PTR_NEW([input,    $
	       REPLICATE(STRING(REPLICATE(32B,80)),rest) ])
	    self.keyWords = PTR_NEW([STRMID(input,0,8), $
	       REPLICATE(STRING(REPLICATE(32B,8)),rest)])
      endelse
   endif           ; Input is string array or File LUN
   if (isFix) then input = oldInput  ; restore input if LUN
   iErr = 0
RETURN,1
END

FUNCTION fitsHeader::stHead, type, extName, endPos
;****************************************************************
;#function#
;stHead
;#description#
;returns stored FITSHEADER object in string form
;along with some information
;#call#
;headerRecords = headerObject->sthead(type, extName, endPos)
;#outputs#
;\anArg{type}{string}{optional:'PRIMARY' or 'EXTENSION}
;\anArg{extName}{string}{optional:if extension returns name}
;\anArg{endPos}{long}{optional:offset to end of header data, bytes}
;#return#
;\anArg{-}{strArray}{all the card image records in the header}
;#end_function#
;*****************************************************************
;
   type    = self.type
   endPos  = self.endPos
   extName = self.extName
   RETURN, *self.head
END

PRO fitsHeader::writeToDisk, unit, iErr=iErr
;********************************************************************
;#procedure#
;writeToDisk
;#description#
;Unformated dump of contents of stored header
;into file, presumably opened for writing
;#call#
;headerObj->writeToDisk, unit, iErr=iErr
;#inputs#
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ keyList }  { strarr }{ A list of keyWords to examine }
;\anArg{}{}{These may also be ESO HIERARCH keywords}
;\anArg{ extName }{ string }{ which extension to examine;default=primary}
;\anArg{ extNumber }{ int }{ which extension to examine;default=primary}
;#end_procedure#
;********************************************************************
;
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   iErr = 0
   WRITEU, unit, *self.head
RETURN
END

PRO fitsHeader::expand, nBlock
;****************************************************************
;#procedure#
;expand
;#description#
;   Increase the internally stored header 
;  area in blocks of 2880 bytes
;   this has no effect on any disk storage
;#call#
;header->expand(nBlock)
;#inputs#
;\anArg{nBlock}{int}{# of requested expansion blocks:1 if not specified}
;#end_procedure#
;****************************************************************
   if(N_PARAMS() GE 1) then nb = nBlock else nb = 1
   tmpHead = [*self.head, REPLICATE(STRING(REPLICATE(32B,80)),36L*nb)]
   PTR_FREE,self.head
   self.head  = PTR_NEW(tmpHead)
   tmpKey = [*self.keyWords, REPLICATE(STRING(REPLICATE(32B,8)),36L*nb)]
   PTR_FREE,self.keyWords
   self.keyWords=PTR_NEW(tmpKey)
   self.lines = self.lines + 36*nb
   RETURN
END

PRO fitsHeader::append, newLines
;****************************************************************
;#procedure#
;append
;#description#
;Append new records to internal header 
;without checking for duplication.
;In general these will be blank, HISTORY or COMMENT records
;This has no effect on any disk files
;#call#
;header->append(newLines)
;#inputs#
;\anArg{newLines}{strarr}{array of 80 byte header lines}
;****************************************************************
;#end_procedure#
   nLines = N_ELEMENTS(newLines)
   isString = siZE(newLines)
   isString = isString(2)
; Check if this is a string array
   if (isString NE 7) then RETURN
; Is there enough space without expansion.  If not
; make the space
   freeLines = self.lines - self.endPos
   if (nLines GT freeLines) then begin
      nb = 1 + (nLines - freeLines)/36
      self->expand,nb
   endif
; Construct END and blank lines
   endLine = 'END'+STRING(REPLICATE(32B,77))
   blank   = STRING(REPLICATE(32B,80))
; Position of current END record
   bLine = self.endPos
; Fill out or truncate lines to 80 characters
   for I=0, nLines - 1 do begin
      tLine = newLines[i]
      sl = 80 - strLen(tLine)
      if (sl GT 0) then tline = tLine + strMid(blank,0,sl) $
         else if (sl LT 0) then tLine = strMid(tLine,0,80)
; Insert new line at end and update pointer
      (*self.head)[bLine] = tLine
      (*self.keyWords)[bLine] = strmid(tLine,0,8)
      bLine = bLine + 1
   endfor
; Put END line back in place and update permanent pointer
   (*self.head)[bLine] = endLine
   (*self.keyWords)[bLine] = 'END     '
   self.endPos = bLine
   RETURN
   END

FUNCTION fitsHeader::extract_keys, keyWords
;****************************************************************
;#function#
;extract_keys
;#description#
;Extract all records from the header with keywords whose
;initial characters agree with any of those in the strarr KEYWORDS
;#call#
;headerRecords=headerObj->extract_keys(keyWords)
;#inputs#
;\anArg{keyWords}{strArr}{list of keywords to be sought}
;#return#
;\anArg{-}{strArr}{list of header lines }
;#end_function#
;****************************************************************
   n_key = N_ELEMENTS(keyWords)
   FOR iKey = 0, n_key - 1 do begin
;  Condition KEY
       key = strUpCase(keyWords(iKey))
       if(strLen(key) GT 8) then key = strMid(key,0,8)
;  Look for KEY in header, if you find it, concatenate positions
       keyPos = where(strPos((*self.keyWords), key) EQ 0)
       if (keyPos[0] GE 0) then if (N_ELEMENTS(allKeys) EQ 0) $
          then allKeys = keyPos else allKeys = [allKeys, keyPos]
    endfor                   ; iKey loop
;Sort output so they are in original order
    allKeys = allKeys(SORT(allKeys))
    RETURN, (*self.head)[allKeys]
    END

FUNCTION fitsHeader::byteSize
;****************************************************************
;#function#
;byteSize
;#description#
;   get Long size of data/table file defined by header in bytes
;#call#
;dataLength = headerObj->byteSize()
;#return#
;\anArg{-}{long}{length of data/table section}
;#end_function#
;****************************************************************
;
;   
    bitPix = self->getPar('BITPIX')
    naxis  = self->getPar('NAXIS')
    gCount = self->getPar('GCOUNT')  &  if gcount EQ 0 then gcount = 1
    pcount = self->getPar('PCOUNT')
    if naxis gt 0 then begin
       dims = self->getPar('NAXIS*')           ;Read dimensions
       nData = dims[0]
    endif else nData = 0
    if naxis gt 1 then for i=2,naxis do nData = nData*dims[i-1] $
       else nData = 0
    nBytes = (ABS(bitPix) / 8) * gCount * (pCount + nData)
    nRec = LONG((nBytes + 2879) / 2880)
    RETURN, 2880*nRec
    END


FUNCTION fitsHeader::type
; RETURNS string whether header is primary, binTable or otherwise
   RETURN, self.type
END
FUNCTION fitsHeader::extName
; RETURNS, string extension name of
   RETURN,self.extName
END
FUNCTION fitsHeader::isPriHead
; RETURNS, bool TRUE if this header is a primary header
   if(self.type NE 'PRIMARY') then RETURN,0 $
      else RETURN,self->getPar('EXTEND')
END

;****************************************************************
;#function#
;type, extName, isPriHead
;#description#
; miniRoutines to return internal variables
;#call#
;headerType = headerObj->type()
;headerExt = headerObj->extName()
;pri_bin = headerObj->isPriHead()
;#return#
;\anArg{type}{string}{'PRIMARY','BINTABLE'...}
;\anArg{extName}{string}{Extension Name}
;\anArg{isPriHead}{byte}{1 if primary, 0 otherwise}
;#end_function#
;****************************************************************

FUNCTION fitsHeader::getPar, name, matches, comment=comments
;****************************************************************
;#function#
;getPar
;#description#
;get value(s) pointed to by keyword(s) in header
;  This is largely pre-existing IDL-astro code that has
;  been repackaged in object form
;#call#
;headerValue = headerObj->getPar(keyword, matches, comment=comments)
;#return#
;{-}{depends}{value behind equal sign in keyword/value pair}
;#end_function#
;****************************************************************
; Inputs      : 
;	NAME	= String name of the parameter to return.  If NAME is of the
;		  form 'keyword*' then an array is returned containing values
;		  of keywordN where N is an integer.  The value of keywordN
;		  will be placed in RESULT(N-1).  The data type of RESULT will
;		  be the type of the first valid match of keywordN found.
;                 Also allow keywords with several words
;                 This will be assumed to be a "HIERARCH ESO" header record
; Outputs     : 
;	The returned value of the function is the value(s) associated with the
;	requested keyword in the header array.
;
;	If the parameter is complex, double precision, floating point, long or
;	string, then the result is of that type.  Apostrophes are stripped from
;	strings.  If the parameter is logical, 1 is returned for T, and 0 is
;	returned for F.
;
;	If NAME was of form 'keyword*' then a vector of values are returned.
; Opt. Outputs: 
;	None.
; Keywords    : 
;	matches	= Optional keyword to return a value equal to the number of
;		  parameters found by FXPAR.
;	COMMENTS= Array of comments associated with the returned values.
; Calls       : 
;	GETTOK, VALID_NUM
; Common      : 
;	None.
; Restrictions: 
;	None.
; Side effects: 
;	Keyword COUNT returns the number of parameters found.
;
;	The system variable !err is set to -1 if parameter not found, 0 for a
;	scalar value returned.  If a vector is returned it is set to the number
;	of keyword matches found.
;
;	If a keyword occurs more than once in a header, a warning is given,
;	and the first occurence is used.  However, if the keyword is "HISTORY",
;	"COMMENT", or "        " (blank), then multiple values are returned.
; Category    : 
;	Data Handling, I/O, FITS, Generic.
;------------------------------------------------------------------------------
;
;  Check the number of parameters.
;
;  Convert the selected keyword NAME to uppercase.
;
         matches = 0
         nam = STRTRIM(STRuPcASE(name),2)
;  Consider whether this is ESO HIERARCH
         split = STRSPLIT(nam,' ',/EXTRACT)
         ESO = N_ELEMENTS(split) GT 1
         value = 0
;
;  Determine if NAME is of form 'keyword*'.  If so, then strip off the '*', and
;  set the VECTOR flag.  One must consider the possibility that NAM is an empty
;  string.
;
         nameLength1 = (strLen(NAM) - 1) > 1
         if strPos( NAM, '*' ) EQ nameLength1 then begin    
            nam = STRMID( nam, 0, nameLength1)  
            vector = 1	;Flag for vector output  
            nameLength = strLen(NAM)  	;Length of name 
            numLength = 8 - nameLength 	;Max length of number portion  
            if numLength LE 0 then MESSAGE= $
              'Keyword length must be 8 characters or less'
;
;  Otherwise, extend NAME with blanks to eight characters.
;
         endif else begin
            while strLen(NAM) LT 8 do nam = nam + ' '
            vector = 0
         endelse
;
;  If of the form 'keyword*', then find all instances of 'keyword' followed by
;  a number.  Store the positions of the located keywords in nFound, and the
;  value of the number field in number.
;
         keyword = *self.keyWords
         if vector then begin
            nFound = where(strPos(keyword,nam) GE 0, matches)
            if ( matches GT 0 ) then begin
               numSt= STRMID((*self.head)[nFound], nameLength, numLength)
               number = INTARR(matches)-1
               FOR i = 0, matches-1 do	$
                  if VALID_NUM( numSt[I], num) then number[i] = num
                  iGood = where(number GE 0, matches)
		if matches GT 0 then begin
		    nFound = nFound[iGood]
		    number = number[iGood]
		endif
	    endif
;
;  Otherwise, find all the instances of the requested keyword.  If more than
;  one is found, and NAME is not one of the special cases, then print an error
;  message.
;
        endif else begin
; worry about special format of ESO keywords
           if (ESO) then begin
              nFound = MIN(WHERE(STRPOS(*self.head, nam) GE 0, matches))
              matches = matches < 1
           endif else nFound = WHERE(keyword EQ nam, matches)
	   if (matches GT 1) AND (nam NE 'HISTORY ') AND		$
              (nam NE 'COMMENT ') AND (nam NE '') then	$
              MESSAGE,/INFORMATIONAL, 'WARNING- Keyword ' +	$
              nam + 'located more than once ' 
        endelse
;
;  Extract the parameter field from the specified header lines.  If one of the
;  special cases, then done.
;
	if matches GT 0 then begin
	    LINE = (*self.head)[nFound]
	    if(ESO) then begin
               split = STRSPLIT(LINE[0],'=',/EXTRACT)
               Svalue = STRTRIM(split(1),2)
            endif else Svalue = STRTRIM( STRMID(LINE,9,71),2)
	    if (nam EQ 'HISTORY ') OR (nam EQ 'COMMENT ') OR	$
		    (nam EQ '        ') then begin
		value = STRTRIM( STRMID(LINE,8,72),2)
		comments = STRARR(N_ELEMENTS(value))
;
;  Otherwise, test to see if the parameter contains a string, signalled by
;  beginning with a single quote character (') (apostrophe).
;
	    END else FOR i = 0,matches-1 do begin
		if ( STRMID(Svalue[I],0,1) EQ "'" ) then begin
		    TEST = STRMID( Svalue[I],1,strLen( Svalue[I] )-1)
		    NEXT_CHAR = 0
		    value = ''
;
;  Find the next apostrophe.
;
NEXT_apost:
		    ENDAP = strPos(TEST, "'", NEXT_CHAR)
		    if ENDAP LT 0 then MESSAGE,		$
			'Value of '+name+' invalid ' 
		    value = value + STRMID( TEST, NEXT_CHAR, ENDAP-NEXT_CHAR )
;
;  Test to see if the next character is also an apostrophe.  If so, then the
;  string isn't completed yet.  Apostrophes in the text string are signalled as
;  two apostrophes in a row.
;
		    if STRMID( TEST, ENDAP+1, 1) EQ "'" then begin    
	     		value = value + "'"
	      		NEXT_CHAR = ENDAP+2	 
			GOTO, NEXT_apost
	  	    endif
;
;  Extract the comment, if any.
;
		    slash = strPos(TEST, "/", ENDAP)
		    if slash LT 0 then comment = '' else	$
			comment = STRMID(TEST, slash+1, strLen(TEST)-slash-1)
;
;  If not a string, then separate the parameter field from the comment field.
;
                endif else begin
		    TEST = Svalue[I]
		    slash = strPos(TEST, "/")
		    if slash GT 0 then begin
			comment = STRMID(TEST, slash+1, strLen(TEST)-slash-1)
			TEST = STRMID(TEST, 0, slash)
		    END else comment = ''
;
;  Find the first word in TEST.  Is it a logical value ('T' or 'F')?
;
		    TEST2 = TEST
		    value = GETTOK(TEST2,' ')
		    TEST2 = STRTRIM(TEST2,2)
		    if ( value EQ 'T' ) then begin
			value = 1
		    END else if ( value EQ 'F' ) then begin
			value = 0
		    END else begin
;
;  Test to see if a complex number.  It's a complex number if the value and the
;  next word, if any, both are valid numbers.
;
			if strLen(TEST2) EQ 0 then GOTO, NOT_COMPLEX
			value2 = GETTOK(TEST2,' ')
			if VALID_NUM(value,val1) AND VALID_NUM(value2,val2) $
				then begin
			    value = COMPLEX(val1,val2)
			    GOTO, GOT_value
			endif
;
;  Not a complex number.  Decide if it is a floating point, double precision,
;  or integer number.  If an error occurs, then a string value is returned.
;
NOT_COMPLEX:
			ON_IOERROR, GOT_VALUE
			value = test
			if not VALID_NUM(value) then GOTO, GOT_VALUE
			if (strPos(value,'.') GE 0) OR (strPos(value,'E') $
				GE 0) OR (strPos(value,'D') GE 0) then begin
			    if ( strPos(value,'D') GT 0 ) OR $
				    ( strLen(value) GE 8 ) then begin
				value = DOUBLE(value)
			    END else value = FLOAT(value)
			END else value = LONG(value)
;
GOT_VALUE:
			ON_IOERROR, NULL
		    endelse
		endelse		; if string
;
;  Add to vector if required.
;
		if vector then begin
		    maxNum = MAX(number)
		    if ( i EQ 0 ) then begin
			sz_value = SIZE(value)
			result = MAKE_ARRAY( maxNum, type=SZ_value[1])
			comments = STRARR(maxNum)
		    endif 
		    result[   number[i]-1 ] =  value
		    comments[ number[i]-1 ] =  comment
		endif else begin
		    comments = comment
		endelse
	    endfor
;
;  Set the value of !ERR for the number of matches for vectors, or simply 0
;  otherwise.
;
	    if vector then begin
		!ERR = matches
		RETURN, result
	    endif else !ERR = 0
;
;  Error point for keyword not found.
;
	endif 
;
RETURN, value
END

FUNCTION fitsHeader::getParStruct, keyList, iErr=iErr
;***************************************************************
;#function#
;getParStruct
;#description#
;get values for a list of keywords, return as IDL structure
;structure tags are same as keywords except that some
;IDL-illegal characters are changed ("-"->"_") and
;ESO keywords are compacted (blanks removed)
;#call#
; valueStruct = headObj->getParStruct(keyList)
;#inputs#
;\anArg{keyList}{strArr}{list of keywords to be found}
;#return#
;\anArg{-}{struct}{an anonymous idl structure with tags as described above}
;\anArg{}{}{and scalar values set equal to the values of the keys in the header}
;#end_function#
;************************************************************
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
      RETURN,0
   endif
   nKeys = N_ELEMENTS(keyList)
   for iKey = 0, nKeys-1 do begin
      key = keyList[iKey]
      value = self->getPar(key, match)
      if (match LT 1) then midiSetError,'key ',key,' not present in header'
;generate column name by removing blanks and - signs
      split = STRSPLIT(key,' -',/EXTRACT)
      colName = split[0]
      for iN = 1, N_ELEMENTS(split)-1 do colName=colName+split(iN)
      if (iKey EQ 0) then rowStruct = CREATE_STRUCT(colName, value[0]) else $
         rowStruct=CREATE_STRUCT(rowStruct, colName, value[0])
   endfor
   iErr = 0
   RETURN, rowStruct
END

PRO fitsHeader::addPar, name, value, iErr=iErr, comment=COMMENT, $
   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;*****************************************************************
;#procedure#
;addPar
;#description#
;add/update a keyword/value pair in header
;This is largely pre-existing IDL-astro code that has
;been repackaged in object form
;#call#
;headObj->addPar,keyName, keyValue, comment=COMMENT, BEFORE=BEFORE,
;   AFTER=AFTER, FORMAT=FORMAT
;#inputs#
;\anArg{keyName}{string}{keyword name to insert or update}
;\anArg{keyValue}{poly}{scalar value to associate with keyword}
;\anArg{COMMENT}{string}{optional:comment to insert after value and '/' sign}
;\anArg{BEFORE}{string}{optional:keyword: insert new record just before this one}
;\anArg{AFTER}{string}{optional:keyword: insert new record just after this one}
;\anArg{FORMAT}{string}{optional:FORTRAN formatting description for putting data}
;#end_procedure#
;*****************************************************************
;+
; Project     : SOHO - CDS
;
; Name        : 
;	FXADDPAR
; Purpose     : 
;	Add or modify a parameter in a FITS header array.
; Explanation : 
;
; Use         : 
;	FXADDPAR, HEADER, NAME, value, COMMENT
; Inputs      : 
;	HEADER	= String array containing FITS header.  The maximum string
;		  length must be equal to 80.  If not defined, then FXADDPAR
;		  will create an empty FITS header array.
;
;	NAME	= Name of parameter.  If NAME is already in the header the
;		  value and possibly comment fields are modified. Otherwise a
;		  new record is added to the header.  If NAME is equal to
;		  either "COMMENT" or "HISTORY" then the value will be added to
;		  the record without replacement.  In this case the comment
;		  parameter is ignored.
;
;	value	= Value for parameter.  The value expression must be of the
;		  correct type, e.g. integer, floating or string.  String
;		  values of 'T'	or 'F' are considered logical values.
;
; Opt. Inputs : 
;	COMMENT	= String field.  The '/' is added by this routine.  Added
;		  starting in position 31.  If not supplied, or set equal to ''
;		  (the null string), then any previous comment field in the
;		  header for that keyword is retained (when found).
; Outputs     : 
;	HEADER	= Updated header array.
; Opt. Outputs: 
;	None.
; Keywords    : 
;	BEFORE	= Keyword string name.  The parameter will be placed before the
;		  location of this keyword.  For example, if BEFORE='HISTORY'
;		  then the parameter will be placed before the first history
;		  location.  This applies only when adding a new keyword;
;		  keywords already in the header are kept in the same position.
;
;	AFTER	= Same as BEFORE, but the parameter will be placed after the
;		  location of this keyword.  This keyword takes precedence over
;		  BEFORE.
;
;       FORMAT	= Specifies FORTRAN-like format for parameter, e.g. "F7.3".  A
;		  scalar string should be used.  For complex numbers the format
;		  should be defined so that it can be applied separately to the
;		  real and imaginary parts.
; Calls       : 
;	FXPAR, FXPARPOS
;
;
;
; Define a blank line and the END line
;
   iErr = 0
   endLine = 'END' + STRING(REPLICATE(32B,77))	;END line
   blank = STRING(REPLICATE(32B,80))		;blank line
;
;  If no comment was passed, then use a null string.
;
   if (NOT KEYWORD_SET(COMMENT)) then comment = ''
;
;  Check the HEADER array.
;
;
;  Make sure NAME is 8 characters long
;
   nn = STRING(REPLICATE(32B,8))	;8 char name
   STRPUT,nn,STRUPCASE(name)       ;Insert name
;
;  Check value.
;
   s = size(value)		;get type of value parameter
   sType = s[s[0]+1]
   if s[0] NE 0 then begin
	MESSAGE,'Keyword Value (third parameter) must be scalar'
   end else if sType EQ 0 then begin
		MESSAGE,'Keyword Value (third parameter) is not defined'
   end else if sType EQ 8 then begin
	MESSAGE,'Keyword Value (third parameter) cannot be structure'
   endif
;
;  Extract first 8 characters of each line of header, and locate END line
;
   keyWrd = *self.keyWords
;
;
;  History, comment and "blank" records are treated differently from the
;  others.  They are simply added to the header array whether there are any
;  already there or not.
;
   if (nn eq 'COMMENT ') OR (nn EQ 'HISTORY ') OR		$
	(nn EQ '        ') then begin
;
;  If the header array needs to grow, then expand it in increments of 36 lines.
;
  	if (self.endPos GE self.lines) then self->expand,1
;
;  Format the record.
;
	newLine = blank
	STRPUT,newLine,nn+STRING(value),0
;
;  If a history record, then append to the record just before the end.
;
	if nn EQ 'HISTORY ' then begin
		(*self.head)[self.endPos] = newline ;add history rec.
		(*self.keyWords)[self.endPos] = STRMID(newline,0,8)
; move end  and pointer up
		(*self.head)[self.endPos+1]=endLine		
		(*self.keyWords)[self.endPos+1] = STRMID(endLine,0,8)
;
;  The comment record is placed immediately after the last previous comment
;  record, or immediately before the first history record, unless overridden by
;  either the BEFORE or AFTER keywords.
;
	end else if nn EQ 'COMMENT ' then begin
           i = FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE=BEFORE)
           if i EQ self.endPos then i =	$
              FXPARPOS(keyWrd,self.endPos,AFTER='COMMENT',BEFORE='HISTORY')
           (*self.head)[i+1] = (*self.head)[i:self.lines-2];move rest up
           (*self.head)[i] = newLine		;insert comment
           (*self.keyWords)[i+1] = (*self.keyWords)[i:self.lines-2];move rest up
           (*self.keyWords)[i] = STRMID(newLine,0,8)
;
;  The "blank" record is placed immediately after the last previous "blank"
;  record, or immediately before the first comment or history record, unless
;  overridden by either the BEFORE or AFTER keywords.
;
       end else begin
          i = FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE=BEFORE)
          if i EQ self.endPos  then i =	$
             FXPARPOS(keyWrd,self.endPos,AFTER='',BEFORE='COMMENT')<$
             FXPARPOS(keyWrd,self.endPos,AFTER='',BEFORE='HISTORY')
             (*self.head)[i+1] = (*self.head)[i:self.lines-2];move rest up
             (*self.head)[i] = newLine		;insert "blank"
             (*self.keyWords)[i+1] = (*self.keyWords)[i:self.lines-2]
             (*self.keyWords)[i] = STRMID(newLine,0,8)
      endelse
      self.endPos = self.endPos + 1
      RETURN
   endif				;history/comment/blank
;
;  Find location to insert keyword.  If the keyword is already in the header,
;  then simply replace it.  If no new comment is passed, then retain the old
;  one.
;
   iPos  = WHERE(keyWrd EQ nn,nFound)
   if nFound GT 0 then begin
      i = iPos[0]
      if comment EQ '' then begin
         slash = strPos((*self.head)[i],'/')
         quote = strPos((*self.head)[i],"'")
         if (quote GT 0) AND (quote LT slash) then begin
            quote = strPos((*self.head)[i],"'",quote+1)
            if quote LT 0 then slash = -1 else	$
               slash = strPos((*self.head)[i],'/',quote+1)
            endif
            if slash NE -1 then	$
            comment = STRMID((*self.head)[i],slash+1,80) else $
               comment = STRING(REPLICATE(32B,80))
            endif
            GOTO, REPLACE
	endif
;
;  Start of section dealing with the positioning of required FITS keywords.  If
;  the keyword is SIMPLE, then it must be at the beginning.
;
	if nn EQ 'SIMPLE  ' then begin
            i = 0
            GOTO, INSERT
	endif
;
;  In conforming extensions, if the keyword is XTENSION, then it must be at the
;  beginning. 
;
	if nn EQ 'XTENSION' then begin
            i = 0
            GOTO, INSERT
	endif
;
;  If the keyword is BITPIX, then it must follow the either SIMPLE or XTENSION
;  keyword.
;
	if nn EQ 'BITPIX  ' then begin
            if (keyWrd[0] NE 'SIMPLE  ') AND		$
               (keyWrd[0] NE 'XTENSION') then MESSAGE,	$
		'Header must start with either SIMPLE or XTENSION'
            i = 1
            GOTO, INSERT
	endif
;
;  If the keyword is NAXIS, then it must follow the BITPIX keyword.
;
	if nn EQ 'NAXIS   ' then begin
           if keyWrd[1] NE 'BITPIX  ' then MESSAGE,	$
		'Required BITPIX keyword not found'
           i = 2
           GOTO, INSERT
	endif
;
;  If the keyword is NAXIS1, then it must follow the NAXIS keyword.
;
	if nn EQ 'NAXIS1  ' then begin
		if keyWrd[2] NE 'NAXIS   ' then MESSAGE,	$
			'Required NAXIS keyword not found'
		i = 3
		GOTO, INSERT
	endif
;
;  If the keyword is NAXIS<n>, then it must follow the NAXIS<n-1> keyword.
;
	if STRMID(nn,0,5) EQ 'NAXIS' then begin
		numAxis = FIX(STRMID(nn,5,3))
		prev = STRING(REPLICATE(32B,8))		;Format NAXIS<n-1>
		STRPUT,prev,'NAXIS',0			;Insert NAXIS
		STRPUT,prev,STRTRIM(numAxis-1,2),5	;Insert <n-1>
		if keyWrd[numAxis+1] NE prev then MESSAGE,	$
			'Required '+prev+' keyword not found'
		i = numAxis + 2
		GOTO, INSERT
	endif
;
;  If the first keyword is XTENSION, and has the value of either 'TABLE' or
;  'BINTABLE', then there are some additional required keywords.
;
	if keyWrd[0] EQ 'XTENSION' then begin
		XTEN = self->getPar('XTENSION')
		if (XTEN EQ 'TABLE   ') OR (XTEN EQ 'BINTABLE') then begin
;
;  If the keyword is PCOUNT, then it must follow the NAXIS2 keyword.
;
			if nn EQ 'PCOUNT  ' then begin
				if keyWrd[4] NE 'NAXIS2  ' then MESSAGE, $
					'Required NAXIS2 keyword not found'
				i = 5
				GOTO, INSERT
			endif
;
;  If the keyword is gCount, then it must follow the PCOUNT keyword.
;
			if nn EQ 'GCOUNT  ' then begin
				if keyWrd[5] NE 'PCOUNT  ' then MESSAGE, $
					'Required PCOUNT keyword not found'
				i = 6
				GOTO, INSERT
			endif
;
;  If the keyword is TFIELDS, then it must follow the gCount keyword.
;
			if nn EQ 'TFIELDS ' then begin
				if keyWrd[6] NE 'gCount  ' then MESSAGE, $
					'Required gCount keyword not found'
				i = 7
				GOTO, INSERT
			endif
		endif
	endif
;
;  At this point the location has not been determined, so a new line is added
;  at the end of the FITS header, but before any blank, COMMENT, or HISTORY
;  keywords, unless overridden by the BEFORE or AFTER keywords.
;
	i = FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE=BEFORE)
	if i EQ self.endPos then i =					  $
	    FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE='')		< $
	    FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE='COMMENT')	< $
	    FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE='HISTORY')
;
;  A new line needs to be added.  First check to see if the length of the
;  header array needs to be extended.  Then insert a blank record at the proper
;  place.
;
INSERT:
         if (self.endPos EQ (self.lines - 1))  then self->EXPAND,1
         (*self.head)[i+1] = (*self.head)[i:self.lines-2]
         (*self.head)[i] = blank
         (*self.keyWords)[i+1] = (*self.keyWords)[i:self.lines-2]
         (*self.keyWords)[i] = '        '
         self.endPos = self.endPos + 1
;
;  Now put value into keyword at line I.
;
REPLACE: 
	H=blank			;80 blanks
	STRPUT,H,nn+'= '	;insert name and =.
	apost = "'"	        ;quote (apostrophe) character
	type = SIZE(value)	;get type of value parameter
;
;  Store the value depending on the data type.  If a character string, first
;  check to see if it is one of the logical values "T" (true) or "F" (false).
;
	if type[1] EQ 7 then begin		;which type?
		upVal = STRUPCASE(value)	;force upper case.
		if (upVal EQ 'T') OR (upVal EQ 'F') then begin
			STRPUT,h,upVal,29	;insert logical value.
;
;  Otherwise, remove any tabs, and check for any apostrophes in the string.
;
		END else begin
			val = DETABIFY(value)
			NEXT_CHAR = 0
			REPEAT begin
				ap = strPos(val,"'",NEXT_CHAR)
				if AP GE 66 then begin
					val = STRMID(val,0,66)
				END else if AP GE 0 then begin
					val = STRMID(val,0,ap+1) + apost +   $
						STRMID(val,ap+1,80)
					next_char = ap + 2
				endif
			endrep until ap LT 0
;
;  If a long string, then add the comment as soon as possible.
;
			if strLen(val) GT 18 then begin
				STRPUT,h,apost+STRMID(val,0,68)+apost+ $
					' /'+comment,10
		                (*self.head)[i] = h
		                (*self.keyWords)[i] = STRMID(h,0,8)
				RETURN
;
;  If a short string, then pad out to at least eight characters.
;
			END else begin
				STRPUT,h,apost+val,10
				STRPUT,h,apost,11+(strLen(val)>8)
			endelse
		endelse
;
;  If complex, then format the real and imaginary parts, and add the comment
;  beginning in column 51.
;
	END else if type[1] EQ 6 then begin
		if N_ELEMENTS(format) EQ 1 then begin	;use format keyword
			vr = STRING(FLOAT(value),    '('+format+')')
			vi = STRING(IMAGINARY(value),'('+format+')')
		END else begin
			vr = STRTRIM(FLOAT(value),2)
			vi = STRTRIM(IMAGINARY(value),2)
		endelse
		sr = strLen(vr)  &  STRPUT,h,vr,(30-sr)>10
		si = strLen(vi)	 &  STRPUT,h,vi,(50-si)>30
		STRPUT,h,' /'+comment,50
		(*self.head)[i] = h
		(*self.keyWords)[i] = STRMID(h,0,8)
		RETURN
;
;  If not complex or a string, then format according to either the FORMAT
;  keyword, or the default for that datatype.
;
	END else begin
;  IDL STRING routine cant distinguish byte from character, so
;  in this case convert it to long
                if (type[1] EQ 1) then iValue = long(value) else iValue=value
		if (N_ELEMENTS(format) EQ 1) then $ ;use format keyword
			v = STRING(iValue,'('+format+')' ) else $
			v = STRTRIM(iValue,2)	;default format
		s = strLen(v)                 ;right justify
		STRPUT,h,v,(30-s)>10          ;insert
	endelse
;
;  Add the comment, and store the completed line in the header.
;
	STRPUT,h,' /',30	;add ' /'
	STRPUT,h,comment,32	;add comment
	(*self.head)[i]=h		;save line
	(*self.keyWords)[i] = STRMID(h,0,8)
;
	RETURN
	END



PRO fitsHeader::addESOPar, keys, value, iErr=iErr, comment=COMMENT, $
   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;*****************************************************************
;#procedure#
;addESOPar
;#description#
;add/update a ESO hierarchic keywords/value pair in header
;This is largely pre-existing IDL-astro code that has
;been repackaged in object form
;#call#
;headObj->addESOPar,keyName, keyValue, comment=COMMENT, BEFORE=BEFORE,
;   AFTER=AFTER, FORMAT=FORMAT
;#inputs#
;\anArg{keys}{strarr}{set of keywords to insert or update}
;\anArg{keyValue}{poly}{scalar value to associate with keyword}
;\anArg{COMMENT}{string}{optional:comment to insert after value and '/' sign}
;\anArg{BEFORE}{string}{optional:keyword: insert new record just before this one}
;\anArg{AFTER}{string}{optional:keyword: insert new record just after this one}
;\anArg{FORMAT}{string}{optional:FORTRAN formatting description for putting data}
;#end_procedure#
;*****************************************************************
; Define a blank line and the END line
;
   iErr = 0
   endLine = 'END' + STRING(REPLICATE(32B,77))	;END line
   blank = STRING(REPLICATE(32B,80))		;blank line
;
;  If no comment was passed, then use a null string.
;
   if (NOT KEYWORD_SET(COMMENT)) then comment = ''
;
;  Get array of current keywords array.
;
   keyWrd = *self.keyWords
;construct beginning of input line
   inLine = 'HIERARCH ESO'
   for iKey = 0, N_ELEMENTS(keys)-1 do inLine=inLine+$
      ' '+STRUPCASE(STRTRIM(keys(iKey),2))
;
;  Check value.
;
   s = size(value)		;get type of value parameter
   sType = s[s[0]+1]
   if s[0] NE 0 then begin
	MESSAGE,'Keyword Value (third parameter) must be scalar'
   end else if sType EQ 0 then begin
		MESSAGE,'Keyword Value (third parameter) is not defined'
   end else if sType EQ 8 then begin
	MESSAGE,'Keyword Value (third parameter) cannot be structure'
   endif
;
;
;
;
;  Find location to insert keyword.  If the keyword is already in the header,
;  then simply replace it.  If no new comment is passed, then retain the old
;  one.
;
   iPos  = MIN(WHERE(STRPOS(*self.head, inLine) GE 0))
   if iPos GT 0 then begin
      i = iPos[0]
      if comment EQ '' then begin
         slash = strPos((*self.head)[i],'/')
         quote = strPos((*self.head)[i],"'")
         if (quote GT 0) AND (quote LT slash) then begin
            quote = strPos((*self.head)[i],"'",quote+1)
            if quote LT 0 then slash = -1 else	$
               slash = strPos((*self.head)[i],'/',quote+1)
            endif
            if slash NE -1 then	$
            comment = STRMID((*self.head)[i],slash+1,80) else $
               comment = STRING(REPLICATE(32B,80))
            endif
            GOTO, REPLACE
	endif
;
;  At this point the location has not been determined, so a new line is added
;  at the end of the FITS header, but before any blank, COMMENT, or HISTORY
;  keywords, unless overridden by the BEFORE or AFTER keywords.
;
	i = FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE=BEFORE)
	if i EQ self.endPos then i =					  $
	    FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE='')		< $
	    FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE='COMMENT')	< $
	    FXPARPOS(keyWrd,self.endPos,AFTER=AFTER,BEFORE='HISTORY')
;
;  A new line needs to be added.  First check to see if the length of the
;  header array needs to be extended.  Then insert a blank record at the proper
;  place.
;
INSERT:
         if (self.endPos EQ (self.lines - 1))  then self->EXPAND,1
         (*self.head)[i+1] = (*self.head)[i:self.lines-2]
         (*self.head)[i] = blank
         (*self.keyWords)[i+1] = (*self.keyWords)[i:self.lines-2]
         (*self.keyWords)[i] = '        '
         self.endPos = self.endPos + 1
;
;  Now put value into keyword at line I.
;
REPLACE: 
	H=blank			;80 blanks
        inPos = (1+STRLEN(inLine))>29
	STRPUT,H,inLine 	;insert name and =.
	STRPUT,H,'= ',inPos 	;insert name and =.
	inPos=inPos+2
        room = 80 - inPos
	apost = "'"	        ;quote (apostrophe) character
	type = SIZE(value)	;get type of value parameter
;
;  Store the value depending on the data type.  If a character string, first
;  check to see if it is one of the logical values "T" (true) or "F" (false).
;
	if type[1] EQ 7 then begin		;which type?
		upVal = STRUPCASE(value)	;force upper case.
		if (upVal EQ 'T') OR (upVal EQ 'F') then begin
			STRPUT,h,upVal,inPos	;insert logical value.
;
;  Otherwise, remove any tabs, and check for any apostrophes in the string.
;
		ENDif else begin
			val = DETABIFY(value)
			NEXT_CHAR = 0
			REPEAT begin
				ap = strPos(val,"'",NEXT_CHAR)
				if AP GE 66 then begin
					val = STRMID(val,0,66)
				END else if AP GE 0 then begin
					val = STRMID(val,0,ap+1) + apost +   $
						STRMID(val,ap+1,80)
					next_char = ap + 2
				endif
			endrep until ap LT 0
;length of string
                        inLen = STRLEN(val)
; truncate if too long
                        if (room - inLen LT 2) then $
                           val = apost+STRMID(val,0,room-2)+apost $
                        else val = STRMID(apost+val+apost+' /'+comment,0,room)
                        STRPUT,h,val, inPos
		        (*self.head)[i] = h
		        (*self.keyWords)[i] = 'HIERARCH'
                        RETURN
		endelse
;
;  If complex, then format the real and imaginary parts, and add the comment
;  beginning in column 51.
;
	END else if type[1] EQ 6 then begin
		if N_ELEMENTS(format) EQ 1 then begin	;use format keyword
			vr = STRING(FLOAT(value),    '('+format+')')
			vi = STRING(IMAGINARY(value),'('+format+')')
		END else begin
			vr = STRTRIM(FLOAT(value),2)
			vi = STRTRIM(IMAGINARY(value),2)
		endelse
		STRPUT,h,STRMID(vr+' '+vi+' /'+comment,0,room),inPos
		(*self.head)[i] = h
		(*self.keyWords)[i] = 'HIERARCH'
		RETURN
;
;  If not complex or a string, then format according to either the FORMAT
;  keyword, or the default for that datatype.
;
	END else begin
;  IDL STRING routine cant distinguish byte from character, so
;  in this case convert it to long
                if (type[1] EQ 1) then iValue = long(value) else iValue=value
		if (N_ELEMENTS(format) EQ 1) then $ ;use format keyword
			v = STRING(iValue,'('+format+')' ) else $
			v = STRTRIM(iValue,2)	;default format
	endelse
;
;  Add the comment, and store the completed line in the header.
;
      STRPUT,h,STRMID(v+' /'+comment,0, room),inPos
      (*self.head)[i]=h		;save line
      (*self.keyWords)[i] = 'HIERARCH'
;
RETURN
END



PRO fitsHeader::FILL
;********************************************************************
;#procedure#
;fill
;#description#
;Fill all header records out to 80 char 
;This is mostly a fix to FXBHEAD that doesn't do this right
;#call#
;headObj->fill
;#end_procedure#
;******************************************************************
   blank = STRING(REPLICATE(32B,80))
   FOR i = 0, self.lines -1 do              $
      if (strLen((*self.head)[i]) LT 80)  then $
         (*self.head)[i] = (*self.head)[i]+STRMID(blank,0, $
	    80-strLen((*self.head)[i]))
RETURN
END

FUNCTION fitsHeader::headerLength
;***************************************************************
;#function#
;headerLength
;#description#
;return size of the current header in bytes, including padding
;#call#
; headerLength = headObj->headerLength()
;#return#
;\anArg{-}{long}{header length}
;#end_function#
;************************************************************
;
RETURN, 80*self.lines
END

FUNCTION fitsHeader::clone
;***************************************************************
;#function#
;clone
;#description#
;create a literal copy of current header
;#call#
;newHeadObj = oldHeadObj->clone()
;#return#
;\anArg{-}{fitsHeaderObject}{copy}
;#end_function#
;************************************************************
;create a new fitsheader with identical contents
   header = obj_new('fitsHeader', *self.head)
RETURN, header
END

