;********************************************************
;#class#
;fbtFitsTable
;#description#
;generic routines dealing with Fits Binary Tables
;read 'em; write 'em, manipulate their headers
;#inheritance#
;fbtFitsExtension
;#end_class#
;*******************************************************

PRO FitsTable__DEFINE
;define structure associated with FitsTable Class
a5={FitsTable, $       ;#class members#
INHERITS FITSEXTENSION, $;#class inheritance#
table:PTR_NEW(),     $ ;ptr->structure & internal IDL representation of \cr
                       ;table data
strucTemp:PTR_NEW(), $ ;ptr->structure & internal rep. of one row
subStrucTemp:PTR_NEW(), $;ptr->structure & internal rep. of selected columns
subStructCol:PTR_NEW(),  $;ptr->int array & which columns selected
floatCol:PTR_NEW(),$ ;ptr->int array & colmns with float conversion
naxis1:0L, $         ;long   & bytes per row
naxis2:0L, $         ;long   & number of rows in file
naxis2_POS:0L, $     ;long   & position of NAXIS2 keyword in file
tfields:0L, $        ;long   & number of columns
heap:0L, $           ;long   & not currently used
dheap:0, $           ;long   & not currently used
ttype:PTR_NEW(),  $  ;ptr->str array & names of columns
format:PTR_NEW(), $  ;ptr->str array & column formatting info
idlType:PTR_NEW(),$  ;ptr->int array& idl types of each column
nElem:PTR_NEW(), $   ;ptr->long array & number of elements per column
tScale:PTR_NEW(), $  ;ptr->dbl array &rescaling per column
tZero:PTR_NEW(),  $  ;ptr->dbl array &offsets per column
maxVaL:PTR_NEW(), $  ;ptr & ? not implemented
nDims:PTR_NEW(), $   ;ptr->long  &array dimensions of each column
bytOff:PTR_NEW(), $  ;ptr->long  &position in file of column rel. start of row
buffSize:0L,      $  ;long   & max size of i/o buffer (bytes)
oRead:0L,         $  ;long   & size of last read block of data (bytes)
dataTemp:PTR_NEW() $ ;ptr    & buffer containing data just read
}
END

PRO FITSTABLE::cleanup
;Project:   MIDI
;FITSTABLE class destructor
;DeAllocate IDL Heap space for the variable length records in table def
;
   PTR_FREE,self.Table
   PTR_FREE,self.Ttype
   PTR_FREE,self.format
   PTR_FREE,self.IDLtype
   PTR_FREE,self.nElem
   PTR_FREE,self.tScale
   PTR_FREE,self.tZero
   PTR_FREE,self.maxVal
   PTR_FREE,self.strucTemp
   PTR_FREE,self.subStrucTemp
   PTR_FREE,self.subStructCol
   PTR_FREE,self.floatCol
   PTR_FREE,self.bytOff
   PTR_FREE,self.nDims
   PTR_FREE,self.dataTemp
;cleanup at higher level
   self->fitsExtension::cleanup
RETURN
END
;
FUNCTION FITSTABLE::init, input, extName=extName, $
      extNumber=extNumber, logCol=logCol, table=table, $
      buffSize=buffSize, iErr=iErr
;Project:   MIDI
;Constructor for FITSTABLE (fits binary table) class
;The chief function is to set internal member variables that
;describe the format of the underlying fits table
;
;This constructor is polymorphic, its source of data and actions depends
;on the type of the input, which is checked at run time
;
;INPUTS:
;   input     ?       generic input
;      The choices for input are:
;      (1) A scalar string giving the the disk filename
;          of a disk fits binary table that is the data source.
;          In this case extName and/or extNumber describe which
;          extension contains the table.  These work as follows:
;          extNumber defaults to 1 if not given.  This describes
;          how many extensions of type extName to look for before
;          grabbing one.  extName defaults to '' (i.e. blank), which
;          matches any extension.  Thus if neither extName or
;          extNumber are given, the first extension is picked.
;          If extNumber is given (=n) but not extName, the nth
;          extension, of any type, is taken.
;
;          In any case a FITSFILE object is created for the file,
;          the disk is read through to the appropriate extension
;          and the formatting data is read into internal members
;
;      (2) an existing FITSFILE object, opened for READ.  This
;          is treated as above.
;
;      (3) an existing IDL structure, which will probably be
;          a template for writing arrays of such structures to disk.
;          In this case the IDL information on the structure is
;          parsed.  NOTE:  IF THE IDL STRUCTURE CONTAINS STRINGS,
;          THE FITSTABLE STRINGSIZE FOR THAT ELEMENT IS SET TO THE
;          MAXIMUM OF THE IDL STRINGLENGTHS, OR 8 IF THE TEMPLATE
;          STRINGS ARE ALL BLANK.  IT IS WISE TO FILL IN AN
;          ADEQUATELY LONG STRING IN THE TEMPLATE.
;
;   In the second case, there are two optional keyword inputs:
;
;   extName  String    an extension name to be associated with this table
;   extNumber int      sequence number of extensions of type extName
;   logCol   intarr    specifies columns to be considered boolean variable.
;                      IDL does not distinguish internally between
;                      byte and boolean types.  An internal structure
;                      containing byte variables will be considered as
;                      byte type, unless the columns containing the
;                      boolean variables are specified here as an array
;                      of integer column numbers
;
;   table              if this is set (e.g. /table, or table=1) for
;                      cases 1 or 2 (input) then after reading in
;                      formatting information, also read the entire
;                      actual table into self.table.  This can be
;                      convenient for small tables.
;
;  buffSize            size in bytes of read/write buffer.  If not
;                      specified a default will be chosen
;  Check the type of the input parameter.
;
   iErr = 0
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN, 0
   endif         ; handle actual error
   if (N_ELEMENTS(buffSize) EQ 1) then self.buffSize = buffSize $
      else buffSize = 1024L*256L    ; default 256KB.  Not
;     allocated until needed
;
   sInput = SIZE(input)
;case 1: input is a scalar string
   isFileName = sinput[0] eq 0 AND (sinput[1] eq 7)
   sInput = sInput[sInput[0]+1]
;case 3; input is a structure
   isStruct = sinput EQ 8
;case 2; input is an object, should be a FITSFILE object
   isObj    = sinput EQ 11
   if (isObj) then isObj = (OBJ_CLASS(input) eq 'FITSFILE')
;handle cases 1 and 2
   if (NOT (isFileName OR isStruct OR isObj)) then $
      midiSetError, 'Input is neither file or IDL structure'
   if (isStruct) then stName = TAG_NAMES(input,/STRUCTURE_NAME)
;
;case 1: Input specifies an (old) filename and
;an extension name or number.  Open the file.  and find extension.
;The primary header is then in self.file.priHead
;the extension header is in self.head
;and the input file is positioned just after the extension header
   if (isFileName OR isObj) then begin
;Read header into internal variables
       if (0 EQ self->FITSEXTENSION::init(input, iErr=iErr, $
          extName = extName, extNumber = extNumber)) then $
     midiSetError,'',/notInitial
;Dereference the header for FXBtForm/FIND.  We won't need to do this
;if we update them to deal with pointers
       head = self.head->stHead()
       self.headerStatus = 'DISK'
       tFields      = self.head->getPar('TFIELDS')
       self.tFields = tFields
;
;Gather the necessary information
;
      FXBtForm,head,bytOff0,idlType0,format0,nElem0,maxVal0
      FXBFIND,head,'Ttype',columns,Ttype0,nfound,''
      FXBFIND,head,'tScal',columns,tScal0,nfound,1.
      FXBFIND,head,'tzero',columns,tZero0,nfound,0.
;
;Get the information from the required keywords.
;
      self.tType   = PTR_NEW(tType0)
      self.format  = PTR_NEW(format0)
      self.idlType = PTR_NEW(idlType0)
      self.nElem   = PTR_NEW(nElem0)
      self.tZero   = PTR_NEW(tZero0)
      self.tScale   = PTR_NEW(tScal0)
      self.maxVal  = PTR_NEW(maxVal0)
      self.bytOff  = PTR_NEW(bytOff0)
      self.nDims   = PTR_NEW(LONARR(9, tFields))
      self.naxis1  = self.head->getPar('NAXIS1')
      self.naxis2  = self.head->getPar('NAXIS2')
;
;If theap is not present, then set it equal to the size of the table.
;
      theap = self.head->getPar('THEAP', isHeap)
      if (isHeap GT 0) then tHeap = self.naxis1*self.naxis2
      self.heap = tHeap
;
;Store the information about the columns.
;
;
;If not a variable length array, then get the dimensions associated with each
;column from the TDIMn keywords.  If not found, then assume to be the number
;of elements.
;
      for iCol = 0,tFields-1 do if (*self.maxVal)[iCol] EQ 0 then begin
         tDim = self.head->getPar('TDIM'+strTrim(iCol+1,2), istDim)
         tDimused = (istDim GT 0)
         if tDimused then dims = FIX(fxbtDim(tdim))   $
        else dims = (*self.nElem)[iCol]
    dims = [N_ELEMENTS(dims),dims]
;
;If the datatype is a bit array, then no dimensions are applied to the data.
;
   if (*self.format)[iCol] EQ 'X' then dims =     $
        [1,(*self.nElem)[iCol]]
   (*self.nDims)[0:N_ELEMENTS(dims)-1,iCol] = dims
;
;For those columns which are character strings, then the number of
;characters, nChar, is the first dimension, and the number of elements is
;actually nElem/nChar.
;
   if (*self.IDLtype)[iCol] EQ 7 THEN     $
      (*self.nElem)[iCol] = (*self.nElem)[iCol] / dims[1]
       endif        ; not variable length arrays
;do we read in data right now?
       if (KEYWORD_SET(table)) then begin
          self.table = PTR_NEW(self->readRows(iErr=iErr))
     if (iErr NE 0) then $
        midiSetError,'reading table data failed',/notInitial
       endif
       RETURN,1
    endif           ; filename for input

;
;Case 3: Input is a structure, assume that it
;is a template IDL structure and create the appropriate definitions
;and header
    if (isStruct) then begin
;create dummy header
      if (0 EQ self->FITSEXTENSION::init(extName = extName)) $
          then midiSetError,'',/notInitial
      tags  = strTrim(TAG_NAMES(input), 2)
      tFields = N_ELEMENTS(tags)
      self.tFields = tFields
      if (KEYWORD_SET(extName)) then self.extName = extName
;Allocate space for internal variables
      self.Ttype   = PTR_NEW(tags)
      self.format  = PTR_NEW(STRARR(tFields))
      self.IDLtype = PTR_NEW(INTARR(tFields))
      self.nElem   = PTR_NEW(LONARR(tFields))
      self.nDims   = PTR_NEW(LONARR(9,tFields))
      self.tScale  = PTR_NEW(DBLARR(tFields))
      self.tZero   = PTR_NEW(DBLARR(tFields))
      self.bytOff  = PTR_NEW(LONARR(tFields))
      self.maxVal  = PTR_NEW(LONARR(tFields))
;See if logCol is set, i.e. that there are some byte arguments
;that are supposed to be logicals.  If so, parse the argument
;to determine which columns are specified
      if(KEYWORD_SET(logCol)) then begin
;check logCol type
      logType = SIZE(logCol)
      logType = logType[1+logType[0]]
;if numbers just copy; if strings check against column names
      if (logType GE 1 AND (logType LE 3)) then logColN = logCol $
      else if (logType EQ 7) then begin
         nLogCol = N_ELEMENTS(logCol)
         logColN = INTARR(nLogCol)
         for iCol = 0, nLogCol-1 do logColN[iCol] = $
            WHERE(strUpCase(strTrim(logCol[iCol],2)) EQ strTrim(tags,2))
    endif            ; string logCols
      endif               ; logCols set
;Set tFields in header
      if (self.extName NE '') then self.head->addPar,'EXTNAME',self.extName
      self.head->addPar,'TFIELDS', tFields
;Initialize bytOff etc in object.
      (*self.bytOff)[0] = 0
      self.naxis1    = 0
;Loop over fields geting information
      for iTag = 0, tFields - 1 do begin
         sTag = strTrim(iTag+1,2)
         s = SIZE(input.(iTag))
         type = s[s[0]+1]
         nElem = N_ELEMENTS(input.(iTag))
         CASE type OF
            1: begin
               nBytes = nElem
               tForm  = 'B'
;Check logicals
               if (KEYWORD_SET(logCol)) then if $
             (MIN(WHERE(iTag EQ logColN)) GE 0) then tForm = 'L'
            END
            2: begin
               nBytes = 2*nElem
               tForm  = 'I'
            END
            3: begin
               nBytes = 4*nElem
               tForm  = 'J'
            END
            4: begin
               nBytes = 4*nElem
               tForm  = 'E'
            END
            5: begin
               nBytes = 8*nElem
               tForm  = 'D'
            END
            6: begin
               nBytes = 8*nElem
               tForm  = 'C'
            END
       7: begin
          stLen  = MAX(STRLEN(input.(iTag)))
          if (stLen eq 0) then stLen = 8
          nBytes = stLen * nElem
          nElem  = nBytes
          tForm  = 'A'
          s = [s[0]+1, stLen, s[1:*]]
       END
            9: begin
               nBytes = 16*nElem
               tForm   = 'M'
            END
         ENDCASE
;Add element count to tForm
         tForm = strTrim(STRING(nElem),2) + tForm
    (*self.format)[iTag]  = tForm
    (*self.idlType)[iTag] = type
    (*self.nElem)[iTag]   = nElem
    self.naxis1           = self.naxis1 + nBytes
;Compute tDim
         if (S[0] GT 1) then begin
            tDim = "("+strTrim(S[1],2)
       for I = 2, s[0] do tDim=tDim+","+strTrim(s[I],2)
       tDim = tDim+")"
       self.head->addPar, 'TDIM'+sTag, tDim
    endif
    (*self.nDims)[0:S[0], iTag] = S[0:S[0]]
         if(iTag LT tFields-1) then (*self.bytOff)[iTag+1] =  $
        nBytes + (*self.bytOff)[iTag]

         self.head->addPar,'TTYPE'+STAG, TAGS(iTag)
    self.head->addPar, 'TFORM'+STAG, tForm
       endfor       ; iTag loop
       self.head->addPar,'NAXIS1', self.naxis1
       self.strucTemp = PTR_NEW(input[0])
       self.dataStatus = 'EMPTY'
       self.headerStatus = 'INTERNAL'
       RETURN,1
    endif           ; IDLstruct template input
RETURN,0
END

FUNCTION FITSTABLE::head
;RETURNS FITSHEADER pointer      fitsheader object for extension header
   RETURN,self.head
END

FUNCTION FITSTABLE::table
;returns pointer to internally stored table (if any)
   RETURN,self.table
END

PRO FITSTABLE::setTable, table
;sets table pointer to given table
   self.table = PTR_NEW(table)
   RETURN
END

FUNCTION FITSTABLE::template, COLTAG = COL_TAG, iErr=iErr
; Project : MIDI
;  Create an empty template IDL structure representing one row
;       of a binary table based on the internally stored information
;       in the current object.
;   INPUTS:
;       COLTAG     boolean Normally the tags in the returned rowStruct are
;                  set to the FITS bintable column names (Ttypen), if defined,
;                  and to COLn if blank or not defined.  If COLTAG is set, the
;                  second option is taken for all columns.  Remember
;                  that FITS is 1-relative and IDL is 0-relative.
;   OUTPUTS:  iErr              the usual
;   RETURNS:  IDL structure     the requested empty template
;
;
;
   iErr = 0
;do I already have the answer
   if (PTR_VALID(self.strucTemp)) then RETURN,*(self.strucTemp)
;Store the information about the columns.
;
   iErr = 1
   if (self.tFields LE 0) then RETURN,0
   ; Test for real array
   if ((*self.NELEM)[0] GT 0) then begin
      rowEl = midiDumEl ( (*self.idlType)[0],  (*self.nDims)[*,0])
      if ((strTrim((*self.tType)[0],2) EQ '') OR KEYWORD_SET(COL_TAG)) then $
            rowStruct = CREATE_STRUCT('COL1', rowEl) else                   $
       rowStruct = CREATE_STRUCT(strTrim((*self.Ttype)[0],2), rowEl)
      endif
   FOR iCol = 1, self.tFields - 1 do if ((*self.nElem)[iCol] GT 0) $
      then begin
      rowEl = midiDumEl ((*self.idlType)[iCol], (*self.nDims)[*,iCol])
      if ((strTrim((*self.tType)[iCol],2) EQ '') OR KEYWORD_SET(COL_TAG)) $
         then rowStruct = CREATE_STRUCT(rowStruct, $
       'COL'+strTrim(STRING(iCol+1),2), rowEl)  $
    else rowStruct = CREATE_STRUCT(rowStruct, $
       strTrim((*self.tType)[iCol],2), rowEl)
      endif

;
   iErr = 0
   PTR_FREE,self.strucTemp
   self.strucTemp = PTR_NEW(rowStruct)
RETURN, rowStruct
END

FUNCTION FITSTABLE::readRows, Rows, columns=columns, $

   floatCol=floatCol, iErr=iErr
;  fetch a given set of rows from the disk fits binary table mapped
;  to this object
;
;  INPUTS:
;         rows  intarr or longarr arrays of row numbers to be read, 1-relative.
;                                 if omitted, 1st row is assumed
;         columns array of numbers or of strings
;                                 if specified, a substructure with only these
;                                 columns is returned.  This does not save
;                                 i/o time since all data is read anyway,
;                                 but it might save space.  Remember
;                                 that columns given as numbers are
;                                 1-Relative
;         floatCol array of numbers of strings
;                                 if specified these columns are converted
;                                 to float with application of tzero and
;                                 tscale.
;OUTPUTS: iErr
;   RETURNS  an array of IDL structures containing the table values

   iErr = 0

   errMsg = ''
;establish error handler

   cErr = 0

   catch, cErr

   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN, 0
   endif         ; handle actual error
;

   iLun = (self.file)->unit()

   if (iLun EQ 0) OR ((self.file)->state() EQ 0) then $
      midiSetError, 'Binary Table not opened properly'
;If no rows specified; get all rows (is this wise?)

   if (n_params() LT 1) then row = 1+lindgen(self.naxis2) else Row = Rows
;
;check row range
;

   if (total(row LE 0 OR (row GT (self.naxis2))) GT 0) then $
      midiSetError, 'ROWS must be between 1 and ' +   $
          strTrim(self.naxis2,2)

   nRow = N_ELEMENTS(row)
;sort rows if specified as input
   if (N_PARAMS() GE 1) then begin 
      tRow = sort(row)
      sRow = row(tRow)
   endif else begin
      sRow = row
      tRow = sRow - 1
   endelse
   minRow = tRow(0)
   maxRow = tRow(nRow-1)

;do we want conversion to float and or scaling?

   doFloat = 0

   floatNo = -1

   doScale = 0

   scaleNo =-1

   if (N_ELEMENTS(floatCol) GT 0) then begin

      doFloat = 1

      floatSize = size(floatCol)

      floatSize = floatSize[1+floatSize[0]]
;string column names

      if (floatSize EQ 7) then begin
         floatNo = self->columnNumbers(STRUPCASE(STRTRIM(floatCol,2)),$
            iErr=iErr) ; 1-relative
      if (iErr NE 0) then $
            midiSetError, 'Misformed set of float column names'

      endif else if (floatSize GE 1 AND (floatSize LE 5)) $
         then floatNo = floatCol $                          ; 1-rel
      else midiSetError, 'floatCol input not string or number'

      nFloat = N_ELEMENTS(floatNo)
; keep a list of columns that are already float but need to be
; rescaled

      scaleNo = REPLICATE(-1, nFloat)

      for iCol = 0, nFloat -1 do begin
;check if already float; is so is rescaling necessary?

         if ((*self.idltYPE)[floatNo(iCol)-1] EQ 4) then begin
            if (((*self.tScale)[floatNo(iCol)-1] NE 1) OR $
               ((*self.tZero)[floatNo(iCol)-1] NE 0)) then $
               scaleNo(iCol) = floatNo(iCol)
            floatNo(iCol) = -1
         endif
      endfor
   endif    ; floatCol has been specified
;cull the flock, do we really need to float or rescale?

   doFloat = total(floatNo NE -1) GT 0

   if (doFloat) then floatNo = floatNo(WHERE(floatNo GT 0))

   doScale = total(ScaleNo NE -1) GT 0

   if (doScale) then begin
      scaleNo = scaleNo(WHERE(scaleNo GT 0))
      tZero = (*self.tZero)[scaleNo-1]
      tScale = (*self.tScale)[scaleNo-1]
   endif
;do we want to select some columns?

   doCols = 0
   colNo = 1 + indgen(self.tFields)
   nCols = self.tFields
   if (N_ELEMENTS(columns) GT 0) then begin
      doCols = 1
      colSize = size(columns)
      colSize = colSize[1+colSize[0]]
;string column names
      if (colSize EQ 7) then begin
         colNo = self->columnNumbers(STRUPCASE(STRTRIM(columns,2)), iErr=iErr)
; 1-relative
    if (iErr NE 0) then $
            midiSetError, 'Misformed set of column names'
      endif else if (colSize GE 1 AND (colSize LE 5)) $
         then colNo = columns $                          ; 1-rel
      else midiSetError, 'column input not string or number'
      nCols = N_ELEMENTS(colNo)
   endif    ; columns have been specified
;do we need an output structure different than already assumed?
   if ((doCols NE 0) OR (doFloat )) then begin
      beenHere = ( 1 EQ 1)
      if (doCols NE 0) then begin
         beenHere = PTR_VALID(self.subStructCol)
         if (beenHere) then beenHere = (N_ELEMENTS(colNo) $
            EQ N_ELEMENTS(*self.subStructCol))
         if (beenHere) then beenHere = $
            total(colNo NE *self.subStructCol) EQ 0
      endif
      if (doFloat ) then begin
         beenHere = beenHere AND PTR_VALID(self.floatCol)
         if (beenHere) then beenHere = (N_ELEMENTS(floatNo) $
            EQ N_ELEMENTS(*self.floatCol))
         if (beenHere) then beenHere = $
            total(floatNo NE *self.floatCol) EQ 0
      endif else begin  ; no float check last time
         beenHere = beenHere AND (NOT PTR_VALID(self.floatCol))
         PTR_FREE,self.floatCol
      endelse
;need to create new substructure template
      if (NOT beenHere) then begin
         PTR_FREE, self.subStructCol
         PTR_FREE, self.subStrucTemp
         PTR_FREE, self.floatCol
         if (doCols NE 0) then self.subStructCol = PTR_NEW(colNo)
         if (doFloat) then self.floatCol = PTR_NEW(floatNo)
;make sure I have a full template
         if (NOT PTR_VALID(self.strucTemp)) then $
            dummy = self->template () ;->template puts result in self.structTemp
         subNames = self->columnNames()
;col numbers are 1-relative convert to 0 relative for idl
         subNames = subNames[colNo-1]
         if (total(colNo[0] EQ floatNo) EQ 0) then $
            subStrucTemp = CREATE_STRUCT(subNames[0], $
            (*self.strucTemp).(colNo[0]-1))$
         else subStrucTemp = CREATE_STRUCT(subNames[0], $
            float((*self.strucTemp).(colNo[0]-1)))
         for iNames = 1, nCols-1 do if(total(colNo[iNames] EQ floatNo) EQ 0) $
            then subStrucTemp = CREATE_STRUCT(subStrucTemp, subNames[iNames], $
               (*self.strucTemp).(colNo[iNames]-1)) $
            else subStrucTemp = CREATE_STRUCT(subStrucTemp, subNames[iNames], $
               float((*self.strucTemp).(colNo[iNames]-1)))
         self.subStrucTemp = PTR_NEW(subStrucTemp)
      endif       ; already knew substructure
   endif    ; select columns or floats
;
;Create a single template output structure with required structure
   if (NOT PTR_VALID(self.strucTemp)) then $
            dummy = self->template () ;->template puts result in self.structTemp
;Create output array of structures , check whether substructures wanted
   if ((doCols EQ 0) AND (doFloat EQ 0)) then $
      if (nRow GT 1) then dataStruct = REPLICATE(*self.strucTemp, nRow) $
         else dataStruct = *self.strucTemp $
   else $
      if (nRow GT 1) then dataStruct = REPLICATE(*self.subStrucTemp,nRow)$
         else dataStruct = *self.subStrucTemp
   bR = 0L
;loop over rows
   while (bR LT nRow) do begin
;find a group of continuous rows
      eR = bR
      if (eR LT (nRow -2)) then while ((eR LT (nRow - 2)) AND (sRow[eR+1] EQ (sRow[eR]+1))) do eR = eR + 1 $
      else if (eR EQ (nRow -2)) then if (sRow[nRow-1] EQ (sRow[nRow-2]+1)) then eR = nRow-1
      nRead = eR - bR + 1       ; how many contiguous rows
;is read buffer already allocated? otherwise do so.
;  *** because of problems with length of strings, reallocate memory every time
;  *** the problem is that the string lengths in self.dataTemp depend on
;  *** the size of the strings actually read in, so the allocated length
;  *** is different from what they are n the template
;      if (nRead NE self.oRead) then begin
         PTR_FREE,self.dataTemp
         self.dataTemp = PTR_NEW(REPLICATE(*self.strucTemp, nRead))
         self.oRead = nRead
;      endif
;Find the position of the first byte of the data array in the file.
      offSet = self.dataStart + self.naxis1*(sRow(bR)-1)
      POINT_LUN,iLun,offSet
;read in data
       READU, iLun, *self.dataTemp
       IEEE_TO_HOST, *self.dataTemp
;put data in output array
;no column selection
      if ((doCols EQ 0) AND (doFloat EQ 0)) then $
         for iR = bR, eR do dataStruct[tRow[iR]] = (*self.dataTemp)[iR-bR] $
      else for iR = bR, eR do $   ; do column selection and/or floating
       for iCol = 0, nCols -1 do begin


      inCol = colNo(iCol)-1  ; zero relative

             if (total((inCol+1) EQ floatNo) EQ 0) then $
             dataStruct[tRow[iR]].(iCol) = $
                   (*self.dataTemp)[iR-bR].(inCol) else $
                dataStruct[tRow[iR]].(iCol) = $
                   (*self.dataTemp)[iR-bR].(inCol) * $
                   (*self.tScale)[inCol] + (*self.tZero)[inCol]
          endfor
;is rescaling necessary?


    if (doScale) then for iCol = 0, N_ELEMENTS(scaleNo)-1 do begin
         inCol = MIN(WHERE(scaleNo(iCol) EQ colNo))
         if (inCol GE 0) then $
         (*self.dataTemp).(inCol) = (*self.dataTemp).(inCol)*$
            tScale[iCol] + tZero[iCol]
      endfor

      self.dataCurrent = self.dataStart + self.naxis1*(tRow(eR)-1)

      bR = eR + 1
   endwhile          ; loop over rows

   RETURN, dataStruct
END

PRO FITSTABLE::addUnits, columns, units, iErr=iErr
;
;   If the UNITS descriptors should be added or modified
;   do it here (if IDL structures were initial inputs, they
;   do not have UNITS information)
;
;   INPUTS:
;   columns   intarr or strarr   columns numbers or labels to which
;                                UNITS should be added; numbers are 1-rel
;   units     strarr             The UNITS to be added (must be
;                                same length as columns
;
;   OUTPUTS   iErr
;
;
;   check inputs
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;check inputs
   if(N_ELEMENTS(columns) NE N_ELEMENTS(units)) then $
      midiSetError, 'Column and units arrays are not the same size'
;  Units should be string
   Sunit = SIZE(units)
   Stype = Sunit[Sunit[0]+1]
   if( Stype NE 7) then midiSetError, 'Units must be string array'
;columns numbers or column names
   sCol = SIZE(columns)
   nCol = N_ELEMENTS(columns)
;Numbers
   cType = sCol[sCol[0]+1]
   if (cType GE 1 AND (cType LE 3)) then cols = columns $
   else if (cType EQ 7) then begin
      cols = INTARR(nCol)
      FOR iCol = 0, nCol-1 do cols[iCol] = $
         WHERE(strUpCase(strTrim(columns[iCol],2)) EQ strTrim(*self.Ttype,2))
      if (TOTAL(cols LT 0) GT 0) then $
         midiSetError, 'Specified columns names not present in table'
;Fits is 1-relative
      cols = 1 + cols
   endif
   FOR iCol = 0, nCol-1 do self.head->addPar,'TUNIT'+strTrim(cols[iCol],2),$
      strTrim(strUpCase(units[iCol]),2)
   iErr = 0
RETURN
END

PRO FITSTABLE::addScales, columns, tScale, tZero, iErr=iErr
;
;  Set TSCAL and TZERO internal members parameters for table columns
;  This is used if input source is IDL table without scaling
;
;   INPUTS:
;   columns  intarr or strarr   list of affected columns
;   tScale   dblarr             scaleing factors
;   tZero   dblarr              zero offsets
;
;   OUTPUTS:
;   iErr
;
;   check inputs
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   nCol = N_ELEMENTS(columns)
   if(nCol NE N_ELEMENTS(TSCALE) OR (nCol NE N_ELEMENTS(TZERO))) then $
      midiSetError, 'Column and scale and size arrays are not the same size'
;columns numbers or column names
   sCol = SIZE(columns)
   nCol = N_ELEMENTS(columns)
;Numbers
   cType = sCol[sCol[0]+1]
   if (cType GE 1 AND (cType LE 3)) then cols = columns $
   else if (cType EQ 7) then begin
      cols = INTARR(nCol)
      for iCol = 0, nCol-1 do cols[iCol] = $
         where(strUpCase(strTrim(columns(iCol),2)) EQ *self.TtypeS)
      if (TOTAL(cols LT 0) GT 0) then $
         midiSetError, 'Specific columns names not present in table'
;FITS is 1-relative
      cols = 1 + cols
   endif
   for iCol = 0, nCol-1 do begin
;set the parameters in local storage and write them in header
      (*self.tScale)[cols[iCol]-1] = tScale[iCol]
      (*self.tZero)[cols[iCol]-1]  = tZero[iCol]
      self.head->addPar,'TSCAL'+strTrim(cols[iCol],2),tScale[iCol]
      self.head->addPar,'TZERO'+strTrim(cols[iCol],2),tZero[iCol]
   endfor
   iErr = 0
RETURN
END

PRO FITSTABLE::newFile, fileName, iErr=iErr, table=table, $
   priHead=priHead
;Open a new output fitsfile with given name
;write a minimal primary header
;then write out the current extension header and
;be prepared to write data.
;
;if /table is specified, then write out internal stored table
;INPUTS:
;   fileName   String     file name of output file
;   table      anything   if set, internal table written automatically
;   priHead    fitsHead   if given, use this as primary header in new file
;                         otherwise use priHead stored in "self"
;OUTPUTS
;   iErr       Int
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   iErr = 1
   errMsg = ''
;
;Open file.  Copy local version of primary header into file and reset pointer
;
   if (NOT KEYWORD_SET(priHead)) then priHead = self.priHead else begin
      errMsg = 'primary header not valid'
      if (NOT OBJ_VALID(priHead)) then midiSetError, errMsg
      if (OBJ_CLASS(priHead) NE 'FITSHEADER') then midiSetError,errMsg
      if (NOT priHead->isPriHead()) then midiSetError,errMsg
   endelse
   self.file = OBJ_NEW('FITSFILE', filename, 'WRITE', priHead=priHead, $
      /noself, iErr=iErr)
   if (iErr ne 0) then midiSetError,'',/notInitial
   (self.file)->addExt, self
   self.priHead = OBJ_NEW()
   unit = (self.file)->unit()
;Write primary header to file
   WRITEU, unit, ((self.file)->PRIhead())->stHead()
   POINT_LUN,-unit, point
   self.headStart = point
   WRITEU, unit, (self.head)->stHead()
   self.headerStatus = 'DISK'
;Set extnumber and pointer to beginning of data
   self.extNumber = 1
   point = 0L
   POINT_LUN,-unit, point
   self.dataStart   = point
   self.dataCurrent = point
   self.dataEnd     = point
   self.dataStatus  = 'WRITE'
   (self.file)->setEof,point
   (self.file)->setLastExt,self
;reset number of lines on disk
   self.naxis2 = 0
   iErr = 0
   if (KEYWORD_SET(table)) then begin
      self->writeRows, *self.table, 1, iErr=iErr
      if (iErr NE 0) then midiSetError, 'writing table to file failed', $
         /notInitial
   endif
RETURN
END

PRO FITSTABLE::appendToFile, input, iErr=iErr, table=table
;  For a given, open, fits file, prepare to write a table at
;  the end.  Write out the current extension header and
;  prepare to write data.
;
;  If keyword table is specified, write out the internal table too.
;
;   INPUTS:
;   input    FITSFILE object or FITSEXTENSION object specifying file
;
;   OUTPUTS:  iErr
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;
;input can either be a FITSFILE object or a FITSEXTENSION object
   sinput = SIZE(input)
   sinput = sinput[sinput[0]+1]
   if (sinput NE 11) then $
      midiSetError, 'INPUT must be a FITSFILE or FITSEXTENSION object'
;check that it's the right kind of object
   inClass = OBJ_CLASS(input)
   inClass = inClass[0]
;find highest order superclass of inclass
   sc      = inClass
   while (sc NE '') do begin
      superClass = sc
      sc = OBJ_CLASS(sc,/ super)
      sc = sc[0]
   endwhile
   superClass = superClass[0]
   if (superClass NE 'FITSFILE' AND (inClass NE 'FITSEXTENSION') $
      AND (superClass NE 'FITSEXTENSION'))  then $
      midiSetError, 'INPUT class of '+inClass+' incorrect'
;OK got the right kind of input
   if (inClass EQ 'FITSEXTENSION' OR (superClass EQ 'FITSEXTENSION')) $
      then self.file = input->file() $
      else if (superClass EQ 'FITSFILE') then self.file = input
; Check whether we can write onto end of file
   openStatus = (self.file)->openStatus()
   if (openStatus NE 'WRITE' AND openStatus NE 'APPEND') then $
      midiSetError,'FITS file '+(self.file)->fileName()+ ' was not opened ' $
         + 'for WRITE or APPEND'
; Check if last section correctly terminated ***
   lastExt = (self.file)->lastExt()
;No previous extension
   if (NOT OBJ_VALID(lastExt)) then self.file->pad2880, rest, iErr=iErr $
      else begin
;previous extension.  If FITSTABLE use writefinish, otherwise just pad
         inClass = OBJ_CLASS(lastExt)
    isTable = inClass EQ 'FITSTABLE'
;is this a subclass of FITSTABLE?
    if (not isTable) then begin
       ic = inClass
       while (ic NE '') do begin
          ic = OBJ_CLASS(ic,/super)
          ic = ic[0]
          isTable = isTable OR (ic EQ 'FITSTABLE')
       endwhile
    endif
         if (isTable) then lastExt->writeFinish,iErr=iErr $
       else self.file->pad2880, rest, iErr=iErr
      endelse
   if (iErr ne 0) then midiSetError, 'writeFinish error',/notInitial
;Go to end of file
   unit = (self.file)->unit()
   point = (self.file)->eof()
   POINT_LUN, unit, point
   self.headStart = point
;Write table header to disk and set status pointers
   WRITEU, unit, (self.head)->stHead()
   self.headerStatus = 'DISK'
   self.extNumber = 1 + (self.file)->numExt()
   (self.file)->setNumExt,self.extNumber
   POINT_LUN,-unit, point
   self.dataStart   = point
   self.dataStatus  = 'WRITE'
   (self.file)->setLastExt,self
;reset naxis2
   self.naxis2 = 0
   iErr = 0
   if (KEYWORD_SET(table)) then begin
      self->writeRows, *self.table, 1, iErr=iErr
      if (iErr NE 0) then midiSetError, $
         'writing table to file failed', /notInitial
      POINT_LUN,-unit, point
   endif
   self.dataCurrent = point
   self.dataEnd     = point
   (self.file)->setEof,point
   (self.file)->addExt, self
RETURN
END

PRO FITSTABLE::OVERWRITE, FITSFILE, ierr
;  For a given, open, fits file, position the file
;  at the beginning of a given extension header.
;  Overwrite that header with the current header
;  and invalidate all data from here to the eof
;  Be prepared to write new data at end of header
RETURN
END

PRO FITSTABLE::writeRows, input, startRow, iErr=iErr
;  Translate an IDLstructure array of one or more elements
;  into a number of rows in a FITS Binary Table, and
;  write them out.
;
;   INPUTS:
;      input     IDL array of structures
;      startRow  int or long    starting row;
;                            if omitted or le 0; start on end of current
;                            table (or beginning if there is not yet data)
;                            Can overwrite existing data
;   OUTPUTS:
;     iErr
   errMsg = ''
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;
;Check that input agrees in structure with internal description
;
   sinput = SIZE(input)
   Stype  = sinput[sinput[0] +1]
   if (Stype NE 8) then midiSetError, 'Input must be a structure array'
;do I now what output should look like
   if (NOT PTR_VALID(self.strucTemp)) then $
      dummy = self->template () ;->template puts result in self.structTemp
   errMsg = 'Input structure different than initial description'
   inTags  = strTrim(TAG_NAMES(input), 2)
   outTags  = strTrim(TAG_NAMES(*self.strucTemp),2)
   nTags = N_ELEMENTS(inTags)
   if (nTags NE N_ELEMENTS(outTags)) then midiSetError, errMsg
   for iTag = 0, nTags - 1 do $
      if((inTags[iTag] NE outTags[iTag]) OR $
      (total( SIZE(input[0].(iTag)) NE SIZE((*self.strucTemp).(iTag))) NE 0))$
         then midiSetError, errMsg
;Check if there is a file attached
   if (NOT OBJ_VALID(self.file)) then $
      midiSetError, 'No valid FITS file attached to this table'
;Check if file is really open in correct status
   unit = (self.file)->unit()
   openStatus = (self.file)->openStatus()
   if ((unit LE 0) OR (openStatus EQ 'READ')) then $
      midiSetError, 'File '+(self.file)->fileName()+$
      ' not open or opened for READ only'
;Where do we start
   if (N_PARAMS() LT 2) then sRow = 0 else sRow = startRow
   if (sRow LE 0) then sRow = self.naxis2 + 1
   if (sRow GT (self.naxis2+1)) then $
      midiSetError, 'Starting Row beyond end of file'
   rowOffset = self.dataStart + self.naxis1*(sRow - 1)
   nRow = N_ELEMENTS(input)
   POINT_LUN,unit,rowOffSet
;copy and convert to IEEE
   data = input
   HOST_TO_IEEE, data
   WRITEU, unit, data
   self.dataCurrent = rowOffset + self.naxis1*nRow
   self.dataEnd = (self.dataCurrent > self.dataEnd)
   (self.file)->maxEof, self.dataCurrent
   self.naxis2  = (self.naxis2 > (sRow + nRow -1))
   iErr = 0
RETURN
END

PRO FITSTABLE::writeFinish, iErr=iErr
;  when writting to disk, close off current extension (but not file)
;  by padding out to 2880 byte blocks and writing the current value
;  of naxis2 into the disk header
;
;  OUTPUTS:
;    iErr
;
;  Calculate position of naxis2 keyword and update it
   errMsg = ''
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   if (NOT OBJ_VALID(self.file)) then $
      midiSetError, 'No valid FITS file attached to this table'
   unit = (self.file)->unit()
   pointer = self.headStart + 4*80
   n2 = strTrim(self.naxis2,2)
   l2 = strLen(N2)
   newLine = 'NAXIS2  ='+STRING(REPLICATE(32B,21-L2))+N2+' /Number of rows'+$
      STRING(REPLICATE(32B,34))
   POINT_LUN, unit, pointer
   WRITEU, unit, newLine
; Correctly terminate this extension and mark it as fixed
   bottom = 0
   self.file->pad2880, rest, iErr=iErr
   if (iErr ne 0) then midiSetError,'',/notInitial
   self.dataStatus = 'DISK'
   iErr = 0
RETURN
END


PRO FITSTABLE::close, iErr=iErr
;correctly finish up this table before closing file
   iErr = 0
   status = self.file->openStatus()
   if (status EQ '') then RETURN
   if (status NE 'READ') then self->writeFinish, iErr=iErr
   if (iErr EQ 0) then self->FITSEXTENSION::close, iErr=iErr
   if (iErr EQ 0) then RETURN
   midiSetError,'fitstable close failed',/notInitial
RETURN
END

FUNCTION FITSTABLE::columnNames, iErr=iErr
;return an array of column names, if known
;INPUTS:
;   none
   iErr = 0
   if (PTR_VALID(self.ttype)) then RETURN,STRTRIM(*self.tType,2)
   iErr = 1
   midiSetError,'Table Column Names not Initialized'
RETURN,''
END

FUNCTION FITSTABLE::columnNumbers, columnNames, iErr=iErr
;given a list of columns names, including wild cards
;return the corresponding column numbers !! 1-RELATIVE
;INPUTS:
;columnNames   string array   list of requested column names
;                             if a names is of the format 'ABC*', all
;                             column numbers beginning with ABC will
;                             be returned.  If a name is not
;                             found, a -1 is returned and iErr is set to 1
;                             but no error message is written
;RETURNS:
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN, -1
   endif         ; handle actual error
;   array of found column numbers 1-RELATIVE (FITS convention)
   iErr = 1
   nNames = N_ELEMENTS(columnNames)
   if (nNames LT 1) then midiSetError, 'no Input'
   inSize = size(columnNames)
   inSize = inSize[inSize[0]+1]
   if (inSize NE 7) then midiSetError, 'Input not string'
   myNames = self->columnNames()
   columns = [-1]
   for iName = 0, nNames -1 do begin
      name = STRTRIM(columnNames[iName],2)
      lName = STRLEN(name)
;chop off "*"
      if (STRMID(name,lName-1,1) EQ '*') then begin
         wild = 1
         name = STRMID(name,0,lName-1)
    lName = lName - 1
      endif else wild = 0
      if (wild) then $
         columns = [columns, WHERE(name EQ STRMID(myNames,0,lName))] $
      else columns = [columns, WHERE(name EQ myNames)]
   endfor
   iErr = TOTAL(columns NE (-1)) EQ 0
;columns in FITS are 1 relative
RETURN, columns[1:*] + 1
END


FUNCTION FITSTABLE::tZero
RETURN, *self.tZero
END

FUNCTION FITSTABLE::tScale
RETURN, *self.tScale
END
