FUNCTION vinciRawFiles, reducedFiles, rawDir=rawDir, iErr=iErr
;****************************************************
;#function#
;vinciFiles
;#description#
;return all raw Files used in a set of reduced files
;in the form of a fitsFileList object.
;#call#
;RawFileList= vinciRawFiles(reducedFileSpec, iErr=iErr)
;#inputs#
;\anArg{reducedFiles}{poly}{specifies file: can either be a}
;\anArg{-}{-}{file name, array of file names, or a fitsFileList object }
;\anArg{rawDir}{string}{A  directory in which to look for raw files}
;\anArg{-}{-}{If not specified I will try the environmental variable:vinciRawData}
;\anArg{-}{-}{If I dont find this, I will use the input file names as is:}
;\anArg{-}{-}{they may have indeterminate directories attached}
;#return#
;\anArg{-}{fitsFileListObject}{contains list of raw files}
;#end_function#
;****************************************************
   inType = SIZE(reducedFiles,/type)
;fileList input
   if (inType EQ 11) then begin 
      if (OBJ_CLASS(reducedFiles) NE 'FITSFILELIST') then begin
         print,'input is not a fitsFileList object'
         RETURN,0
      endif
      inFiles = reducedFiles->files()
   endif else if (inType EQ 7) then $
      inFiles = reducedFiles $
   else begin
      print,'input type not recognized'
      RETURN,0
   endelse
   nIn = N_ELEMENTS(inFiles)
   outFiles = ''
   for iIn = 0, nIn-1 do begin
      fitsFile = OBJ_NEW('fitsfile', inFiles[iIn], iErr=iErr)
      if (iErr NE 0) then begin
         print,'cant open input file'
         RETURN,0
      endif
      priHead = fitsFile->priHead()
;check if this is really reduced
      if ('GLUED' NE STRTRIM(priHead->getPar('PRO TYPE'),2)) $
      and ('RED' NE STRTRIM(priHead->getPar('PRO TYPE'),2)) then begin
         print, 'input file is not a reduced PRO file'
         OBJ_DESTROY, priHead
         OBJ_DESTROY, fitsFile
      RETURN,0
      endif
;open batch table
      batch = OBJ_NEW('fitsTable', fitsFile, extName='BATCH',iErr=iErr)
      if (iErr NE 0) then begin
         print,'cant open input batch table '
         OBJ_DESTROY, priHead
         OBJ_DESTROY, fitsFile
         RETURN,0
      endif
      batchTable = batch->readrows()
;count number of valid batches
      nBatch = total(batchTable.nScan NE 0)
      for iBatch = 0, nBatch - 1 do begin
         outFiles = [outFiles,STRTRIM(batchTable[iBatch].file_Off,2)]
         outFiles = [outFiles,STRTRIM(batchTable[iBatch].file_A,2)]
         outFiles = [outFiles,STRTRIM(batchTable[iBatch].file_B,2)]
         outFiles = [outFiles,STRTRIM(batchTable[iBatch].file_On,2)]
      endfor
      OBJ_DESTROY, priHead
      OBJ_DESTROY, fitsFile
   endfor   ; list of input files
   outFiles = outFiles[1:*]
;got outFile list, now parse directories
;if no directory specified just stuff em
   dir = ''
   if (KEYWORD_SET(rawDir)) then dir = rawDir else $ 
      dir = GETENV('vinciRawDir')
   if (dir EQ '') then RETURN,OBJ_NEW('fitsFileList', outFiles)
;do directory searches to find files
;try both given directory and one level higher
   dir = dir+'/'
   dirUp = dir+'*/'
   for iFile = 0, N_ELEMENTS(outFiles)-1 do begin
      fullFile = outFiles(iFile)
      split = STRSPLIT(fullFile,'/',/EXTRACT)
      fullFile = split(N_ELEMENTS(split)-1)
      File = findFile(dir+fullFile)
      if (File[0] EQ '') then File = findFile(dirUp+fullFile)
      if (File[0] EQ '') then begin
	 print,'file ',outFiles(iFile),' couldnt be found in directory structure'
	 RETURN,0
      endif else outFiles(iFile) = File[0]
   endfor
RETURN,OBJ_NEW('fitsFileList', outFiles)
END

FUNCTION vinciRawScans, fileSpec, iErr=iErr
;****************************************************
;#function#
;vinciRawScans
;#description#
;return VINCI IMAGING_DATA from a set of raw batchs
;The data is in the form of a float array(nFrames, nScans, 5, nFiles)
;where nFrames is frames/scan
;nScans is number of scans
;and 5 image planes are "INT_1","INT_2","P_A","P_B", and
;a band-pass filtered version of INT_1 - INT_2
;and nFiles is the number of input files.
;These must all be of the same size (nFrames and nScans equal)
;#call#
;vinciScanData = vinciRawScans(fileSpec, iErr=iErr)
;#inputs#
;\anArg{fileSpec}{poly}{specifies file: can either be a}
;\anArg{-}{-}{file name an array of filenames, or a fitsFileList object}
;#return#
;\anArg{-}{-}{raw data}
;#end_function#
;****************************************************
;figure out input type
   iErr = 1
   fileList = ''
   inType = SIZE(fileSpec,/type)
   if (7 EQ inType) then fileList = fileSpec $
   else if (11 eq inType) then begin 
         if ('FITSFILELIST' EQ OBJ_CLASS(fileSpec)) then $
            fileList = fileSpec->files() 
      endif
   if (fileList[0] EQ '') then begin
      print,' input must be a file, list of files or non-empty'
      print,'fileList object'
      RETURN,0
   endif
   nFiles = N_ELEMENTS(fileList)
   if (nFiles LE 0) then RETURN, 0
;open 1st file and get formatting stuff
   dataTable = OBJ_NEW('fitsTable', fileList[0], $
      extName='IMAGING_DATA', iErr=iErr)
   if (iErr NE 0) then begin
      print,'cant open input file ',fileList[0]
      RETURN,0
   endif
   priHead = dataTable->priHead()
   nScan = priHead->getPar('NC NSCAN')
   nFrames = priHead->getPar('RM NFRAM')
   dataTable->close
   OBJ_DESTROY,dataTable
   OBJ_DESTROY,priHead
;create output array
   output = fltarr(nFrames, nScan, 5, nFiles)
   for iFile = 0, nFiles-1 do begin
      dataTable = OBJ_NEW('fitsTable', fileList[iFile], $
	 extName='IMAGING_DATA', iErr=iErr)
      if (iErr NE 0) then begin
         print,'cant open input file '+fileList[iFile]
         RETURN,output[*,*,*,0:iFile-1]
      endif
      priHead = dataTable->priHead()
      nF= priHead->getPar('RM NFRAM')
      nS= priHead->getPar('NC NSCAN')
      if (nF NE nFrames OR (nS NE nScan)) then begin
         print, 'file ', iFile, ' format not consistent with previous files '
         RETURN,output[*,*,*,0:iFile-1]
      endif
      data = dataTable->readRows(iErr=iErr)
      if (iErr NE 0) then begin
         print,'cant read input data from file ', iFile
         RETURN,output[*,*,*,0:iFile-1]
      endif
      for iChan = 0, 4 do output[*,*,iChan,iFile] = $
         REFORM(data.(iChan+1), nFrames, nScan)
      dataTable->close
      OBJ_DESTROY,dataTable
      OBJ_DESTROY,priHead
   endfor
   output = REFORM(output)
RETURN, output
END

PRO vinciGetBatchTables, fileSpec, source, batch, iErr=iErr
;*******************************************************
;#procedure#
;vinciGetRedTables
;#description#
;retrieve the SOURCE and BATCH tables from one or more
;reduced vinci files.  If multiple files are specified 
;concat the tables.  The source names are appended onto
;the batch files for easy reading
;#call#
;vinciGetBatchTables, fileName, SourceTable, batchTable, iErr=iErr
;vinciGetBatchTables, fileNameArray, SourceTable, batchTable, iErr=iErr
;vinciGetBatchTables, fileListObject, SourceTable, batchTable, iErr=iErr
;#inputs#
;\anArg{fileSpec}{input file specification:filename, array of filenames,}
;\anArg{-}{-}{or fitsFileList object}
;#outputs#
;\anArg{source}{idl struct}{SOURCE tables.  If more than}
;\anArg{-}{-}{one input file, concatenate tables and update numbers}
;\anArg{batch}{idl struct}{Batch tables.  If more than}
;\anArg{-}{-}{one input file, concatenate tables and update source numbers}
;#end_procedure#
;*******************************************************
;figure out input type
   iErr = 1
   fileList = ''
   inType = SIZE(fileSpec,/type)
   if (7 EQ inType) then fileList = fileSpec $
   else if (11 eq inType) then begin 
         if ('FITSFILELIST' EQ OBJ_CLASS(fileSpec)) then $
            fileList = fileSpec->files() 
      endif
   if (fileList[0] EQ '') then begin
      print,' input must be a file, list of files or non-empty'
      print,'fileList object'
      RETURN
   endif
   nFiles = N_ELEMENTS(fileList)
;array for updating source numbers on concatenation
   sourceOffset = lonarr(nFiles)
   for iFile = 0, nFiles-1 do begin
      fileName = fileList[iFile]
      sourceTable = OBJ_NEW('fitstable', fileName, extName='SOURCE', iErr=iErr)
      if (iErr NE 0) then begin
         print,' coudnt open SOURCE table for file ',filename
         RETURN
      endif
;check if this is a reduced file
      priHead = sourceTable->priHead()
      redType = priHead->getPar('PRO TYPE',match)
      typeOK = match EQ 1
      if (typeOK) then redType=STRTRIM(redType,2)
      typeOK = ((redType EQ 'RED') OR (redType EQ 'GLUED'))
      if (NOT typeOK) then begin
         print,'input file is not reduced:',filename
         RETURN
      endif
;read in source table into temporary structure
      tSource = sourceTable->readrows()
;trim off empty rows
      tSource = tSource(WHERE(tSource.SOURCE_ID GT 0))
;increment source numbers, storing offset
      if (iFile EQ 0) then begin
	 nsTags = N_ELEMENTS(TAG_NAMES(tSource))
	 source = tSource 
	 sourceStruct = source[0]
      endif else begin
	 nSource = N_ELEMENTS(tSource)
	 sourceOffset[iFile] = MAX(source.source_id)
	 tSource.source_id = tSource.source_id + sourceOffset[iFile]
	 tS = REPLICATE(sourceStruct,nSource)
	 for iS = 0, nsTags-1 do tS.(iS) = tSource.(iS)
	 source = [source, tS]
      endelse             ; first file
;open batch table
      batchTable = OBJ_NEW('fitstable',sourceTable->fileName(),extName='BATCH',$
        iErr=iErr)
;clean up a little bit
      sourceTable->close
      OBJ_DESTROY,sourceTable
      if (iErr NE 0) then begin
         print,' coudnt open BATCH table for file ',filename
         RETURN
      endif
;get batch table
      tBatch = batchTable->readrows(ierr=ierr)
      if (iErr EQ 0) then begin
      tBatch = tBatch(WHERE(tBatch.batch_id GT 0))
;initialize output if iFile = 0; create output structure
; append column for source name.  Note that some reduction
; files do not have batch tables, so you cant assume that
; iFile = 0 contains the first legitimate batch file
      if (N_ELEMENTS(outStruct) LE 0) then begin
	 outStruct = CREATE_STRUCT(tBatch[0], 'sourceName','')
	 nTags = N_ELEMENTS(TAG_NAMES(tBatch))
	 nBatch = N_ELEMENTS(tBatch)
	 batch = replicate(outStruct, nBatch)
	 source_id = source.source_id
	 for iTag = 0, nTags-1 do batch.(iTag) = tBatch.(iTag)
	 for ib = 0, nBatch-1 do batch[ib].sourceName = $
	    source[WHERE(source.source_id EQ batch[ib].source)].source
      endif else begin
	 nBatch = N_ELEMENTS(tBatch)
	 tB = REPLICATE(outStruct, nBatch)
	 for iTag = 0, nTags-1 do tB.(iTag) = tBatch.(iTag)
	 for ib = 0, nBatch-1 do tB[ib].sourceName = $
	    source[WHERE(source.source_id EQ tB[ib].source)].source
	 for ib = 0, nBatch-1 do tB[ib].source = tB[ib].source +$
	    sourceOffset[iFile]
         batch = [batch, tB]
      endelse
      endif   ; successfully read batch rows
      batchTable->close
      OBJ_DESTROY,batchTable
   endfor   ; loop over files
RETURN
END

PRO vinciAssembleRaw, fileSpec, OffData, Adata, Bdata, OnData, $
   frameRate, nx, fringeSpd, obName, insMode, obsDate,  iErr=iErr
;********************************************************************
;#procedure#
;vinciAssembleRaw
;#description#
;assemble the data from 4  batches (OFF/A/B/ON) in one legitimate vinci group
;#call#
;vinciAssembleRaw, fileSpec, OffData, Adata, Bdata, OnData, $
;  frameRate, nx, fringeSpd, obName, insMode, iErr=iErr
;#inputs#
;\anArg{fileSpec}{poly file specification}{can be an array of 4 files}
;\anArg{-}{-}{a FileList containing 4 files, or a single reduced file}
;#outputs#
;\anArg{OffData}{fltarr}{Data from OFF batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{AData}{fltarr}{Data from A batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{BData}{fltarr}{Data from B batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{OnData}{fltarr}{Data from On batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{frameRate}{float}{frame rate}
;\anArg{nx}{int}{window pixel size}
;\anArg{fringeSpd}{float}{fringe speed (m/s)}
;\anArg{obName}{string}{OB name}
;\anArg{insMode}{string}{INS MODE }
;\anArg{obsDate}{string}{obs-date}
;#end_procedure#
;******************************************************************** 
;parse file spec; it can be one .Pro file name, four file names or
;a fitsFileList with 4 files.
   COMPILE_OPT HIDDEN
   iErr = 1
   fileList = ''
   inType = SIZE(fileSpec,/type)
   if (7 EQ inType) then fileList = fileSpec $
   else if (11 eq inType) then begin
      if ('FITSFILELIST' EQ OBJ_CLASS(fileSpec)) then $
         fileList = fileSpec->files()
   endif
   if (fileList[0] EQ '') then begin
      print,' input must be a file, list of files or non-empty'
      print,'fileList object'
      RETURN
   endif
   nFiles = N_ELEMENTS(fileList)
;a single file; test if this is a .PRO. file and then
;get the reduced files
   if (nFiles EQ 1) then begin
      if (STRPOS(fileList[0],'PRO') LT 0) then begin
         print,'Single input files is not of reduced type'
         RETURN
      endif else begin
        rawFiles = vinciRawFiles(fileList[0],iErr=iErr)
        if (iErr NE 0) then begin
           print,'Couldnt recover rawfiles from reduced file'
           RETURN
        endif
        fileList = rawFiles->files() 
        nFiles = N_ELEMENTS(fileList)
	OBJ_DESTROY, rawFiles
      endelse
   endif
   if (nFiles NE 4) then begin
      print, 'Input needs to be 4 files'
      RETURN
   endif
;got 4 files
;because we're lazy, reassemble them into a fileList object
   rawFiles = OBJ_NEW('fitsFileList', fileList)
;  keyList = ['obs name','ql framerate','dpr catg','ql fringespd',$
;     'win1 nx','ins mode','date-obs']
   keyList = ['obs name','ql framerate','dpr type','ql fringespd',$
      'win1 nx','ins mode','date-obs']
   headKeys = rawFiles->getHeaderKeys(keyList)
   obName=headKeys[0].(1)
   consistent = total(headKeys.(1) NE obName) EQ 0
   frameRate=headKeys[3].(2)
;  consistent = consistent AND (total(headKeys.(1) NE frameRate) EQ 0)
;ESO hasn't got the frame rates right in the non-On data
   nx=headKeys[0].(5)
   consistent = consistent AND (total(headKeys.(5) NE nx) EQ 0)
   fringeSpd=headKeys[3].(4)
;  consistent = consistent AND (total(headKeys.(3) NE fringeSpd) EQ 0)
   insMode = headKeys[0].(6)
   obsDate = headKeys[3].(7)
   fileTypes = STRTRIM(headKeys.(3),2)
   nOff = MIN(WHERE(fileTypes EQ 'CALIB,OFFSOURCE'))
   nA   = MIN(WHERE(fileTypes EQ 'CALIB,BEAM_A'))
   nB   = MIN(WHERE(fileTypes EQ 'CALIB,BEAM_B'))
   nOn  = MIN(WHERE(fileTypes EQ 'SCIENCE,OBJECT'))
; Modified by cah
   nOff = MIN(WHERE(fileTypes EQ 'OFFSOURCE'))
   nA   = MIN(WHERE(fileTypes EQ 'BEAM_A'))
   nB   = MIN(WHERE(fileTypes EQ 'BEAM_B'))
   nOn  = MIN(WHERE(fileTypes EQ 'OBJECT'))
   if nOn lt 0 then nOn  = MIN(WHERE(fileTypes EQ 'STD'))
   consistent=consistent AND (nOff GE 0) AND (nA GE 0) AND $
      (nB GE 0) AND (nOn GE 0)
   if (NOT consistent) then begin
      print,' four input files do not agree in keywords'
      print,'or do not contain all for obs types'
stop
      RETURN
   endif
   OffData = vinciRawScans(fileList(nOff), iErr=iErr)
   if (iErr EQ 0) then AData = vinciRawScans(fileList(nA), iErr=iErr)
   if (iErr EQ 0) then BData = vinciRawScans(fileList(nB), iErr=iErr)
   if (iErr EQ 0) then ONData = vinciRawScans(fileList(nOn), iErr=iErr)
   if (iErr NE 0) then print,' failed to acquire all four data files'
RETURN
END

FUNCTION vinci_OF_read, infile, iErr=iErr
;********************************************************************
; Read a FITS table and return an ObsFile
;#function#
; vinci_OF_read
;#call#
; vinci_OF_read(infile, iErr=iErr)
;#description#
; Reads the specified FITS file and converts it into an ObsFile
; structure.
;#inputs#
;\anArg{ infile  }{ string  }{ name of input FITS file}
;#outputs#
;\anArg{ iErr }{ int }{ Return code, 0 => OK, else failed}
;#return#
; VINCIObsFile* = structure created (or something very much like it)
;#end_function#
;********************************************************************
; Get primary header
  file  = obj_new("fitsfile",infile,"READ")
  if (OBJ_VALID(file) LE 0) then begin
    print,"File ",infile," cannot be opened "
    ierr = 1
    return,0L
  endif
; get primary header
  oldHead=file->prihead()
; make copy of header
  newHead = oldHead->clone()
; obj_destroy,newHead
; obj_destroy,oldHead
; obj_destroy,file

; Get component tables
; read ARRAY_GEOMETRY table
  geom_table = OBJ_NEW('fitstable', infile, extName='ARRAY_GEOMETRY')
  geom_data = geom_table->readRows()
; obj_destroy,geom_table

; read OPTICAL_TRAIN table, if there
  ot_table = OBJ_NEW('fitstable', infile, extName='OPTICAL_TRAIN')
; is it there?
  if (OBJ_VALID(ot_table)) then begin
    ot_data = ot_table->readRows()
  endif else begin ot_data = 0L
  endelse
; obj_destroy,ot_table

; read SOURCE table
  source_table = OBJ_NEW('fitstable', infile, extName='SOURCE')
  source_data = source_table->readRows()
  ; select valid values
  source_data = source_data[where(source_data.source_id GT 0)]
  ; set default frequency (2.128 mu) if necessary
  index = where (source_data.freq[0] LT 1.0)
  if (index[0] GE 0) then begin
    source_data[index].freq[0] = 2.997924562e8 / 2.128e-6
  endif
; obj_destroy,source_table

; read BATCH table
  batch_t = OBJ_NEW('fitstable', infile, extName='BATCH')
  batch_d = batch_t->readrows()
  ; find how many batches are actually populated
  index = where (batch_d.batch_id GE 1, nbatch)
  ; select valid values
  batch_d = batch_d[index]
  ; get revision number
  revision = batch_t->getPar("REVISION")
; obj_destroy,batch_t

; read FRINGE_A table
  fringe_a_t = OBJ_NEW('fitstable', infile, extName='FRINGE_A')
  fringe_a_d = fringe_a_t->readrows()
; obj_destroy,fringe_a_t

; create VINCI batches and reorganize
  index = where (fringe_a_d.batch EQ batch_d[0].batch_id, nscan)
  template = {nscans:nscan, isCal:0L, batch_data: batch_d[0], $
              scan_data:PTR_NEW(fringe_a_d[index])}
  batches = replicate(template, nbatch)
  for i = 0, nbatch-1 do begin
    batches[i].nscans = nscan
    ; is this a calibrator (nonblank cal code)?
    index = where (source_data.source_id EQ (batch_d[i].source)(0))
    if (source_data[index[0]].calcode NE "    ") then begin
      batches[i].isCal = 1
    endif
    index = where (fringe_a_d.batch EQ (batch_d[i].batch_id)(0), nscan)
    if (nscan GT 0) then begin
      batches[i].nscans = nscan
      batches[i].scan_data = PTR_NEW(fringe_a_d[index])
      ; copy batch_data values
      for j = 0, n_tags(batches[i].batch_data)-1 do begin
         batches[i].batch_data.(j) = batch_d[i].(j)
      endfor
    endif else begin
      batches[i].nscans = 0
      batches[i].batch_data.batch_id = -1
      batches[i].scan_data = PTR_NEW(fringe_a_d[0])
    endelse
  endfor
;
; create output structure (like VINCIObsFile)
; Note: this should have the same structure as the similar
; definition in vinci_OF_extract_cals
   out = {infile:infile, outfile:'  ', rawdata:'',            $
          revision:revision, nbatch:nbatch, priHead:newHead,  $
          source_table:source_table, source_data:source_data, $
          ot_table:ot_table, ot_data:ot_data,                 $
          geom_table:geom_table, geom_data:geom_data,         $
          batch_table:batch_t,                                $
          fringe_a_table:fringe_a_t,  batches:batches} 
;
; cleanup (CAH added 3 Oct 2003)
;
RETURN, out
END   ; end of vinci_OF_read

PRO vinci_OF_write, outfile, ObsFile, iErr=iErr
;********************************************************************
; Write an ObsFile to a FITSfile
;#procedure#
; vinci_OF_write
;#call#
; vinci_OF_write, outfile, ObsFile, iErr=iErr
;#description#
; Writes an ObsFile to the specified FITS file
; structure.
;#inputs#
;\anArg{ outfile  }{ string  }{ name of input FITS file}
;\anArg{ ObsFile }{ VINCIObsFile-like }{ memory resident version }
;#outputs#
;\anArg{ iErr }{ int }{ Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
; apply current calibration to be sure
  vinci_OF_apply_cal, ObsFile

; write ARRAY_GEOMETRY table
  ObsFile.geom_table->newfile, outfile, prihead=ObsFile.priHead
;  ObsFile.geom_table->setSize, 0L
  ObsFile.geom_table->writeRows, ObsFile.geom_data, 1
  ObsFile.geom_table->writeFinish

; write OPTICAL_TRAIN table, if there
  if (OBJ_VALID(ObsFile.ot_table)) then begin
    ObsFile.ot_table->appendToFile, ObsFile.geom_table
    ObsFile.ot_table->writeRows, ObsFile.ot_data, 1
    ObsFile.ot_table->writeFinish
  endif

; write SOURCE table
  ObsFile.source_table->appendToFile, ObsFile.geom_table
;  ObsFile.source_table->setSize, 0L
  ObsFile.source_table->writeRows, ObsFile.source_data, 1
  ObsFile.source_table->writeFinish

; Batch table
  ObsFile.batch_table->appendToFile, ObsFile.source_table
;  ObsFile.batch_table->setSize, 0L
  for i = 0, ObsFile.nbatch-1 do begin
    ObsFile.batch_table->writeRows, ObsFile.batches[i].batch_data
  endfor
  ObsFile.batch_table->writeFinish

; Fringe_A table
  ObsFile.fringe_a_table->appendToFile, ObsFile.batch_table
;  ObsFile.fringe_a_table->setSize, 0L
; loop over batches writing fringe_a table info
  for i = 0, ObsFile.nbatch-1 do begin
    ObsFile.fringe_a_table->writeRows, *ObsFile.batches[i].scan_data
  endfor
  ObsFile.fringe_a_table->writeFinish

; close file
  ObsFile.fringe_a_table->close
END   ; end of vinci_OF_write

