;************************************************************************
;#class#
;VINCI PIPE
;#description#
;provide a prototype "pipeline" for VINCI raw data reduction
;that takes a series of four VINCI-batches (OFF/A/B/ON)
;reads in the data, calculated power spectra, makes
;some plots and maybe ;even estimates visibilities.
;This is not a definitive pipeline, but a "template" to
;allow users to make a copy of this, rename it, and
;make modifications as desired
;#end_class#
;****************************************************************

PRO vinciUnOff, OffData, AData, BData, ONData, OffStat
;********************************************************************
;#procedure#
;vinciUnOff
;#description#
;remove the average of the OffData from all 5 channels
;#call#
;vinciUnOff, OffData, Adata, Bdata, OnData, OffStat
;#inputs#
;\anArg{OffData}{fltarr}{Data from OFF batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{AData}{fltarr}{Data from A batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{BData}{fltarr}{Data from B batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{OnData}{fltarr}{Data from On batch, 5 deep (I1,I2,PA,PB,D5)}
;#outputs#
;\anArg{OffStat}{fltarr 5x2}{Mean and RMS of 5 OffData channels}
;#end_procedure#
;******************************************************************** 
   COMPILE_OPT HIDDEN
   OffStat = fltarr(5,2)
   for iChan = 0,4 do begin
      mom = moment(OffData(*,*,iChan))
      OffStat(iChan, 0) = mom(0)
      OffStat(iChan, 1) = sqrt(mom(1))
      AData(*,*,iChan) = AData(*,*,iChan) - OffStat(iChan,0)
      BData(*,*,iChan) = BData(*,*,iChan) - OffStat(iChan,0)
      OnData(*,*,iChan) = OnData(*,*,iChan) - OffStat(iChan,0)
   endfor
RETURN
END

FUNCTION vinciPower, data, filter, average=average
;*******************************************************
;#function#
;vinciPower
;#description#
;computer the power spectra of an array of scanData
;#call#
;PowerSpectra = vinciPower(dataArray, filter, /average=average)
;#inputs#
;\anArg{data}{fltarr (2 or 3-d)}{An array of vinci scans}}
;\anArg{filter}{fltarr(1-d)}{optional: an array  to multiply each row}
;\anArg{-}{-}{of data before computer power spectrum}
;\anArg{average}{bool}{if set computer average Power spectrum of all scans}
;\anArg{-}{-}{by default computer Power spectrum of each row separately}
;#return#
;\anArg{-}{fltarr}{for each row the power spectrum or the average power}
;\anArg{-}{-}{spectrum of all rows.  The rows are only half as long as}
;\anArg{-}{-}{the rows of the input data because the input is real}
;#end_function#
;********************************************************
   dataSize = size(data)
   nX = dataSize(1)
   hX = nX/2
   nDim = dataSize(0)
   nY = dataSize(2)
   if (nDim EQ 3) then nZ = dataSize(3) else nZ = 1
   if(N_PARAMS() EQ 1) THEN filter = replicate(1., nX)
   outPower = fltarr(hX, nY, nZ)
   fs = complexarr(nX)
   for iZ = 0, nZ -1 do begin
   for iY = 0, nY -1 do begin
      fs(*) = fft(filter*data(*,iY,iZ),1)
      outPower(*,iY,iZ) = (float(fs(0:hX-1)))^2 + (imaginary(fs(0:hX-1)))^2
   endfor
   endfor
   if (KEYWORD_SET(average)) then outPower = total(outPower,2)/nY
   if (nDim EQ 2) then outPower=reform(outPower)
RETURN, outPower
END

FUNCTION vinciKappa, AData, BData
;********************************************************************
;#function#
;vinciKappa
;#description#
;determine cross-coupling coefficients assuming
;input is already debiased (UnOffed)
;#call#
;kappa = vinciKappa(Adata, Bdata )
;#inputs#
;\anArg{AData}{fltarr}{De-zeroed Data from A batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{BData}{fltarr}{De-zeroed Data from B batch, 5 deep (I1,I2,PA,PB,D5)}
;#return#
;\anArg{kappa}{fltarr 2x2}{[d1/dA, d1/dB, d2/dA, d2/dB]}
;#end_function#
;******************************************************************** 
   COMPILE_OPT HIDDEN
   kappa = fltarr(4)
   kappa(0) = total(Adata(*,*,0))/total(Adata(*,*,2))
   kappa(1) = total(Bdata(*,*,0))/total(Bdata(*,*,3))
   kappa(2) = total(Adata(*,*,1))/total(Adata(*,*,2))
   kappa(3) = total(Bdata(*,*,1))/total(Bdata(*,*,3))
RETURN, kappa
END

FUNCTION vinciUnPhoto, OnData, kappa
;********************************************************************
;#function#
;vinciUnPhoto
;#description#
;Remove photometric fluctuations from cross correlated Data
;input is already debiased (UnOffed)
;#call#
;cleanData = vinciUnPhoto (OnData, kappa)
;#inputs#
;\anArg{OnData}{fltarr}{De-zeroed Data from On batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{kappa}{fltarr}{correlation coefficients d1/dA, d1/dB, d2/dA, d2/dB}
;#return#
;\anArg{cleanData}{fltarr }{Indata - kappa_A * A - kappa_B * B}
;#end_function#
;******************************************************************** 
   COMPILE_OPT HIDDEN
   onSize = size(OnData)
   outData = fltarr(onSize[1], onSize[2],2)
   outData[*,*,0] = onData[*,*,0] - kappa[0]*onData[*,*,2] - kappa[1]*onData[*,*,3]
   outData[*,*,1] = onData[*,*,1] - kappa[2]*onData[*,*,2] - kappa[3]*onData[*,*,3]
RETURN,outData
END

FUNCTION vinciApoPower, OnData, apoFilter, avPower
;********************************************************************
;#function#
;vinciApoPower
;#description#
;calculate power spectrum from OnData removing baseline
;calculted from off-fringe part of data
;result is, per scan, PS (ON*apoFilter) - PS(ON*(1-apoFilter))
;#call#
;apoPower = vinciApoPower (OnData, apoFilter, avPower)
;#inputs#
;\anArg{OnData}{fltarr}{De-zeroed Data from On batch, 5 deep (I1,I2,PA,PB,D5)}
;\anArg{apoFilter}{fltarr}{a time-domain weighting function}
;#return#
;\anArg{ApoPower}{fltarr }{Power Spectrum}
;\anArg{AvPower}{fltarr }{Power Spec averaged over scans}
;#end_function#
;******************************************************************** 
   COMPILE_OPT HIDDEN
;normalize filter
   filter = apoFilter/max(apoFilter)
   mFilter = mean(filter^2)
;assuming flat spectrum noise, calculate weight of off channel to
;make noise disappear
   if (mFilter lt .95) then noiseWt = mfilter/(1-mFilter) else noiseWt = 0.
   dataSize = SIZE(Ondata)
   nDim = dataSize(0)
   if (nDim EQ 3) then nData = dataSize[3] else nData = 1
   outPower = fltarr(dataSize[1], dataSize[2], nData)
   onPower = vinciPower(onData, filter)
   if (noiseWt NE 0.) then offPower = vinciPower(onData, 1-filter) else offPower=0.
;reweight for difference in areas under filtered and 1-filtered data
   outPower = onPower - noiseWT * offPower
   outPower = REFORM(outPower)
   powerSize = size(outPower)
   avPower = total(outPower,2)/powerSize[2]
RETURN, outPower
END

FUNCTION vinciABStat, onData
;********************************************************************
;#function#
;vinciMeanAB
;#description#
;calculate mean of A*B, A^2, B^2 from on source data
;#call#
;meanAB = vinciMeanAB (OnData)
;#inputs#
;\anArg{OnData}{fltarr}{De-zeroed Data from On batch, 5 deep (I1,I2,PA,PB,D5)}
;#return#
;\anArg{meanAb}{float(3) }{Mean of AB, A^2, B^2}
;#end_function#
;********************************************************************
   COMPILE_OPT HIDDEN
   outStat = fltarr(3)
   nPix1 = 1./N_ELEMENTS(onData(*,*,0))
   outStat[0] = nPix1*total(onData[*,*,2] * onData[*,*,3])
   outStat[1] = nPix1*total(onData[*,*,2] * onData[*,*,2])
   outStat[2] = nPix1*total(onData[*,*,3] * onData[*,*,3])
RETURN, outStat
END

PRO vinciPipe, fileList, datadisp=datadisp, powerdisp=powerdisp, apo=apo
;***************************************************
;#procedure#
;vinciPipe
;#call#
;vinciPipe, fileListObject, diskFile=diskFile, dir=dir, output=output,
; apo=apo
;#description#
;runs a standard set of reductions on
;a group of Vinci raw data files
;#inputs#
;\anArg{fileList}{fitsFileListObject}{object containing a set of}
;\anArg{-}{-}{files to be processed; these must be in sets of 4-Off:A:B:ON}
;#end_procedure#
;***************************************************
   ll = ''
   unit=-1
   !x.style=1
   !y.style=1
;assume files are in sets of 4
   nFiles = fileList->nFiles()
   nObs = nFiles/4
   fileArr = fileList->files()
   fileTypes = ['Off','A ','B ','ON']
   pForm = "(a20,5f10.3)"
;loop over multiple observations
   for iObs = 0, nObs-1 do begin
      bFile = 4*iObs
      files4=fileArr[bFile:bFile+3]
      a = systime(1)
      print,'Reading in data'
      vinciAssembleRaw, files4, offData, Adata, Bdata, Ondata, $
	 frameRate, winx, fSpeed, obName, insMode, onDate
      print,'That took ',systime(1)-a,' seconds',format="(a,f10.2,a)"
;print id info
      printf,unit,'obName:     ',obName
      printf,unit,'obsDate:    ', onDate
      printf,unit,frameRate, format="('Frame Rate:', f8.2)"
      printf,unit,'Ins mode:   ', insMode
      printf,unit, winx, format="('Win1 Nx:  ', i3)"
      printf,unit, fSpeed, format="('FringeSpeed: ', f8.6)"
;make data even sized to speed up ffts
      dataSize = size(onData)
      nX = dataSize(1)
      printf,unit, nX, format="('nFrames   ', i6)"
      if (nX NE 2*(nX/2)) then begin
	nX = nX -2
	offData=offData(0:nX,*, *)
	AData=AData(0:nX,* , *)
	BData=BData(0:nX,*, *)
	onData=onData(0:nX,*, *)
      endif
;subtract OFF data
      vinciUnOff, OffData, Adata, Bdata, onData, Offstat
      printf,unit, 'mean of Off channels    ', Offstat(*,0), format=pForm
      printf,unit, 'variance of Off channels', Offstat(*,1)^2, format=pForm
;calculate kappa coefficients
      kappa = vinciKappa(Adata, bData)
      printf,unit, 'kappa', kappa, format=pForm
;calculate mean photometric signals in on data
      allMean = fltarr(4, 3)
      allRMS  = fltarr(4, 3)
      for ichan = 0, 3 do begin
	 m = moment(AData(*,*,iChan))
	 allmean(iChan,0)=m(0)
	 allrms(iChan,0) = sqrt(m(1))
	 m = moment(BData(*,*,iChan))
	 allmean(iChan,1)=m(0)
	 allrms(iChan,1) = sqrt(m(1))
	 m = moment(onData(*,*,iChan))
	 allmean(iChan,2)=m(0)
	 allrms(iChan,2) = sqrt(m(1))
     endfor
     for iChan = 0, 2 do begin
	printf,unit, fileTypes(iChan+1),'Means:',allmean(*, iChan),format='(2a,4f8.2)'
	printf,unit, fileTypes(iChan+1),'RMS:  ',allrms(*, iChan),format='(2a,4f8.2)'
     endfor
      meanOn = 5*total(total(ondata,1),1)/N_ELEMENTS(onData)
      dataSize = size(onData)
      nFrame = dataSize(1)
;convert positions in FFT to freq (Hz) and frequency of incoming light
      powerFreq = frameRate * findgen(nFrame/2)/nFrame
      lightFreq = powerFreq * 2.9979e8/fspeed ; frequency of incoming light
;compute raw average power spectra
;guess areas where there is and isnt signal
      inFreq = WHERE((lightFreq gt 1.0e14) AND (lightFreq LT 1.8e14))
      outfreq = WHERE((lightFreq gt 1.8e14) AND (lightFreq LT 2.6e14))
      dFreq = lightFreq(1)-lightFreq(0)
;
;consider clipping in the time domain
      timeFilter=replicate(1,nFrame)
      if (keyword_set(apo)) then timefilter= $
	 exp(-.5*(2.*apo*(findgen(nFrame)-.5*nFrame)/nFrame)^2> (-10))
      ps = fltarr(nFrame)
      nScan = dataSize(2)
      tSpec = vinciApoPower(onData, 1., rawave)
      rawSpec = rawave(*,0:3)
      rawSpec(*,2:3) = rawave(*,2:3)
;estimate hifreq noise
      hiRawNoise = fltarr(4)
      for ichan = 0, 3 do hiRawNoise(ichan) = mean(rawSpec(outFreq))
;choose bottom of plot
      ymin = FLOOR(MIN(alog10(rawSpec)))-.5
      ymax = CEIL(MAX(alog10(rawSpec(10:*,*)))) > (ymin+3)
;remove photometric total power fluctuations from INT channels
      intData = vinciUnPhoto(onData, kappa)
      if(keyword_SET(datadisp)) then begin
	 erase
	 loadct,0
	 tvscl,rebin(intData(*,*,0),nFrame,2*nScan)
	 tvscl,rebin(intData(*,*,1),nFrame,2*nScan),nframe,0
         plots,findgen(nframe),2*nScan*timefilter,psym=3,/dev
         plots,nframe+findgen(nframe),2*nScan*timefilter,psym=3,/dev
         read,ll
      endif
;compute powerspectra of these without background subtraction
	 intSpec = vinciApoPower(intData, 1., intSpecAve)
      if(keyword_set(powerDisp)) then begin
	 erase
         tvscl,rebin(sqrt(intSpec(*,*,0)>0),nFrame,2*nScan)
         tvscl,rebin(intSpec(*,*,1),nFrame,2*nScan),nFrame,0
	 plots,replicate(2*min(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 plots,replicate(2*max(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 plots,replicate(nFrame+2*min(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 plots,replicate(nFrame+2*max(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 read,ll
      endif 
      plot,powerFreq,alog10(rawSpec(*,0)),yr = [ymin, ymax], xTitle='Hertz'
      for ichan = 1,3 do oplot, powerFreq,alog10(rawSpec(*,ichan)),line=iChan+1
;estimate hifreq noise
      hiIntNoise = fltarr(2)
      for ichan = 0,1 do hiIntNoise(ichan) = mean(IntSpec(outFreq))
      xyouts,powerFreq(nFrame/4),ymax-.4,obName,/data
      xyouts,powerFreq(nFrame/4),ymax-.5,onDate,/data
      xyouts,powerFreq(nFrame/4),ymax-.6,strtrim(framerate,2)+'  '+$
	 strtrim(winx,2),/data
;try it with removal of background
      backSpecback = vinciApoPower(intData, timeFilter, backSpecAve)
      backSpecAve(*,0)=backspecAve(*,0)/(kappa(0)*kappa(2))
      backSpecAve(*,1)=backspecAve(*,1)/(kappa(1)*kappa(3))
      tek_color
      oplot,powerFreq,alog10(backSpecAve(*,0)),thick=2,color=2
      oplot,powerFreq,alog10(backSpecAve(*,1)),thick=2,color=4
;sort of integrate
      noisePower = fltarr(2)
      powerOn = total(backSpecAve(inFreq,*),1)
      powerOff = total(backSpecAve(outFreq,*),1)
      m = moment(backSpecAve(outFreq,0))
      noisePower(0) = sqrt(m[1] * N_ELEMENTS(inFreq) *2)
      m = moment(backSpecAve(outFreq,1))
      noisePower(1) = sqrt(m[1] * N_ELEMENTS(inFreq) *2)
      dPower = powerOn - powerOff*(N_ELEMENTS(inFreq)/N_ELEMENTS(outFreq))
      cLab = ['I_1 power', 'I_2 power']
      for iS = 0,1 do printf,unit,cLab[iS],powerOn(is), poweroff(is), $
	 dpower(is), noisepower(is),format='(a, 4e10.2)'
      meanAB = vinciAbStat (onData)
      bandWidth = .5*dFreq * N_ELEMENTS(inFreq)
      norm = 2* (nFrame)^2*(dFreq/bandWidth)*meanAB[0]
      mu = dPower/norm
      emu= mu * noisepower/dpower
      mlab = ['mu12 emu12 ','mu22 emu22']
      for iS = 0,1 do printf,unit, mlab[iS], mu(iS), emu(iS), $
	 format='(a, 2f8.4)'
   endfor         ; looping over observations
RETURN
END

FUNCTION vinciOldPipe, fileList, DiskFile=DiskFile, $
   output=output, short=short,$
   datadisp=datadisp, powerdisp=powerdisp, apo=apo
;***************************************************
;#function#
;vinciOldPipe
;#call#
;vinciOldPipe, fileListObject, diskFile=diskFile, dir=dir, output=output,
; apo=apo
;#description#
;runs a standard set of reductions on
;a group of Vinci raw data files
;#inputs#
;\anArg{fileList}{fitsFileListObject}{object containing a set of}
;\anArg{-}{-}{files to be processed; these must be in sets of 4-Off:A:B:ON}
;\anArg{diskFile}{string}{if set make text and postscript files, use this}
;\anArg{-}{-}{parameter as root of file name}
;\anArg{output}{bool}{create output structure with assorted data}
;#return#
;\anArg{-}{-}{an assorted structure with kappas etc}
;#end_function#
;***************************************************
   ll = ''
   !x.style=1
   !y.style=1
   doDisk =KEYWORD_SET(DiskFile)
;assume files are in sets of 4
   nFiles = fileList->nFiles()
   nObs = nFiles/4
   doOut = (KEYWORD_SET(output))
;if doOut is set, create a set of output structures
   if (doOut) then begin
      outStruct = {target:'', $
      obsDate:'',$
      frameRate:0., $
      insMode:'',$
      nx:0b,$
      fSpeed:0.,$
      nFrames:0l, $
      kappa:fltarr(4), $
      meanoff:fltarr(4), $
      meanon:fltarr(4), $
      powerFreq:PTR_NEW(), $
      lightFreq:PTR_NEW(), $
      rawspec:PTR_NEW(), $
      ihspec:PTR_NEW(), $
      apBandwidth:0.,  $
      mu12:fltarr(2), $
      mu22:fltarr(2), $
      emu12:0., $
      emu22:0. $
      }
      outData = replicate(outStruct, nObs)
   endif else outData = 0               ; doOut
   if (doDisk) then begin
      set_plot,'ps'
      device,file=STRTRIM(diskFile,2)+'.ps',xs=15,ys=8
      openw,unit,STRTRIM(diskFile,2)+'.txt',/get_lun
   endif else unit = -1
   fileList->start
   fileTypes = ['Off','A ','B ','ON']
   keyList = ['obs name','date-obs','ql framerate','ins mode', $
	 'dpr catg','win1 nx', 'ql fringespd']
   pForm = "(a20,5f10.3)"
;which columns to insist on consistency
;   keyEqual = [0, 2, 3, 5]
    keyEqual = [0, 3, 5]  ; forget frame rate because vinci writes down wrong
    ;                           rate for off batchs
   nEqual = N_ELEMENTS(keyEqual)
   keyValueTable = fileList->getHeaderKeys(keyList,iErr=iErr)
   if (iErr NE 0) then begin
      print, 'couldnt retrieve headers from all the specified files'
      ewsPrintErrMsg
      RETURN,0
   endif
;loop over multiple observations
   fileList->start
   for iObs = 0, nObs-1 do begin
      bFile = 4*iObs
;check for consistency
      consistent = 1
      for iEq = 0, nEqual-1 do begin
	 consistent = consistent AND $
	 TOTAL(keyValueTable[bFile+[1,2,3]].(keyEqual[iEq]) NE $
	       keyValueTable[bFile].(keyEqual[iEq])) EQ 0
         if(NOT consistent) then begin
	    print,'keyword ',keyList[keyEqual[iEq]],' inconsistent'
            print,'values are ',$
	    keyValueTable[bFile+indgen(4)].(keyEqual[iEq])
	    RETURN,0
         endif
      endfor         ; we seem to be consistent
      target    = keyValueTable[bFile+3].(0)
      onDate    = keyValueTable[bFile+3].(1)
      frameRate = keyValueTable[bFile+3].(2)
      insMode   = keyValueTable[bFile+3].(3)
      winx      = keyValueTable[bFile+3].(5)
      fSpeed    = keyValueTable[bFile+3].(6)
;print id info
      printf,unit,'Target:     ',target
      printf,unit,'obsDate:    ', onDate
      printf,unit,frameRate, format="('Frame Rate:', f8.2)"
      printf,unit,'Ins mode:   ', insMode
      printf,unit, winx, format="('Win1 Nx:  ', i3)"
      printf,unit, fSpeed, format="('FringeSpeed: ', f8.6)"
;put into structure
      if(doOut) then begin
	 outData[iObs].target=target
	 outData[iObs].obsDate=onDate
         outData[iObs].frameRate=frameRate
         outData[iObs].insMode=insMode
         outData[iObs].nx=winx
         outData[iObs].fSpeed=fSpeed
      endif
      a = systime(1)
      print,'Reading in data'
;get data 
      offData = vinciRawScans(fileList->getFileNames(), iErr=iErr)
      AData = vinciRawScans(fileList->getFileNames(), iErr=iErr)
      BData = vinciRawScans(fileList->getFileNames(), iErr=iErr)
      ABData = vinciRawScans(fileList->getFileNames(), iErr=iErr)
;make data even sized to speed up ffts
      dataSize = size(ABData)
      nX = dataSize(1)
      printf,unit, nX, format="('nFrames   ', i6)"
      if (nX NE 2*(nX/2)) then begin
	nX = nX -2
	offData=offData(0:nX,*, *)
	AData=AData(0:nX,* , *)
	BData=BData(0:nX,*, *)
	ABData=ABData(0:nX,*, *)
      endif
      print,'That took ',systime(1)-a,' seconds',format="(a,f10.2,a)"
;calculate mean and variance of off channels
      meanOffs = fltarr(5)
      varOffs  = fltarr(5)
      for iChan=0,4 do begin
         m=moment(offData(*,*,iChan))
         meanOffs(iChan) = m[0]
         varOffs(iChan)  = m[1]
      endfor
      printf,unit, 'mean of Off channels    ', meanOffs, format=pForm
      printf,unit, 'variance of Off channels', varOffs, format=pForm
;subtract means offsets from all other channels
      for iChan = 0, 4 do begin
	 AData(*,*,iChan)  = AData(*,*,iChan)-meanOffs(iChan)
	 BData(*,*,iChan)  = BData(*,*,iChan)-meanOffs(iChan)
	 ABData(*,*,iChan) = ABData(*,*,iChan)-meanOffs(iChan)
      endfor
;calculate kappa coefficients
      Atot = total(total(AData,1),1)
      Btot = total(total(BData,1),1)
      dDatadA = Atot/Atot(2)
      dDatadB = Btot/Btot(3)
;
;     printf,unit,'d_signal/d_A', dDatadA, format=pForm
;     printf,unit,'d_signal/d_B', dDatadB, format=pForm
      kappa = [dDataDA[0:1],dDatadB[0:1]]
      printf,unit, 'kappa', kappa, format=pForm
;calculate mean photometric signals in on data
      allMean = fltarr(4, 3)
      allRMS  = fltarr(4, 3)
      for ichan = 0, 3 do begin
	 m = moment(AData(*,*,iChan))
	 allmean(iChan,0)=m(0)
	 allrms(iChan,0) = sqrt(m(1))
	 m = moment(BData(*,*,iChan))
	 allmean(iChan,1)=m(0)
	 allrms(iChan,1) = sqrt(m(1))
	 m = moment(ABData(*,*,iChan))
	 allmean(iChan,2)=m(0)
	 allrms(iChan,2) = sqrt(m(1))
     endfor
     for iChan = 0, 2 do begin
	printf,unit, fileTypes(iChan+1),'Means:',allmean(*, iChan),format='(2a,4f8.2)'
	printf,unit, fileTypes(iChan+1),'RMS:  ',allrms(*, iChan),format='(2a,4f8.2)'
     endfor
      meanOn = 5*total(total(ABdata,1),1)/N_ELEMENTS(ABData)
      dataSize = size(ABData)
      nFrame = dataSize(1)
      if(doOut) then begin
	 outData[iObs].kappa=kappa
	 outData[iObs].meanoff=meanoffs[0:3]
	 outData[iObs].nFrames=nFrame
	 outData[iObs].meanOn = meanOn[0:3]
      endif
;compute raw average power spectra
      powerFreq = frameRate * findgen(nFrame/2)/nFrame
      lightFreq = powerFreq * 2.9979e8/fspeed ; frequency of incoming light
;guess areas where there is and isnt signal
      inFreq = WHERE((lightFreq gt 1.0e14) AND (lightFreq LT 1.8e14))
      outfreq = WHERE((lightFreq gt 1.8e14) AND (lightFreq LT 2.6e14))
      dFreq = lightFreq(1)-lightFreq(0)
;
      rawspec = fltarr(nFrame/2,4)
;consider clipping in the time domain
      timeFilter=replicate(1,nFrame)
      if(keyword_set(short)) then begin
         timefilter(0:nFrame/4)=0
         timefilter(3*nFrame/4:*)=0
      endif
      if (keyword_set(apo)) then timefilter= $
	 exp(-.5*(2.*apo*(findgen(nFrame)-.5*nFrame)/nFrame)^2> (-10))
      ps = fltarr(nFrame)
      nScan = dataSize(2)
      rawSpec(*,0:1) = vinciPower(ABData(*,*,0:1), timeFilter,/ave)
      rawSpec(*,2:3) = vinciPower(ABDATA(*,*,2:3),/ave)
;estimate hifreq noise
      hiRawNoise = fltarr(4)
      for ichan = 0, 3 do hiRawNoise(ichan) = mean(rawSpec(outFreq))
      if(doOut) then begin
	 outData[iObs].rawSpec = PTR_NEW(rawspec)
	 outData[iObs].powerFreq = PTR_NEW(powerFreq)
	 outData[iObs].lightFreq = PTR_NEW(lightFreq)
      endif
;choose bottom of plot
      ymin = FLOOR(MIN(alog10(rawSpec)))-.5
      ymax = CEIL(MAX(alog10(rawSpec(10:*,*)))) > (ymin+3)
;remove photometric total power fluctuations from INT channels
      intData = fltarr(nFrame, nScan, 2)
      intData (*,*,0) = ABData(*,*,0)-dDatadA(0)*ABData(*,*,2)  $
	 - dDatadB(0)*ABData(*,*,3)
      intData (*,*,1) = ABData(*,*,1)-dDatadA(1)*ABData(*,*,2)  $
	 - dDatadB(1)*ABData(*,*,3)
      if(keyword_SET(datadisp)) then begin
	 erase
	 tvscl,rebin(intData(*,*,0),nFrame,2*nScan)
	 tvscl,rebin(intData(*,*,1),nFrame,2*nScan),nframe,0
; plots,replicate(min(where(timeFilter GT 0)),2*nScan),$
;    lindgen(2*nScan),psym=3,/dev
; plots,replicate(max(where(timeFilter GT 0)),2*nScan),$
;    lindgen(2*nScan),psym=3,/dev
; plots,replicate(nFrame+min(where(timeFilter GT 0)),2*nScan),$
;    lindgen(2*nScan),psym=3,/dev
; plots,replicate(nFrame+max(where(timeFilter GT 0)),2*nScan),$
;    lindgen(2*nScan),psym=3,/dev
         plots,findgen(nframe),2*nScan*timefilter,psym=3,/dev
         plots,nframe+findgen(nframe),2*nScan*timefilter,psym=3,/dev
;        plot,nframe+findgen(nframe),timefilter,xr=[0,2*nframe],xstyle=5,ystyle=5,$
;    chars=.01,/noerase
         read,ll
      endif
;compute powerspectra of these
      if(keyword_set(powerDisp)) then begin
	 erase
         intSpec = vinciPower(intData, timeFilter)
;        tvscl,rebin(intSpec(*,*,0),nFrame,2*nScan)
;        tvscl,rebin(sqrt(intSpec(*,*,1)>0),nFrame,2*nScan),nFrame,0
         tvscl,rebin(sqrt(intSpec(*,*,0)>0),nFrame,2*nScan)
         tvscl,rebin(intSpec(*,*,1),nFrame,2*nScan),nFrame,0
	 plots,replicate(2*min(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 plots,replicate(2*max(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 plots,replicate(nFrame+2*min(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 plots,replicate(nFrame+2*max(inFreq),2*nScan),$
	    lindgen(2*nScan),psym=3,/dev
	 intSpec = total(intSpec,2)/nScan
	 read,ll
      endif else intSpec = vinciPower(intData, timeFilter,/ave)
      plot,powerFreq,alog10(rawSpec(*,0)),yr = [ymin, ymax], xTitle='Hertz'
      for ichan = 1,3 do oplot, powerFreq,alog10(rawSpec(*,ichan)),line=iChan+1
;estimate hifreq noise
      hiIntNoise = fltarr(2)
      for ichan = 0,1 do hiIntNoise(ichan) = mean(IntSpec(outFreq))
      oplot,powerFreq,alog10(intSpec(*,0)),thick=2
      oplot, powerFreq,alog10(intSpec(*,1)),thick=2
;combine 1 and 2 to minimize noise
      wtNoise = 1./hiIntNoise
      dData = (intData(*,*,0)*wtNoise(0)-intData(*,*,1)*wtNoise(0)) /$
	 total(wtNoise)
      dSpec = vinciPower(dData,timeFilter,/ave)
;try it again without subtracting photometric fluctuations
;this has better hf noise characteristics but worse 1/f noise
      wtNoise = 1./hiRawNoise(0:1)
      eData = (wtNoise(0)*ABData(*,*,0) - wtNoise(1)*ABData(*,*,1))/$
	 total(wtNoise)
      eSpec = vinciPower(eData, timeFilter,/ave)
;do it one more time.  This time do center and
;edges of window separately and subtract results.
;     timefilter(*) = 1
;     timefilter(0:nFrame/4)=0
;     timefilter(3*nFrame/4:*)=0
      psOn = vinciPower(eData, timeFilter, /ave)
      psOff = vinciPower(eData, 1-timeFilter, /ave)
      fSpec = psOn - psOff
      if (doDisk) then begin
         oplot,powerFreq,alog10(dSpec(*,0)),thick=3,line=2
         oplot,powerFreq,alog10(eSpec(*,0)),thick=3
      endif else begin
         oplot,powerFreq,alog10(dSpec(*,0)),thick=3,color=230,line=2
         oplot,powerFreq,alog10(eSpec(*,0)),thick=3,color=230
         oplot,powerFreq,alog10(fSpec(*,0)),thick=3,color=230,line=3
      endelse
      xyouts,powerFreq(nFrame/4),ymax-.4,target,/data
      xyouts,powerFreq(nFrame/4),ymax-.5,onDate,/data
      xyouts,powerFreq(nFrame/4),ymax-.6,strtrim(framerate,2)+'  '+$
	 strtrim(winx,2),/data
;consider espec and dspec and fspec
      bspec = [[dspec],[espec],[fspec]]
      if(doOut) then outData[iObs].rawSpec = PTR_NEW(rawspec)
      plabel = ['dspec','espec','fspec']
      PowerOn = fltarr(3)
      PowerOff = fltarr(3)
      dPower = fltarr(3)
      noisePower=fltarr(3)
      for ispec=0,2 do begin
	 powerOn(iSpec)=  dFreq * total(bspec(inFreq,iSpec))
	 powerOff(iSpec)= dFreq * total(bspec(outFreq,iSpec))
	 dPower(iSpec) = powerOn(iSpec)-powerOff(iSpec)
	 m=moment(bspec(outFreq,iSpec))
	 noisePower(iSpec) = dFreq * sqrt(m[1] * N_ELEMENTS(inFreq) *2)
	 printf,unit,plabel(iSpec), powerOn(ispec),poweroff(ispec), $
	    dpower(ispec), noisepower(ispec)
      endfor
;make an attempt to normalize
;yet one more attempt to evaluate integral
      Ihspec = vincipower(intData, timefilter,/av) - vincipower(intData,1-timefilter,/av)
      for i=0,1 do begin 
; powerOn(i)  = dFreq * total(ihspec(inFreq,i))
; powerOff(i) = dFreq * total(ihspec(outFreq,i))
         powerOn(i)  = total(ihspec(inFreq,i))
         powerOff(i) = total(ihspec(outFreq,i))
	 dPower(i) = powerOn(i)-powerOff(i)
	 m=moment(ihspec(outFreq,i))
	 noisePower(i) = sqrt(m[1] * N_ELEMENTS(inFreq) *2)
      endfor
      meanA = mean(ABData(*,*,2)) 
      meanB = mean(ABData(*,*,3)) 
      meanAB = mean(ABData(*,*,2) * ABData(*,*,3))
      printf,unit,'AB mean ratio ', meanAB/(meanA*meanB)
      ihspec(*,0) = ihspec(*,0)/(kappa(0)*kappa(2))
      ihspec(*,1) = ihspec(*,1)/(kappa(1)*kappa(3))
      bandWidth = .5*dFreq * N_ELEMENTS(inFreq)
      nband = 2*dFreq*total(ihspec(inFreq,*))/max(ihspec(infreq,*))
      printf,unit, 'apostiori bandwidth ',bandwidth, nband
      norm = 2*(nFrame)^2*(dFreq/bandWidth)*meanA*meanB
      mu12 = dPower(0)/(norm*kappa(0)*kappa(2))
      mu22 = dPower(1)/(norm*kappa(1)*kappa(3))
      xyouts,powerFreq(nFrame/4),ymax-.7,string(mu12,format='(f8.3)')
      xyouts,powerFreq(nFrame/4),ymax-.8,string(mu22,format='(f8.3)')
      em12 = mu12 * noisePower(0)/dPower(0)
      em22 = mu22 * noisePower(1)/dPower(1)
      printf,unit,'mu12 ',mu12, mu12*bandwidth/nband, em12
      printf,unit,'mu22 ',mu22, mu22*bandwidth/nband, em22
      if (doOut ) then begin
         outData[iObs].ihspec = PTR_NEW(ihspec)
         outData[iObs].apBandwidth = nBand
	 outData[iObs].mu12 = [mu12, mu12*bandwidth/nband]
	 outData[iObs].mu22 = [mu22, mu22*bandwidth/nband]
	 outData[iObs].emu12 = em12
	 outData[iObs].emu22 = em22
      endif
      if (NOT doDisk AND (not doOut)) then read,ll
;close up this plot and printout
   endfor         ; looping over observations
   if (doDisk) then begin
     device,/close
     set_plot,'x'
     free_lun, unit
   endif
RETURN, outData
END
