pro zodipic, fnu, pixsize, lambda, inclination=inclination, radin=radin, radout=radout, starname=starname, albedo=albedo, isotropic=isotropic, nodisplay=nodisplay,$
   distance=dist, addstar=addstar, zodis=zodis, ring=ring, blob=blob, earthlong=earthlong, bands=bands, nofan=nofan, noiterate=noiterate, radring=radring,$
   pixnum=pixnum, positionangle=positionangle, iras=iras, rstar=rstar, tstar=tstar, lstar=lstar, offsetx=offsetx, offsety=offsety, offsetz=offsetz, alpha=alpha, delta=delta, dustsize=dustsize

; Makes an image of what the Earth's zodiacal cloud
; would look like if it were around another star
; based on a model of the solar zodiacal cloud
; as seen by COBE

; example:  make an image of the Solar system dust viewed 
; edge-on by the HST NICMOS coronagraph from 10 pc, including
; the earth ring and the dust bands, and put that image in fnu
;  zodipic, fnu, 75, 1, /ring, /bands

; example:  to make an image of what Vega would look like if it had a
; solar-type zodiacal cloud, extended out to 10 AU, as seen face-
; on by MIRLIN (a mid-IR camera) at Keck, at 12 microns, and
; include the stellar flux in the 
; center pixel
;  zodipic, fnu, 137, 5, starnum=8, theta=90, rout=10, /addstar

; here's a model for Epsilon Eridani as in Greaves et al. 1998, ApJ, 506,L133 
;zodipic, fnu, 400, 2.2, starname='Epsilon_Eridani', radin=35, radout=70, inc=25.0, zodis=67000.0

; Also try:
;zodipic, fnu, 75, 1.1, star='HR_4796', radout=100, inc=73.1, pos=26.8, radring=70.4, /nofan, /noiter, pixnum=64, ring=10000, offsety=3

; * fnu is the output image, 
;   each pixel contains a flux, in Jy 
;   fnu is an array with dimensions pixnum x pixnum
;   computation time goes as pixnum^3

; * pixsize is the size of a pixel, in milliarcsec
;   You might want to oversample to get a more accurate picture of the disk.
;   If your output matrix is smaller than about 100x100, the computed
;   excess will be wrong by maybe a huge amount.

; * lambda is the wavelength, in microns

;*****************Optional Parameters**********************

; * pick a star from one of the following list using starname
;   or edit the code to add new ones
;   the Sun at 10 pc  THIS IS THE DEFAULT
;   starname='Alpha_Centauri_B' (the nearest KV star)
;   starname='Epsilon_Eridani' (the nearest KV Keck can see) 
;   starname='Alpha_Centauri_A' (the nearest GV star other than the sun)
;   starname='Tau_Ceti'    (the nearest GV Keck can see, other than the sun)
;   starname='Procyon'     (the nearest FV star...Keck can see it)
;   starname='Sirius'      (the nearest AV star...Keck can see it)
;   starname='Altair'
;   starname='Vega'
;   starname='Fomalhaut'
;   starname='HR_4796'

; * radin is the inner radius of the disk, in AU
;   The default is 0.0

; * radout is the outer radius of the disk, in AU
;   Our asteroid belt stops at 3.28 AU, the 2:3 resonance with Jupiter
;   That's the default setting.
;   The DIRBE model isn't a good model of our zodiacal cloud
;   past there, but if another star has a more distant asteroid belt
;   it might be a good model for that star's disk

; *  inclination is the inclination angle in degrees
;    (0 is face-on).

; * distance is the distance to the observer, in pc
;   setting starnum overrides this parameter

; * Set addstar to add the stellar flux to the center pixel. 
;   The flux is calculated assuming the star is a blackbody.

; * Set zodis to the number of zodis you want, and I'll multiply the
;   dust emission by that number.
;   Note that for about 1000 zodis, dust destruction by mutual collisions
;   becomes important, and this model does not include this physics

; * pixnum is the desired size of the image
;   the default is 144x144 (pixnum=144)
;   pixnum must be set to a multiple of 16
;   If you choose pixnum < 112, the model will be calaulated using
;   a internal grid that is at least 112x112 anyway

; * alpha is the radial power-law for the dust <n sigma>

; * delta is the temperature power-law for the dust <n sigma>

; The following parameters, when set, add additional detail to the
; model, and increase the running time significantly.

; * dustsize is the effective size of the dust grains, in microns
;   Setting this parameter turns on a subroutine which 
;   calculates the temperature of the dust grains
;   by iteratively solving the thermal equilibrium equation
;   assuming p=q=1 (see Backman & Paresce 1993, in Protostars and Planets III).
;   It does not affect any other properties of the cloud 
;   (like optical depth).

; * set positionangle to rotate the image positionangle degrees E of N

; * Set ring=1 to add the ring and wake of dust associated with the Earth
;   according to the DIRBE model.
;   Ring is the density of the ring compared to the real ring

; * earthlong is the angle that determines the location of the
;   Earth, or rather, the Earth's wake, in degrees
;   measured in the plane of the disk from the ascending node
;   The wake trails clockwise when the disk is face-on.

; * Set bands=1 to add the bands associated with major asteroid families.

; * Set nofan to remove the fan component of the zodiacal cloud
;   Useful for visualizing the bands and ring

; * offsetx, offsety, offsetz  shift the dust from the center of the frame
;   x and y are in the plane of the disk, y points to the ascending node

; * radring is the radius of the ring;  the default (from the DIRBE model)
;   is 1.03 AU.  The width and height of the ring and blob all
;   scale with radring.

; * set noiterate if you want to force zodipic not to iterate

; * set nodisplay if you don't want zodipic to display your image

; Written 2/99 by

; Marc J. Kuchner
; Harvard-Smithsonian Center for Astrophysics
; Mail Stop #20
; 60 Garden St. 
; Cambridge, MA 02138
; (617)496-4773
; mkuchner@cfa.harvard.edu

; revised 2/01
; temperaturecalc added 3/02

if n_params() lt 3 then begin

print, ' '
print, 'Syntax:'
print, ' '
print,'zodipic, fnu, pixsize, lambda, [inclination=, radin=, radout=, starname=, distance=, addstar=, addring=, addbands=, nofan=]'
print, ' '
print, 'fnu is the name of the matrix containing the output image
print, ' '
print, 'pixsize is the size of a pixel, in milliarcseconds.'
print, ' '
print, 'lambda is the wavelength, in microns'
print, ' '

return
end


;*************Stellar Parameters**********************


; lstar luminosity of the star in solar luminosities
; tstar is effective temperature of star, kelvin
; rstar is the radius of the star in solar radii
; dist is in pc

if not keyword_set(starname) then starname='Sun'
stellarparam, starname, urstar, ulstar, utstar, udist

if keyword_set(dist) then udist=dist
if keyword_set(rstar) then urstar=rstar
if keyword_set(tstar) then utstar=tstar
if keyword_set(lstar) then ulstar=lstar

; The stellar parameters we're actually going to USE are
; ulstar, udist, urstar and utstar
print, 'Stellar Parameters'
print, 'Luminosity (Solar Lumin.):', ulstar, '  Distance (pc):', udist
print, 'Radius (Solar radii):', urstar, '  Temperature (K):', utstar
print, ' '

print, 'ZODIPIC  by Marc Kuchner---Last revised 2/01'

;***************Fundamantal Constants**********************
pi=3.141592
k=1.38066d-16
c=2.9979d10
h=6.62608d-27
sigma=5.6705d-5
l10=2.30259

; tsolar is the effective temperature of the sun 
tsolar=5770.0

; radsolar is the radius of the sum in cm
radsolar=6.96d10

; lsolar is the solar luminosity in ergs/sec
lsolar=3.86d33

;*********************More Definitions********************
; rstarcm is the radius of the star in cm
rstarcm=radsolar*urstar

; rstarau is the radius of the star in au
rstarau=rstarcm/1.49597d13

; wavelength in microns, of all the various DIRBE bands
; 0.5, 1.25, 2.2, 3.5, 4.9, 12.0, 25.0, 60.0, 100.0, 140.0, 240.0


if not keyword_set(radout) then radout=3.28
if not keyword_set(ring) then ring=0
if not keyword_set(blob) then blob=0
if not keyword_set(bands) then bands=0
if not keyword_set(radin) then radin=0.0
if not keyword_set(offsetx) then offsetx=0
if not keyword_set(offsety) then offsety=0
if not keyword_set(offsetz) then offsetz=0
if not keyword_set(positionangle) then positionangle=0
if not keyword_set(earthlong) then earthlong=0


useralpha=0
if keyword_set(alpha) then begin
  useralpha=alpha
  print, 'Alpha=', useralpha
endif
 
userdelta=0
userdustsize=0
if keyword_set(dustsize) then begin
  userdustsize=dustsize
  print, 'Effective size of dust grains=', userdustsize
endif else begin
 if keyword_set(delta) then begin
   userdelta=delta
   print, 'Delta=', userdelta
 endif 
endelse


; assume no oversampling
oversample=1.0  

; upixnum is the number of pixels we're actually going to use
if keyword_set(pixnum) then begin
  if pixnum/16.0 ne fix(pixnum/16.0) then begin
    print, ' '
    print, 'Please set pixnum to a multiple of 16.'
    print, ' '
  return
  end
; do not allow user to use fewer than 112 pixels in the model
; even if the output is to be smaller than that
  if pixnum lt 112.0 then oversample=fix(112.0/pixnum)+1
  upixnum=pixnum*oversample
endif else begin
  ; default grid size
  pixnum=144
  upixnum=144
endelse

print, 'The internal matrix will be', fix(upixnum), ' by', fix(upixnum)
print, 'The output matrix will be', pixnum, ' by', pixnum


upixnum=float(upixnum)
num=upixnum/2.0
gum=num-1
pixsize=float(pixsize)
radin=float(radin)
radout=float(radout)


stepau=pixsize*udist/(1000.0*oversample)

if 1.415*num*stepau lt radin then begin
  print, ' '
  print, '*************************************************'
  print,"    You are looking into the disk's central hole."
  print, '*************************************************'
  print, ' '
endif

if num*stepau gt 3.0*radout then begin
  print, ' '
  print, '*************************************************'
  print,'    The disk is much smaller than the frame.'
  print, '*************************************************'
  print, ' '
endif

;l0 is the wavelength, in cm
lambda=float(lambda)
l0= lambda*1e-4
print, 'Wavelength in microns: ', lambda

; add scattered light if the wavelength is less than this value (in microns)
scatterwavelength=4.2 ; microns

;*****************************************************
; Find the radius where the dust sublimates
delta=0.467
T0=286.0
if keyword_set(iras) then begin
  T0=266.20
  delta=0.359
endif
tsublime=1500.0
if keyword_set(userdustsize) then begin
 print, 'Finding the radius where the dust sublimates.'
 ; start at a little beyond two times the sublimation radius of a blackbody
 logsubrau=alog10(2.1*(ulstar^0.5)*(T0/tsublime)^2.0)
 rok=0

 ; work our way inwards till the dust temperature (tgrain)
 ; exceeds the sublimation temperature 
 while rok eq 0 do begin  ; loop through radii
  temperaturecalc, ulstar, utstar, userdustsize, 10.0^logsubrau+fltarr(1), tgrain 

  tgrain=tgrain(0)  ; since rau and t in temperaturecalc are matrices 
  if tgrain gt tsublime then begin rok=1
  endif else begin
   ;print, 10.0^(logsubrau), tgrain
   logsubrau=logsubrau-0.008  ; no sublimation?  decrease the trial radius by 1.859%
  endelse

endwhile
rsublime=10.0^(logsubrau)

endif else begin
 rsublime=(ulstar^0.5)*(T0/tsublime)^(1.0/delta)
endelse
;*****************************************************


if rsublime gt radin then begin
  print, 'Dust sublimation temperature:', tsublime, ' K'
  print, 'Disk inner radius set to', rsublime, ' AU, where the dust sublimates.' 
  radin=rsublime
endif else begin
  print, 'Inner radius, in AU:', radin
endelse
print, 'Outer radius, in AU:', radout


if not keyword_set(zodis) then zodis=1.0
zodis=float(zodis)
print, 'Synthesizing an image of a Solar type zodiacal cloud x', zodis
; Worry about whether the dust will be destroyed by mutual collisons
alpha = 1.34
n0=1.13d-7
ep=1.5-alpha
dustsizecm=5d-4  ; 5 microns in diameter
dustdensity=3.5 ; grams per cc
dustpar=3.55d-8/(dustsizecm*dustdensity)
;  Assume dust traverses the scale height of the disk twice per revolution,
; a is the heliocentric distance, and tau is the optical depth traversed
; by a dust particle since it was created at radout.
; dtau/da = dtau/dt / da/dt
; dtau/dt = 2tau0/P (P is the period=a^3/2, tau0 is the face-on optical depth) 
; da/dt = -2 dustpar/a  (see Wyatt, S. P., & Whipple, F. L. 1950, ApJ, 111, 134)  ; dtau/da = -tau0 a^(-1/2) / dustpar
; rcoll is the radius where a particle released at radout
; has traversed an optical depth of unity
; The integral of dtau/da from a=radout to a=rcoll = 1
; assume tau0=0.1*n0 * rau^(1-alpha)
; and do the integral and solve for rcoll
freepathpar=dustpar*ep/(0.1*n0*zodis)
if freepathpar le radout^ep then begin
  rcoll=(radout^ep-freepathpar)^(1.0/ep)
  print, ' '
  print, 'Heliocentric distance where a typical 5 micron radius dust particle will be detroyed on its way in by a collision with another grain:', rcoll, ' AU'
  if radin lt rcoll then print, 'You might want to truncate the disk at this inner radius.'
  print, ' '
endif else begin
  rcoll=0.0
endelse


print, 'Step size=', stepau, ' AU'

if keyword_set(inclination) then begin
  print, 'Inclination:', inclination, ' degrees from face-on'
  inclination=float(inclination)
endif else inclination=0.0

if keyword_set(nofan) then begin
  if ring eq 0 and bands eq 0 then begin
    print, 'You selected nofan and turned off the bands and the rings; there will be no dust.'
    zodis=0 ; make sure there's no dust
  endif
endif

;************************************************************

; scatterflag makes sure you run the scattering calculation when you need it
scatterflag=0
if lambda lt scatterwavelength then begin
scatterflag=1
if keyword_set(isotropic) then begin
  pfunc500=fltarr(500)+1.0/(4.0*pi)
endif else begin
print, 'Calculating phase function.'
; calculate the phase function for our zodiacal cloud
; use the information in Hong, S.S. 1985, A&A, 146, 67 and the fact that
; our zodi has alpha near 1.35
hongphasefunction, 1.35, pfunc500
; use some of these lines for the dirbe phase function
; dirbeband=1 ; (1 is 1.25 microns, 2 is 2.2, 3 is 3.5)
; dirbephasefunction, dirbeband, pfunc500
; albedo=0.204

endelse
endif

;*********************start the action*********************
;get Fnu for the dust

; how much to magnify on each iteration:
iterfactor=8.0

; decide how many iterations to do
; make sure there are at least minpix pixels across radin
minpix=14.0
; minpix had better be less than upixnum!
; after one iteration there are radin/stepau steps
; after maxi iterations, there are iterfactor^maxi times as many
; so we want (iterfactor^maxi)*radin/stepau > minpix
;    or   10^(alog10(iterfactor)*maxi) > minpix*stepau/radin
;    alog10(iterfactor)*maxi > alog10(minpix*stepau/radin)
;    maxi > alog10(minpix*stepau/radin)/alog10(iterfactor)
maxi = fix(alog10(minpix*stepau/radin)/alog10(iterfactor))+1.0

; don't iterate if the user asks you not to
if keyword_set(noiterate) then maxi=1

for i=1, maxi do begin

; Start with the smallest radii and move to
; larger scales with each iteration.

; how much we are magnifying by 
magfactor=iterfactor^(maxi-i)
print, ' '
print, 'Iteration ', i, ' out of ', fix(maxi)
print, 'magnification x', fix(magfactor)

; first shrink whatever we had from last iteration 
; to our new larger scale
if i gt 1 then begin
fnuold=rebin(fnu,upixnum/iterfactor,upixnum/iterfactor)  ; i couldn't resist the pun
endif
; then go get a new image 

; if we are on the smallest iteration, don't leave a hole
; in the center
scube=upixnum/(2.0*iterfactor)
if i eq 1 then scube=0

;shortcut allows you to use faster models for special position angles
shortcut=0
if ((positionangle mod 90) eq 0) then shortcut=1

;----------FIRST CASE
; use this program for the whole full-on salami with bands, rings etc.
if (offsetx ne 0) or (offsety ne 0) or (offsetz ne 0) or (ring ne 0) or (bands ne 0) or (scatterflag eq 1 and shortcut eq 0) then begin
  xfullzodimodel, ulstar, utstar, urstar, num, fnu, stepau/magfactor, inclination, positionangle, lambda, radin, radout, pfunc500, albedo=albedo, ring, blob,$
    earthlong, bands, nofan=nofan, offsetx, offsety, offsetz, iras=iras, scatterflag=scatterflag, useralpha=useralpha, userdelta=userdelta, scube=scube, radring=radring, userdustsize=userdustsize
endif else begin

;----------SECOND CASE
if (scatterflag eq 1 and shortcut eq 1) then begin
; use this program to do position angle = 0 at scattering wavelengths
  xscatteringzodimodel, ulstar, utstar, urstar, num, fnu, stepau/magfactor, inclination, 0.0, lambda, radin, radout, pfunc500, albedo=albedo, iras=iras, scatterflag=scatterflag, useralpha=useralpha, userdelta=userdelta, scube=scube, userdustsize=userdustsize
  if positionangle ne 0 then begin
; we took the shortcut, but we might still have some rotating to do
; it's only by a multiple of 90 degrees, though
    fnu=rot(fnu,positionangle) 
  endif
endif else begin

;----------THIRD CASE
if (scatterflag eq 0 and shortcut eq 0) then begin
; use this program to do position angle rotation for a thermal model
  xscatteringzodimodel, ulstar, utstar, urstar, num, fnu, stepau/magfactor, inclination, positionangle, lambda, radin, radout, pfunc500, albedo=albedo, iras=iras, scatterflag=scatterflag, useralpha=useralpha, userdelta=userdelta,scube=scube, userdustsize=userdustsize
endif else begin

;----------FOURTH CASE
; fast no-frills, symmetrical, thermal-emission-only model
xthermalzodimodel, ulstar, utstar, num, fnu, stepau/magfactor, inclination, lambda, radin, radout, iras=iras, useralpha=useralpha, userdelta=userdelta, scube=scube, userdustsize=userdustsize
if positionangle ne 0 then begin
; we took the shortcut, but we might still have some rotating to do
; it's only by a multiple of 90 degrees, though
  fnu=rot(fnu,positionangle) 
endif
endelse
endelse
endelse

; Compute total flux from this iteration, in Jy
fnuit = total(fnu)*1d23*((pixsize/oversample)/(magfactor*1000.0*206265.0))^2.0
print, 'Flux added this iteration:', fnuit 

; add the old image to the new image
if i gt 1 then begin
gscube=upixnum/(2.0*iterfactor)-1.0
fnu(gum-gscube:num+gscube,gum-gscube:num+gscube)=fnu(gum-gscube:num+gscube,gum-gscube:num+gscube)+fnuold
endif


endfor   ; end iteration loop
print, ' '

; Convert disk surface brightness from cgs to Jy/ster
fnu=fnu*1d23

; now convert that to Jy
fnu=fnu*((pixsize/oversample)/(1000.0*206265.0))^2.0

fnu=fnu*zodis

fdisk=total(fnu)
print, 'Total flux (Jy):', fdisk

; if we oversampled, bin back down
if upixnum ne pixnum then begin
  fnu=rebin(fnu, pixnum, pixnum)
  fnu=fnu*fdisk/total(fnu)   ; make sure that the total flux is correct
endif

; redefine num because we are now working again with a 
; pixnum x pixnum grid
  num=pixnum/2.0

;Calculate flux from star from Planck spectrum 
; Bnu (erg s^-1 cm^-2 ster^-1 Hz^-1)
  nu = c/l0
  xb=h*nu/(k*utstar)
  bnu=xb^3.0/(exp(xb)-1.0)
  bnu=bnu*2.0*((k*utstar)^3.0)/((c*h)^2.0)

distcm=udist*3.085678d18  ; distance to star in cm
fstar = 1e23 * pi * bnu * ((rstarcm/distcm)^2.0)

; Add star to the image, if required
if keyword_set(addstar) then begin
  if (2*rstarau) gt stepau then begin
    print, "Warning: Star is bigger than one pixel."
  endif
  
  fnu(num-1,num-1)=fnu(num-1,num-1)+fstar/4.0
  fnu(num,num-1)=fnu(num,num-1)+fstar/4.0
  fnu(num-1,num)=fnu(num-1,num)+fstar/4.0
  fnu(num,num)=fnu(num,num)+fstar/4.0
  print, 'The Stellar flux has been divided evenly among pixels'
  print, fix(num-1), ',', fix(num-1)
  print, fix(num), ',', fix(num-1)
  print, fix(num-1), ',', fix(num)
  print, fix(num), ',', fix(num)
endif

print, 'Stellar flux', fstar, ' Jy'
print, 'Excess (Disk flux/Stellar Flux)', fdisk/fstar

; Display the result 
; if the grid is small, make it look bigger
case 1 of
num le 9: rfactor=16 
(num le 18) and (num gt 9) : rfactor=8 
(num le 37) and (num gt 18) : rfactor=8
(num le 73) and (num gt 37) : rfactor=4 
(num le 145) and (num gt 73) : rfactor=2 
else: rfactor=1
endcase

if pixnum lt 600 and not keyword_set(nodisplay) then begin
pic=rebin(fnu, pixnum*rfactor, pixnum*rfactor, /sample)
sz=size(pic)
pic=rotate(pic,2)  ; to make North up in the display

wtitle='Flux'
window, 0, title=wtitle,xsize=sz(1),ysize=sz(2)
tvscl, pic

wtitle='Log Flux'
window, 2, title=wtitle,xsize=sz(1),ysize=sz(2)
places=where(pic gt 0)
if places(0) ne -1 then begin
  amin=min(pic(places))
  amin=amin > 1e-20
  pic=pic > amin
  tvscl, alog10(pic)
endif

loadct, 3

endif


end
