/***********************************************************************
 * This divides the projected outline of a Roche spheroid into an array
 * of pixels along and perpendicular to the (baseline) direction, Theta
 * (measured East from North). "iB" is the resolution along the
 * baseline.  The resolution perpendicular for each strip is set so that
 * the pixel size is about square.  
 *
 * Once the pixels are defined, the values of Teff, log_g and mu are
 * evaluated at their centers. 
 *
 * This is cannibalized extensively from the original StripBright.c
 * subroutine. 22/12/03
 *
 * If iB < 0, forces a complete dump! 
 *
 * This version, 1.5, divides the outline as described but evaluates
 * the physical variables at the edges of the strips.  Because we
 * expect this to be used for Simpson's rule type integrations we need
 * points at the extreme ends of the baselines.  This requires special
 * treatment of those two end points.  3/8/04
 *
 * Still blowing up finding (r,th,phi) for the limb points.  Will do it
 * differently. After obtaining the outline, as we locate the outline
 * intervals containing each X[i], go back and subdivide 10x finer,
 * saving the output (r,th,phi)'s.  Then interpolate them to get even
 * better values of (r,th,phi).  This allows us to skip the step of
 * going back with (X,Y) values to get the body coordinates for all
 * points on the limb.  Life is then much easier for RThetaPhifromXY
 * which is used only for points at least one full pixel in from the
 * limb.  01/09/04
 *
 * Version 1.6 It turns out that while the above works very well most
 * of the time, there is a small range of rotation rates near critical
 * where certain PA's cause RThetafromXY to blow up occasionally, one
 * pixel in from the limb.  The problem gets worse with finer
 * resolution, so there's no way around fixing it.
 *
 * We'll try using RThetafromXY to generate the (Theta,Phi) values at
 * the center of each strip and then work our way out from the center
 * in either direction using the current values of (Theta,Phi) as the
 * first guesses for the next.  This may take subdividing the steps as
 * you get near the limb, but the need for that is easy to detect and
 * implement.  We'll use simple first order Newton-Raphson, implemented
 * as XY-NR.c.  21/12/04
 *
 * This seems to work for pretty much everything except the limb points
 * (where Edge.c takes the load), and it seems to work faster.  I've
 * kept the original RThetaPhifromXY.c routine to do the cases where
 * XYxNR fails.  Between the two, it should be pretty robust.  26/12/04
 **********************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "roche.h"

int Surface(Roche * R, Disk * D, float Theta) {

  float cTh = cos(Theta), sTh = sin(Theta), incl=R->fIncl, ci = cos(incl),
    si = sin(incl);
  int IfWrite;
  //int IfStop=0;
  int i, j, k, iB;
  int iYfound=0, j1, j2; 
  int OutNum=72;

  /* For gravity/temperature calculation */
  float gpb= GRAV*R->fMass*MSUN/R->fRpB/R->fRpB/RSUN/RSUN;
  float ttc = 0.2962963, tts = 0.087791495, fact, w = R->fwOmega, w2=w*w;
  //float hck = 1.43878e7; // units: nm*K
  float gEff, tEff; // hclkt, eMiss;
  float xp, yp, thp, thout, phiout, rout2, z;
  float dlrdth, mu; // SurfBp, SurfB, 

  float Outthp[72], Outxp[72], Outyp[72], delth = 360./OutNum*(float)RADDEG;
  float OutX[72], OutY[72], OutR[72], OutTh[72], OutPhi[72], OutZ[72];
  float delsth, r1, r2, th1, th2, phi1, phi2;
  float y, dy, ynot, ysum, area; int iNy, iNy2;
  float th, phi, rout, sth, sth2, cth, cth2,sph, cph;
  float xmin=0., xmax=0., yxmin, yxmax; int ixmin=-1, ixmax=-1;
  float y1, y2, wtp, wtm; int iy1, iy2;
  float x1, x2, x, dx, xnot, z1, z2;

  float Rout[ISTRIPMAX][ISTRIPMAX+1], Thout[ISTRIPMAX][ISTRIPMAX+1];
  float Phiout[ISTRIPMAX][ISTRIPMAX+1], Zout[ISTRIPMAX][ISTRIPMAX+1];
  float R2, Th2, Phi2, X, Y1, Y2, Z2;
  float dxp, dyp, dyeff, r, OutTh1, OutTh2;
  int iB1, iDivide=5, IfConverge;

  iB = D->iB;
  iB1 = abs(iB)-1;
  //IfWrite = 1;
  IfWrite = 0;
  //IfWrite = 1;
  if (iB < 0) IfWrite=1;
  /* Get the star's outline.  Every 360/OutNum degrees. */
  if (IfWrite) {
    printf("Surface for Theta=%1.6g", Theta*(float)DEGRAD);
    printf(" cos(Theta)=%1.6g sin(Theta)=%1.6g\n", cTh, sTh);
  } // end IfWrite

  for (i=0; i < OutNum; i++) {
    if (Edge(R, Outthp[i]=delth*i, &th, &phi, &rout) == EXIT_FAILURE)
      return EXIT_FAILURE;
    sth = sin(th);
    cth = cos(th);
    sph = sin(phi);
    cph = cos(phi);
    /* Projected (xp,yp) only with polar axis pointed North */
    Outxp[i] = rout*(sth*cph*ci-cth*si);
    Outyp[i] = rout*sth*sph;
    /* Rotate polar axis through Theta, get (OutX,OutY). */
    OutX[i] = -Outxp[i]*cTh - Outyp[i]*sTh;
    OutY[i] =  Outxp[i]*sTh - Outyp[i]*cTh;
    OutR[i] = rout;
    OutTh[i] = th;
    OutPhi[i] = phi;
    OutZ[i] = rout*(sth*cph*si+cth*ci);
    if (IfWrite) {
      printf("Th'=%3.5g X'=%8.5g", Outthp[i]*(float)DEGRAD, Outxp[i]); 
      printf("Y'=%8.5g Th=%4.5g", Outyp[i], th*(float)DEGRAD);
      printf("Phi=%4.5gR=%1.5g", phi*(float)DEGRAD, rout);
      printf(" X=%8.5gY=%8.5gZ=%8.5g\n", OutX[i],OutY[i],OutZ[i]);
    } // end IfWrite
    /*
    if (IfWrite) {
      CkTh = atan2(OutY[i], OutX[i]);
      if (CkTh < 0.) CkTh = CkTh + (float)TWOPI;
      printf("    CkTh=%1.6g", CkTh*(float)DEGRAD);
      printf(" X=%1.6g Y=%1.6g\n", OutX[i], OutY[i]);
    } // end IfWrite
    */
  } 

  /* Get Min/Max X of Outline */
  for (i = 0; i < OutNum; i++) {
    if (OutX[i] > xmax) {
      xmax = OutX[i];
      ixmax = i;
      Rout[iB1][0] = OutR[i];
      Thout[iB1][0] = OutTh[i];
      Phiout[iB1][0] = OutPhi[i];
      Zout[iB1][0] = OutZ[i];
    }
    if (OutX[i] < xmin) {
      xmin = OutX[i];
      ixmin = i;
      Rout[0][0] = OutR[i];
      Thout[0][0] = OutTh[i];
      Phiout[0][0] = OutPhi[i];
      Zout[0][0] = OutZ[i];
    }
  }
  if (iB < 0) IfWrite=1;
  if (IfWrite) printf("xmax=%g at ixmax=%i xmin=%g at ixmin=%i\n",xmax,ixmax,
		      xmin,ixmin);
  delsth=delth/10;
  /* Subdivide around imax */
  if (IfWrite) printf("\nSubdivide around xmax\n");
  for (i=0; i<21; i++) {
    thp=Outthp[ixmax]-delth+delsth*i;
    if (Edge(R, thp, &th, &phi, &rout) == EXIT_FAILURE)
      return EXIT_FAILURE;
    sth = sin(th);
    cth = cos(th);
    sph = sin(phi);
    cph = cos(phi);
    /* Projected (xp,yp) only with polar axis pointed North */
    xp = rout*(sth*cph*ci-cth*si);
    yp = rout*sth*sph;
    /* Rotate polar axis through Theta, get (OutX,OutY). */
    X = -xp*cTh - yp*sTh;
    if (X > xmax) {
      xmax = X;
      yxmax =  xp*sTh - yp*cTh;
      Rout[iB1][0] = rout;
      Thout[iB1][0] = th;
      Phiout[iB1][0] = phi;
      Zout[iB1][0] = rout*(sth*cph*si+cth*ci);
      if (IfWrite) {
	printf("thp'=%3.5g X'=%8.5g", thp*(float)DEGRAD, xp); 
	printf("Y'=%8.5g Th=%4.5g", yp, th*(float)DEGRAD);
	printf("Phi=%4.5gR=%1.5g", phi*(float)DEGRAD, rout);
	printf(" X=%8.5gY=%8.5gZ=%8.5g\n", X,yxmax,Zout[iB1][0]);
      } // end IfWrite
    }
  }
  D->iNy[iB1] = 1;
  //IfWrite=0;
  if (iB < 0) printf("0: Rout[%i][0]=%g\n",iB1,Rout[iB1][0]);

  /* Subdivide around imin */
  if (IfWrite) printf("\nSubdivide around xmin\n");
  for (i=0; i<21; i++) {
    thp=Outthp[ixmin]-delth+delsth*i;
    if (Edge(R, thp, &th, &phi, &rout) == EXIT_FAILURE)
      return EXIT_FAILURE;
    sth = sin(th);
    cth = cos(th);
    sph = sin(phi);
    cph = cos(phi);
    /* Projected (xp,yp) only with polar axis pointed North */
    xp = rout*(sth*cph*ci-cth*si);
    yp = rout*sth*sph;
    /* Rotate polar axis through Theta, get (OutX,OutY). */
    X = -xp*cTh - yp*sTh;
    if (X < xmin) {
      xmin = X;
      yxmin =  xp*sTh - yp*cTh;
      Rout[0][0] = rout;
      Thout[0][0] = th;
      Phiout[0][0] = phi;
      Zout[0][0] = rout*(sth*cph*si+cth*ci);
      if (IfWrite) {
	printf("thp'=%3.5g X'=%8.5g", thp*(float)DEGRAD, xp); 
	printf("Y'=%8.5g Th=%4.5g", yp, th*(float)DEGRAD);
	printf("Phi=%4.5gR=%1.5g", phi*(float)DEGRAD, rout);
	printf(" X=%8.5gY=%8.5gZ=%8.5g\n", X,yxmin,Zout[0][0]);
      } // end IfWrite
    }
  }
  D->iNy[0] = 1;
  if (iB == 0) IfWrite=0;

  /*
  printf("End Outline: iNy[0]=%i iNy[16]=%i iNy[%i]=%i\n",D->iNy[0],
	 D->iNy[16],iB1,D->iNy[iB1]);
  */

  if (iB < 0) IfWrite=1;
  if (IfWrite) {
    printf("xmin=%1.6g xmax=%1.6g\n", xmin, xmax);
  } // end IfWrite

  dx = (xmax-xmin)/(iB1); // increment
  D->fdx = dx;
  xnot = xmin;
  if (IfWrite) {
    printf("Find strips: dx=%1.6g xnot=%1.6g\n", dx, xnot);
  } // end IfWrite
  //IfWrite=0;

  /* Do, one strip at a time */
  for (i = 0; i < abs(iB); i++) {
    ysum=0.; 
    area=0.;
    iYfound=0;
    x = xnot + dx*i;
    D->afX[i] = x;
    if (IfWrite) printf("BEGIN: i=%2i x=%1.6g\n",i,x);
    /*
    if (IfWrite) {
      printf("i=%2i x=%1.6g\n", i, x);
    }
    */

    /* Not end points, find the two Y intercepts */
    j=0;
    /*
    printf("i=%2i iNy[0]=%i Rout[0][0]=%1.6g ",i,D->iNy[0],Rout[0][0]);
    printf("Thout[0][0]=%3.6g Phiout[0][0]=%3.6g\n",(float)DEGRAD*Thout[0][0],
	   (float)DEGRAD*Phiout[0][0]);
    */
    if (i > 0 && i < iB1) {
      while (j < OutNum && iYfound < 2) {
	IfWrite=0;
	j1=j;
	j2=j+1;
	if (j2 == OutNum) j2=0;
	//if (j != 0) IfWrite = 0;
	if ((OutX[j1]<=x && OutX[j2]>x) || (OutX[j1]>=x && OutX[j2]<x)) {
	  if (IfWrite) {
	    printf("j=%2i j1=%2i j2=%2i\n", j, j1, j2);
	    printf(" OutX[j1]=%1.6g OutX[j2]=%1.6g", OutX[j1], OutX[j2]);
	    printf(" X[j]=%1.6g X[j2]=%1.6g\n", OutX[j], OutX[j2]);
	  } // end IfWrite
	  if (iYfound == 0) {
	    /* Divide X interval by 2^iDivide */
	    x1 = OutX[j1];
	    x2 = OutX[j2];
	    y1 = OutY[j1];
	    y2 = OutY[j2];
	    r1 = OutR[j1];
	    r2 = OutR[j2];
	    th1 = OutTh[j1];
	    th2 = OutTh[j2];
	    phi1 = OutPhi[j1];
	    phi2 = OutPhi[j2];
	    OutTh1 = Outthp[j1];
	    OutTh2 = Outthp[j2];
	    if (j2 < j1) OutTh2 += (float)TWOPI;  
	    for (k=0; k<iDivide; k++) {
	      if (IfWrite) {
		printf("x1=%1.6g x2=%1.6g y1=%1.6g y2=%1.6g ",x1,x2,y1,y2);
		printf("th1=%3.6g th2=%3.6g\n",th1,th2);
	      } // end IfWrite
	      thp = (OutTh1+OutTh2)/2.;
	      if (Edge(R, thp, &th, &phi, &rout) == EXIT_FAILURE)
		return EXIT_FAILURE;
	      sth = sin(th);
	      cth = cos(th);
	      sph = sin(phi);
	      cph = cos(phi);
	      /* Projected (xp,yp) only with polar axis pointed North */
	      xp = rout*(sth*cph*ci-cth*si);
	      yp = rout*sth*sph;
	      z = rout*(sth*cph*si+cth*ci);
	      /* Rotate polar axis through Theta, get (OutX,OutY). */
	      X = -xp*cTh - yp*sTh;
	      if ((x1 < x && x < X) || (X < x && x < x1)) {
		x2 = X;
		y2 =  xp*sTh - yp*cTh;
		z2 = z;
		r2 = rout;
		th2 = th;
		phi2 = phi;
		OutTh2 = thp;
	      }
	      if ((x2 < x && x < X) || (X < x && x < x2)) {
		x1 = X;
		y1 =  xp*sTh - yp*cTh;
		z1 = z;
		r1 = rout;
		th1 = th;
		phi1 = phi;
		OutTh1 = thp;
	      }
	    }
	    /* and then linearly interpolate */
	    wtp=(x-x2)/(x1-x2);
	    wtm=1.-wtp;
	    iy1=j1;
	    Y1 = y1*wtp + y2*wtm;
	    Rout[i][0] = r1*wtp + r2*wtm;
	    Thout[i][0] = th1*wtp + th2*wtm;
	    Phiout[i][0] = phi1*wtp + phi2*wtm;
	    Zout[i][0] = z1*wtp + z2*wtm;
	    D->afY[i][0] = Y1;
	    if (IfWrite) {
	      printf("iYfound=%2i Y1=%1.6g", iYfound, Y1);
	      printf(" j1=%1i j2=%1i wtp=%1.6g wtm=%1.6g\n",j1,j2,wtp,wtm);
	    }
	    iYfound++;
	  }
	  else if (iYfound == 1) {
	    /* Divide X interval by 2^iDivide */
	    x1 = OutX[j1];
	    x2 = OutX[j2];
	    y1 = OutY[j1];
	    y2 = OutY[j2];
	    z1 = OutZ[j1];
	    z2 = OutZ[j2];
	    r1 = OutR[j1];
	    r2 = OutR[j2];
	    th1 = OutTh[j1];
	    th2 = OutTh[j2];
	    phi1 = OutPhi[j1];
	    phi2 = OutPhi[j2];
	    OutTh1 = Outthp[j1];
	    OutTh2 = Outthp[j2];
	    if (j2 < j1) OutTh2 += (float)TWOPI;  
	    for (k=0; k<iDivide; k++) {
	      if (IfWrite) {
		printf("x1=%1.6g x2=%1.6g y1=%1.6g y2=%1.6g ", x1, x2, y1, y2);
		printf("th1=%3.6g th2=%3.6g\n",th1, th2);
	      } // end IfWrite
	      thp = (OutTh1+OutTh2)/2.;
	      if (Edge(R, thp, &th, &phi, &rout) == EXIT_FAILURE)
		return EXIT_FAILURE;
	      sth = sin(th);
	      cth = cos(th);
	      sph = sin(phi);
	      cph = cos(phi);
	      /* Projected (xp,yp) only with polar axis pointed North */
	      xp = rout*(sth*cph*ci-cth*si);
	      yp = rout*sth*sph;
	      z = rout*(sth*cph*si+cth*ci);
	      /* Rotate polar axis through Theta, get (OutX,OutY). */
	      X = -xp*cTh - yp*sTh;
	      if ((x1 < x && x < X) || (X < x && x < x1)) {
		x2 = X;
		y2 =  xp*sTh - yp*cTh;
		z2 = z;
		r2 = rout;
		th2 = th;
		phi2 = phi;
		OutTh2 = thp;
	      }
	      if ((x2 < x && x < X) || (X < x && x < x2)) {
		x1 = X;
		y1 =  xp*sTh - yp*cTh;
		z1 = z;
		r1 = rout;
		th1 = th;
		phi1 = phi;
		OutTh1 = thp;
	      }
	    }
	    /* linearly interpolate */
	    wtp=(x-x2)/(x1-x2);
	    wtm=1.-wtp;
	    iy2=j1;
	    Y2 = y1*wtp + y2*wtm;
	    Z2 = z1*wtp + z2*wtm;
	    R2 = r1*wtp + r2*wtm;
	    Th2 = th1*wtp + th2*wtm;
	    Phi2 = phi1*wtp + phi2*wtm;
	    if (IfWrite) {
	      printf("iYfound=%2i Y2=%1.6g", iYfound, Y2);
	      printf(" j1=%1i j2=%1i wtp=%1.6g wtm=%1.6g\n",j1,j2,wtp,wtm);
	    }
	    iYfound++;
	  }
	}
	j++;
      } // j<OutNum & iYfound < 2

      if (IfWrite) {
	printf("i=%2i x=%1.6g iy1=%2i Y1=%1.6g iy2=%2i Y2=%1.6g dx=%g\n", 
	       i, x, iy1, Y1, iy2, Y2, dx);
      } // end IfWrite

      /* Over Y along each strip */
      iNy = 1 + (int) fabs((Y2-Y1)/dx); // keep dy about the size of dx
      if ((iNy/2)*2 == iNy) iNy++; // make an odd number
      if (iNy > ISTRIPMAX + 1) { // limit size
	iNy = ISTRIPMAX;
	if ((ISTRIPMAX/2)*2 == ISTRIPMAX) iNy = ISTRIPMAX + 1; 
      } // keep it odd
      D->iNy[i] = iNy;

      /* Save the second outside points */
      Rout[i][iNy-1] = R2;
      Thout[i][iNy-1] = Th2;
      Phiout[i][iNy-1] = Phi2;
      Zout[i][iNy-1] = Z2;
      D->afY[i][iNy-1] = Y2;

      dy = 0.;
      if (iNy > 1) dy = (Y2-Y1)/(iNy-1);
      D->fdy[i] = dy;

      /* Do center first */
      iNy2 = iNy/2; // center index
      ynot = Y1+iNy2*dy;
      D->afY[i][iNy2] = ynot;
      if (IfWrite) {
	printf(" iNy=%2i iNy2=%2i dy=%1.6g ynot=%1.6g\n", iNy, iNy2, dy, ynot);
      } // end IfWrite

      /* Fundamental Plane */
      xp = -x*cTh + ynot*sTh;
      yp = -x*sTh - ynot*cTh;

      /* Then get body coordinates */
      /*
      */
      if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout, 0) 
	  == EXIT_FAILURE) {
      /*
	if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout) 
	== EXIT_FAILURE) {
      */
	printf("Surf1-center:i=%2i x=%1.6g j=%2i y=%1.6g ", i, x, j, ynot);
	printf("cosTh=%1.6g sinTh=%1.6g xp=%1.6g yp=%1.6g\n",cTh,sTh,xp,yp);
	return EXIT_FAILURE;
      }
      if (IfWrite) {
	if (i == 1) {
	  printf("Surf1-center: x1=%8.5f ynot=%8.5f x'in=%8.5f y'in=%8.5f\n",
		 x,ynot,xp,yp);
	  printf("              rout=%7.5f thout=%6.2f phiout=%7.2f\n",rout,
		 (float)DEGRAD*thout,(float)DEGRAD*phiout);
	  sth=sin(thout);
	  cth=cos(thout);
	  sph=sin(phiout);
	  cph=cos(phiout);
	  xp=rout*(sth*cph*ci-cth*si);
	  yp=rout*sth*sph;
	  z=rout*(sth*cph*si+cth*ci);
	  printf("             x'out=%8.5f y'out=%8.5f z'out=%8.5f\n",xp,yp,z);
	}
      }
      Rout[i][iNy2] = rout;
      Thout[i][iNy2] = thout;
      Phiout[i][iNy2] = phiout;
      if (!(rout >= 0)) {
	printf("Surf1-center:i=%2i x=%1.6g j=%2i y=%1.6g ", i, x, j, ynot);
	printf("cosTh=%1.6g sinTh=%1.6g xp=%1.6g yp=%1.6g\n",cTh,sTh,xp,yp);
	printf("R=%7.4f Theta=%7.2f Phi=%7.2f\n",rout,(float)DEGRAD*thout,
	       (float)DEGRAD*phiout);
	return EXIT_FAILURE;
      }
	

      /* Now, back toward the beginning of the strip */
      for (j = 1; j < iNy2; j++) {
	k=iNy2-j;
	y = ynot - dy*j;
	D->afY[i][k] = y;
	/* set initial guesses */
	thout = Thout[i][k+1];
	phiout = Phiout[i][k+1];

	/* Fundamental Plane */
	xp = -x*cTh + y*sTh;
	yp = -x*sTh - y*cTh;
      
	/* Then get body coordinates */
	/*
	  if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout, 0) 
	  == EXIT_FAILURE) {
	*/
	IfConverge = XYxNR(w, incl, xp, yp, &thout, &phiout, 0); 
	if (IfConverge <= 0) {
	  if (IfWrite) {
	    printf("Surf1-back: IfConverge=%2i\n",IfConverge);
	    printf("Surf1-back: i=%2i x=%1.6g j=%2i y=%1.6g\n", i, x, j, y);
	    printf("            cosTh=%1.6g sinTh=%1.6g xp=%1.6g yp=%1.6g\n",
		   cTh,sTh,xp,yp);
	  }
	  if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout, 0) 
	  == EXIT_FAILURE) return EXIT_FAILURE;
	  if (rout > 0) {
	    if (IfWrite) {
	      printf("RThPhfromXY got: xp=%7.5f yp=%7.5f th=%5.2f phi=%6.2f\n",
		     xp,yp,(float)DEGRAD*thout,(float)DEGRAD*phiout);
	    }
	  }
	  else {
	    printf("RThPhfromXY failed: x=%7.5f y=%7.5f xp=%7.5f yp=%7.5f\n",
		   x,y,xp,yp);
	    return EXIT_FAILURE;
	  }
	}
	sth = sin(thout);
	Rout[i][k] = FractRadius(w2,sth*sth);
	Thout[i][k] = thout;
	Phiout[i][k] = phiout;
      }

      /* and forward to the end of the strip */
      for (j = 1; j < iNy2; j++) {
	k = iNy2+j;
	y = ynot + dy*j;
	D->afY[i][k] = y;

	/* set initial guesses */
	thout = Thout[i][k-1];
	phiout = Phiout[i][k-1];


	/* Fundamental Plane */
	xp = -x*cTh + y*sTh;
	yp = -x*sTh - y*cTh;
      
	/* Then get body coordinates */
	/*
	  if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout, 0) 
	  == EXIT_FAILURE) {
	  printf("Surf1-forth: i=%2i x=%1.6g j=%2i y=%1.6g ", i, x, j, y);
	  printf("cosTh=%1.6g sinTh=%1.6g xp=%1.6g yp=%1.6g\n",cTh,sTh,xp,yp);
	  return EXIT_FAILURE;
	  }
	*/
	IfConverge = XYxNR(w, incl, xp, yp, &thout, &phiout, 0); 
	if (IfConverge <= 0) {
	  if (IfWrite) {
	    printf("Surf1-forwd: IfConverge=%2i\n",IfConverge);
	    printf("Surf1-forwd: i=%2i x=%1.6g j=%2i y=%1.6g\n", i, x, j, y);
	    printf("            cosTh=%1.6g sinTh=%1.6g xp=%1.6g yp=%1.6g\n",
		   cTh,sTh,xp,yp);
	  }
	  if (RThetaPhifromXY(xp, yp, R, &rout, &thout, &phiout, 0) 
	  == EXIT_FAILURE) return EXIT_FAILURE;
	  if (rout > 0) {
	    if (IfWrite) {
	      printf("RThPhfromXY got: xp=%7.5f yp=%7.5f th=%5.2f phi=%6.2f\n",
		     xp,yp,(float)DEGRAD*thout,(float)DEGRAD*phiout);
	    }
	  }
	  else {
	    printf("RThPhfromXY failed: x=%7.5f y=%7.5f xp=%7.5f yp=%7.5f\n",
		   x,y,xp,yp);
	    return EXIT_FAILURE;
	  }
	}
	sth = sin(thout);
	Rout[i][k] = FractRadius(w2,sth*sth);
	Thout[i][k] = thout;
	Phiout[i][k] = phiout;
      }
    }
  }

  /* Now get surface values of Teff, log_g, and mu */

  for (i=0; i<abs(iB); i++) {
    IfWrite = 0;
    iNy = D->iNy[i];
    if (IfWrite) printf("i=%2i iNy=%2i\n",i,iNy);
    for (j=0; j<iNy; j++) {
      rout = Rout[i][j];
      thout = Thout[i][j];
      phiout = Phiout[i][j];
      sth = sin(thout);
      sth2 = sth*sth;
      cth = cos(thout);
      cth2 = cth*cth;
      cph = cos(phiout);
      Zout[i][j] = rout*(sth*cph*si+cth*ci);
      rout2 = rout*rout;
      if (IfWrite) {
	printf("j=%2i Rout=%1.6g Thout=%3.6g Phiout=%3.6g\n",j,Rout[i][j],
	       (float)DEGRAD*Thout[i][j],(float)DEGRAD*Phiout[i][j]);
	printf("sth=%1.6g sth2=%1.6g cth=%1.6g ",sth,sth2,cth); 
	printf("cth2=%1.6g cph=%1.6g rout2=%5.6g\n",cth2,cph,rout2);
      }
      //IfWrite=0;

      /* Now can get local effective gravity and effective temperature */
      fact = (ttc*w2*rout*sth2-1/rout2);
      gEff = sqrt(fact*fact + tts*w2*w2*rout2*sth2*cth2); // scaled by gpb
      tEff = R->fTeffpB*sqrt(sqrt(gEff));
      D->afTLgMu[i][j][0] = tEff;
      D->afTLgMu[i][j][1] = log10(gEff*gpb);
      dlrdth=(ttc*w2*rout2*rout*sth*cth)/(1.-ttc*w2*rout2*rout*sth2);
      mu=(dlrdth*(sth*ci-cth*cph*si)+cth*ci+sth*cph*si)/
	sqrt(dlrdth*dlrdth+1.);
      if (!(mu >= 0.)) mu=0.; // fix noise in the numerical procedures
      D->afTLgMu[i][j][2] = mu;
      if (IfWrite == 1) {
	printf("fact=%g gEff=%g tEff=%g dlrdth=%g mu=%g\n",
	       fact,gEff,tEff,dlrdth,mu);
      }
    }
  }

  /* Fix effective heights of end points */
  iNy = D->iNy[1];
  dy = fabs(D->afY[1][iNy-1] - D->afY[1][0])/2.;
  dx = D->fdx;
  r = (dy*dy+dx*dx)/2./dx;
  dxp = dx/2.;
  dyp = sqrt(r*dx-dxp*dxp);
  dyeff = (r*r*asin(dyp/r)-(r-dxp)*dyp)/dx;
  D->fdy[0] = dyeff;
  D->fdy[iB1] = dyeff;

  if (iB < 0)  IfWrite = 1;

  //IfWrite=1;
  /* Write out results */
  if (IfWrite) {
    printf("End points: iNy=%i Y[1][iNy-1]=%g Y[1][0]=%g dy=%g dx=%g\n",iNy,
	   D->afY[1][iNy-1],D->afY[1][0],dy,dx);
    printf("r=%g dxp=%g dyp=%g dyeff=%g\n",r,dxp,dyp,dyeff);
    //return EXIT_FAILURE;

    printf("Surf1:\n  i  iY     X    j    y      Teff   logg   mu  ");
    printf("     R      Th      Phi     z\n");
    for (i=0; i<abs(iB); i++) {
      //if (i < 2 || i > abs(iB)-4) {
      if (i > abs(iB)-12) {
	printf("%3i%3i %8.5f\n",i,D->iNy[i], D->afX[i]);
	for (j=0; j<D->iNy[i]; j++) {
	  printf("                %2i %8.5f ",j,D->afY[i][j]);
	  printf("%5.0f  %5.2f %6.4f",D->afTLgMu[i][j][0],
		 D->afTLgMu[i][j][1],D->afTLgMu[i][j][2]);
	  printf("  %5.4f  %6.3f  %7.3f",Rout[i][j],
		 (float)DEGRAD*Thout[i][j],(float)DEGRAD*Phiout[i][j]);
	  printf(" %8.5f\n",Zout[i][j]);
	}
      }
    }
  }
  //return EXIT_FAILURE;
  //if (IfStop) return EXIT_FAILURE;
  return EXIT_SUCCESS;
}
