/* 
 * Copyright 1998 UC Regents, All Rights Reserved
 * Written by Eric Korpela 
 *
 * Permission to modify and distribute this file is granted, provided
 * the copyright notices in this file remain intact.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The authors
 * and copyright holders may not be held liable for any damages resulting
 * from the use or misuse of this software.
 *
 * Bug Reports to korpela@ssl.berkeley.edu
 *
 * $Id: varray.c,v 1.5 2000/04/18 15:29:15 korpela Exp $
 */



#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/uio.h>
#include <errno.h>
#include <sys/stat.h>
#include <signal.h>
#include "export.h"

#ifndef MS_SYNC
#define MS_SYNC 0
#endif
#ifndef MAP_FILE
#define MAP_FILE 0
#endif
#ifndef MAP_VARIABLE
#define MAP_VARIABLE 0
#endif


#ifdef DEBUG
#define D(x) x
#else 
#define D(x)
#endif
#define MAX_MAPPED_FILES 32
#define MAX_PATH_LEN 256

int ele_size[IDL_NUM_TYPES]={
#ifdef __GNUC__
       [IDL_TYP_UNDEF]        0,
       [IDL_TYP_BYTE]         sizeof(UCHAR),
       [IDL_TYP_INT]          sizeof(short),
       [IDL_TYP_LONG]         sizeof(IDL_LONG),
       [IDL_TYP_FLOAT]        sizeof(float),
       [IDL_TYP_DOUBLE]       sizeof(double),
       [IDL_TYP_COMPLEX]      sizeof(IDL_COMPLEX),
       [IDL_TYP_STRING]       0,
       [IDL_TYP_STRUCT]       0,
       [IDL_TYP_DCOMPLEX]     sizeof(IDL_DCOMPLEX)
#else
      /*Warning, order of these could change with IDL version!*/
      /*IDL_TYP_UNDEF*/       0,
      /*IDL_TYP_BYTE*/        sizeof(UCHAR),
      /*IDL_TYP_INT*/         sizeof(short),
      /*IDL_TYP_LONG*/        sizeof(IDL_LONG),
      /*IDL_TYP_FLOAT*/       sizeof(float),
      /*IDL_TYP_DOUBLE*/      sizeof(double),
      /*IDL_TYP_COMPLEX*/     sizeof(IDL_COMPLEX),
      /*IDL_TYP_STRING*/      0,
      /*IDL_TYP_STRUCT*/      0,
      /*IDL_TYP_DCOMPLEX*/    sizeof(IDL_DCOMPLEX)
#endif
};

static int map_file(IDL_STRING *file_name, int ele_type, IDL_LONG dim[], int n_dim, int flags);
static void unmap_file(char *data);
static void unmap_all(void);

typedef struct str_mapped_file {
  char filename[MAX_PATH_LEN];
  int fd;
  int flags;
  char *data;
  IDL_ULONG length;
  IDL_VPTR vptr;
} mapped_file;

static mapped_file *files[MAX_MAPPED_FILES];
static int num_open_files;
static int is_first=1;

static void first_call() {
  if (is_first) {
    fprintf(stderr,"Memory mapped file support for IDL, v1.04\n");
    fprintf(stderr,"Copyright 1998, UC Regents, all rights reserved\n");
    fprintf(stderr,"Programmed by Eric Korpela\n");
    IDL_ExitRegister(unmap_all);
    is_first=0;
  }
}


static int map_file(IDL_STRING *filename, int ele_type, IDL_LONG dim[], int n_dim, int flags)
{
  struct stat buf;
  int map=MAP_FILE|MAP_VARIABLE,file_len,fd,i;

  if (num_open_files==MAX_MAPPED_FILES) {
    fprintf(stderr,"Too many memory mapped files\n");
    return(-1);
  }

  if (!(files[num_open_files]=
           (mapped_file *)IDL_MemAlloc(sizeof(mapped_file),0,IDL_MSG_RET)))
  {
    fprintf(stderr,"Unable to allocate memory!\n");
    return(-1);
  }

  if ((fd=open(filename->s,flags))==-1) {
    if (flags != O_RDONLY) {
      D(fprintf(stderr,"creating file . . ."));
      if ((fd=open(filename->s,flags|O_CREAT,0666))==-1) {
        fprintf(stderr,"File creation error!\n");
        IDL_MemFree(files[num_open_files],0,IDL_MSG_RET);
        return(-1);
      } 
    } else {
      fprintf(stderr,"File not found error!\n");
      IDL_MemFree(files[num_open_files],0,IDL_MSG_RET);
      return(-1);
    }
  }

  strncpy(files[num_open_files]->filename,filename->s,MAX_PATH_LEN);
  files[num_open_files]->fd=fd;
  files[num_open_files]->flags=flags;
 
  if (!(file_len=ele_size[ele_type])) {
    fprintf(stderr,"Error, unsupported type!\n");
    IDL_MemFree(files[num_open_files],0,IDL_MSG_RET);
    return(-1);
  }
  for (i=0;i<n_dim;i++)
    file_len*=dim[i];
  files[num_open_files]->length=file_len;
  if (fstat(fd,&buf) || (buf.st_size < file_len )) {
    if (flags == O_RDONLY) {
      fprintf(stderr,"Warning read only file is shorter than mapped array.\n");
    } else {
      ftruncate(fd,file_len);
    }
  }
  lseek(fd,0,SEEK_SET);

  D(fprintf(stderr,"mapping . . .\n"));
  if (flags==O_RDONLY) {
    map|=MAP_PRIVATE;
  } else {
    map|=MAP_SHARED; 
  }
  if ((files[num_open_files]->data=
         (char *)mmap(0,file_len,PROT_READ|PROT_WRITE,map,fd,0))
           ==(char *)-1) {
    fprintf(stderr,"mmap() failed with code %d\n",errno);
    IDL_MemFree(files[num_open_files],0,IDL_MSG_RET); 
    return(-2);
  }

  num_open_files++;
  if (!(files[num_open_files-1]->vptr=
                                  IDL_ImportArray(n_dim,dim,
                                    ele_type,(files[num_open_files-1]->data),
                                    (unmap_file), 0))) {
    fprintf(stderr,"IDL_ImportArray failed\n");
    unmap_file(files[num_open_files-1]->data);
  } 

  return(0);
}

static void unmap_file(char *data) {
  int i=0,j;

  first_call();
  while ((files[i]->data != data) && (i<num_open_files)) i++;
  
  if (i==num_open_files) {
    fprintf(stderr,"IDL is trying to free a nonexistant variable\n");
    return;
  }

  if (files[i]->flags != O_RDONLY) {
    if (!fork()) {
      fprintf(stderr,"syncing\n");
      fflush(stderr);
      if (files[i]->data && msync(files[i]->data,files[i]->length,MS_SYNC)) {
	fprintf(stderr,"msync(0x%x,0x%x,0x%x) failed with %d\n",
		   files[i]->data,files[i]->length,MS_SYNC,errno);
	kill(getpid(),SIGKILL);
      }
      kill(getpid(),SIGKILL);
    }
  }

  if (files[i]->data && munmap(files[i]->data,files[i]->length)) {
    fprintf(stderr,"munmap() failed\n");
    return;
  }

  fsync(files[i]->fd);
  close(files[i]->fd);
 
  IDL_MemFree(files[i],0,IDL_MSG_RET); 

  for (j=i;j<(num_open_files-1);j++) {
    files[j]=files[j+1];
  }
  num_open_files--;
}

static void unmap_all(void) {
  while (num_open_files) unmap_file(files[0]->data);
}

static IDL_VARIABLE reterr={IDL_TYP_UNDEF};

IDL_VPTR varray(int argc, IDL_VPTR argv[], char argk[])
{
  IDL_VPTR filename=NULL,rawargs[11];
  int type;
  IDL_LONG dim[IDL_MAX_ARRAY_DIM];
  IDL_STRING *s;
  int first=0,i,n_dim,flags;
  char buff[25];
  static int specified[2]; 
  static IDL_LONG status;
  static IDL_LONG writable;
  static IDL_KW_PAR kw_pars[] = 
    {{ "WRITABLE", IDL_TYP_LONG, 1, 0, specified, IDL_CHARA(writable) },
     { "STATUS", IDL_TYP_LONG, 1, 0, specified+1, IDL_CHARA(status) },
     { NULL }};

  first_call();
  D(fprintf(stderr,"VARRAY Called with %d parameters",argc));
  writable=0;
  status=0;
  IDL_KWGetParams(argc,argv,argk,kw_pars,rawargs,1);
  if (specified[0]+specified[1]) {
    D(fprintf(stderr," and one keyword"));
    argc-=(specified[0]+specified[1]);
  }
  D(fprintf(stderr,".\n"));
  if (writable) {
    flags=O_RDWR;
  } else {
    flags=O_RDONLY;
  }

  if (argv[0]->type != IDL_TYP_STRING) {
     s=(IDL_STRING *)IDL_GetScratch(&filename,1,sizeof(IDL_STRING));
     srandom(time(0));
     sprintf(buff,"/tmp/%8.8x",random());
     IDL_StrStore(s,buff);
     first=0;
     writable=1;
  } else {
    s=&(argv[0]->value.s);
    first=1;
  }
  type=argv[first++]->type;
  for (i=first;i<argc;i++) {
    if (argv[i]->type != IDL_TYP_LONG) {
      dim[i-first]=IDL_CvtLng(1,&(argv[i]))->value.l;
    } else {
      dim[i-first]=argv[i]->value.l;
    }
    if (!dim[i-first]) {
      fprintf(stderr,"ERROR! Dimensions must be non-zero.");
      return(&reterr);
    }
    D(fprintf(stderr,"dimension %d is %ld\n",i-first,dim[i-first]));
  }
  n_dim=argc-first;
  
  if (map_file(s,type,dim,n_dim,flags))
    return(&reterr);

  if (filename) IDL_Deltmp(filename);

  if (status) {
    printf("%d files open and mapped\n",num_open_files);
  }

  return(files[num_open_files-1]->vptr);
}

/*
 * $Log: varray.c,v $
 * Revision 1.5  2000/04/18  15:29:15  korpela
 * *** empty log message ***
 *
 * Revision 1.4  2000/04/18  15:26:35  korpela
 * Fixed fprintf bug in line 202
 *
 * Revision 1.3  1998/09/02  17:59:43  korpela
 * Initial RCS revision
 *
 */
